use core:lang;
use lang:bs;
use lang:bs:macro;
use core:sync;
use core:asm;

/**
 * Definition of an effect type.
 */
class EffectType extends Class {
	init(SrcPos pos, Scope scope, Str name, SEffectTypeBody body) {
		init(TypeFlags:typeClass, pos, scope, name, body);
	}

	protected Bool loadAll() : override {
		if (!super:loadAll())
			return false;

		// For each effect function, also add a corresponding handler that the handler can override.
		Named[] toAdd;
		for (x in this) {
			if (x as EffectFun) {
				toAdd << x.handlerFn;
				if (x.unsafe)
					toAdd << x.makeUnsafeWrapper();
			}
		}

		for (x in toAdd)
			add(x);

		true;
	}
}

// Create effect types. Note: we don't need a Decl for this, we can just create it directly since we
// want it to be created before someone might need it!
Named createEffectType(Scope scope, SStr name, SEffectTypeBody body) {
	SrcPos pos = name.pos;
	EffectType t(pos, fileScope(scope, pos), name.v, body);
	return Nonmovable(null, t);
}


/**
 * An actual effect function.
 */
class EffectFun extends Function {
	init(SrcPos pos, Class memberOf, Bool unsafe, Str name, Value result, ValParam[] params) {
		var effect = if (memberOf as EffectType) {
			memberOf;
		} else {
			throw InternalError("EffectFun must be a member of an effect!");
		};

		var vals = params.values();

		// Make us a member function.
		vals.insert(0, thisPtr(memberOf));

		var resultType = if(found = named{}.find("EffectResult", result, Scope()) as Type) {
			found;
		} else {
			throw InternalError("Could not find EffectFrame!");
		};

		// Note: we intentionally ignore the possible this parameter.
		var resumeType = ResumeHandlerType(effect, name, params.values());

		init(pos, result, name, vals) {
			unsafe = unsafe;
			resumeType = resumeType;
			resultType = resultType;
			handlerFn = BSAbstractFn(Value(), SStr("@handle-" + name), [
										thisParam(memberOf),
										ValParam(named{HandlerFrame}, "@result"),
										ValParam(named{Continuation}, "@continuation"),
										ValParam(resumeType, "@params")
									]);
		}

		setCode(LazyCode(&this.generate()));
	}

	// Custom type used to store the parameters to this effect.
	package ResumeHandlerType resumeType;

	// Type used for results. Points to a subclass of EffectResult.
	private Type resultType;

	// Store the function that handles this effect in the parent class.
	package Function handlerFn;

	// Is this effect an unsafe effect? This just means that we should generate a wrapper that don't
	// accept a 'this' parameter.
	package Bool unsafe;

	private CodeGen generate() {
		CodeGen gen(this);

		// The 'this' parameter. Specifies the handler to invoke.
		var me = gen.l.createParam(ptrDesc);

		core:asm:Var[] actuals;
		for (i, x in params) {
			// Skip param #0, its the this param.
			if (i > 0)
				actuals << gen.l.createParam(x.desc);
		}

		gen.l << prolog();

		// Create our effect result object.
		var resultObj = gen.allocObject(resultType);

		// Create a handler resume object and store the parameters we received:
		var resumeObj = gen.l.createVar(gen.block, sPtr);
		gen.l << fnParam(ptrDesc, resumeType.typeRef);
		gen.l << fnCall(BuiltIn:alloc.ref, false, ptrDesc, resumeObj);

		// Call the base class' constructor:
		var ctorParams = Value:[named{ResumeHandler}, named{EffectResult}];
		if (ctor = named{ResumeHandler}.find("__init", ctorParams, Scope()) as Function) {
			gen.l << fnParam(ptrDesc, resumeObj);
			gen.l << fnParam(ptrDesc, resultObj);
			gen.l << fnCall(ctor.ref, true);
		} else {
			throw InternalError("Failed to find resume type ctor!");
		}

		// Store the handler.
		gen.l << mov(ptrA, resumeObj);
		gen.l << mov(ptrRel(ptrA, resumeType.handler.offset), me);

		// Now, copy the parameters:
		for (i, actual in actuals) {
			var v = resumeType.vars[i];
			if (v.type.isAsmType) {
				gen.l << mov(ptrA, resumeObj);
				gen.l << mov(xRel(v.type.size, ptrA, v.offset), actual);
			} else {
				gen.l << mov(ptrA, resumeObj);
				gen.l << add(ptrA, ptrConst(v.offset));
				gen.l << lea(ptrC, actual);
				gen.l << fnParam(ptrDesc, ptrA);
				gen.l << fnParam(ptrDesc, ptrC);
				gen.l << fnCall(v.type.copyCtor, true);
			}
		}

		// Finally, set the vtable properly!
		if (vtable = resumeType.vtable)
			vtable.insert(gen.l, resumeObj);

		// Call 'post' on the resume handler to make it ready to capture the continuation.
		gen.l << fnParam(ptrDesc, resumeObj);
		gen.l << fnParam(ptrDesc, me);
		gen.l << fnCall(named{ResumeHandler:post<ResumeHandler, Object>}.ref, true);

		// Call 'wait' to make things happen!
		gen.l << fnParam(ptrDesc, resultObj);
		gen.l << fnCall(named{EffectResult:wait<EffectResult>}.ref, true);

		// If we have a result, find the result and return it. Note, we abuse the fact that a Maybe
		// type has the same representation as an instance of an ordinary type when it is non-null
		// (except for the possibility of a few additional bytes).
		if (result.any) {
			unless (resultVar = resultType.find("result", resultType, Scope()) as MemberVar)
				throw InternalError("Failed to find 'result' in EffectFrame!");

			gen.l << add(resultObj, ptrConst(resultVar.offset));
			gen.l << fnRetRef(resultObj);
		} else {
			gen.l << fnRet();
		}

		gen;
	}

	// Make a wrapper for an unsafe effect that calls this effect.
	Function makeUnsafeWrapper() {
		unless (effect = this.params[0].type as EffectType)
			throw InternalError("Inconsistent types!");

		Array<Value> params(this.params);
		params.remove(0);
		Function f(pos, result, name, params);
		f.make(FnFlags:static);

		var resultDesc = result.desc;
		Listing l(false, resultDesc);

		l << prolog();
		l << fnParam(ptrDesc, effect.typeRef);
		l << fnCall(named{findEffectHandler<EffectType>}.ref, false, ptrDesc, ptrA);

		l << fnParam(ptrDesc, ptrA);

		// Forward parameters:
		for (x in params) {
			var desc = x.desc;
			var v = l.createParam(desc);
			l << fnParam(desc, v);
		}

		var resultVar = l.createVar(l.root, resultDesc);
		l << fnCall(this.ref, true, resultDesc, resultVar);

		l << fnRet(resultVar);

		f.setCode(DynamicCode(l));

		f;
	}
}

// Helper:
package EffectFun effectFun(SrcPos pos, Class memberOf, Bool unsafe, SStr name, SrcName result, NameParam[] params) on Compiler {
	var res = memberOf.scope.value(result);
	var par = params.resolve(memberOf.scope);
	return EffectFun(pos, memberOf, unsafe, name.v, res, par);
}
