/*
 *  $Id: mask_shift.c 22934 2020-08-28 14:26:19Z yeti-dn $
 *  Copyright (C) 2020 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <stdlib.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/arithmetic.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils.h>
#include "preview.h"

#define MASKSHIFT_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

enum {
    MASKSHIFT_EXTERIOR_EMPTY  = 1024,
    MASKSHIFT_EXTERIOR_FILLED = 1025,
};

typedef struct {
    gint exterior;
    gint hmove;
    gint vmove;
} MaskShiftArgs;

typedef struct {
    MaskShiftArgs *args;
    GtkWidget *dialog;
    GwyContainer *mydata;
    GtkWidget *view;
    GtkObject *hmove;
    GtkObject *vmove;
    GtkWidget *exterior;
    GtkWidget *color_button;
} MaskShiftControls;

static gboolean module_register        (void);
static void     mask_shift             (GwyContainer *data,
                                        GwyRunType run);
static gboolean maskshift_dialog       (MaskShiftArgs *args,
                                        GwyContainer *data,
                                        GwyDataField *dfield,
                                        GwyDataField *mask,
                                        gint id);
static void     exterior_changed       (GtkComboBox *combo,
                                        MaskShiftControls *controls);
static void     hmove_changed          (GtkAdjustment *adj,
                                        MaskShiftControls *controls);
static void     vmove_changed          (GtkAdjustment *adj,
                                        MaskShiftControls *controls);
static void     preview                (MaskShiftControls *controls);
static void     maskshift_do           (GwyDataField *mfield,
                                        MaskShiftArgs *args);
static void     maskshift_sanitize_args(MaskShiftArgs *args);
static void     maskshift_load_args    (GwyContainer *settings,
                                        MaskShiftArgs *args);
static void     maskshift_save_args    (GwyContainer *settings,
                                        MaskShiftArgs *args);

static const MaskShiftArgs maskshift_defaults = {
    GWY_EXTERIOR_BORDER_EXTEND,
    0, 0,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Shift mask with respect to the image."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2020",
};

GWY_MODULE_QUERY2(module_info, mask_shift)

static gboolean
module_register(void)
{
    gwy_process_func_register("mask_shift",
                              (GwyProcessFunc)&mask_shift,
                              N_("/_Mask/_Shift..."),
                              GWY_STOCK_MASK_SHIFT,
                              MASKSHIFT_RUN_MODES,
                              GWY_MENU_FLAG_DATA_MASK | GWY_MENU_FLAG_DATA,
                              N_("Shift mask"));

    return TRUE;
}

static void
mask_shift(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield, *mfield;
    MaskShiftArgs args;
    GQuark quark;
    gint id, xres, yres;

    g_return_if_fail(run & MASKSHIFT_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_MASK_FIELD, &mfield,
                                     GWY_APP_MASK_FIELD_KEY, &quark,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);
    g_return_if_fail(mfield);

    maskshift_load_args(gwy_app_settings_get(), &args);

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    args.hmove = CLAMP(args.hmove, -(xres+1)/2, (xres+1)/2);
    args.vmove = CLAMP(args.vmove, -(yres+1)/2, (yres+1)/2);

    if (run == GWY_RUN_IMMEDIATE
        || maskshift_dialog(&args, data, dfield, mfield, id)) {
        gwy_app_undo_qcheckpointv(data, 1, &quark);
        maskshift_do(mfield, &args);
        gwy_data_field_data_changed(mfield);
        gwy_app_channel_log_add_proc(data, id, id);
    }
    maskshift_save_args(gwy_app_settings_get(), &args);
}

static gboolean
maskshift_dialog(MaskShiftArgs *args,
                 GwyContainer *data,
                 GwyDataField *dfield, GwyDataField *mask,
                 gint id)
{
    MaskShiftControls controls;
    GtkWidget *dialog;
    GtkWidget *hbox, *align, *table;
    gint response, xres, yres, row = 0;

    controls.args = args;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);

    dialog = gtk_dialog_new_with_buttons(_("Shift Mask"), NULL, 0,
                                         NULL);
    gtk_dialog_add_button(GTK_DIALOG(dialog), _("_Reset"), RESPONSE_RESET);
    gtk_dialog_add_button(GTK_DIALOG(dialog),
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
    gtk_dialog_add_button(GTK_DIALOG(dialog),
                          GTK_STOCK_OK, GTK_RESPONSE_OK);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);
    controls.dialog = dialog;

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox,
                       FALSE, FALSE, 4);

    controls.mydata = gwy_container_new();
    gwy_container_set_object_by_name(controls.mydata, "/0/data", dfield);
    gwy_container_set_object_by_name(controls.mydata, "/1/mask", mask);
    gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                            GWY_DATA_ITEM_PALETTE,
                            GWY_DATA_ITEM_MASK_COLOR,
                            GWY_DATA_ITEM_RANGE,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            0);
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SIZE, TRUE);

    align = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
    gtk_container_add(GTK_CONTAINER(align), controls.view);
    gtk_box_pack_start(GTK_BOX(hbox), align, FALSE, FALSE, 4);

    table = gtk_table_new(4, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 4);

    controls.exterior
        = gwy_enum_combo_box_newl(G_CALLBACK(exterior_changed), &controls,
                                  args->exterior,
                                  gwy_sgettext("exterior|Empty"),
                                  MASKSHIFT_EXTERIOR_EMPTY,
                                  gwy_sgettext("exterior|Filled"),
                                  MASKSHIFT_EXTERIOR_FILLED,
                                  gwy_sgettext("exterior|Border"),
                                  GWY_EXTERIOR_BORDER_EXTEND,
                                  gwy_sgettext("exterior|Mirror"),
                                  GWY_EXTERIOR_MIRROR_EXTEND,
                                  gwy_sgettext("exterior|Periodic"),
                                  GWY_EXTERIOR_PERIODIC,
                                  NULL);
    gwy_table_attach_adjbar(table, row, _("_Exterior type:"), NULL,
                            GTK_OBJECT(controls.exterior),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    gtk_table_set_row_spacing(GTK_TABLE(table), row, 8);
    row++;

    controls.hmove = gtk_adjustment_new(args->hmove, -(xres+1)/2, (xres+1)/2,
                                        1, 10, 0);
    gwy_table_attach_adjbar(table, row, _("_Horizontal shift:"), _("px"),
                            controls.hmove, GWY_HSCALE_LINEAR);
    g_signal_connect(controls.hmove, "value-changed",
                     G_CALLBACK(hmove_changed), &controls);
    row++;

    controls.vmove = gtk_adjustment_new(args->vmove, -(yres+1)/2, (yres+1)/2,
                                        1, 10, 0);
    gwy_table_attach_adjbar(table, row, _("_Vertical shift:"), _("px"),
                            controls.vmove, GWY_HSCALE_LINEAR);
    g_signal_connect(controls.vmove, "value-changed",
                     G_CALLBACK(vmove_changed), &controls);
    row++;

    controls.color_button = create_mask_color_button(controls.mydata, dialog,
                                                     0);
    gwy_table_attach_adjbar(table, row++, _("_Mask color:"), NULL,
                            GTK_OBJECT(controls.color_button),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    gtk_widget_show_all(dialog);
    preview(&controls);

    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            return FALSE;
            break;

            case RESPONSE_RESET:
            gtk_adjustment_set_value(GTK_ADJUSTMENT(controls.hmove), 0);
            gtk_adjustment_set_value(GTK_ADJUSTMENT(controls.vmove), 0);
            break;

            case GTK_RESPONSE_OK:
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gwy_app_sync_data_items(controls.mydata, data, 0, id, FALSE,
                            GWY_DATA_ITEM_MASK_COLOR,
                            0);
    gtk_widget_destroy(dialog);

    return TRUE;
}

static void
exterior_changed(GtkComboBox *combo,
                 MaskShiftControls *controls)
{
    MaskShiftArgs *args = controls->args;

    args->exterior = gwy_enum_combo_box_get_active(combo);
    preview(controls);
}

static void
hmove_changed(GtkAdjustment *adj,
              MaskShiftControls *controls)
{
    MaskShiftArgs *args = controls->args;

    args->hmove = gwy_adjustment_get_int(adj);
    preview(controls);
}

static void
vmove_changed(GtkAdjustment *adj,
              MaskShiftControls *controls)
{
    MaskShiftArgs *args = controls->args;

    args->vmove = gwy_adjustment_get_int(adj);
    preview(controls);
}

static void
preview(MaskShiftControls *controls)
{
    GwyContainer *mydata = controls->mydata;
    GwyDataField *src, *dest = NULL;

    src = gwy_container_get_object_by_name(mydata, "/1/mask");
    if (gwy_container_gis_object_by_name(mydata, "/0/mask", &dest))
        gwy_data_field_copy(src, dest, FALSE);
    else {
        dest = gwy_data_field_duplicate(src);
        gwy_container_set_object_by_name(mydata, "/0/mask", dest);
        g_object_unref(dest);
    }
    maskshift_do(dest, controls->args);
    gwy_data_field_data_changed(dest);
}

static void
maskshift_do(GwyDataField *mask, MaskShiftArgs *args)
{
    guint exterior = args->exterior;
    GwyDataField *extended;
    gint xres, yres, hmove = args->hmove, vmove = args->vmove;
    gdouble fillvalue = 0.0;

    if (exterior == MASKSHIFT_EXTERIOR_EMPTY) {
        exterior = GWY_EXTERIOR_FIXED_VALUE;
        fillvalue = 0.0;
    }
    else if (exterior == MASKSHIFT_EXTERIOR_FILLED) {
        exterior = GWY_EXTERIOR_FIXED_VALUE;
        fillvalue = 1.0;
    }

    xres = gwy_data_field_get_xres(mask);
    yres = gwy_data_field_get_yres(mask);
    extended = gwy_data_field_extend(mask,
                                     MAX(hmove, 0), MAX(-hmove, 0),
                                     MAX(vmove, 0), MAX(-vmove, 0),
                                     exterior, fillvalue, FALSE);
    gwy_data_field_area_copy(extended, mask,
                             MAX(-hmove, 0), MAX(-vmove, 0),
                             xres, yres, 0, 0);
    g_object_unref(extended);
    gwy_data_field_data_changed(mask);
}

static const gchar exterior_key[] = "/module/mask_shift/exterior";
static const gchar hmove_key[]    = "/module/mask_shift/hmove";
static const gchar vmove_key[]    = "/module/mask_shift/vmove";

static void
maskshift_sanitize_args(MaskShiftArgs *args)
{
    if (args->exterior != MASKSHIFT_EXTERIOR_EMPTY
        && args->exterior != MASKSHIFT_EXTERIOR_FILLED
        && args->exterior != GWY_EXTERIOR_BORDER_EXTEND
        && args->exterior != GWY_EXTERIOR_MIRROR_EXTEND
        && args->exterior != GWY_EXTERIOR_PERIODIC)
        args->exterior = GWY_EXTERIOR_BORDER_EXTEND;
    /* Moves are sanitised with respect to data size. */
}

static void
maskshift_load_args(GwyContainer *settings,
                    MaskShiftArgs *args)
{
    *args = maskshift_defaults;
    gwy_container_gis_enum_by_name(settings, exterior_key, &args->exterior);
    gwy_container_gis_int32_by_name(settings, hmove_key, &args->hmove);
    gwy_container_gis_int32_by_name(settings, vmove_key, &args->vmove);
    maskshift_sanitize_args(args);
}

static void
maskshift_save_args(GwyContainer *settings,
                    MaskShiftArgs *args)
{
    gwy_container_set_enum_by_name(settings, exterior_key, args->exterior);
    gwy_container_set_int32_by_name(settings, hmove_key, args->hmove);
    gwy_container_set_int32_by_name(settings, vmove_key, args->vmove);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
