function CreateBitsSharpEDID(outputEDID, inputEDID)
% CreateBitsSharpEDID - Create proper EDID file for connected display for CRS Bits#
%
% Usage: CreateBitsSharpEDID(outputEDID [, inputEDID='MON_EDID.edid']);
%
% 'inputEDID' filename of raw EDID input file.
% 'outputEDID' filename of to be written processed EDID output file.
%
% The function reads a raw EDID file, marks the EDID as DVI-D display
% device, so the computer sends DVI-D data to the Bits#, regardless if the
% Bits# is connected to a analog VGA CRT monitor or some DVI-D display.
% Then it adds a proper checksum and writes the new EDID file, ready for
% use with the Bits#.
%
% The Bits# provides the DDC detected EDID of a connected display in the
% Firmware folder under MON_EDID.edid. It expects a valid EDID file for use
% in the EDID folder.
%

% History:
% 15.03.2013  mk  Written.
%

if nargin < 1 || isempty(outputEDID) || ~ischar(outputEDID)
    error('You must specify outputEDID parameters as filename of output EDID file!');
end

% Use the file generated by Bits# based on DDC detected EDID data as
% default input file:
if nargin < 2 || isempty(inputEDID) || ~ischar(inputEDID)
    inputEDID = 'MON_EDID.edid';
    fprintf('Using default Bits# DDC detected EDID file MON_EDID.edid as input file.\n');
end

% Read raw EDID from input file:
fin = fopen(inputEDID, 'r');
edid = fread(fin);
fclose(fin);

% Mark it as a DVI-D device, even if it is an analog VGA CRT, so computer
% thinks a DVI-D sink is connected and uses DVI-D encoders instead of DVI-A
% or VGA output:
edid(21) = 129;

% Calculate proper checksum to make EDID valid:
edid(end) = 256 - mod(sum(edid(1:length(edid)-1)), 256);

% Write new EDID file:
fout = fopen(outputEDID, 'w');
fwrite(fout, edid);
fclose(fout);

fprintf('New EDID file derived from %s written to file %s. Bye.\n\n', inputEDID, outputEDID);

return;

end
