///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/space.h"
#include "rheolef/space_numbering.h"
#include "rheolef/space_mult.h"
#include "rheolef/piola_util.h"
#include "rheolef/piola_on_pointset.h"

namespace rheolef {

// ======================================================================
// space_base_rep
// ======================================================================
template <class T, class M>
space_base_rep<T,M>::space_base_rep (
    const geo_basic<T,M>&    omega_in,
    std::string              approx,
    std::string              valued)
  : _constit(omega_in,
	     (approx == "" || valued == "scalar") ? approx : valued+"("+approx+")"),
    _xdof(),
    _have_freezed(false),
    _idof2blk_dis_iub(),
    _has_nt_basis(),
    _normal(),
    _iu_ownership(),
    _ib_ownership()
{
  // omega_in is compressed by space_scalar_constitution() allocator
  // when it is a domain: then it becomes a geo_domain, with compressed numbering
  // so, omega_in can be different from omega:
  if (approx == "") return; // empty element => default cstor
  _idof2blk_dis_iub.resize (_constit.ownership());
  _has_nt_basis.resize     (_constit.ownership()); // TODO: _constit.scalar_ownership()
  _normal.resize           (_constit.ownership());
  init_xdof();
}
template <class T, class M>
space_base_rep<T,M>::space_base_rep (
    const geo_basic<T,M>&    omega_in,
    const basis_basic<T>&    b)
  : _constit(omega_in, b.name()),
    _xdof(),
    _have_freezed(false),
    _idof2blk_dis_iub(),
    _has_nt_basis(),
    _normal(),
    _iu_ownership(),
    _ib_ownership()
{
  // omega_in is compressed by space_scalar_constitution() allocator
  // when it is a domain: then it becomes a geo_domain, with compressed numbering
  // so, omega_in can be different from omega:
  if (! b.is_initialized()) return; // empty element => default cstor
  _idof2blk_dis_iub.resize (_constit.ownership());
  _has_nt_basis.resize     (_constit.ownership());
  _normal.resize           (_constit.ownership());
  init_xdof();
}
template <class T, class M>
space_base_rep<T,M>::space_base_rep (const space_constitution<T,M>& constit)
  : _constit(constit),
    _xdof(),
    _have_freezed(false),
    _idof2blk_dis_iub(),
    _has_nt_basis(),
    _normal(),
    _iu_ownership(),
    _ib_ownership()
{
  distributor idof_ownership = _constit.ownership();
  _idof2blk_dis_iub.resize (idof_ownership);
  _has_nt_basis.resize     (idof_ownership);
  _normal.resize           (idof_ownership);
  init_xdof();
}
template <class T, class M>
space_base_rep<T,M>::space_base_rep (const space_mult_list<T,M>& expr)
  : _constit(expr),
    _xdof(),
    _have_freezed(false),
    _idof2blk_dis_iub(),
    _has_nt_basis(),
    _normal(),
    _iu_ownership(),
    _ib_ownership()
{
  _idof2blk_dis_iub.resize (_constit.ownership());
  _has_nt_basis.resize     (_constit.ownership());
  _normal.resize           (_constit.ownership());
  init_xdof();
}
template <class T, class M>
void
space_base_rep<T,M>::init_xdof()
{
  if (_constit.is_hierarchical()) {
    trace_macro ("init_xdof: hierarchical space: xdof undefined");
    return;
  }
  const geo_basic<T,M>& omega = get_geo();
  const basis_basic<T>& b = get_basis();
  size_type dis_nnod = space_numbering::dis_nnod (b, omega.sizes(), omega.map_dimension());
  size_type     nnod = space_numbering::    nnod (b, omega.sizes(), omega.map_dimension());
  communicator   comm = ownership().comm();
  distributor nod_ownership (dis_nnod, comm, nnod);
  _xdof.resize (nod_ownership);
  piola_on_pointset<T> pops;
  pops.initialize (omega.get_piola_basis(), b, integrate_option());
  std::vector<size_type> dis_inod_tab;
  for (size_type ie = 0, ne = omega.size(); ie < ne; ie++) {
    const geo_element& K = omega [ie];
    const Eigen::Matrix<piola<T>,Eigen::Dynamic,1>& piola = pops.get_piola (omega, K);
    space_numbering::dis_inod (b, omega.sizes(), K, dis_inod_tab);
    for (size_type loc_inod = 0, loc_nnod = piola.size(); loc_inod < loc_nnod; ++loc_inod) {
      _xdof.dis_entry (dis_inod_tab[loc_inod]) = piola [loc_inod].F;
    } 
  }
  _xdof.dis_entry_assembly();
  // add external nodes on current element:
  if (is_distributed<M>::value && comm.size() > 1) {
    index_set dis_inod_set;
    for (size_type ie = 0, ne = omega.size(); ie < ne; ++ie) {
      const geo_element& K = omega [ie];
      space_numbering::dis_inod (b, omega.sizes(), K, dis_inod_tab);
      for (size_type loc_inod = 0, loc_nnod = dis_inod_tab.size(); loc_inod < loc_nnod; ++loc_inod) {
        size_type dis_inod = dis_inod_tab[loc_inod];
        if (nod_ownership.is_owned(dis_inod)) continue;
        check_macro (dis_inod < dis_nnod, "invalid dis_inod="<<dis_inod<<" out of range [0:"<<dis_nnod<<"[");
        dis_inod_set.insert (dis_inod);
      }
    }
    _xdof.set_dis_indexes (dis_inod_set);
  }
}
template <class T, class M>
void
space_base_rep<T,M>::base_freeze_body () const
{
  _constit.neighbour_guard(); // geo could uses S.master(i), a neighbour connnectivity
  // -----------------------------------------------------------------------
  // 1) loop on domains: mark blocked dofs
  // -----------------------------------------------------------------------
  disarray<size_type,M> blocked_flag = _constit.build_blocked_flag();

  // copy the blocked_flag into the dis_iub disarray, as the "blocked" bit:
  for (size_type idof = 0, ndof = blocked_flag.size(); idof < ndof; idof++) {
    _idof2blk_dis_iub [idof].set_blocked (blocked_flag[idof]);
  }
  // -----------------------------------------------------------------------
  // 2) init numbering
  // -----------------------------------------------------------------------
  size_type n_unknown = 0;
  size_type n_blocked = 0;
  for (size_type idof = 0, ndof = _idof2blk_dis_iub.size(); idof < ndof; idof++) {
    bool blk = _idof2blk_dis_iub [idof].is_blocked();
    if (! blk) {
      _idof2blk_dis_iub[idof].set_dis_iub (n_unknown);
      n_unknown++;
    } else {
      _idof2blk_dis_iub[idof].set_dis_iub (n_blocked);
      n_blocked++;
    }
  }
  size_type dis_n_unknown = n_unknown;
  size_type dis_n_blocked = n_blocked;
#ifdef _RHEOLEF_HAVE_MPI
  if (is_distributed<M>::value) {
    dis_n_unknown = mpi::all_reduce (comm(), dis_n_unknown, std::plus<T>());
    dis_n_blocked = mpi::all_reduce (comm(), dis_n_blocked, std::plus<T>());
  }
#endif // // _RHEOLEF_HAVE_MPI
  _iu_ownership = distributor (dis_n_unknown, comm(), n_unknown);
  _ib_ownership = distributor (dis_n_blocked, comm(), n_blocked);
}
// ----------------------------------------------------------------------------
// still used by geo_subdivide.cc and internally in space.cc
// ----------------------------------------------------------------------------
template <class T, class M>
void
space_base_rep<T,M>::dis_idof (const geo_element& K, std::vector<size_type>& dis_idof) const
{
  freeze_guard();
  _constit.assembly_dis_idof (get_geo(), K, dis_idof);
}
// ----------------------------------------------------------------------------
// name : e.g. "P1(square)", for field_expr<Expr> checks
// ----------------------------------------------------------------------------
template <class T, class M>
std::string
space_base_rep<T,M>::name() const
{
  return _constit.name();
}
// ----------------------------------------------------------------------------
// u["left"] 
// => build here the requiresd temporary indirect disarray
// ----------------------------------------------------------------------------
/**
 Implementation note: there is two numbering styles

   bgd_idof : from the current space
   dom_idof : from a space compacted to the domain "dom", as geo_domain does

  This function returns the renumbering disarray "dom_idof2bgd_idof".
  It is a temporary, used at the fly when using the u[dom] syntax.
 */
template <class T, class M>
disarray<typename space_base_rep<T,M>::size_type, M>
space_base_rep<T,M>::build_indirect_array (
    const space_base_rep<T,M>& Wh,	// Wh = space on domain
    const std::string&       dom_name   // redundant: contained in Wh.get_geo()
  ) const
{
  const geo_basic<T,M>& bgd_gamma = get_geo()[dom_name];
  return build_indirect_array (Wh, bgd_gamma);
}
template <class T, class M>
disarray<typename space_base_rep<T,M>::size_type, M>
space_base_rep<T,M>::build_indirect_array (
    const space_base_rep<T,M>& Wh,	// Wh = space on domain
    const geo_basic<T,M>&      bgd_gamma2
  ) const
{
  // TODO: move it to the call ?
  const geo_basic<T,M>& bgd_gamma = bgd_gamma2.get_background_domain();

  // TODO: verifier que le domaine bgd_gamma est compatible:
  // => il doit y avoir un meme maillage de base a bgd_gamma & Wh.get_geo
  const space_base_rep<T,M>& Vh = *this;  // Vh = space on background mesh
  Vh.freeze_guard();
  Wh.freeze_guard();
  const geo_basic<T,M>& dom_gamma = Wh.get_geo();
  size_type map_dim = dom_gamma.map_dimension();
  check_macro (dom_gamma.size() == bgd_gamma.size(), "incompatible domains");
  distributor dom_ownership = Wh.ownership();
  distributor bgd_ownership = Vh.ownership();
trace_macro("Vh="<<Vh.name()<<", Wh="<<Wh.name()<<", dom_idof2bgd_idof.size=Wh.size="<<dom_ownership.size()<<"...");
  size_type first_dom_dis_idof = dom_ownership.first_index();
  size_type first_bgd_dis_idof = bgd_ownership.first_index();
  std::vector<size_type> dom_dis_idofs, bgd_dis_idofs;
  disarray<size_type, M> dom_idof2bgd_idof (dom_ownership, std::numeric_limits<size_type>::max());
  for (size_type ige = 0, nge = dom_gamma.size(); ige < nge; ige++) {
    const geo_element& dom_S = dom_gamma[ige];
    const geo_element& bgd_S = bgd_gamma[ige];
    Wh.dis_idof (dom_S, dom_dis_idofs);
    Vh.dis_idof (bgd_S, bgd_dis_idofs);
trace_macro("ige="<<ige<<": dom_S="<<dom_S.name()<<dom_S.dis_ie()<<", bgd_S="<<bgd_S.name()<<bgd_S.dis_ie()
	<< ": dom_dis_idofs.size="<<dom_dis_idofs.size()
	<< ", bgd_dis_idofs.size="<<bgd_dis_idofs.size());
    for (size_type loc_idof = 0, loc_ndof = dom_dis_idofs.size(); loc_idof < loc_ndof; loc_idof++) {
      size_type dom_dis_idof = dom_dis_idofs [loc_idof];
      size_type bgd_dis_idof = bgd_dis_idofs [loc_idof];
      dom_idof2bgd_idof.dis_entry (dom_dis_idof) = bgd_dis_idof;
trace_macro("ige="<<ige<<": dom_idof2bgd_idof["<<dom_dis_idof<<"] = "<<bgd_dis_idof);
    }
  }
  dom_idof2bgd_idof.dis_entry_assembly();
  // move to local numbering:
  for (size_type dom_idof = 0, dom_ndof = dom_idof2bgd_idof.size(); dom_idof < dom_ndof; dom_idof++) {
      size_type bgd_dis_idof = dom_idof2bgd_idof [dom_idof];
      size_type dom_dis_idof = dom_idof + first_dom_dis_idof;
      check_macro (bgd_ownership.is_owned (bgd_dis_idof), "bgd_dis_idof="<<bgd_dis_idof<<" is out of range ["<<first_bgd_dis_idof<<":"<< bgd_ownership.last_index()<<"[");
      size_type bgd_idof = bgd_dis_idof - first_bgd_dis_idof;
      dom_idof2bgd_idof [dom_idof] = bgd_idof;
  }
trace_macro("Vh="<<Vh.name()<<", Wh="<<Wh.name()<<", dom_idof2bgd_idof.size=Wh.size="<<dom_ownership.size()<<" done");
  return dom_idof2bgd_idof;
}
#define _RHEOLEF_space_real(M)                             	\
template <class T>						\
space_basic<T,M>						\
space_basic<T,M>::real()					\
{								\
  return space_basic<T,M> (geo_basic<T,M>(details::zero_dimension()), "P1"); \
}
_RHEOLEF_space_real(sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_space_real(distributed)
#endif // _RHEOLEF_HAVE_MPI
#undef _RHEOLEF_space_real
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T,M)                             \
template class space_base_rep<T,M>;				\
template space_basic<T,M> space_basic<T,M>::real();

_RHEOLEF_instanciation(Float,sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_instanciation(Float,distributed)
#endif // _RHEOLEF_HAVE_MPI
#undef _RHEOLEF_instanciation

} // namespace rheolef
