/*____________________________________________________________________________
	Copyright (C) 1994-1998 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: PGPDiskEncryptDecrypt.c,v 1.4 1999/03/10 02:24:40 heller Exp $
____________________________________________________________________________*/

#include "PGPDiskEncryptDecrypt.h"
#include "SHA.h"

#include "CipherContext.h"
#include "PGPDiskCastCFB.h"

#include "MacMemory.h"

const UInt32	kBlockSize			= 512;
const UInt32	kBlockSizeInUInt32	= kBlockSize / sizeof( UInt32 );

#define Move512(src, dest)	BlockMoveData( src, dest, kBlockSize )


#if CAST5_INTERLEAVED_ENCRYPT // {

	static void
EncryptBlock512(
	const CipherContext *	context,
	ItemCount				blockNumber,
	const UInt32 *			src,
	UInt32 *				dest)
{
	int			i;
	UInt32		sum0, sum1;
	UInt32		iv0, iv1;
	
	pgpAssert( sizeof(context->salt.saltLongs[0]) == sizeof(UInt32) );
		
	sum0	= context->salt.saltLongs[ 0 ] + blockNumber;
	sum1	= context->salt.saltLongs[ 1 ] + blockNumber;

	for (i = 0; i < kBlockSizeInUInt32 - 2; ++i) 
	{
		sum1	+= src[i];
		sum0	+= sum1;		// Simple Fletcher checksum of block
	}
	
	iv0 = sum0 ^ src[ kBlockSizeInUInt32 - 2 ];
	iv1 = sum1 ^ src[ kBlockSizeInUInt32 - 1 ];
	
	CAST5encryptCFBdbl(context->castCFB.expandedKey.keyWords, iv0, iv1, ~iv0,
			~iv1, src, dest, kBlockSize / ( 8 + 8) );

	dest[ kBlockSizeInUInt32 - 2 ] ^= sum0;
	dest[ kBlockSizeInUInt32 - 1 ] ^= sum1;
}


	static void
DecryptBlock512(
	const CipherContext *	context,
	ItemCount				blockNumber,
	const UInt32 *			src,
	UInt32 *				dest)
{
	const int kSplitPointInUInt32 = kBlockSizeInUInt32/2;
	const size_t kBytesBeforeSplit = kSplitPointInUInt32 * sizeof(UInt32);
	const size_t kBytesAfterSplit = (kBlockSizeInUInt32 - kSplitPointInUInt32) *
					sizeof(UInt32);
	int	i;
	UInt32	iv0, iv1;

	/* This must be done in two halves to recover the IV */	
	CAST5decryptCFBdbl(context->castCFB.expandedKey.keyWords,
		src[ kSplitPointInUInt32 - 4 ], src[ kSplitPointInUInt32 - 3 ],
		src[ kSplitPointInUInt32 - 2 ], src[ kSplitPointInUInt32 - 1 ],
		src + kSplitPointInUInt32, dest + kSplitPointInUInt32,
				kBytesAfterSplit/(8 + 8));
	iv0 = dest[ kBlockSizeInUInt32 - 2 ];
	iv1 = dest[ kBlockSizeInUInt32 - 1 ];
	CAST5decryptCFBdbl(context->castCFB.expandedKey.keyWords, iv0, iv1, ~iv0,
			~iv1, src, dest, kBytesBeforeSplit/(8 + 8) );

	/* Undo the checksum */
	iv0	= context->salt.saltLongs[ 0 ] + blockNumber;
	iv1	= context->salt.saltLongs[ 1 ] + blockNumber;
	for (i = 0; i < kBlockSizeInUInt32 - 2; ++i)
	{
		iv1		+= dest[ i ];
		iv0		+= iv1;
	}

	dest[ kBlockSizeInUInt32 - 2 ]	^= iv0;
	dest[ kBlockSizeInUInt32 - 1 ]	^= iv1;
}

#else // } !CAST5_INTERLEAVED_ENCRYPT {

	static void
EncryptBlock512(
	const CipherContext *	context,
	ItemCount				seedValue,
	const UInt32 *			src,
	UInt32 *				dest)
{
	short			i;
	UInt32			iv0, iv1;
	
	pgpAssert( sizeof( context->salt.saltLongs[ 0 ] ) == sizeof( UInt32 ) );
		
	iv0 = 0;
	iv1	= 0;
	for ( i = 0; i < kBlockSizeInUInt32 - 2; ++i )
	{
		iv1	+= src[ i ];
		iv0	+= iv1;		// Simple Fletcher checksum of block
	}
	
	iv0	^= context->salt.saltLongs[ 0 ];
	iv1	^= context->salt.saltLongs[ 1 ];
	
	iv0	+= seedValue;
	iv1	+= seedValue;

	CAST5encryptCFB(context->castCFB.expandedKey.keyWords,
		iv0 ^ src[kBlockSizeInUInt32 - 2],
		iv1 ^ src[kBlockSizeInUInt32 - 1],
		src, dest, kBlockSize/8);
	dest[kBlockSizeInUInt32 - 2] ^= iv0;
	dest[kBlockSizeInUInt32 - 1] ^= iv1;
}

	static void
DecryptBlock512(
	const CipherContext *	context,
	ItemCount				seedValue,
	const UInt32 *			src,
	UInt32 *				dest)
{
	const int kSplitPointInUInt32 = kBlockSizeInUInt32/2;
	const size_t kBytesBeforeSplit = kSplitPointInUInt32 * sizeof(UInt32);
	const size_t kBytesAfterSplit = (kBlockSizeInUInt32 -
				kSplitPointInUInt32) * sizeof(UInt32);
	int	i;
	UInt32	iv0, iv1;

	/* This must be done in two halves to recover the IV */	
	CAST5decryptCFB( context->castCFB.expandedKey.keyWords,
		src[ kSplitPointInUInt32 - 2 ], src[ kSplitPointInUInt32 - 1 ],
		&src[ kSplitPointInUInt32 ], &dest[ kSplitPointInUInt32 ],
		kBytesAfterSplit/8 );
	CAST5decryptCFB( context->castCFB.expandedKey.keyWords,
		dest[ kBlockSizeInUInt32 - 2 ], dest[ kBlockSizeInUInt32 - 1 ],
		src, dest, kBytesAfterSplit/8 );

	/* Undo the checksum */
	iv0	= 0;
	iv1	= 0;
	for (i = 0; i < kBlockSizeInUInt32 - 2; ++i)
	{
		iv1		+= dest[ i ];
		iv0		+= iv1;
	}
	iv0		^= context->salt.saltLongs[ 0 ];
	iv1		^= context->salt.saltLongs[ 1 ];
	iv0		+= seedValue;
	iv1		+= seedValue;

	dest[ kBlockSizeInUInt32 - 2 ]	^= iv0;
	dest[ kBlockSizeInUInt32 - 1 ]	^= iv1;
}

#endif // } !CAST5_INTERLEAVED_ENCRYPT


	void
CipherBlocks(
	const CipherContext *	context,
	ItemCount				startBlockIndex,
	ItemCount				numBlocks,
	const void *			src,
	void *					dest,
	CipherOp				op)
{
	UInt32				blockIndex;
	const uchar *		curSrc		= (const uchar *)src;
	uchar *				curDest		= (uchar *)dest;

	pgpa((
		pgpaAddrValid( context, CipherContext),
		pgpaAddrValid( src, VoidAlign ),
		pgpaAddrValid( dest, VoidAlign ),
		pgpaAssert( numBlocks != 0 )
		));

	for( blockIndex = 0; blockIndex < numBlocks; blockIndex++)
	{
		ulong	seedValue;

		seedValue	= startBlockIndex + blockIndex;

		if ( op == kCipherOp_Decrypt )
			{
			DecryptBlock512( context, seedValue, (const UInt32 *)curSrc,
					(UInt32 *)curDest );
			}
		else
			{
			EncryptBlock512( context, seedValue, (const UInt32 *)curSrc,
					(UInt32 *)curDest );
			}

		curSrc	+= kBlockSize;
		curDest	+= kBlockSize;
	}
}


	void
HashBuf( const uchar *buf, ushort length, EncryptedCASTKey *key)
{
	SHA			hash;
	SHA::Digest	hashResult;
		
	pgpAssertAddrValid( buf, uchar );
	pgpAssertAddrValid( key, EncryptedCASTKey );
	
	hash.Update(buf, length);
	hash.Final( &hashResult );
	
	BlockMoveData( hashResult.bytes, key->keyBytes, sizeof( key->keyBytes ) );
}

	void
SaltPassphrase(
	const EncryptedCASTKey	*keyin,
	EncryptedCASTKey 		*keyout,
	const PassphraseSalt 	*salt,
	short 					*hashReps)
{
	uchar		j;
	short		i, k;
	long		endTicks;
	SHA			hash;
	SHA::Digest	hashResult;
	
	pgpAssertAddrValid( keyin, EncryptedCASTKey );
	pgpAssertAddrValid( keyout, EncryptedCASTKey );
	pgpAssertAddrValid( salt, uchar );
	pgpAssertAddrValid( hashReps, short );
	
	hash.Update(salt->saltBytes, sizeof( salt->saltBytes ));
	if( *hashReps != 0)
	{
		// Hash the passphrase and a rotating byte counter the specified
		// number of times into the hash field with the 8 bytes of salt from
		// above in order to form the session key to decrypt the master key.

		k = *hashReps;

		for( i = 0, j = 0; i < k; i++, j++ )
		{
			hash.Update(keyin->keyBytes, sizeof( keyin->keyBytes ) );
			hash.Update(&j, 1);
		}
	}
	else
	{
		// This is a new disk or we are changing the passphrase.  We hash the
		// passphrase in with a rotating counter byte an arbitrary number of
		// times based on the processing power of the computer we're running
		// on up to a maximum of 16000.

		endTicks = LMGetTicks() + 30;
		for( i = 0, j = 0; ( LMGetTicks() < endTicks ) && ( i < 16000 );
					i++, j++ )
		{
			hash.Update(keyin->keyBytes, sizeof( keyin->keyBytes ) );
			hash.Update(&j, 1);
		}
		
		*hashReps = i;
	}
	hash.Final( &hashResult );
	
	BlockMoveData( hashResult.bytes, keyout->keyBytes,
			sizeof( keyout->keyBytes ) );
}
