/* ======================================================================

	Functions to manage string buffers
	This supports:
		Pascal Strings - one length byte plus string
		C Strings      - Null terminated string in buffer
		Buffers        - String data and a length

	These mostly return the length of the result

	Filename:			CopyCat.c
	Last Edited:		March 7, 1997
	Authors:			Laurence Lundblade, Myra Callen
	Copyright:			1995, 1996 QUALCOMM Inc.
	Technical support:	<emsapi-info@qualcomm.com>
*/

#include <ctype.h>
#include "copycat.h"


/* Copy a pascal string into a buffer, return length */
long CopyPB(const StringPtr source, char *dest)
{
	BlockMoveData(source + 1, dest, *source);
	return (*source);
}

/* Copy a buffer to a Pascal string, return length */
long CopyBP(const char *source, long sourceLen, StringPtr dest)
{
	BlockMoveData(source, dest + 1, sourceLen);
	*dest = sourceLen;
	return sourceLen;
}

/* Make a copy of a Pascal string, return length */
long CopyPP(const StringPtr source, StringPtr dest)
{
	BlockMoveData(source+1, dest + 1, *source);
	*dest = *source;
	return (*dest);
}

/* Copy a Pascal string into a C string, return length */
long CopyPC(const StringPtr source, char *dest)
{
	BlockMoveData(source+1, dest, *source);
	*(dest + *source) = '\0';
	return (*source);
}

/* Copy C string into a Pascal string, return length */
long CopyCP(const char *source, StringPtr dest)
{
	char *s, *d;

	s = (char*) source;
	for (d = (char*) dest + 1; *s; *d++ = *s++)
		;
	*dest = s - source;
	return (*dest);
}

#pragma mark -

/* Append a pascal string into a buffer, return length */
long CatPB(const StringPtr addition, long destLen, char *dest)
{
	BlockMoveData(addition + 1, dest + destLen, *addition);
	destLen += *addition;
	return destLen;
}

/* Append a buffer to a Pascal string, return length */
long CatBP(const char *addition, long additionLen, StringPtr dest)
{
	BlockMoveData(addition, dest + 1 + *dest, additionLen);
	*dest += additionLen;
	return (*dest);
}

/* Append a Pascal string to another, return length */
long CatPP(const StringPtr addition, StringPtr dest)
{
	BlockMoveData(addition + 1,  dest + 1 + *dest, *addition);
	*dest += *addition;
	return (*dest);
}

#pragma mark -

/* ----------------------------------------------------------------------
	A case insensitive strncmp() like thing

	Args:	o - First string to compare (null terminated)
			r - Second buffer (string) to compare
			n - length of second string
			caseignore

	Result: integer indicating which is greater
 */
short CmpCB(const char *o, const char *r, long n, short caseignore)
{
	if (r == nil && o == nil)
		return 0;
	if (o == nil)
		return 1;
	if (r == nil)
		return -1;

	n--;
	while (n && *o && (caseignore ? ((isupper(*o) ? tolower(*o) : *o) ==
					  (isupper(*r) ? tolower(*r) : *r)) : (*o == *r))) {
		o++;
		r++;
		n--;
	}
	return (caseignore ? ((isupper(*o) ? tolower(*o) : *o) -
						  (isupper(*r) ? tolower(*r) : *r)) : (*o - *r));
}

/* ----------------------------------------------------------------------
	A case insensitive strncmp() like thing

	Args:	o - First string to compare - a PASCAL string
			r - Second buffer (string) to compare
			n - length of second string
			caseignore

	Result: integer indicating which is greater
 */
short CmpPB(const StringPtr o, const char *r, long n, short caseignore)
{
	char	*oc, *o_end;

	if (r == nil && o == nil)
		return 0;
	if (o == nil)
		return 1;
	if (r == nil)
		return -1;

	oc = (char*) o;
 	o_end = (char*) oc + (*oc + 1);
	n--;
	oc++;
	while (n && (oc < o_end) &&
				(caseignore ? ((isupper(*oc) ? tolower(*oc) : *oc) ==
				(isupper(*r) ? tolower(*r) : *r)) : (*oc == *r))) {
		oc++;
		r++;
		n--;
	}
	return (caseignore ? ((isupper(*oc) ? tolower(*oc) : *oc) -
						  (isupper(*r) ? tolower(*r) : *r)) : (*oc - *r));
}
