/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: CPassphraseEdit.cp,v 1.23 1999/03/10 02:35:15 heller Exp $
____________________________________________________________________________*/

#include <Sound.h>

#include <LView.h>
#include <PP_KeyCodes.h>
#include <PP_Messages.h>
#include <UDrawingUtils.h>
#include <UGAColorRamp.h>
#include <UGraphicUtils.h>

#include "MacEnvirons.h"
#include "MacStrings.h"
#include "pgpMacMemory.h"
#include "pgpMem.h"
#include "pgpPassphraseUtils.h"

#include "CPassphraseEdit.h"

UInt32	CPassphraseEdit::mNumActiveFields = 0;
UInt8	CPassphraseEdit::mSpaceCountList[kMaxPassphraseLength];

const uchar	kOptionSpaceCharacter = 0xCA;	// <<<--- This is an option-Space!

CPassphraseEdit::CPassphraseEdit(LStream	*inStream)
		: LEditText(inStream)
{
	// This must be TRUE for GetDescriptor to work
	pgpAssert( kMaxPassphraseLength < sizeof( Str255 ) );
	
	mKeyFilter = sPassphraseKeyFilter;
	
	mPassphraseHidden 		= TRUE;
	mPassphraseLength		= 0;
	mSpacePassphraseLength	= 0;

	pgpClearMemory( mPassphrase, sizeof( mPassphrase ) );

	if( VirtualMemoryIsOn() )
	{
		HoldMemory( mPassphrase, sizeof( mPassphrase ) );
	}
		
	// Pre-fill an array of space characters for simple showing and
	// hiding of the passphrase
	pgpFillMemory( mSpacePassphrase,
			sizeof( mSpacePassphrase ), kOptionSpaceCharacter );
	
	SetMaxChars( kMaxPassphraseLength );
	
	++mNumActiveFields;
	if( mNumActiveFields == 1 )
	{
		// Generate a "random" amount of Space characters to represent the
		// entered character. This is stored so that backspacing appears to
		// work "correctly". The number of Spaces is derived using the crappy
		// Macintosh Random() routine.
			
		for( short index = 0; index < kMaxPassphraseLength; index++ )
		{
			mSpaceCountList[index] =
				( (ulong) Random() % kMaxSpacesPerCharacter ) + 1;
		}
	}
}

CPassphraseEdit::~CPassphraseEdit()
{
	// For security reasons, clear the passphrase items
	
	mPassphraseLength = 0;
	
	pgpAssert( mNumActiveFields > 0 );
	--mNumActiveFields;
	
	pgpClearMemory( mPassphrase, sizeof( mPassphrase ) );
	
	if( VirtualMemoryIsOn() )
	{
		UnholdMemory( mPassphrase, sizeof( mPassphrase ) );
	}
}

// ---------------------------------------------------------------------------
//	SetHideStatus
// ---------------------------------------------------------------------------
//	Show or hide the passphrase text

	void
CPassphraseEdit::SetHideTyping(Boolean hide)
{
	if( mPassphraseHidden != hide )
	{
		FocusDraw();
		
		if( hide )
		{
			::TESetText( mSpacePassphrase, mSpacePassphraseLength, mTextEditH);
		}
		else
		{
			::TESetText( mPassphrase, mPassphraseLength, mTextEditH);
		}
		
		/*
		** The passphrase edit field may be hidden when this is called. Do not
		** arbitrarily show it.
		*/
		
		if( IsVisible() )
			DrawSelf();

		mPassphraseHidden = hide;
	}
}

// ---------------------------------------------------------------------------
//	FindCommandStatus
// ---------------------------------------------------------------------------
//	Disable the Edit menu options for the password field.

	void
CPassphraseEdit::FindCommandStatus(
	CommandT	inCommand,
	Boolean&	outEnabled,
	Boolean&	outUsesMark,
	Char16&		outMark,
	Str255		outName)
{
	switch (inCommand) {

		case cmd_Cut:
		case cmd_Copy:
		case cmd_Paste:
		case cmd_Clear:
		case cmd_SelectAll:
			outEnabled = FALSE;
			break;
			
		default:
			LEditText::FindCommandStatus(inCommand,
				outEnabled, outUsesMark, outMark, outName);
			break;
	}
}

// ---------------------------------------------------------------------------
//	ClickSelf
// ---------------------------------------------------------------------------
//	Disallow mouse selection within the password field. The user can backspace,
//	but keyboard and mouse navigation are not allowed. Clicking in the field
//	will target it and set the insertion point to just beyond the last character
//	in the field.

	void
CPassphraseEdit::ClickSelf( const SMouseDownEvent &inMouseDown)
{
	#pragma unused( inMouseDown )

	BroadcastMessage('ckUI', (void *)mPaneID);

	if( !IsTarget() )
	{
		FocusDraw();
		::TESetSelect( max_Int16, max_Int16, mTextEditH );
		SwitchTarget( this );
	}
}

// ---------------------------------------------------------------------------
//	AdjustCursorSelf
// ---------------------------------------------------------------------------
//	Do not let LEditField change the cursor to the I-beam.

void
CPassphraseEdit::AdjustCursorSelf(Point inPortPt, const EventRecord &inMacEvent)
{
#pragma unused( inPortPt, inMacEvent )
}

// ---------------------------------------------------------------------------
//	GetPassphraseLength
// ---------------------------------------------------------------------------

	PGPUInt32
CPassphraseEdit::GetPassphraseLength(void)
{
	return( mPassphraseLength );
}

// ---------------------------------------------------------------------------
//	EstimatePassphraseQuality
// ---------------------------------------------------------------------------

	PGPUInt32
CPassphraseEdit::EstimatePassphraseQuality(void)
{
	return( pgpEstimatePassphraseQuality( mPassphrase ) );
}

// ---------------------------------------------------------------------------
//	GetDescriptor
// ---------------------------------------------------------------------------

	StringPtr
CPassphraseEdit::GetDescriptor(Str255 outDescriptor) const
{
	BlockMoveData( mPassphrase, &outDescriptor[1], mPassphraseLength );
	outDescriptor[0] = mPassphraseLength;
	
	return outDescriptor;
}

// ---------------------------------------------------------------------------
//	GetPassphrase
// ---------------------------------------------------------------------------

	void
CPassphraseEdit::GetPassphrase(char passphrase[256])
{
	CopyCString( mPassphrase, passphrase );
}

// ---------------------------------------------------------------------------
//	PassphraseKeyFilter
// ---------------------------------------------------------------------------

	EKeyStatus
CPassphraseEdit::sPassphraseKeyFilter(
	TEHandle	inMacTEH,
	Char16	inKeyCode,
	Char16	&ioCharCode,
	UInt16	inModifiers)
{
	EKeyStatus	keyStatus;
	
	if( UKeyFilters::IsNavigationKey( inKeyCode ) ||
		UKeyFilters::IsExtraEditKey( inKeyCode ))
	{
		keyStatus = keyStatus_Reject;
	}
	else
	{
		keyStatus = UKeyFilters::PrintingCharField( inMacTEH,
					inKeyCode, ioCharCode, inModifiers );
					
		if( keyStatus == keyStatus_PassUp )
		{
			// Reject all non-pringing pass up characters except action keys
			if( ! UKeyFilters::IsActionKey( inKeyCode ) )
			{
				keyStatus = keyStatus_Reject;
			}
		}
	}

	return( keyStatus );
}

// ---------------------------------------------------------------------------
//	HandleKeyPress
// ---------------------------------------------------------------------------
//	This is the total logic (and most of the code) from
//	LEditField::HandleKeyPress. We copy the code here to address security
//	concerns. LEditField posts LTETypingAction objects to support
//	undo/redo. The objects contain copies of the current edit text and are
//	uncontrolled. Since we support no undo, this functionality is eliminated.
//	
//	Because we have copied the code from PowerPlant, it is important to keep
//	in sync with future developments of the LBroadcasterEditField and
//	LGAEditText classes. We do this by breaking compilation if the
//	PowerPlant version changes

#if __PowerPlant__ > 0x01908000	// Version 1.9
//	!!! Recheck code copied from parent classes. See comment above!
#endif

	Boolean
CPassphraseEdit::HandleKeyPress(
	const EventRecord& inKeyEvent)
{

	// Intercept Enter and Return key for immediate value change.

	EKeyStatus	theKeyStatus 	= keyStatus_Input;
	Boolean		keyHandled 		= TRUE;
	Int16		theKey 			= inKeyEvent.message;
	
	if (inKeyEvent.modifiers & cmdKey)
	{										// Always pass up when the command
		theKeyStatus = keyStatus_PassUp;	//   key is down
	}
	else if( mKeyFilter != nil )
	{
		Char16	newCharCode	= inKeyEvent.message & charCodeMask;
		
		theKeyStatus = (*mKeyFilter)( mTextEditH,
							theKey,
							newCharCode,
							inKeyEvent.modifiers );
	}
	
	StFocusAndClipIfHidden	focus(this);

	switch (theKeyStatus)
	{
		case keyStatus_Input:
		{
			UInt32	numSpaces;
			
			// Check if we are at the character limit
			// ### Not two-PGPByte char compatible
			
			if( TooManyCharacters( 1 ) )
			{
				SysBeep(1);
				break;
			}
			
			numSpaces 				= mSpaceCountList[mPassphraseLength];
			mSpacePassphraseLength	+= numSpaces;
			
			mPassphrase[mPassphraseLength++] 	= theKey;
			mPassphrase[mPassphraseLength]		= 0;
			
			if( mPassphraseHidden )
			{
				UInt32	SpaceIndex;
				
				// If the passphrase is hidden, add the Spaces
				for( SpaceIndex = 0; SpaceIndex < numSpaces; SpaceIndex++ )
				{
					TEKey( kOptionSpaceCharacter, mTextEditH );
				}
			}
			else
			{
				TEKey(theKey, mTextEditH);
			}
			
			UserChangedText();
			break;
		}
		
		case keyStatus_TEDelete:
		{
			if ((**mTextEditH).selEnd > 0)
			{
				UInt32	backspaceCount;
				UInt32	index;
				UInt32	numSpaces;
				
				FocusDraw();
				
				pgpAssert( mPassphraseLength > 0 );

				--mPassphraseLength;
				mPassphrase[mPassphraseLength] = 0;
				
				numSpaces = mSpaceCountList[mPassphraseLength];
				mSpacePassphraseLength	-= numSpaces;
				
				if( mPassphraseHidden )
				{
					backspaceCount = numSpaces;
				}
				else
				{
					backspaceCount = 1;
				}
				
				for( index = 0; index < backspaceCount; index++ )
				{
					::TEKey(char_Backspace, mTextEditH);
				}

				UserChangedText();
			}
			
			break;
		}
			
		case keyStatus_TECursor:
			pgpDebugMsg( "CPassphraseEdit::HandleKeyPress: "
				"Cursor key was not filtered!" );
			break;
			
		case keyStatus_ExtraEdit:
			pgpDebugMsg( "CPassphraseEdit::HandleKeyPress: "
				"Edit key was not filtered!" );
			break;
			
		case keyStatus_Reject:
			SysBeep(1);
			break;
			
		case keyStatus_PassUp:
			// Skip all parent classes up to LCommander
			keyHandled = LCommander::HandleKeyPress(inKeyEvent);
			break;
	}
	
	return keyHandled;
}

// ---------------------------------------------------------------------------
//	SelectAll
// ---------------------------------------------------------------------------
//	Don't allow select all text. Instead, position the insertion point at the
//	end of the text.

	void
CPassphraseEdit::SelectAll(void)
{
	StFocusAndClipIfHidden	focus(this);
	::TESetSelect(max_Int16, max_Int16, mTextEditH);
}

// ---------------------------------------------------------------------------
//	TooManyCharacters
// ---------------------------------------------------------------------------
//	The default implementation uses the onscreen representation to get the
//	length of the text. This does not work when Spaces are being displayed.


	Boolean
CPassphraseEdit::TooManyCharacters(Int32 inCharsToAdd)
{
	return( mPassphraseLength + inCharsToAdd > mMaxChars );
}

// ---------------------------------------------------------------------------
//	SetDescriptor
// ---------------------------------------------------------------------------
//	Don't allow this call for passphrase fields. They always start out empty.

	void
CPassphraseEdit::SetDescriptor(ConstStr255Param inDescriptor)
{
	#pragma unused( inDescriptor )
	
	pgpDebugMsg( "CPassphraseEdit::SetDescriptor(): "
		"This should never be called" );
}

// ---------------------------------------------------------------------------
//	ClearPassphrase
// ---------------------------------------------------------------------------
//	Clear out the passphrase and redraw.

	void
CPassphraseEdit::ClearPassphrase(void)
{
	mPassphraseLength		= 0;
	mSpacePassphraseLength	= 0;

	LEditText::SetDescriptor( "\p" );
	
	pgpClearMemory( mPassphrase, sizeof( mPassphrase ) );
	
	Draw( nil );
	BroadcastValueMessage();
}

// ---------------------------------------------------------------------------
//	ClearPassphrase
// ---------------------------------------------------------------------------
//	Identical to LEditText::ObeyCommand for the supported command, however
//	our SelectAll() implementation is used instead.

	Boolean
CPassphraseEdit::ObeyCommand(CommandT inCommand, void *ioParam)
{
	Boolean		cmdHandled = true;
	
	switch( inCommand )
	{
		case msg_TabSelect:
			if( ! IsEnabled() )
			{
				cmdHandled = false;
				break;
			} // else FALL THRU to SelectAll()
			
		case cmd_SelectAll:
			SelectAll();
			break;
		
		default:
			cmdHandled = LEditText::ObeyCommand( inCommand, ioParam);
			break;
	}
	
	return cmdHandled;
}

