/* gdb.c - PIW breakpoint support.
 *
 ****************************************************************
 * Copyright (C) 1999 Tom Lord
 * 
 * See the file "COPYING.PIW" for further information about
 * the copyright and warranty status of this work.
 */


#include "hackerlab/bugs/panic.h"
#include "hackerlab/piw/gdb.h"



/************************************************************************
 *(h1 "Using GDB With PIW"
 *    :includes ("hackerlab/piw/piw.h"))
 * 
 * PIW is designed to be used in combination with a symbolic debugger.
 * This documentation uses GNU GDB for its examples.
 * 
 */

/*(h2 "Breakpoints and Backtraces")
 * 
 * When the PIW run-time system detects a bug in your program, it
 * calls a "breakpoint function".  A breakpoint function has no
 * side-effects (except, optionally, printing a warning message on the
 * standard error stream), but is a good place to set a debugger
 * breakpoint.
 * 
 * All breakpoint functions call the function `piw_breakpoint'
 * Therefore, setting a breakpoint in `piw_breakpoint' is effectively
 * the same as setting breakpoints in each individual breakpoint
 * function:
 * 
 *   # Set a breakpoint which is reached whenever PIW detects
 *   # an error:
 *   #
 *   (gdb) br piw_breakpoint
 * 
 */

/*(c piw_msg_breakpoint)
 * void piw_msg_breakpoint (t_uchar * msg);
 * 
 * Print a panic message (`msg') on the standard error stream
 * and call piw_breakpoint.  If `msg' is 0, a generic message
 * is printed.
 * 
 * This function uses `panic_msg' to print `msg'.  See xref:"Panic".
 */
void
piw_msg_breakpoint (t_uchar * msg)
{
  /*
   * void piw_breakpoint (void);
   *
   * This function is called whenever PIW detects an error.
   * To learn more about a particular error, read the comment in the
   * function that called this one.  To find the location in your
   * program where the error was detected, examine a backtrace
   * of the program at this point.
   *
   * A PIW log entry was generated for this error.  To see such
   * log messages as they are generated, consider using the
   * "--real-time" feature of piw-analyze.
   */

  if (!msg)
    msg = "piw_breakpoint reached (indicates a malloc related error)";
  panic_msg (msg);
  piw_breakpoint ();
}



/*(c piw_breakpoint)
 * void piw_breakpoint (void);
 * 
 * This function is called whenever PIW detects an error.  To learn
 * more about a particular error, read the comments in the functions
 * that called this one.  To find the location in your program where
 * the error was detected, examine a backtrace of the program at this
 * point.
 */
/* The redundant comment below is to provide documentation
 * visible from the gdb `list' command.
 */
void
piw_breakpoint (void)
{
  /*
   * void piw_breakpoint (void);
   *
   * This function is called whenever PIW detects an error.
   * To learn more about a particular error, read the comments in the
   * functions that called this one.  To find the location in your
   * program where the error was detected, examine a backtrace
   * of the program at this point.
   *
   */
}


/*(h2 "Adding Your Own Breakpoints")
 * 
 * When implementing new kinds of instrumentation, it is a good idea
 * to define new breakpoint functions for each kind of error your
 * instrumentation can detect at run-time.  New breakpoint functions
 * should should have names that describe the kind of error detected,
 * and should call `piw_breakpoint' or `piw_msg_breakpoint'.  A
 * breakpoint function called when *xyzzy* errors are detected could
 * be written:
 * 
 * 	#include "hackerlab/piw/piw.h"
 * 
 * 	void
 *	xyzzy_error ()
 *	{
 *	  piw_breakpoint ();
 *	}
 * 
 * If you want your breakpoint function to produce a message on the 
 * standard error stream, use `piw_msg_breakpoint' instead:
 * 
 * 	void
 *	xyzzy_error ()
 *	{
 *	  piw_msg_breakpoint ("An xyzzy error was detected.");
 *	}
 * 
 */
