use core:asm;
use core:lang;

class AddInstr extends AsmAtom {
	AsmOperand dest;
	AsmOperand src;

	init(AsmOperand dest, AsmOperand src) {
		init() { dest = dest; src = src; }
	}

	void output(CodeGen to, Labels l) {
		to.l << add(dest.v(l), src.v(l));
	}

	Str toS() {
		"add " # dest # ", " # src;
	}
}

class SubInstr extends AsmAtom {
	AsmOperand dest;
	AsmOperand src;

	init(AsmOperand dest, AsmOperand src) {
		init() { dest = dest; src = src; }
	}

	void output(CodeGen to, Labels l) {
		to.l << sub(dest.v(l), src.v(l));
	}

	Str toS() {
		"sub " # dest # ", " # src;
	}
}

class MovInstr extends AsmAtom {
	AsmOperand dest;
	AsmOperand src;

	init(AsmOperand dest, AsmOperand src) {
		init() { dest = dest; src = src; }
	}

	void output(CodeGen to, Labels l) {
		to.l << mov(dest.v(l), src.v(l));
	}

	Str toS() {
		"mov " # dest # ", " # src;
	}
}

class LeaInstr extends AsmAtom {
	AsmOperand dest;
	AsmOperand src;

	init(AsmOperand dest, AsmOperand src) {
		init() { dest = dest; src = src; }
	}

	void output(CodeGen to, Labels l) {
		to.l << lea(dest.v(l), src.v(l));
	}

	Str toS() {
		"lea " # dest # ", " # src;
	}
}

class CmpInstr extends AsmAtom {
	AsmOperand a;
	AsmOperand b;

	init(AsmOperand a, AsmOperand b) {
		init() { a = a; b = b; }
	}

	void output(CodeGen to, Labels l) {
		to.l << cmp(a.v(l), b.v(l));
	}

	Str toS() {
		"cmp " # a # ", " # b;
	}
}

class JmpInstr extends AsmAtom {
	AsmOperand to;
	CondFlag flag;

	init(AsmOperand to) {
		init() { to = to; flag = CondFlag:ifAlways; }
	}

	init(AsmOperand to, CondFlag c) {
		init() { to = to; flag = c; }
	}

	void output(CodeGen g, Labels l) {
		g.l << jmp(to.v(l).label, flag);
	}

	Str toS() {
		"jmp " # flag # ", " # to;
	}
}

class SetCondInstr extends AsmAtom {
	AsmOperand to;
	CondFlag flag;

	init(AsmOperand to, CondFlag c) {
		init() { to = to; flag = c; }
	}

	void output(CodeGen g, Labels l) {
		g.l << setCond(to.v(l), flag);
	}

	Str toS() {
		"setCond " # to # ", " # flag;
	}
}
