/*____________________________________________________________________________
	CTranslator.cp
	
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$ld$
____________________________________________________________________________*/
#include <string.h>
#include <Icons.h>
#include <TextUtils.h>

#include "MacStrings.h"
#include "MacIcons.h"
#include "MacFiles.h"
#include "MacEvents.h"
#include "WarningAlert.h"
#include "MacErrors.h"

#include "pgpUtilities.h"
#include "pgpKeys.h"
#include "pgpErrors.h"
#include "pgpUserInterface.h"
#include "pgpStrings.h"

#include "CTranslator.h"
#include "PassphraseCache.h"
#include "PGPUserInterface.h"
#include "TranslatorIDs.h"
#include "MyMIMEUtils.h"
#include "TranslatorStrings.h"
#include "TranslatorUtils.h"
#include "TranslatorPrefs.h"
#include "pgpVersionHeader.h"
#include "MacMenus.h"
#include "CSecureMemory.h"
#include "pgpSharedEncryptDecrypt.h"
#include "pgpClientLib.h"
#include "pgpUserInterface.h"
#include "pgpClientErrors.h"
#include "PlugInLibUtils.h"

static void		AddCommandKeys( void );

typedef struct TranslatorInfo
{
	ConstStringPtr	description;
	SInt32			type;
	SInt32			id;	// aka subtype
	UInt32			flags;
	short			iconSuiteID;	// 0 if none
} TranslatorInfo;

const ResID		kDoesntHaveIconSuite	= 0;

static TranslatorInfo	sTranslatorInfo[] =
{
	{
		"\pPGP Plugin Encrypt",
		EMST_PREPROCESS,
		kEncryptTranslatorID,
		( EMSF_Q4_TRANSMISSION | EMSF_WHOLE_MESSAGE |
			EMSF_REQUIRES_MIME | EMSF_GENERATES_MIME | EMSF_BASIC_HEADERS ),
		kEncryptIconSuiteID
	},
	
	{
		"\pPGP Plugin Sign",
		EMST_SIGNATURE,
		kSignTranslatorID,
		( EMSF_Q4_TRANSMISSION | EMSF_WHOLE_MESSAGE |
			EMSF_REQUIRES_MIME | EMSF_GENERATES_MIME | EMSF_BASIC_HEADERS ),
		kSignIconSuiteID
	},
	
	
	{
		"\pPGP Plugin Encrypt & Sign",
		EMST_COALESCED,
		kEncryptAndSignTranslatorID,
		( EMSF_Q4_TRANSMISSION | EMSF_WHOLE_MESSAGE |
			EMSF_REQUIRES_MIME | EMSF_GENERATES_MIME | EMSF_BASIC_HEADERS ),
		kDoesntHaveIconSuite
	},
	
	{
		"\pPGP Plugin Verify",
		EMST_SIGNATURE,
		kVerifyTranslatorID,
		( EMSF_ON_ARRIVAL | EMSF_ON_DISPLAY | EMSF_REQUIRES_MIME |
			EMSF_GENERATES_MIME ),
		kSignIconSuiteID
	},
	
	{
		"\pPGP Plugin Decrypt",
		EMST_PREPROCESS,
		kDecryptTranslatorID,
		( EMSF_ON_ARRIVAL | EMSF_ON_DISPLAY | EMSF_WHOLE_MESSAGE |
			EMSF_REQUIRES_MIME | EMSF_GENERATES_MIME ),
		kEncryptIconSuiteID
	},
	
	{
		"\pPGP Decrypt/Verify",
		EMST_PREPROCESS,
		kManualDecryptVerifyTranslatorID,
		( EMSF_ON_REQUEST | EMSF_WHOLE_MESSAGE  ),
		kDoesntHaveIconSuite
	},
	
	{
		"\pPGP Add Keys",
		EMST_PREPROCESS,
		kManualAddKeyTranslatorID,
		( EMSF_ON_REQUEST | EMSF_WHOLE_MESSAGE  ),
		kDoesntHaveIconSuite
	},
	
	{
		"\pPGP Encrypt/Sign",
		EMST_PREPROCESS,
		kManualEncryptSignTranslatorID,
		( EMSF_ON_REQUEST | EMSF_WHOLE_MESSAGE  ),
		kDoesntHaveIconSuite
	},
	
	{
		"\pPGP Sign",
		EMST_PREPROCESS,
		kManualSignTranslatorID,
		( EMSF_ON_REQUEST | EMSF_WHOLE_MESSAGE  ),
		kDoesntHaveIconSuite
	},
	
	{
		"\pPGP Encrypt",
		EMST_PREPROCESS,
		kManualEncryptTranslatorID,
		( EMSF_ON_REQUEST | EMSF_WHOLE_MESSAGE  ),
		kDoesntHaveIconSuite
	},
};


CTranslator::CTranslator(
	PGPContextRef		context,
	PGPtlsContextRef	tlsContext,
	emsProgress			progressHook )
{
	mContext		= context;
	mTLSContext		= tlsContext;
	mKeySet			= kInvalidPGPKeySetRef;
	
	/* may be NULL */
	mEudoraProgress	= progressHook;
}


CTranslator::~CTranslator( )
{
	if ( IsntNull( mKeySet ) )
	{
		PGPFreeKeySet( mKeySet );
		mKeySet	= NULL;
	}
}

	PluginError
CTranslator::GetInfo(emsTranslatorP transInfo)
{
	CComboError				err;
	const TranslatorInfo *	info = nil;
	UInt32					index;
	
	if ( transInfo->id == kManualAddKeyTranslatorID )
	{
		AddCommandKeys();
	}
	
	// find info for the specified translator
	for( index = kFirstTranslatorID; index < kLastTranslatorIDPlusOne; ++index )
	{
		info = &sTranslatorInfo[ index - 1 ];
		if ( info->id == transInfo->id )
			break;
		info = nil;
	}
	
	pgpAssert( IsntNull( info ) );
	
	transInfo->type	 = info->type;
	transInfo->flags = info->flags;
	
	MacLeaks_Suspend();
		
	transInfo->desc = (StringHandle) NewHandle( StrSize( info->description ) );
		
	if ( IsntNull( transInfo->desc ) )
	{
		CopyPString( info->description, *transInfo->desc );
	}
	else
	{
		err.err	= memFullErr;
	}
	
	// The icon
	if( err.IsntError() && info->iconSuiteID != kDoesntHaveIconSuite )
	{
		Handle	theSuiteH = nil;
	
		err.err = GetIconSuite( &theSuiteH, info->iconSuiteID,
					svAllAvailableData);
		if( err.IsntError() && IsntNull( theSuiteH ) )
		{
			err.err = DuplicateIconSuite( theSuiteH, &transInfo->icon );
			
			DisposeIconSuite( theSuiteH, true );
		}
		
	}
	
	MacLeaks_Resume();
	
	return( CComboErrorToEudoraError( err ) );
}



	PGPError
CTranslator::LoadDefaultKeySet( PGPBoolean	writeable )
{
	PGPError				err	= kPGPError_NoErr;
	PGPKeyRingOpenFlags		openFlags	= 0;
	
	if ( writeable )
		openFlags	|= kPGPKeyRingOpenFlags_Mutable;
		
	if ( IsNull( mKeySet ) )
	{
		err	= PGPOpenDefaultKeyRings( mContext, openFlags, &mKeySet );
	}
	
	return( err );
}

  
/*____________________________________________________________________________
	Caller must call PGPFreeKey() on *signingKeyOut
____________________________________________________________________________*/
	CComboError
CTranslator::GetSigningPassBuffer(
	PGPPassBufferRef	*passBufferOut,
	PGPKeyRef *			signingKeyOut )
{
	CComboError			err;
	PGPPassBufferRef	passBuffer;
	PGPKeyRef			signingKey;
	
	if ( IsNull( passBufferOut ) || IsNull( signingKeyOut ) )
	{
		err.pgpErr	= kPGPError_BadParams;
		return( err );
	}

	passBuffer = kInvalidPGPPassBufferRef;
	signingKey = kInvalidPGPKeyRef;
	
	if ( IsNull( mKeySet ) )
		err.pgpErr = LoadDefaultKeySet( FALSE );
		
	if ( err.IsntError() )
	{
		err.pgpErr = PGPNewPassBuffer( PGPGetContextMemoryMgr( mContext ),
							&passBuffer );
		if( err.IsntError() )
		{
			CSecureCString256	data;
			PGPBoolean			isPassphrase;
			PGPSize				dataLength;

			if ( gSigningPassphraseCache->GetPassphraseOrPasskey( mKeySet,
					data, &isPassphrase, &dataLength, &signingKey ) )
			{
				/* already in cache */
				pgpAssert( ! err.IsError() );
				
				if( isPassphrase )
				{
					err.pgpErr = PGPPassBufferSetPassphrase( passBuffer,
										data );
				}
				else
				{
					err.pgpErr = PGPPassBufferSetPassKey( passBuffer,
										(PGPByte *) (char *) data, dataLength);
				}
			}
			else
			{
				char							msg[ 256 ];
				PGPGetPassphraseSettings		userSettings;
				PGPGetPassphraseSettings		settings	= 0;
				const PGPGetPassphraseOptions	options		= 
					kPGPGetPassphraseOptionsHideFileOptions;
				
				GetIndCString( msg, kTranslatorErrorStringsResID,
					kPleaseEnterSigningPassphraseStrIndex );

				LoadDefaultKeySet( FALSE );
				
				err.pgpErr	= PGPClientSigningPassphraseDialog( mContext,
									mKeySet, msg, options, settings, NULL,
									data, &userSettings, &signingKey );
				if( err.IsntError() )
				{
					err.pgpErr = PGPPassBufferSetPassphrase( passBuffer,
										data );
				}
				else if( err.pgpErr == kPGPError_KeyUnusableForSignature )
				{
					PGPByte	*passKey = NULL;
					PGPSize	passKeySize;
					
					// User has chosen split key. Show reconstitution dialog
					err.pgpErr = PGPReconstitutionDialog( signingKey, mKeySet,
											kInvalidPGPtlsContextRef,
											&passKey, &passKeySize);
					if( err.IsntError() )
					{
						err.pgpErr = PGPPassBufferSetPassKey( passBuffer,
											passKey, passKeySize );
						
						PGPFreeData( passKey );
					}
				}
			}
			
			if( err.IsError() )
			{
				PGPFreePassBuffer( passBuffer );
				
				passBuffer = kInvalidPGPPassBufferRef;
				signingKey = kInvalidPGPKeyRef;
			}
		}
	}
	
	*passBufferOut	= passBuffer;
	*signingKeyOut 	= signingKey;
	
	return( err );
}




	PluginError
CTranslator::CanTranslateMIMEType(
	long					transContext,
	ConstemsMIMETypeHandle	mimeType,
	ConstStr255Param		type,
	ConstStr255Param		subType,
	ConstStr255Param		protocol )
{
	PluginError	err	= EMSR_CANT_TRANS;
	
	if ( IsntNull( mimeType ) )
	{
		Boolean		haveValidContext;
		const long	kValidContextMask	= EMSF_ON_DISPLAY | EMSF_ON_ARRIVAL;
		
		haveValidContext	= ( transContext & kValidContextMask ) != 0;
		
		if ( haveValidContext )
		{
			Boolean		haveMatch;
			
			haveMatch	= MatchMIMEType( mimeType, type, subType );
			
			if ( haveMatch && IsntNull( protocol ) )
			{
				Str255			paramValue;
				
				haveMatch	= false;
				if ( GetMIMEParameter( mimeType, "\pprotocol", paramValue ) )
				{
					haveMatch	= PStringsAreEqual( paramValue, protocol );
				}
				
				pgpAssertMsg( haveMatch,
					"CTranslator::CanTranslateMIMEType: missing param" );
			}
			
			if ( haveMatch )
			{
	 		    if( transContext == EMSF_ON_ARRIVAL )
	 		    {
		   	        err = EMSR_NOT_NOW;
		   	    }
		   	    else
		   	    {
		   		    err = EMSR_NOW;
		   		}
			}
		}
	}
	
	return( err );
}
					
					
	PluginError
CTranslator::CanTranslate(
	emsTranslatorP 		trans,
	emsDataFileP 		inTransData,
    emsResultStatusP 	transStatus)
{
	PluginError	err	= EMSR_NOW;
	Str255		paramValue;
	
	(void) trans;
	(void) transStatus;
	
	pgpDebugMsgIf( (inTransData->context & EMSF_Q4_COMPLETION ),"EMSF_Q4_COMPLETION");
	
	if ( GetMIMEParameter( inTransData->mimeInfo,
			kPGPAlreadyProcessedParamName, paramValue ) )
	{
		// it has already been translated
		err	= EMSR_CANT_TRANS;
	}
	else
	{
		err = EMSR_NOW;
	}
	
	return( err );
}


	CComboError
CTranslator::PrepareToTranslate( void )
{
	CComboError	err;
	
	err.pgpErr	= LoadDefaultKeySet( FALSE );
	if ( err.IsError() )
	{
		if ( err.pgpErr == kPGPError_FileNotFound ||
			err.pgpErr == kPGPError_CantOpenFile )
		{
			WarningAlert( kWANoteAlertType, kWAOKStyle,
				kTranslatorErrorStringsResID, kKeyringsNotFoundStrIndex);
		}
		else
		{
			ReportError( err );
		}
	}
	
	return( err );
}


	PluginError
CTranslator::TranslateFile(
	emsTranslatorP 		trans,
	emsDataFileP 		inFile,
    emsDataFileP 		outFile,
    emsResultStatusP 	transStatus)
{
	PluginError	err = EMSR_INVALID_TRANS;

	(void) trans;
	(void) inFile;
	(void) outFile;
	(void) transStatus;

	return( err );
}

/*____________________________________________________________________________
	Caller should pass any desired options in 'optionsIn'. Examples:
		PGPOOutputLineEndType
		PGPOPGPMIMEEncoding
	
	All relavant options are set in this routine.
____________________________________________________________________________*/
	CComboError
CTranslator::HandleEncryptSign(
	const FSSpec *		inSpec,
	const FSSpec *		outSpec,
	emsHeaderDataP		header,
	Boolean				doSign,
	PGPOptionListRef	optionsIn,
	Boolean				isPGPMime )
{
	CComboError			err;
	PGPKeySetRef		recipientSet	= kInvalidPGPKeySetRef;
	Boolean				gotRecipients	= false;
							
	Boolean			doConventionalEncrypt	= FALSE;
	Boolean			reportedError			= false;
	Boolean			fyeo					= false;
	char			*conventionalPassphrase	= NULL;
	PGPKeySetRef	newKeysToAdd			= kInvalidPGPKeySetRef;

	DebugCopyToRAMDisk( PGPGetContextMemoryMgr( mContext ), inSpec,
				"\pEncryptIn" );

	err.pgpErr	= LoadDefaultKeySet( FALSE );
	if ( err.IsntError() )
	{
		PGPRecipientSettings	settings	= 0;
	
		err	= GetRecipientList( mContext, mTLSContext, header,
					mKeySet, isPGPMime, &recipientSet, &settings, &newKeysToAdd );
					
		doConventionalEncrypt	=
			( settings & kPGPRecipientSettingsConvEncrypt ) != 0;
		fyeo = ( settings & kPGPRecipientSettingsFYEO ) != 0;
	}
	
	if ( err.IsntError() && doConventionalEncrypt )
	{
		err.pgpErr	= PGPConventionalEncryptionPassphraseDialog( mContext,
						PGPOUIOutputPassphrase( mContext, &conventionalPassphrase ),
						PGPOLastOption( mContext ) );
	}
	
	/* enter this block if we are doing either kind of encrypt */
	if ( err.IsntError() &&
		( recipientSet != kInvalidPGPKeySetRef || doConventionalEncrypt ) )
	{
		PGPKeyRef			signingKey	= kInvalidPGPKeyRef;
		PGPContextRef		c	= mContext;
		PGPPassBufferRef	passBuffer = kInvalidPGPPassBufferRef;
		
		if ( doSign )
		{
			err = GetSigningPassBuffer( &passBuffer, &signingKey );
		}
		
		if ( err.IsntError() )
		{
			EncryptSignEventHandlerData	data( this );
			char				comment[ 256 ];
			PGPOptionListRef	optionList	= kInvalidPGPOptionListRef;
			
			PrefGetComment( comment );
			
			// all these options are the same, regardless of whether it's
			// PGP/MIME or not
			err.pgpErr	= PGPBuildOptionList( c, &optionList,
				PGPOInputFileFSSpec( c, inSpec ),
				PGPOOutputFileFSSpec( c, outSpec ),
				PGPOLocalEncoding( c, kPGPLocalEncoding_None ),
				PGPOArmorOutput(c, TRUE),
				PGPODataIsASCII( c, TRUE ),
				PGPOEventHandler( c, sPGPEncodeEventHandler, &data ),
				PGPOSendNullEvents( c, TRUE ),
				PGPOCommentString( c, comment ),
				PGPOVersionString( c, pgpVersionHeaderString ),
				PGPOForYourEyesOnly( c, fyeo ),
				PGPOLastOption(c)
				);
			if ( err.IsntError() )
			{
				if ( doConventionalEncrypt  )
				{
					err.pgpErr	= PGPAppendOptionList( optionList,
							PGPOConventionalEncrypt( c, 
								PGPOPassphrase( c, conventionalPassphrase ),
								PGPOLastOption( c )),
							PGPOCipherAlgorithm( c,
								PrefGetConventionalCipher() ),
							PGPOLastOption( c ) );
				}
				else
				{
					PreferredAlgorithmsStruct	algs;
					
					PrefGetPreferredAlgorithms( &algs );
					if ( algs.numAlgs !=0 )
					{
						err.pgpErr	= PGPAppendOptionList( optionList,
								PGPOPreferredAlgorithms( c,
									algs.algs, algs.numAlgs),
								PGPOLastOption( c ) );
					}
					
					err.pgpErr	= PGPAppendOptionList( optionList,
							PGPOEncryptToKeySet( c, recipientSet ),
							PGPOLastOption( c ) );
				}
			}
			
			if ( err.IsntError() && doSign )
			{
				err.pgpErr	= PGPAppendOptionList( optionList, 
					PGPOSignWithKey( c, signingKey,
						PGPOPassBuffer(c, passBuffer),
						PGPOLastOption(c) ),
					PGPOLastOption( c ) );
			}
			
			if ( err.IsntError() )
			{
				err.pgpErr = PGPGuaranteeMinimumEntropy( c );
			}
			
			if ( err.IsntError() )
			{
				err.pgpErr	= PGPEncode( c, optionList,
					optionsIn, PGPOLastOption( c ) );
			}
			
			if ( optionList != kInvalidPGPOptionListRef )
				PGPFreeOptionList( optionList );
		}
		
		
		DebugCopyToRAMDisk( PGPGetContextMemoryMgr( mContext ),
				outSpec, "\pEncryptFinalOut" );
	
		if ( err.IsntError() && doSign )
		{
			RememberSigningPassBuffer( passBuffer, signingKey );
			
			// remember it for encryption as well (but not vice versa)
			RememberDecryptionPassBuffer( passBuffer, signingKey );
		}
		
		if( PGPPassBufferRefIsValid( passBuffer ) )
			PGPFreePassBuffer( passBuffer );
	}

	if ( recipientSet != kInvalidPGPKeySetRef )
	{
		PGPFreeKeySet( recipientSet );
		recipientSet	= NULL;
	}
	
	/* allow user to add any new keys we fetched while sending */
	if ( PGPKeySetRefIsValid( newKeysToAdd ) )
	{
		if ( err.IsntError() )
		{
			PGPKeySetRef	importSet	= kInvalidPGPKeySetRef;
			char			prompt[ sizeof( Str255 ) ];
			PGPError		tempErr;
			
			GetIndCString( prompt, kTranslatorStringsRID,
				kSelectiveImportAfterSendPromptStrIndex );
			
			tempErr = PGPSelectKeysDialog( mContext,
						kPGPSelectKeysImportVariation,
						prompt, newKeysToAdd,
						mKeySet, &importSet);
			if ( IsntPGPError( tempErr ) )
			{
				PGPSharedAddKeysToDefaultKeyring( importSet );
				PGPFreeKeySet( importSet );
			}
		}
		PGPFreeKeySet( newKeysToAdd );
	}
		
	if ( ! reportedError )
	{
		ReportError( err );
	}

	if( IsntNull( conventionalPassphrase ) )
		PGPFreeData( conventionalPassphrase );
		
	return( err );
}

	void
CTranslator::GetErrorString(
	CComboError		err,
	StringPtr		resultString )
{
	if ( err.HavePGPError() )
	{
		/* pgp error */
		switch( err.pgpErr )
		{
			case memFullErr:
			case kPGPError_OutOfMemory:
				GetIndString( resultString,
					kTranslatorErrorStringsResID, kOutOfMemoryStrIndex );
				break;
			
			default:
			{
				Str255	errString;

				PGPGetClientErrorString( err.pgpErr,
					sizeof( errString ), (char *)errString);
				CToPString( (char *)errString, errString );
					
				GetIndString( resultString, kTranslatorErrorStringsResID,
					kGenericErrorTemplateString );
				
				PrintPString( resultString, resultString, errString );
				break;
			}
		}
	}
	else
	{
		/* regular mac error */
		switch( err.err )
		{
			case memFullErr:
				GetIndString( resultString,
					kTranslatorErrorStringsResID, kOutOfMemoryStrIndex );
				break;
				
			default:
			{
				Str255	errorNumberString;
				
				GetOSErrorString( err.err, errorNumberString);
				NumToString( err.err, errorNumberString );
				
				GetIndString( resultString, kTranslatorErrorStringsResID,
					kGenericErrorTemplateString );
				
				PrintPString( resultString, resultString, errorNumberString );
				break;
			}
		}
	}
}


	void
CTranslator::ReportError( CComboError	err )
{
	if ( err.IsError() && ! err.IsCancelError() )
	{
		Str255	errorMessage;
		
		GetErrorString( err, errorMessage );
		
		WarningAlert( kWANoteAlertType, kWAOKStyle, errorMessage );
	}
}


	PGPError
CTranslator::sPGPDecodeEventHandler(
	PGPContextRef	context,
	PGPEvent *		event,
	PGPUserValue	userValue)
{
	PGPError						err;
	DecryptVerifyEventHandlerData *	data;
	
	data = (DecryptVerifyEventHandlerData *) userValue;
	
	pgpAssert( data->mMagic == EventHandlerData::kMagic );
	err	= data->mTranslator->PGPDecodeEventHandler( context, event, data);
	
	return( err );
}

	PGPError
CTranslator::PGPDecodeEventHandler(
	PGPContextRef					context,
	PGPEvent *						event,
	DecryptVerifyEventHandlerData *	data)
{
	PGPError	err = kPGPError_NoErr;
	
	pgpAssert( context == mContext );
	
	switch( event->type )
	{
		case kPGPEvent_PassphraseEvent:
		{
			PGPBoolean	havePassphrase = FALSE;
			
			if( event->data.passphraseData.fConventional )
			{
				err = PGPClientDecodeEventHandler( context, event,
									&data->mClientHandlerData );
									
				havePassphrase = TRUE;
			}
			else if( ! data->mTriedCachedPassphrase )
			{
				PGPOptionListRef	optionList;
				
				havePassphrase = gDecryptionPassphraseCache->GetPGPOptionList(
										context, &optionList );
				if( havePassphrase )
				{
					err = PGPAddJobOptions( event->job, optionList,
										PGPOLastOption( context ) );
					
					data->mTriedCachedPassphrase = TRUE;
				}
			}
			
			if( IsntPGPError( err ) && ! havePassphrase )
			{
				GetIndCString( data->mClientHandlerData.prompt,
						kTranslatorErrorStringsResID,
						kPleaseEnterDecryptionPassphraseStrIndex );

				err = PGPClientDecodeEventHandler( context, event,
											&data->mClientHandlerData );
			}
	
			break;
		}
		
		case kPGPEvent_SignatureEvent:
		{
			err = PGPClientDecodeEventHandler( context, event,
											&data->mClientHandlerData );
			if( IsntPGPError( err ) )
			{
				data->mSignatureData.sigData =
						data->mClientHandlerData.signatureData;
				data->mHaveSignatureData =
						data->mClientHandlerData.signatureDataValid;
				
				data->mSignatureData.keyIDString[0] = 0;
				
				if( data->mHaveSignatureData )
				{
					(void)PGPGetKeyIDString( 
							&data->mSignatureData.sigData.signingKeyID,
							kPGPKeyIDString_Abbreviated,
							data->mSignatureData.keyIDString );
				}
			}
			
			break;
		}

		case kPGPEvent_NullEvent:
		{
			err = PGPClientDecodeEventHandler( context, event,
											&data->mClientHandlerData );
			if( IsntPGPError( err ) )
			{
				err = HandleProgressEvent( event );
			}
			
			break;
		}
		
		default:
		{
			err = PGPClientDecodeEventHandler( context, event,
											&data->mClientHandlerData );
			break;
		}
	}
	
	return( err );
}

	PGPError
CTranslator::sPGPEncodeEventHandler(
	PGPContextRef	context,
	PGPEvent *		event,
	PGPUserValue	userValue)
{
	PGPError						err;
	EncryptSignEventHandlerData *	data;
	
	data = (EncryptSignEventHandlerData *) userValue;
	
	pgpAssert( data->mMagic == EventHandlerData::kMagic );
	err	= data->mTranslator->PGPEncodeEventHandler( context, event, data);
	
	return( err );
}

	PGPError
CTranslator::PGPEncodeEventHandler(
	PGPContextRef					context,
	PGPEvent *						event,
	EncryptSignEventHandlerData *	data)
{
	PGPError	err = kPGPError_NoErr;
	
	(void) context;
	(void) data;
	
	switch( event->type )
	{
		case kPGPEvent_NullEvent:
		{
			err = HandleProgressEvent( event );
			break;
		}
		
		default:
			break;
	}
	
	return( err );
}

	PGPError
CTranslator::HandleProgressEvent(PGPEvent *event)
{
	PGPError	err = kPGPError_NoErr;
	
	pgpAssert( event->type == kPGPEvent_NullEvent );
	
	/* progress event */
	if ( IsntNull( mEudoraProgress ) )
	{
		PGPUInt32		percentDone	= 0;
		PGPFileOffset	bytesWritten;
		PGPFileOffset	bytesTotal;
		emsProgressData	progressData;
		
		bytesWritten	= event->data.nullData.bytesWritten;
		bytesTotal		= event->data.nullData.bytesTotal;
		if ( bytesTotal != 0 )
		{
			percentDone	= 100 * ((float)bytesWritten / (float)bytesTotal);
		}
		
		progressData.size 		= sizeof( progressData );
		progressData.value 		= percentDone;
		progressData.message 	= NULL;
		
		if ( CallEMSProgressProc( mEudoraProgress, &progressData ) != 0 )
		{
			err	= kPGPError_UserAbort;
		}
	}
	
	return( err );
}

	static MenuHandle
GetMessagePluginsMenu()
{
	MenuHandle	pluginsMenu	= NULL;
	MenuHandle	editMenu	= NULL;
	Str255		itemText;
	
	/* get edit menu */
	editMenu	= GetIndMenuInMenuBar( 2 );
	
	if ( IsntNull( editMenu ) )
	{
		short		numItems	= CountMItems( editMenu );
		Boolean		foundIt	= FALSE;
		short		cmdChar;
		short		cmdIndex;
		
		/* search for command containing "Message Plug-ins" */
		for( cmdIndex = 1; cmdIndex <= numItems; ++cmdIndex )
		{
			char	cText[ 256 ];
			
			GetMenuItemText( editMenu, cmdIndex, itemText);
			PToCString( itemText, cText );
			
			if ( FindSubStringNoCase( cText, "Message Plug\320ins" ))
			{
				foundIt	= TRUE;
				break;
			}
		}
		if ( ! foundIt )
		{
			/* not found; assume hard-coded location */
			cmdIndex	= 16;
		}
		
		GetItemCmd( editMenu, cmdIndex, &cmdChar );
		if ( cmdChar == 0x1B )
		{
			/* it's a submenu */
			short	subMenuID;
			
			GetItemMark( editMenu, cmdIndex, &subMenuID );
			if ( subMenuID != noMark )
			{
				pluginsMenu	= GetMenuHandle( subMenuID );
			}
		}
	}
	
	
	return( pluginsMenu );
}


	static void
AddKeyForCommand(
	ConstStringPtr	cmd,
	char			cmdKey )
{
	MenuHandle			pluginsMenu	= NULL;
	
	pluginsMenu	= GetMessagePluginsMenu();
	if ( IsntNull( pluginsMenu ) )
	{
		short		numItems;
		Str255		itemText;
		
		numItems	= CountMItems( pluginsMenu );
		for( short index	= 1; index <= numItems; ++index )
		{
			GetMenuItemText( pluginsMenu, index, itemText );
			if ( PStringsAreEqual( itemText, cmd ) )
			{
				SetItemCmd( pluginsMenu, index, cmdKey);
				break;
			}
		}
	}
}


	static void
AddCommandKeys()
{
	static Boolean		sAddedKeys	= FALSE;
	
	if ( sAddedKeys )
		return;
	sAddedKeys	= TRUE;
	
	/* commands must match those in translator info */
	AddKeyForCommand( "\pPGP Sign", '2' );
	AddKeyForCommand( "\pPGP Encrypt", '3' );
	AddKeyForCommand( "\pPGP Encrypt/Sign", '4' );
	AddKeyForCommand( "\pPGP Decrypt/Verify", '5' );
}














