/*
 * nlitcl.c --
 *
 * Natural Language Inc interface to Tcl
 *
 * Copyright 1992 Tom Poindexter and U S WEST Enhanced Services, Inc.
 * Portions Copyright 1994 De Clarke (Regents of the U of Calif)
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  
 * Tom Poindexter, U S WEST, De Clarke, and the UC Regents all  make 
 * no representations about the suitability 
 * of this software for any purpose.  It is provided "as is" without express or
 * implied warranty.  By use of this software the user agrees to 
 * indemnify and hold harmless Tom Poindexter, U S WEST, De Clarke, and
 * the UC Regents  from any 
 * claims or liability for loss arising out of such use.
 *
 *-----------------------------------------------------------------------------
 * Version 1.0 July 1994 -- prototype
 * De Clarke, Santa Cruz, CA : UCO/Lick Observatory, UCSC
 * de@lick.ucsc.edu
 *   this code completely based on "nlitcl" by
 *   Tom Poindexter, Denver Colorado
 *   tpoindex@nyx.cs.du.edu   
 *   (thanks Tom for letting me rip off your source)
 *-----------------------------------------------------------------------------
 *
 */

#include "tcl.h"
#include "tclUnix.h"

#include "nlc.h"
#include "nlc_error.h"

#include <string.h>
#include <ctype.h>


typedef struct NatTclProcs {	/* struct for handle entries 	*/
    int         in_use;		/* if this entry is opened 	*/
    Nl_session  *nlc;		/* nlc pointer for this entry   */
    Nl_status   status;		/* return code from last action */
} NatTclProcs;

typedef unsigned char BYTE;

#define NLITCLPROCS       25	/* default number of nlcs available */
#define NLI_BUFF_SIZE	4096	/* conversion buffer size for various needs*/

/* defines for text/image handling - first is our buffer size;other is max */
#define   TEXT_BUFF_SIZE  32768
#define   MAX_SERVER_TEXT  "2147483647"
#define	  NAT_ATT_CODE_CNT 22

static NatTclProcs   NatProcs[NLITCLPROCS];  

static char *NatHandlePrefix = "nli";  /* prefix used to identify handles*/

static char *NatMsgArray = "nlimsg";  /* array to place errors & messages */

static Tcl_Interp *NatInterp;	    /* interpreter access in err&msg handler*/

/* prototype for malloc */

extern char * malloc();
extern int atoi();

/* OK here is the idiot code for nl_get from nli_get tcl func */
static unsigned int NatAttCodes[NAT_ATT_CODE_CNT];

/* prototypes for all (simple) functions */

extern Tcl_CmdProc  Tcl_NatConnect;	/* connect to NLI with command arg */
					/* model this on nliopen	   */
extern Tcl_CmdProc  Tcl_NatAsk;		/* pass query string to NLI	   */
					/* model this on nli_ask	   */
extern Tcl_CmdProc  Tcl_NatStat;	/* get status codes back from NLI  */
extern Tcl_CmdProc  Tcl_NatOpen;	/* have to do this before fetch    */
extern Tcl_CmdProc  Tcl_NatClose;	/* closes fetch operation	   */
extern Tcl_CmdProc  Tcl_NatFetch;	/* fetch results para from NLI     */
extern Tcl_CmdProc  Tcl_NatReply;	/* reply to prompt from NLI	   */
extern Tcl_CmdProc  Tcl_NatBreak;	/* break NLI processing 	   */
extern Tcl_CmdProc  Tcl_NatExit;	/* exit NLI connection		   */
/* superseded by NatTable		*/
/* extern Tcl_CmdProc  Tcl_NatCols;	/* get column info 		   */
extern Tcl_CmdProc  Tcl_NatMode;	/* set table mode data/text	   */
extern Tcl_CmdProc  Tcl_NatTable;	/* get table info		   */
extern Tcl_CmdProc  Tcl_NatRow;		/* get row data		   	   */
extern Tcl_CmdProc  Tcl_NatGet;		/* get row data		   	   */

/* 
 *----------------------------------------------------------------------
 * get_nli_handle
 *    authenticate a handle string 
 *  return: NatProcs index number or -1 on error;
 */

static int
get_nli_handle (handle) 
    char *handle;
{
    int prefix_len = strlen(NatHandlePrefix);
    int h;

    if ( (strlen(handle) > prefix_len) &&
	 (strncmp(handle,NatHandlePrefix,prefix_len) == 0)  &&
	 (isdigit(*(handle + prefix_len))) ) {

	 h = atoi((handle + prefix_len));
	 if (NatProcs[h].in_use) {
	   return (h);
	 } else {
	   return (-1);
	 }
    } 

    return (-1);
}


/*
 *----------------------------------------------------------------------
 * clear_msg --
 *
 * clears all error and message elements in the global array variable
 *
 */

static void
clear_msg(interp)
    Tcl_Interp *interp;
{

    /* indices associated with error and message handlers */
    /*  status : status string from nli 
	handle : current handle name (last opened)
	mode   : 1 if in data mode, 0 if in text mode
	ptype  : type of last para retrieved
	nullvalue : same as in sybtcl
	tabcount : count of tables from last nli_table call
	colcount : count of columns ditto
	collengths : byte lengths of columns
	coltypes : 3char type codes for columns  -- chr int flt dat
	colhdrs : NLI header values for columns
	colnames : sybase or NLI columns names
    */

    Tcl_SetVar2(interp, NatMsgArray, "status",  "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, NatMsgArray, "handle", "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, NatMsgArray, "mode", "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, NatMsgArray, "ptype", "", TCL_GLOBAL_ONLY);
/*	User should set this to something if they want a special null val */
    Tcl_SetVar2(interp, NatMsgArray, "nullvalue", "",TCL_GLOBAL_ONLY);

    /* these indices only meaningful after nli_table command */
    Tcl_SetVar2(interp, NatMsgArray, "tabcount",     "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, NatMsgArray, "colcount",     "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, NatMsgArray, "collengths","", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, NatMsgArray, "coltypes",  "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, NatMsgArray, "colhdrs",  "", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, NatMsgArray, "colnames",  "", TCL_GLOBAL_ONLY);

}



/*
 *----------------------------------------------------------------------
 * nli_prologue
 *
 * does most of standard command prologue, assumes handle is argv[1]
 * returns: handle index  or -1 on failure
 * 
 */

static int
nli_prologue (interp, argc, argv, num_args, err_msg)
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
    int         num_args;
    char       *err_msg;
{
    int         hand;


    /* check number of minimum args*/

    if (argc < num_args) {
	Tcl_AppendResult (interp, "wrong # args: ", argv[0],
			  err_msg, (char *) NULL);
	return (-1);
    }

    /* parse the handle */
    hand = get_nli_handle(argv[1]);

    if (hand == -1) {
	Tcl_AppendResult (interp, argv[0], ": handle ", argv[1],
			 " not open ", (char *) NULL);
	return (-1);
    }

    /* save the interp structure for the error & msg handlers */
    NatInterp = interp;

    /* clear nlimsg array for new messages & errors */
    Tcl_SetVar2(interp, NatMsgArray, "handle",  argv[1], TCL_GLOBAL_ONLY);
/*
    clear_msg(interp);
*/

    return (hand);
}


/*
 *----------------------------------------------------------------------
 * Tcl_KillNat --
 *   perform all cleanup upon any command deletion
 *
 */

void
Tcl_KillNat (clientData)
    ClientData clientData;
{
    int i;


    for (i = 0; i < NLITCLPROCS; i++) {
        nl_exit(NatProcs[i].nlc);
	NatProcs[i].in_use = 0;
    }

}


/*
 *----------------------------------------------------------------------
 * Nattcl_Init --
 *   perform all initialization for the Natase to Tcl interface.
 *   adds additional commands to interp, creates message array
 *
 *   a call to Nattcl_Init should exist in Tcl_CreateInterp or
 *   Tcl_CreateExtendedInterp.
 */

int
Nattcl_Init (interp)
    Tcl_Interp *interp;
{
    int i;

    /* save the interp structure for the error & msg handlers */
    NatInterp = interp;

    /*
     * Initialize nli proc structures 
     */

    for (i = 0; i < NLITCLPROCS; i++) {
	NatProcs[i].in_use        = 0;
	NatProcs[i].nlc        = NULL;
	NatProcs[i].status     = NULL;
    }

/*	init array of attribute codes				*/
    init_attcodes();


    /*
     * Initialize the new Tcl commands
     */

    Tcl_CreateCommand (interp, "nli_connect", Tcl_NatConnect, (ClientData)NULL,
		      Tcl_KillNat);
    Tcl_CreateCommand (interp, "nli_ask",     Tcl_NatAsk,     (ClientData)NULL,
		      Tcl_KillNat);
    Tcl_CreateCommand (interp, "nli_open",    Tcl_NatOpen,    (ClientData)NULL,
		      Tcl_KillNat);
    Tcl_CreateCommand (interp, "nli_fetch",   Tcl_NatFetch,   (ClientData)NULL,
		      Tcl_KillNat);
    Tcl_CreateCommand (interp, "nli_reply",   Tcl_NatReply,   (ClientData)NULL,
		      Tcl_KillNat);
    Tcl_CreateCommand (interp, "nli_mode",    Tcl_NatMode,    (ClientData)NULL,
		      Tcl_KillNat);
    Tcl_CreateCommand (interp, "nli_table",   Tcl_NatTable,   (ClientData)NULL,
		      Tcl_KillNat);
    Tcl_CreateCommand (interp, "nli_row",     Tcl_NatRow,     (ClientData)NULL,
		      Tcl_KillNat);
/* superseded by NatTable
    Tcl_CreateCommand (interp, "nli_cols",    Tcl_NatCols,    (ClientData)NULL,
		      Tcl_KillNat);
*/
    Tcl_CreateCommand (interp, "nli_quit",    Tcl_NatExit,    (ClientData)NULL,
		      Tcl_KillNat);
    Tcl_CreateCommand (interp, "nli_close",   Tcl_NatClose,   (ClientData)NULL,
		      Tcl_KillNat);
    Tcl_CreateCommand (interp, "nli_break",   Tcl_NatBreak,   (ClientData)NULL,
		      Tcl_KillNat);
    Tcl_CreateCommand (interp, "nli_exit",    Tcl_NatExit,    (ClientData)NULL,
		      Tcl_KillNat);
    Tcl_CreateCommand (interp, "nli_stat",    Tcl_NatStat,    (ClientData)NULL,
		      Tcl_KillNat);
    Tcl_CreateCommand (interp, "nli_get",     Tcl_NatGet,     (ClientData)NULL,
		      Tcl_KillNat);

    /*
     * Initialize nlimsg global array, inital null elements
     */
    
    /* other indices - correspond to error and message handler arguments */
    clear_msg(interp);

    return TCL_OK;

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatConnect --
 *    Implements the nliconnect command:
 *    usage: nli_connect ProgramStr
 *	                
 *    results:
 *	handle - a character string of newly open handle
 *      TCL_OK - connect successful
 *      TCL_ERROR - connect not successful - error message returned
 */

int
Tcl_NatConnect (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{

    int        hand = -1;
    int        i;
    char       buf[NLI_BUFF_SIZE];


    /* can't use nli_prologue, nli_connect creates a handle */

    if (argc < 2) {
	Tcl_AppendResult (interp, "wrong # args: ", argv[0],
			  " CommandStr ",(char *) NULL);
	return TCL_ERROR;
    }

    /* find an unused handle */

    for (i = 0; i < NLITCLPROCS; i++) {
	if (NatProcs[i].in_use == 0) {
            hand = i;
	    break;
	}
    }

    if (hand == -1) {
	Tcl_AppendResult (interp, argv[0], ": no nli nlcs available",
			  (char *) NULL);
	return TCL_ERROR;
    }

    /* save the interp structure for the error & msg handlers */
    NatInterp = interp;

    NatProcs[hand].nlc = nl_connect(argv[1],"nowarn");

    if (NatProcs[hand].nlc == NULL) {
	Tcl_AppendResult (interp, argv[0], ": nli_connect failed in dbopen",
			  (char *) NULL);
	return TCL_ERROR;
    }

    NatProcs[i].in_use = 1;	/* handle ok, set in use flag */

    /* construct handle and return */
    sprintf(buf,"%s%d",NatHandlePrefix,hand);

    Tcl_SetVar2(interp, NatMsgArray, "handle",      buf, TCL_GLOBAL_ONLY);
    clear_msg(interp);

    Tcl_SetResult(interp,buf,TCL_VOLATILE);

    return TCL_OK;

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatAsk --
 *
 *    Implements the nli_ask command:
 *    usage: nli_ask handle sql-string 
 *	                
 *    results:
 *	"OK" if query OK, "NEED_REPLY" if more user input needed
 *      TCL_OK - handle is opened, sql executed ok
 *      TCL_ERROR - wrong # args, or handle not opened,  bad sql stmt
 */

int
Tcl_NatAsk (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;
    Nl_status nlret;


/*	If you fail here then you couldn't use the handle at all	*/

    if ((hand = nli_prologue(interp,argc, argv, 3, " handle ask_str")) == -1) {
	return TCL_ERROR;
    }

    /* send query to server and get return code */

    NatProcs[hand].status = nl_ask(NatProcs[hand].nlc, argv[2]);
    nlret = NatProcs[hand].status;

    if (nlret != NL_SUCCESS) {
	Tcl_AppendResult (interp, argv[0], ": nl_ask failed ", 
			  (char *) NULL);
 	nl_seterr(hand,interp);
	return TCL_ERROR;
    }

    nl_seterr(hand, interp);	

    return TCL_OK;
}

int
nl_seterr (hand, interp)
int hand;
Tcl_Interp  *interp;
{
    char *nlerrstr;
    char    buf[NLI_BUFF_SIZE];
    Nl_error nlstat;

    nlerrstr = '\0';

    nlstat = nl_get(NatProcs[hand].nlc,NL_ERROR_CODE);
/*
    fprintf(stderr,"got error code value %x\n",nlstat);
*/

    switch (nlstat) {

    case NLE_SUCCESS: 
	nlerrstr = "SUCCESS";
	break;
    case NLE_ABORT: 
	nlerrstr="ERROR IRRECOVERABLE";
	break;
    case NLE_NOMEM: 
	nlerrstr = "ERROR INSUFF MEMORY";
	break;
    case NLE_CONN_FAIL: 
	nlerrstr = "ERROR CONN FAIL";
	break;
    case NLE_DBLOAD_FAIL: 
	nlerrstr = "ERROR CONN LOAD";
	break;
    case NLE_PROTOCOL: 
	nlerrstr = "ERROR PROTOCOL";
	break;
    case NLE_READ_ERR: 
	nlerrstr = "ERROR LISP SYNTAX";
	break;
    case NLE_EVAL_ERR: 
	nlerrstr = "ERROR LISP EVAL";
	break;
    case NLE_BUFOVF: 
	nlerrstr = "ERROR OUTPUT TOO LONG";
	break;
    case NLE_TABOVF: 
	nlerrstr = "ERROR TABLE > 200 COL";
	break;
    case NLE_NO_MORE: 
	nlerrstr = "NO MORE OUTPUT";
	break;
    case NLE_INV_ARG: 
	nlerrstr = "ERROR NLCLIB CALL";
	break;
    case NLE_NEED_REPLY: 
	nlerrstr = "NEED USER INPUT";
	break;
    case NLE_INPOVF: 
	nlerrstr = "ERROR INPUT TOO LONG";
	break;
    case NLE_INTERRUPTED: 
	nlerrstr = "ERROR INTERRUPT";
	break;
    case NLE_EXITED: 
	nlerrstr = "ERROR ABNORMAL EXIT";
	break;
    case NLE_INPROG: 
	nlerrstr = "ERROR ACTION IN PROG";
	break;
    case NLE_USEREXIT: 
	nlerrstr = "ERROR USER EXIT";
	break;

    }

/*
    fprintf(stderr,"got error string value %s\n",nlerrstr);
*/
    sprintf(buf,"%s",nlerrstr);
    Tcl_SetVar2(interp, NatMsgArray, "status", buf, TCL_GLOBAL_ONLY);

    return TCL_OK;
	
}


/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatOpen --
 *    Implements the nl_open command:
 *    usage: nli_open handle 
 *	                
 *      TCL_OK - handle is opened
 *      TCL_ERROR - wrong # args, or handle not opened
 */

int
Tcl_NatOpen (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;
    Nl_status nlret;
    char    buf[NLI_BUFF_SIZE];
    char *type;


    if ((hand = nli_prologue(interp, argc, argv, 2, " handle")) == -1) {
	return TCL_ERROR;
    }

    nlret = nl_open(NatProcs[hand].nlc,NL_TEXT);

    if (nlret == NL_FAIL) {
	Tcl_AppendResult (interp, argv[0], ": nl_open failed ", 
			  (char *) NULL);
	return TCL_ERROR;
    }
    type = "NOT";
    switch(nl_get(NatProcs[hand].nlc,NL_RESPONSE_TYPE)) {
   	case NL_TEXT:
        type = "TXT";
        break;
        case NL_ECHO:
        type = "ECH";
        break;
        case NL_ANSWER:
        type = "ANS";
        break;
        case NL_TABLE:
        type = "TAB";
        break;
        case NL_PROMPT:
        type = "PRO";
        break;
        case NL_ERROR_MESSAGE:
        type = "ERR";
        break;
        }

    sprintf(buf,"%s",type);
    Tcl_SetVar2(interp, NatMsgArray, "ptype", buf, TCL_GLOBAL_ONLY);
    Tcl_SetResult(interp,buf,TCL_VOLATILE);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatCols --
 *    implements various info gets to return col count, names, etc.
 *    usage: nli_cols handle 
 *	                
 *    results:
 *	latest column names as tcl list, or param columns, or null list
 *      also set nlimsg(collengths) and nlimsg(coltypes) as tcl list
 * 	and nlimsg(colnames) and nlimsg(colhdrs) as well
 *      TCL_OK - handle is opened, and at least one nlinext executed 
 *      TCL_ERROR - wrong # args, or handle not opened,
 */

/* superseded by NatTable
int
Tcl_NatCols (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;
    int     num_cols, num_tbls;
    int     i;
    char    len_buf[NLI_BUFF_SIZE];
    char    typ_buf[NLI_BUFF_SIZE];
    char    nam_buf[NLI_BUFF_SIZE];
    char    hdr_buf[NLI_BUFF_SIZE];
    char    buf2[NLI_BUFF_SIZE];
    char    buf[NLI_BUFF_SIZE];
    int	    version;

    if ((hand = nli_prologue(interp, argc, argv, 2, " handle")) == -1) {
	return TCL_ERROR;
    }

    
    version = nl_get(NatProcs[hand].nlc,NL_CLNT_PVERSION);

    printf("got client version number %d\n", version);

    printf("about to get nlc number %d, NL_TABLE_COUNT val %d\n",
	hand, NL_TABLE_COUNT);
    num_tbls = nl_get(NatProcs[hand].nlc,NL_TABLE_COUNT);
    printf("num tbls is %d\n", num_tbls);
    printf("about to get nlc number %d, NL_COLUMN_COUNT val %d\n",
	hand, NL_COLUMN_COUNT);
    num_cols = nl_get(NatProcs[hand].nlc,NL_COLUMN_COUNT);
    printf("num cols is %d\n", num_cols);

*/
/*
    for (i = 1; i <= num_cols; i++) {

*/
		/* get the return parm length and append to "collengths" */
/*
		sprintf(buf2, (i>1) ? " %d" : "%d",
			    nl_get(NatProcs[hand].nlc,NL_COLUMN_WIDTH,i));
		strcat(len_buf,buf2);

*/
		/* get the column type and append to "coltypes" */
/*
		sprintf(buf2, (i>1) ? " %s" : "%s",
			    nl_get(NatProcs[hand].nlc,NL_COLUMN_TYPE,i));
		strcat(typ_buf,buf2);
*/

		/* get the column name and append to "colnames" */
/*
		sprintf(buf2, (i>1) ? " %s" : "%s",
			    nl_get(NatProcs[hand].nlc,NL_COLUMN_NAME,i));
		strcat(nam_buf,buf2);
*/

		/* get the column type and append to "colhdrs" */
/*
		sprintf(buf2, (i>1) ? " %s" : "%s",
			    nl_get(NatProcs[hand].nlc,NL_COLUMN_HEADER,i));
		strcat(hdr_buf,buf2);
   }

	Tcl_SetVar2(interp, NatMsgArray, "collengths", len_buf,TCL_GLOBAL_ONLY);
	Tcl_SetVar2(interp, NatMsgArray, "coltypes",   typ_buf,TCL_GLOBAL_ONLY);
	Tcl_SetVar2(interp, NatMsgArray, "colnames",   nam_buf,TCL_GLOBAL_ONLY);
	Tcl_SetVar2(interp, NatMsgArray, "colhdrs",    hdr_buf,TCL_GLOBAL_ONLY);

	sprintf(buf, "%d", num_cols);
    	Tcl_SetResult(interp,buf,TCL_VOLATILE);

    return TCL_OK;
}
*/

/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatClose --
 *
 *    Implements the nl_close command:
 *	nli_close handle
 *	                
 *    results:
 *	null string
 *      TCL_OK - handle is opened -- fetch now closed
 *      TCL_ERROR - wrong # args, or handle not opened,
 */

int
Tcl_NatClose (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;
    Nl_status nlret;

    if ((hand = nli_prologue(interp, argc, argv, 2, " handle")) == -1) {
	return TCL_ERROR;
    }

    nlret = nl_close(NatProcs[hand].nlc);

    if (nlret == NL_FAIL) {
	Tcl_AppendResult (interp, argv[0], ": nl_close failed ", 
			  (char *) NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatExit --
 *    Implements the nl_exit command:
 *    usage: nli_quit handle 
 *	                
 *    results:
 *	null string
 *      TCL_OK - handle is closed
 *      TCL_ERROR - wrong # args, or handle not opened,
 */

int
Tcl_NatExit (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;

    if ((hand = nli_prologue(interp, argc, argv, 2, " handle")) == -1) {
	return TCL_ERROR;
    }

    nl_exit(NatProcs[hand].nlc);

    NatProcs[hand].in_use       = 0;
    NatProcs[hand].nlc       = NULL;

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatFetch --
 *    Implements the nl_fetch command:
 *    usage: nli_fetch handle 
 *
 *    results:
 *      nl separated list of lines of text
 *      TCL_OK - handle is opened
 *      TCL_ERROR - wrong # args, or handle not opened
 */
int
Tcl_NatFetch (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{

int hand;
char line[256];

    if ((hand = nli_prologue(interp, argc, argv, 2, " handle")) == -1) {
        return TCL_ERROR;
    }

/*
    printf("Fetching on nlc number %d\n",hand);
*/

    while (nl_fetch(NatProcs[hand].nlc, NL_TEXT, line, sizeof line, NULL) == NL_SUCCESS) {
	Tcl_AppendElement(interp, line);
   }

   nl_seterr(hand, interp);				

   return TCL_OK;

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatReply --
 *    Implements the nl_reply command:
 *    usage: nli_reply handle string
 *
 *    results:
 *      TCL_OK - handle is opened
 *      TCL_ERROR - wrong # args, or handle not opened
 */
int
Tcl_NatReply (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{

int hand;
Nl_status nlret;

    if ((hand = nli_prologue(interp, argc, argv, 3, " handle reply_str")) == -1) {
        return TCL_ERROR;
    }

    nlret = nl_reply(NatProcs[hand].nlc,argv[2]);

    if (nlret != NL_SUCCESS) {
	Tcl_AppendResult (interp, argv[0], ": nl_reply failed ", 
			  (char *) NULL);
	nl_seterr(hand,interp);
	return TCL_ERROR;
    }

    nl_seterr(hand,interp);
    return TCL_OK;

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatBreak --
 *    Implements the nl_exit command:
 *    usage: nli_break handle
 *
 *    results:
 *      null string
 *      TCL_OK - handle is closed
 *      TCL_ERROR - wrong # args, or handle not opened,
 */

int
Tcl_NatBreak (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int     hand;

    if ((hand = nli_prologue(interp, argc, argv, 2, " handle")) == -1) {
        return TCL_ERROR;
    }

    nl_break(NatProcs[hand].nlc);

    return TCL_OK;

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatStat --
 *    Implements the nl_get command:
 *    usage: nli_stat handle code_number (from NatAttCodes array)
 *
 *    results:
 *      TCL_OK - handle is opened
 *      TCL_ERROR - wrong # args, or handle not opened
 */
int
Tcl_NatStat (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{

int hand;
char nlchar[256];
int nlint;
int attind;
char    buf[NLI_BUFF_SIZE];

    if ((hand = nli_prologue(interp, argc, argv, 3, " handle get_code")) == -1)
 {
        return TCL_ERROR;
    }

    nlchar[0] = '\0';
    attind = atoi(argv[2]);
/*
    printf("user gave us an attind of %d\n",attind);
    printf("That means an NLI code of %d\n", NatAttCodes[attind]);
*/

    if ((0 <= attind) && (attind  <=6)) {
    nlint = nl_get(NatProcs[hand].nlc,NatAttCodes[attind]);
    nl_seterr();
    sprintf(nlchar,"%d",nlint);
    } else {
    if ((8 <= attind) && ( attind <= 11)) {
	sprintf(nlchar,"%s", (char *)nl_get(NatProcs[hand].nlc,NatAttCodes[attind]));
    nl_seterr();
    } else {
	printf("Sorry, illegal get code %d\n",attind);
        return TCL_ERROR;
    }
    }

    sprintf(buf,"%s",nlchar);
    Tcl_SetResult(interp,buf,TCL_VOLATILE);

    return TCL_OK;

}

void 
init_attcodes ()

{

/*	first all the ints, then the char *, then lastly the weirdos	*/

	NatAttCodes[0] =  NL_CLNT_PVERSION;	/* type int	*/
/*	printf("NL_CLNT_PVERSION = %d\n",NL_CLNT_PVERSION);	*/
	NatAttCodes[1] =  NL_COLUMN_COUNT;	/* type int	*/
/*	printf("NL_COLUMN_COUNT = %d\n",NL_COLUMN_COUNT);	*/
	NatAttCodes[2] =  NL_COLUMN_NULL;	/* type int	*/
/*	printf("NL_COLUMN_NULL = %d\n",NL_COLUMN_NULL);	*/
	NatAttCodes[3] =  NL_COLUMN_WIDTH;	/* type int	*/
/*	printf("NL_COLUMN_WIDTH = %d\n",NL_COLUMN_WIDTH);	*/
	NatAttCodes[4] =  NL_END_LINES;		/* type int	*/
/*	printf("NL_END_LINES = %d\n",NL_END_LINES);	*/
	NatAttCodes[5] =  NL_SRVR_PVERSION;	/* type int	*/
/*	printf("NL_SRVR_PVERSION = %d\n",NL_SRVR_PVERSION);	*/
	NatAttCodes[6] =  NL_TABLE_COUNT;	/* type int	*/
/*	printf("NL_TABLE_COUNT = %d\n",NL_TABLE_COUNT);	*/
	NatAttCodes[7] =  NL_USER_DATA;		/* type unsigned int */
/*	printf("NL_USER_DATA = %d\n",NL_USER_DATA);	*/
	NatAttCodes[8] =  NL_COLUMN_HEADER;	/* type char *	*/
/*	printf("NL_COLUMN_HEADER = %d\n",NL_COLUMN_HEADER);	*/
	NatAttCodes[9] =  NL_COLUMN_NAME;	/* type char *	*/
/*	printf("NL_COLUMN_NAME = %d\n",NL_COLUMN_NAME);	*/
	NatAttCodes[10] = NL_ERROR_STRING;	/* type char *	*/
/*	printf("NL_ERROR_STRING = %d\n",NL_ERROR_STRING);	*/
	NatAttCodes[11] = NL_LISPERR_STRING;	/* type char *	*/
/*	printf("NL_LISPERR_STRING = %d\n",NL_LISPERR_STRING);	*/
	NatAttCodes[12] = NL_ERROR_PROC;	/* int(*) ()	*/
/*	printf("NL_ERROR_PROC = %d\n",NL_ERROR_PROC);	*/
	NatAttCodes[13] = NL_ERROR_CODE;	/* type Nl_error*/
/*	printf("NL_ERROR_CODE = %d\n",NL_ERROR_CODE);	*/
	NatAttCodes[14] = NL_COLUMN_DATA;	/* type unsigned *	*/
/*	printf("NL_COLUMN_DATA = %d\n",NL_COLUMN_DATA);	*/
	NatAttCodes[15] = NL_COLUMN_TYPE;	/* type Nl_col_type	*/
/*	printf("NL_COLUMN_TYPE = %d\n",NL_COLUMN_TYPE);	*/
	NatAttCodes[16] = NL_RESPONSE_MODE;	/* type Nl_response_item*/
/*	printf("NL_RESPONSE_MODE = %d\n",NL_RESPONSE_MODE);	*/
	NatAttCodes[17] = NL_RESPONSE_TYPE;	/* type Nl_response_item*/
/*	printf("NL_RESPONSE_TYPE = %d\n",NL_RESPONSE_TYPE);	*/
	NatAttCodes[18] = NL_READ_IOSB;		/* unsigned short int VAXonly */
/*	printf("NL_READ_IOSB = %d\n",NL_READ_IOSB);	*/
	NatAttCodes[19] = NL_SYSERR_CODE;	/* int, OS error code	 */
/*	printf("NL_SYSERR_CODE = %d\n",NL_SYSERR_CODE);	*/
	NatAttCodes[20] = NL_SYSERR_STRING;	/* char *, OS err string */
/*	printf("NL_SYSERR_STRING = %d\n",NL_SYSERR_STRING);	*/
	NatAttCodes[21] = NL_WAIT_CHAN;		/* int IO descriptor	 */
/*	printf("NL_WAIT_CHAN = %d\n",NL_WAIT_CHAN);	*/

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatMode --
 *    Implements the nl_set command:
 *    usage: nli_mode handle mode_code ( 0 = text or 1 = data)
 *    switches between raw data mode and text only mode for tables
 *
 *    results:
 *      TCL_OK - handle is opened
 *      TCL_ERROR - wrong # args, or handle not opened
 */
int
Tcl_NatMode (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{

int hand;
int mode;
char    buf[NLI_BUFF_SIZE];

    if ((hand = nli_prologue(interp, argc, argv, 3, " handle get_code")) == -1)
 {
        return TCL_ERROR;
    }

    mode = atoi(argv[2]);
    if (mode) {
    nl_set(NatProcs[hand].nlc, NL_RESPONSE_MODE, NL_TABLE, 0);
    nl_set(NatProcs[hand].nlc, NL_RESPONSE_MODE, NL_TABLE_DATA, 1);
    sprintf(buf,"%s","data");
    Tcl_SetVar2(interp, NatMsgArray, "mode", buf, TCL_GLOBAL_ONLY);
    printf("set mode to data\n");
    } else {
    nl_set(NatProcs[hand].nlc, NL_RESPONSE_MODE, NL_TABLE, 1);
    nl_set(NatProcs[hand].nlc, NL_RESPONSE_MODE, NL_TABLE_DATA, 0);
    sprintf(buf,"%s","text");
    Tcl_SetVar2(interp, NatMsgArray, "mode", buf, TCL_GLOBAL_ONLY);
    printf("set mode to text\n");
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatTable --
 *    Implements the data-mode table commands... supersedes NatCols
 *    usage: nli_table handle
 *	NOTE this will only work if you are in DATA mode, not TEXT mode
 *	 	(see Tcl_NatMode)
 *	results:  return tablecount
 *		  if nonzero, then set rcount, colcount to 0 and return if no data
 *		  if data, set colcount, colnames, colhdrs, coltypes, colsizes
 *			in nlimsg
 *
 *    results:
 *      TCL_OK - handle is opened
 *      TCL_ERROR - wrong # args, or handle not opened
 */
int
Tcl_NatTable (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{

int hand;
Nl_status nlret;
Nl_col_type ctype;
char stype[5];
int ncols;
int  i, nt;
char    buf[NLI_BUFF_SIZE];
char    hdrbuf[NLI_BUFF_SIZE];
char    nambuf[NLI_BUFF_SIZE];
char    typbuf[NLI_BUFF_SIZE];
char    lenbuf[NLI_BUFF_SIZE];

    if ((hand = nli_prologue(interp, argc, argv, 2, " handle ")) == -1)
 {
        return TCL_ERROR;
    }

/*	get the table count, if 0 return 0			*/
   nt = nl_get(NatProcs[hand].nlc, NL_TABLE_COUNT);
/*
   printf("got table count %d from NLI\n",nt);
*/
    
    sprintf(buf,"%d",nt);
    Tcl_SetVar2(interp, NatMsgArray, "tabcount", buf, TCL_GLOBAL_ONLY);

   if (nt == 0) {
	nl_seterr(hand,interp);
    	Tcl_SetResult(interp,buf,TCL_VOLATILE);
	return TCL_OK;
   }

/*	if not 0, assume 1 (could be dangerous) and proceed:	*/

	nlret = nl_open(NatProcs[hand].nlc, NL_TABLE_DATA);

	if (nlret == NL_NO_MORE) {
		sprintf(buf,"%d",0);
		Tcl_SetVar2(interp, NatMsgArray, "rcount", buf, TCL_GLOBAL_ONLY);
		Tcl_SetVar2(interp, NatMsgArray, "colcount", buf, TCL_GLOBAL_ONLY);
		return TCL_OK;
	}

	ncols = nl_get(NatProcs[hand].nlc, NL_COLUMN_COUNT);

/*
	printf("got column count %d from NLI\n",ncols);
*/

	sprintf(buf,"%d",ncols);

        Tcl_SetVar2(interp, NatMsgArray, "colcount", buf, TCL_GLOBAL_ONLY);

	if (ncols == 0) {
		return TCL_OK;
	}

/*
	printf("Got past the set of colcount\n");
*/

/*	get the column headers						*/
    	sprintf(hdrbuf,"%s","");
    	sprintf(nambuf,"%s","");
    	sprintf(typbuf,"%s","");
    	sprintf(lenbuf,"%s","");

/*
	printf("Initialized the 4 buffers\n");
*/

	for (i=1; i<=ncols; i++) {

/*	    fprintf(stderr,"Looping:  i = %d\n",i);	*/
/*            fprintf(stderr,"%s  ", (char *)nl_get(NatProcs[hand].nlc, NL_COLUMN_HEADER, i));	*/
	    sprintf(hdrbuf,"%s {%s}",hdrbuf,(char *)nl_get(NatProcs[hand].nlc, NL_COLUMN_HEADER, i));
/*     	    fprintf(stderr,"Survived get header for col %d\n",i);	*/

/*            fprintf(stderr,"%s  ", (char *)nl_get(NatProcs[hand].nlc, NL_COLUMN_NAME, i));	*/
	    sprintf(nambuf,"%s {%s}",nambuf,(char *)nl_get(NatProcs[hand].nlc, NL_COLUMN_NAME, i));
/*     	    fprintf(stderr,"Survived get name for col %d\n",i);	*/

/*            fprintf(stderr,"%d  ", nl_get(NatProcs[hand].nlc, NL_COLUMN_WIDTH, i));	*/
	    sprintf(lenbuf,"%s %d",lenbuf, nl_get(NatProcs[hand].nlc, NL_COLUMN_WIDTH, i));
/*     	    fprintf(stderr,"Survived get length for col %d\n",i);	*/

            ctype = nl_get(NatProcs[hand].nlc, NL_COLUMN_TYPE, i);

/*	    fprintf(stderr,"Got col type hex %x, dec %d\n",ctype,ctype);  */	

	    switch (ctype) {
		 case NLCOL_INT:
                  sprintf(stype,"int");
                  break;
                case NLCOL_FLOAT:
                  sprintf(stype,"flt");
                  break;
                case NLCOL_CHAR:
                  sprintf(stype,"chr");
                  break;
                case NLCOL_DATE:
                  sprintf(stype,"dat");
                  break;
                default:
                  printf("Unknown datatype (%d)  ",ctype);
                  break;
              } /* end switch */

/*	    fprintf(stderr,"Survived get type for col %d: %s\n",i,stype); */
	    sprintf(typbuf,"%s {%s}",typbuf,stype);

	}

	printf(" ");

	Tcl_SetVar2(interp, NatMsgArray, "colhdrs", hdrbuf, TCL_GLOBAL_ONLY);
	Tcl_SetVar2(interp, NatMsgArray, "colnames", nambuf, TCL_GLOBAL_ONLY);
	Tcl_SetVar2(interp, NatMsgArray, "collengths", lenbuf, TCL_GLOBAL_ONLY);
	Tcl_SetVar2(interp, NatMsgArray, "coltypes", typbuf, TCL_GLOBAL_ONLY);

/*	return table count to Tcl function			*/

	nl_seterr(hand,interp);
    	sprintf(buf,"%d",nt);
    	Tcl_SetResult(interp,buf,TCL_VOLATILE);
        return TCL_OK;

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatRow --
 *    Implements the data-mode table commands...
 *    usage: nli_row handle
 *      NOTE this will only work if you are in DATA mode, not TEXT mode
 *              (see Tcl_NatMode)
 *      results:  return list of row values
 *
 *    results:
 *      TCL_OK - handle is opened
 *      TCL_ERROR - wrong # args, or handle not opened
 */
int
Tcl_NatRow (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
int hand, ncols, i, j;
Nl_status nlret;
Nl_col_type type;
char    buf[NLI_BUFF_SIZE];
char colval[256];
char line[32];
char *data;

/*
	fprintf(stderr,"start NatRow: addrs buf %x line %x colval %x START \n",
			buf, line, colval);
*/

	*buf = '\0';				
        *colval = '\0';
        *line = '\0';

/*
	fprintf(stderr,"addrs buf %x line %x colval %x bufINIT \n",
			buf, line, colval);
        fprintf(stderr,"Put a space into buf\n");	
*/

    if ((hand = nli_prologue(interp, argc, argv, 2, " handle")) == -1) {
	return TCL_ERROR;
    }

    nlret = nl_fetchrow(NatProcs[hand].nlc);

    if(nlret == NL_SUCCESS) {

/*
	fprintf(stderr,"Got a success on row fetch\n");
	fprintf(stderr,"addrs buf %x line %x colval %x gotRow \n",
			buf, line, colval);
*/

	ncols = nl_get(NatProcs[hand].nlc, NL_COLUMN_COUNT);

/*
	fprintf(stderr,"Got a count of cols %d \n",ncols);
*/

        for (i=1; i<=ncols; i++) {

/*
	fprintf(stderr,"addrs buf %x line %x colval %x iteration %d\n",
			buf, line, colval, i);
*/

	if (nl_get(NatProcs[hand].nlc, NL_COLUMN_NULL, i) != 0) {
              sprintf(colval," ");
/*
	      fprintf(stderr,"Got a null colval for col %d\n",i);
*/
        } else {

		type = nl_get(NatProcs[hand].nlc,NL_COLUMN_TYPE, i);
		data = (char *)nl_get(NatProcs[hand].nlc,NL_COLUMN_DATA, i);
	
/*
		fprintf(stderr,"address of data ptr col %d is %x\n",i,data);
		fprintf(stderr,"Got type code %x and data pointer %x\n",
			type, data);
*/

		switch(type) {
                case NLCOL_INT:
                  sprintf(colval,"%d", *(int *)data);
                  break;
                case NLCOL_FLOAT:
                  sprintf(colval,"%g", *(double *)data);
                  break;
                case NLCOL_CHAR:
                  sprintf(colval,"%s", data);
                  break;
                case NLCOL_DATE:
      		  for (j = 0 ; j < 32 ; j++) {
		  	line[j] = 0;
		  }
/*
		  fprintf(stderr,"col %d is a date type line size is %d\n",
			i,sizeof(line));
		  fprintf(stderr,"pre-datechar addrs buf %x line %x colval %x iteration %d\n",
			buf, line, colval, i);
*/
/*
		  sprintf(line,"A Date String");
*/
                  nli_datechar((Nl_datetime *)data, line, sizeof(line), 0);
/*
		  fprintf(stderr,"line is %x\n",line);
		  fprintf(stderr,"We survived the the datechar call\n");
		  fprintf(stderr,"i addr is %x\n",&i);
		  fprintf(stderr,"post-datechar addrs buf %x line %x colval %x iteration %d\n",
			buf, line, colval, i);
		  for (j=0;j<32;j++) fprintf(stderr,"line[%d] = %d;%c\n",
			j,line[j],line[j]);
		  fprintf(stderr,"date string is %s\n",line);
*/
                  sprintf(colval,"%s", line);
/*
  		  fprintf(stderr,"post-sprintfinto colval\n");
		  fprintf(stderr,"i addr is %x\n",&i);
*/
                  break;
                default:
                  fprintf(stderr,"Unknown datatype (%d)  ",type);
		  return TCL_ERROR;
		}

    }
/*
    fprintf(stderr,"Got a colvalue %s for col %d\n",colval,i);
*/
    sprintf(buf,"%s {%s}",buf,colval);

    }


    } else {

    sprintf(buf," ");

    }

    nl_seterr(hand,interp);
    Tcl_SetResult(interp,buf,TCL_VOLATILE);
    return TCL_OK;

}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_NatGet --
 *    Implements the nl_open command and nl_fetch commands in one:
 *    usage: nli_get handle keyword
 *     	     where keyword is one of:  echo, answer, table, prompt, query
 *	     return to tcl:  blank (null) if fail, otherwise results
 *	be aware that NLI responses are ordered and that nl_open can only
 *	access them in their order.  the order is generally ECHO, ANSWER,
 *	TABLE (there may also be some TEXT here and there).  If you go for
 *	ANSWER, you won't get back to ECHO on this query.
 *
 *      TCL_OK - handle is opened
 *      TCL_ERROR - wrong # args, or handle not opened
 */

int
Tcl_NatGet (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{

    int     hand;
    char    buf[NLI_BUFF_SIZE];
    char    line[256];
    Nl_status nlret;
    Nl_response_item nlres;


    if ((hand = nli_prologue(interp, argc, argv, 3, " handle")) == -1) {
        return TCL_ERROR;
    }

    nlres = 0;

    if (strcmp(argv[2],"echo") == 0) nlres = NL_ECHO;
    else if (strcmp(argv[2],"table") == 0) nlres = NL_TABLE;
    else if (strcmp(argv[2],"query") == 0) nlres = NL_QUERY;
    else if (strcmp(argv[2],"prompt") == 0) nlres = NL_PROMPT;
    else if (strcmp(argv[2],"answer") == 0) nlres = NL_ANSWER;
    else
    fprintf(stderr,"Sorry, %s is not a valid option.\nUse one of: echo, table, query, prompt, answer\n",argv[2]);
   

    nlret = nl_open(NatProcs[hand].nlc,nlres);

    if (nlret != NL_SUCCESS) {
	buf[0] = 0;
        nl_seterr(hand,interp);
        Tcl_SetResult(interp,buf,TCL_VOLATILE);
	return TCL_ERROR;
    }


    while (nl_fetch(NatProcs[hand].nlc, nlres, line, sizeof line, NULL) == NL_SUCCESS) {
        Tcl_AppendElement(interp, line);
   }

    nl_seterr(hand,interp);
    return TCL_OK;

}

Nl_status
nli_datechar(date, ans, len, format)
Nl_datetime     *date;
char            *ans;
int             len;
char            *format;
{
        char            buf[25];
        char            *p;
        static char     months[12][4] = { "jan", "feb", "mar", "apr", "may",
                              "jun", "jul", "aug", "sep", "oct", "nov", "dec" };
        
        p= buf;
        if(date->month!=0) {
                sprintf(p, "%02d-%.3s-%d", date->day, months[date->month-1],
                        date->year);
                for(; *p; p++);
                }
        sprintf(p, " %02d:%02d:%06.3f", date->hour, date->minute, date->second);
        strncpy(ans, buf, len);
        return NL_SUCCESS;
}

/* finis */
