/*  _ __ ___ _
 * | |\ /  /| |  $Id: OtclMethod.C,v 1.5 1995/07/24 10:10:19 deans Exp $
 * | | /  / | |  Copyright (C) 1995 IXI Limited.
 * |_|/__/_\|_|  IXI Limited, Cambridge, England.
 *
 * Component   : OtclMethod.C
 *
 * Author      : Dean Sheehan (deans@x.co.uk)
 *
 * Description : Contains the implementation of OtclMethod class and subclasses.
 *
 * License     :
			Object Tcl License & Copyright
			------------------------------

IXI Object Tcl software, both binary and source (hereafter, Software) is copyrighted by IXI Limited (IXI), and ownership remains with IXI. 

IXI grants you (herafter, Licensee) a license to use the Software for academic, research and internal business purposes only, without a fee. Licensee may distribute the binary and source code (if required) to third parties provided that the copyright notice and this statement appears on all copies and that no charge is associated with such copies. 

Licensee may make derivative works. However, if Licensee distributes any derivative work based on or derived from the Software, then Licensee will (1) notify IXI regarding its distribution of the derivative work, and (2) clearly notify users that such derivative work is a modified version and not the original IXI Object Tcl distributed by IXI. IXI strongly recommends that Licensee provide IXI the right to incorporate such modifications into future releases of the Software under these license terms. 

Any Licensee wishing to make commercial use of the Software should contact IXI, to negotiate an appropriate license for such commercial use. Commercial use includes (1) integration of all or part of the source code into a product for sale or license by or on behalf of Licensee to third parties, or (2) distribution of the binary code or source code to third parties that need it to utilize a commercial product sold or licensed by or on behalf of Licensee. 

IXI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE. IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY. IXI SHALL NOT BE LIABLE FOR ANY DAMAGES WHATSOEVER SUFFERED BY THE USERS OF THIS SOFTWARE. 

Copyright (C) 1995, IXI Limited 

By using or copying this Software, Licensee agrees to abide by the copyright law and all other applicable laws of England and the U.S., including, but not limited to, export control laws, and the terms of this license. IXI shall have the right to terminate this license immediately by written notice upon Licensee's breach of, or non-compliance with, any of its terms. Licensee may be held legally responsible for any copyright infringement that is caused or encouraged by Licensee's failure to abide by the terms of this license. 

Comments and questions are welcome and can be sent to
otcl@x.co.uk 

For more information on copyright and licensing issues, contact: 
Legal Department, IXI Limited, Vision Park, Cambridge CB4 4ZR,
ENGLAND. 

 *
 */

// Tcl Includes
#include <tclInt.h>

// Local Includes
#include "OtclError.H"
#include "Otcl.H"
#include "OtclMethod.H"
#include "OtclFormalArg.H"
#include "OtclPart.H"

OtclMethod::OtclMethod (char *n, OtclMethod::Access a, OtclClassOtcl *o)
{
   name = (char*)malloc(strlen(n) + strlen(o->giveName()) + 3);
   sprintf(name,"%s::%s",o->giveName(),n);
   access = a;
   otclClass = o;
   body = NULL;
   formalArgHead = NULL;
   formalArgTail = NULL;
   currentFormalArg = NULL;
   formalArgsSpecified = OTCL_FALSE;
}

OtclMethod::~OtclMethod ()
{
   free(name);
   if (body != NULL)
   {
      free(body);
   }

   // Cleanup FormArgs...
   OtclFormalArg *current = formalArgHead;
   OtclFormalArg *previous = NULL;
   while (current != NULL)
   {
      previous = current;
      current = current->getNext();
      delete previous;
   }
}

int OtclMethod::setFormalArgs (Tcl_Interp *interp, char *args)
{
   if (formalArgsSpecified == OTCL_TRUE)
   {
      return overlayFormalArgs(interp,args);
   }
   formalArgsSpecified = OTCL_TRUE;

   // Break args into argument specifiers.
   int argc;
   char **argv;
   if (Tcl_SplitList(interp,args,&argc,&argv) == TCL_ERROR)
   {
      return TCL_ERROR;
   }

   // Iterate over the argument specifiers
   OtclFormalArg *formalArg;
   int innerArgc;
   char **innerArgv;
   for (int arg = 0; arg < argc; arg++)
   {
      // Break the arg specifier into argument name and default value
      if (Tcl_SplitList(interp,argv[arg],&innerArgc,&innerArgv) == TCL_ERROR)
      {
         free((char*)argv);
         return TCL_ERROR;
      }

      if (innerArgc == 0)
      {
         Otcl::setTclResult(interp,SYNTAX_ERROR_IN_FORMAL_ARG_ERR,arg+1,name,
                            argv[arg]);
         free((char*)argv);
         free((char*)innerArgv);
         return TCL_ERROR;
      }
 
      if (innerArgc > 2)
      {
         // Too many parts to argument specifier
         Otcl::setTclResult(interp,SYNTAX_ERROR_IN_FORMAL_ARG_ERR,arg+1,name,
                            argv[arg]);
         free((char*)argv);
         free((char*)innerArgv);
         return TCL_ERROR;
      }

      if (argExists(innerArgv[0],NULL))
      {
         Otcl::setTclResult(interp,DUP_FORMAL_ARGUMENT_ERR,innerArgv[0],name);
         free((char*)argv);
         free((char*)innerArgv);
         return TCL_ERROR;
      }

      if (innerArgc == 1)
      {
         formalArg = new OtclFormalArg(innerArgv[0]);
      }
      else
      {
         formalArg = new OtclFormalArg(innerArgv[0],innerArgv[1]);
      }

      // Link it into methods formal argument list
      if (formalArgHead == NULL)
      {
         formalArgHead = formalArg;
      }
      else
      {
         formalArgTail->setNext(formalArg);
      }
      formalArgTail = formalArg;

      free((char*)innerArgv);
      
   } 

   free((char*)argv);

   return TCL_OK;
}

int OtclMethod::overlayFormalArgs (Tcl_Interp *interp, char *args)
{
   // Break args into argument specifiers.
   int argc;
   char **argv;
   if (Tcl_SplitList(interp,args,&argc,&argv) == TCL_ERROR)
   {
      return TCL_ERROR;
   }

   // Iterate over the argument specifiers
   OtclFormalArg *formalArg = formalArgHead;
   int innerArgc;
   char **innerArgv;
   for (int arg = 0; arg < argc; arg++)
   {

      if (formalArg == NULL)
      {
         free((char*)argv);
         Otcl::setTclResult(interp,DIFF_NO_OF_FORMAL_ARGS_IN_SPECS_ERR,name);
         return TCL_ERROR;
      }

      // Break the arg specifier into argument name and default value
      if (Tcl_SplitList(interp,argv[arg],&innerArgc,&innerArgv) == TCL_ERROR)
      {
         free((char*)argv);
         return TCL_ERROR;
      }

      if (innerArgc == 0)
      {
         Otcl::setTclResult(interp,SYNTAX_ERROR_IN_FORMAL_ARG_ERR,arg+1,name,
                            argv[arg]);
         free((char*)argv);
         free((char*)innerArgv);
         return TCL_ERROR;
      }
 
      if (innerArgc > 1)
      {
         Otcl::setTclResult(interp,PUBLIC_IMP_DEFAULT_FORMAL_ARGS_ERR,name);
         free((char*)argv);
         free((char*)innerArgv);
         return TCL_ERROR;
      }

      if (argExists(innerArgv[0],formalArg))
      {
         Otcl::setTclResult(interp,DUP_FORMAL_ARGUMENT_ERR,innerArgv[0],name);
         free((char*)argv);
         free((char*)innerArgv);
         return TCL_ERROR;
      }

      // Specify the new name of the formal arg
      formalArg->setName(innerArgv[0]);

      formalArg = formalArg->getNext();

      free((char*)innerArgv);
   } 

   if (formalArg != NULL)
   {
      free((char*)argv);
      Otcl::setTclResult(interp,DIFF_NO_OF_FORMAL_ARGS_IN_SPECS_ERR,name);
      return TCL_ERROR;
   }

   free((char*)argv);

   return TCL_OK;
}

int OtclMethod::argExists (char *n, OtclFormalArg *end)
{
   OtclFormalArg *ofa = formalArgHead;
   while (ofa != end)
   {
      if (strcmp(n,ofa->getName()) == 0)
      {
         return OTCL_TRUE;
      }
      ofa = ofa->getNext();
   }
   return OTCL_FALSE;
}

int OtclMethod::setBody (Tcl_Interp *interp, char *b)
{
   if (body != NULL)
   {
      Otcl::setTclResult(interp,BODY_ALREADY_SPECIFIED_ERR,name);
      return TCL_ERROR;
   }

   body = strdup(b);
   return TCL_OK;
}

int OtclMethod::hasBody (void)
{
   return (body == NULL ? OTCL_FALSE : OTCL_TRUE);
}

int OtclMethod::execute (Tcl_Interp *interp, int argc, char *argv[])
{
   if (bindActualToFormal(interp,argc,argv) != TCL_OK)
   {
      return TCL_ERROR;
   }

   return evaluateMethodBody(interp);
}

int OtclMethod::bindActualToFormal (Tcl_Interp *tclInterp,
                                    int argc, char *argv[])
{
   // Cast the Tcl publicised interp structur to the private (actual)
   // structure
   Interp *interp = (Interp*)tclInterp;
   CallFrame *frame = interp->framePtr;
   frame->argc = argc;
   frame->argv = argv;

   OtclFormalArg *current = formalArgHead;
   int argIndex = 0;
   while (current != NULL)
   {
      // if args finished then run over rest to fill in defaults
      if (argIndex == argc)
      {
         while (current != NULL)
         {
            if (current->createLocal(tclInterp,frame) != TCL_OK)
            {
               char msg[256];
               sprintf(msg,CALLING_INFO_ERR,name, interp->errorLine);
               Tcl_AddErrorInfo(tclInterp,msg);
               return TCL_ERROR;
            }
            current = current->getNext();
         }
         break;
      }

      // If last formal arg and it is called 'args' it take the
      // remainder of the actual args
      if (current->getNext() == NULL && current->takesRemainingActuals())
      {
         if (current->createLocal(tclInterp,frame,(argc - argIndex),
                                &argv[argIndex]) != TCL_OK)
         {
            char msg[256];
            sprintf(msg,CALLING_INFO_ERR,name, interp->errorLine);
            Tcl_AddErrorInfo(tclInterp,msg);
            return TCL_ERROR;
         }
         argIndex = argc; // As we have used up all the args!
         break;
      }

      // Finaly it looks like we have a normal case :-)
      if (current->createLocal(tclInterp,frame,argv[argIndex]) != TCL_OK)
      {
         char msg[256];
         sprintf(msg,CALLING_INFO_ERR,name,interp->errorLine);
         Tcl_AddErrorInfo(tclInterp,msg);
         return TCL_ERROR;
      }

      argIndex++;
      current = current->getNext();
   }

   // if args left then error
   if (argIndex != argc)
   {
      Otcl::setTclResult(tclInterp,TOO_MANY_ARGS_TO_METHOD_ERR,name);

      // Context
      char msg[256];
      sprintf(msg,CALLING_INFO_ERR,name,interp->errorLine);
      Tcl_AddErrorInfo(tclInterp,msg);
      return TCL_ERROR;
   }

   return TCL_OK;
}

int OtclMethod::evaluateMethodBody (Tcl_Interp *tclInterp)
{
   Interp *interp = (Interp*)tclInterp;

   int result = Tcl_Eval(tclInterp,body);

   char msg[256];

   switch (result)
   {
     case TCL_ERROR: 
        // Record information telling where the error occurred.
        sprintf(msg,CALLING_INFO_ERR,name, interp->errorLine);
        Tcl_AddErrorInfo(tclInterp,msg);
        break;

     case TCL_RETURN:
        result = interp->returnCode;
        interp->returnCode = TCL_OK;
        if (result == TCL_ERROR)
        {
           Tcl_SetVar2(tclInterp,"errorCode",NULL,
                     (interp->errorCode != NULL) ? interp->errorCode : "NONE",
                     TCL_GLOBAL_ONLY);
            interp->flags |= ERROR_CODE_SET;
            if (interp->errorInfo != NULL)
            {
               Tcl_SetVar2(tclInterp,"errorInfo",NULL,
                           interp->errorInfo, TCL_GLOBAL_ONLY);
               interp->flags |= ERR_IN_PROGRESS;
            }
        }
        break;

     case TCL_BREAK:
        Otcl::setTclResult(tclInterp,BREAK_NOT_IN_LOOP_ERR);
        result = TCL_ERROR;
        break;

     case TCL_CONTINUE:
        Otcl::setTclResult(tclInterp,CONTINUE_NOT_IN_LOOP_ERR);
        result = TCL_ERROR;
        break;
   }

   return result;
}

int OtclMethod::isAccessible (Tcl_Interp *interp)
{
   if (access == PUBLIC)
   {
      return OTCL_TRUE;
   }

   // Look at the _otcl_class_ variable in the top interp frame
   // if it exists and it point to the class owning this method then
   // its accessible :-)
   char *callingClass = Tcl_GetVar2(interp,OTCL_CLASS_VARIABLE_NAME,NULL,0);
   if (callingClass != NULL)
   {
      OtclClass *callingClassPtr;
      sscanf(callingClass,"%lx",(long**)&callingClassPtr);
      if (callingClassPtr == otclClass)
      {
         return OTCL_TRUE;
      }
   }
   
   return OTCL_FALSE;
}

char *OtclMethod::giveFirstFormalArgName (void)
{
   currentFormalArg = formalArgHead;
   if (currentFormalArg != NULL)
   {
      return currentFormalArg->getName();
   }
   return NULL;
}

char *OtclMethod::giveNextFormalArgName (void)
{
   if (currentFormalArg != NULL)
   {
      currentFormalArg = currentFormalArg->getNext();
   }
   if (currentFormalArg != NULL)
   {
      return currentFormalArg->getName();
   }
   return NULL;
}

OtclInstanceMethod::OtclInstanceMethod (char *n, OtclMethod::Access a,
                                        OtclClassOtcl *o) :
   OtclMethod(n,a,o)
{
}

OtclInstanceMethod::~OtclInstanceMethod ()
{
}

OtclConstructorMethod::OtclConstructorMethod (OtclClassOtcl *o) :
   OtclInstanceMethod(OTCL_CONSTRUCTOR_METHOD_NAME,OtclMethod::PUBLIC,o)
{
   for (int s = 0; s < MAX_SUPERCLASSES; s++)
   {
      parentConstructorArgs[s] = NULL;
   }
}

OtclConstructorMethod::~OtclConstructorMethod ()
{
   for (int s = 0; s < MAX_SUPERCLASSES; s++)
   {
      if (parentConstructorArgs[s] != NULL)
      {
         free((char*)parentConstructorArgs[s]);
      }
   }
}

int OtclConstructorMethod::setParentConstructors (Tcl_Interp *interp,
                                                  char *constructorStr)
{
   char **argv;
   int argc;
   if (Tcl_SplitList(interp,constructorStr,&argc,&argv) == TCL_ERROR)
   {
      return TCL_ERROR;
   }

   int index;
   for (int i = 0; i < argc; i++)
   {
      // The first word in the string is the name of the parent class.
      char *endOfParent = strpbrk(argv[i]," \n\r\t");
      if (endOfParent == NULL)
      {
         Otcl::setTclResult(interp,SYNTAX_ERROR_PARENT_CONSTRUCTOR_ERR,argv[i],
                            otclClass->giveName());
         free((char*)argv);
         return TCL_ERROR;
      }

      *endOfParent = (char)NULL;
      if ((index = otclClass->giveIndexOfSuperclass(argv[i])) == -1)
      {
         Otcl::setTclResult(interp,PARENT_CONSTRUCTOR_UNKNOWN_CLASS_ERR,
                            otclClass->giveName(),argv[i]);
         free((char*)argv);
         return TCL_ERROR;
      }

      if (parentConstructorArgs[index] != NULL)
      {
         // Duplicate call to parent class
         Otcl::setTclResult(interp,DUP_CONSTRUCTOR_ERR,argv[i],
                            otclClass->giveName());
         free((char*)argv);
         return TCL_ERROR;
      }

      parentConstructorArgs[index] = strdup(endOfParent + 1);
   }

   free((char*)argv);
   return TCL_OK;
}

int OtclConstructorMethod::execute (Tcl_Interp *interp, int argc, char *argv[],
                                    OtclPartOtcl *part)
{
   if (bindActualToFormal(interp,argc,argv) != TCL_OK)
   {
      return TCL_ERROR;
   }

   // invoke parts parent constructors
   for (int s = 0; s < MAX_SUPERCLASSES; s++)
   {
      if (part->constructParent(s,interp,parentConstructorArgs[s]) != TCL_OK)
      {
         return TCL_ERROR;
      }
   }

   return evaluateMethodBody(interp);
}

OtclDestructorMethod::OtclDestructorMethod (OtclClassOtcl *o) :
   OtclInstanceMethod(OTCL_DESTRUCTOR_METHOD_NAME,OtclMethod::PUBLIC,o)
{
}

OtclDestructorMethod::~OtclDestructorMethod ()
{
}

int OtclDestructorMethod::setFormalArgs (Tcl_Interp *interp, char *)
{
   Otcl::setTclResult(interp,DESTRUCTOR_CANNOT_HAVE_ARGS_ERR,
                      otclClass->giveName());
   return TCL_ERROR;
}

OtclClassMethod::OtclClassMethod (char *n, OtclMethod::Access a,
                                  OtclClassOtcl *o) :
   OtclMethod(n,a,o)
{
}

OtclClassMethod::~OtclClassMethod ()
{
}
