/*
 * $Id: rl.c,v 1.10 1995/07/25 07:59:52 dimka Exp $
 *
 * $Log: rl.c,v $
 * Revision 1.10  1995/07/25  07:59:52  dimka
 * memory leak found
 *
 * Revision 1.9  1995/07/20  10:57:36  dimka
 * rl_event_handler is now variable which contains
 * actual handler name.
 *
 * Revision 1.8  1995/07/18  22:00:37  dimka
 * Draft Release
 *
 * Revision 1.7  1995/05/28  15:46:03  dimka
 * rl string global variables tracing is changed .
 *
 * Revision 1.6  1995/05/28  15:24:33  dimka
 * bug fixes
 *
 * Revision 1.5  1995/05/28  14:48:55  dimka
 * Many bug fixes and some variable additions
 *
 * Revision 1.4  1995/05/27  21:30:04  dimka
 * Many changes .
 * * Info command is added
 * * undo support is added
 * etc...
 *
 * Revision 1.3  1995/05/26  21:16:18  dimka
 * Some commands are added
 *
 * Revision 1.2  1995/05/26  19:26:27  dimka
 * Key Bindings support is added
 *
 * Revision 1.1  1995/05/18  16:51:33  dimka
 * Initial revision
 *
 *
 * Copyright (C) 1995 Dimitry Kloper . Technion . 
 *
 * rl.c -- this file contains main procedures for readline support for tcl
 * 
 */

#include <tcl.h>
#include <string.h>
#include <stdlib.h>
#include <readline.h>
#include "rl.h"

static
char *rl_instream_name,*rl_outstream_name;

extern int rl_point,rl_end,rl_mark,rl_done,rl_pending_input,rl_completion_query_items;
extern char *rl_line_buffer,*rl_terminal_name,*rl_readline_name;
extern int rl_ignore_completion_duplicates;

#ifdef STRONG_BINDINGS
extern Function *rl_kbdkmap_hook;
#endif

static
Tcl_HashEntry *Rl_RegisterKeymap _ANSI_ARGS_ ((Tcl_Interp *interp,Rl_KeymapTable *keymaps,
					       Keymap map,char* name));
static
int Rl_ManageKeyHook _ANSI_ARGS_((int count,int key));


/* This is bad idea to use global interp , but 
   it used only in Rl_ManageKey* methods where , no 
   interp parameter is passed */

Tcl_Interp *rl_interp;


/* list of integer variables of rl accessed from tcl
   each entry contains : 
        * string with variable name
	* pointer to the variable
	* boolean - true if the variable is writable from tcl
*/
static
Rl_IntVars rl_IntVars[] = { 
  {"rl_point",&rl_point,1} ,
  {"rl_end",&rl_end,1} ,
  {"rl_mark",&rl_mark,1} ,
  {"rl_done",&rl_done,1} ,
  {"rl_pending_input",&rl_pending_input,1} , 
  {"rl_completion_query_items",&rl_completion_query_items,1} ,
  {"rl_ignore_completion_duplicates",&rl_ignore_completion_duplicates,1} ,
  {"rl_filename_completion_desired",&rl_filename_completion_desired,1} ,
  {"rl_filename_quoting_desired",&rl_filename_quoting_desired,1} ,
  {NULL,NULL,1}
};

/* same here for all rl strings */
static
Rl_StrVars rl_StrVars[] = {
  {"rl_line_buffer",&rl_line_buffer,0} ,
  {"rl_terminal_name",&rl_terminal_name,0} ,
  {"rl_readline_name",&rl_readline_name,1} ,
  {"rl_instream",&rl_instream_name,0} ,
  {"rl_outstream",&rl_outstream_name,0} ,
  {"rl_basic_word_break_characters",&rl_basic_word_break_characters,1} ,
  {"rl_completer_word_break_characters",&rl_completer_word_break_characters,1} ,
  {"rl_special_prefixes",&rl_special_prefixes,1} ,
  {"rl_completer_quote_characters",&rl_completer_quote_characters,1} ,
  {NULL,NULL,0}
};

static
char *rl_event_hook_name = NULL;

/* This is table of all used hooks used here */
static
Rl_Hooks rl_hooks[] = {
  {"rl_event_hook",&rl_event_hook_name,&rl_event_hook,Rl_ProceedEvent} ,
  {NULL,NULL,0}
};

/* list of names for default keymaps */
static
char *rl_KeymapNames[] = { 
  "emacs" ,
  "emacs-standard" , 
  "emacs-meta" , 
  "emacs-ctlx" ,
  "vi" ,
  "vi-move" ,
  "vi-command" ,
  "vi-insert" ,
  NULL
};

/* hash table for keymaps , its global because Rl_ManageKey* methods */
static
Rl_KeymapTable *rl_KeymapTable;

#ifdef NEED_LTOA

char *ltoa(x)
long x;
{
  static char tmp[20]; /* really array  must have size  log10(MAXLONG)+1
                                  but I hope this will work */
  *tmp = '\0';

  sprintf(tmp,"%ld",x);
  return(tmp);
}

#endif

#ifdef NEED_STRDUP

char *strdup(str)
char *str;
{
  char *cpy;
  if(str == NULL) return NULL;
  if((cpy = ckalloc(strlen(str)+1)) == NULL) return NULL;
  strcpy(cpy,str);
  return(cpy);
}

#endif
#ifdef NEED_XMALLOC

void *xmalloc(size) 
size_t size;
{
  return(ckalloc(size));
}

void *xrealloc(ptr,size) 
void *ptr;
size_t size;
{
  return(ckrealloc(ptr,size));
}

void xfree(mem)
void* mem;
{
  ckfree(mem);
}

#endif

/*
 * Init function
 */
int Rl_Init(interp)
     Tcl_Interp *interp; /* The current Tcl interpreter */
{
  char *file,**keymap;
  Rl_IntVars *siv;
  Rl_StrVars *ssv;
  Rl_Hooks *hook;

  /* readline() calls this first time it called , but we need this
     already done , because key bindings can be done before first readline() */
  rl_initialize();

  rl_KeymapTable=(Rl_KeymapTable *)ckalloc(sizeof(Rl_KeymapTable));

  rl_interp = interp;

  Tcl_SetVar(interp,"rl_version","$Source: /s/dimka/tclrl/RCS/rl.c,v $ $Revision: 1.10 $",
	     TCL_GLOBAL_ONLY);

  rl_attempted_completion_function = (CPPFunction*)Rl_AlternativeComplete;
  rl_completion_entry_function = (Function*)Rl_CompleteGenerator;
#ifdef STRONG_BINDINGS
  rl_kbdkmap_hook = (Function*)Rl_ManageKeyHook;
  /* create tcl variable rl_strong_bindings for ability detect 
     the feature from the script */
  Tcl_SetVar(interp,"rl_strong_bindings","1",TCL_GLOBAL_ONLY);
#endif
  if(rl_KeymapTable == NULL) {
    Tcl_AppendResult(interp,"Can't allocate memory for rl_KeymapTable",(char*)NULL);
    return TCL_ERROR;
  }

  rl_KeymapTable->count = 0;
  Tcl_InitHashTable(&(rl_KeymapTable->table),TCL_STRING_KEYS);

  if(rl_instream != NULL) {
    Tcl_EnterFile(interp,rl_instream,TCL_FILE_READABLE);
    rl_instream_name = (char*)ckalloc(strlen(interp->result)+1);
    strcpy(rl_instream_name,interp->result);
  } else 
    rl_instream_name = "stdin";
	       
  if(rl_outstream != NULL) {
    Tcl_EnterFile(interp,rl_outstream,TCL_FILE_WRITABLE);
    rl_outstream_name = (char*)ckalloc(strlen(interp->result)+1);
    strcpy(rl_outstream_name,interp->result);
  } else
    rl_outstream_name = "stdout";

  for(siv = rl_IntVars; siv->name != NULL ; siv++) {
    Tcl_SetVar(interp,siv->name,"0",TCL_GLOBAL_ONLY);
    Tcl_TraceVar(interp,siv->name,TCL_GLOBAL_ONLY | TCL_TRACE_READS | TCL_TRACE_WRITES,
		 Rl_TraceInt,(ClientData)NULL);
  }

  for(ssv = rl_StrVars; ssv->name != NULL ; ssv++) {
    Tcl_SetVar(interp,ssv->name,"",TCL_GLOBAL_ONLY);
    Tcl_TraceVar(interp,ssv->name,TCL_GLOBAL_ONLY | TCL_TRACE_READS | TCL_TRACE_WRITES,
		 Rl_TraceStr,(ClientData)NULL);
  }

  for(hook = rl_hooks; hook->name != NULL ; hook++) {
    Tcl_TraceVar(interp,hook->name,TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS,
		 Rl_TraceHook,(ClientData)NULL);
  }

  for(keymap = rl_KeymapNames; *keymap != NULL ; keymap++) {
    Keymap map = rl_get_keymap_by_name(*keymap);
    if(map != NULL) 
      if(Rl_RegisterKeymap(interp,rl_KeymapTable,map,*keymap) == NULL) return TCL_ERROR;
  }

  Tcl_CreateCommand(interp,"readline",Rl_ReadlineCmd,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_make_bare_keymap",Rl_MkBareKeymapCmd,
		    (ClientData)rl_KeymapTable,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_make_keymap",Rl_MkKeymapCmd,
		    (ClientData)rl_KeymapTable,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_copy_keymap",Rl_CopyKeymapCmd,
		    (ClientData)rl_KeymapTable,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_discard_keymap",Rl_DelKeymapCmd,
		    (ClientData)rl_KeymapTable,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_get_keymap",Rl_GetKeymapCmd,
		    (ClientData)rl_KeymapTable,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_set_keymap",Rl_SetKeymapCmd,
		    (ClientData)rl_KeymapTable,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_get_keymap_by_name",Rl_GetByNameKeymapCmd,
		    (ClientData)rl_KeymapTable,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_bind",Rl_BindCmd,
		    (ClientData)rl_KeymapTable,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_parse",Rl_Parse_n_BindCmd,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_message",Rl_MessageCmd,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_display",Rl_DisplayCmd,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_text",Rl_TextCmd,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_info",Rl_InfoCmd,
		    (ClientData)rl_KeymapTable,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_undo",Rl_UndoCmd,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_matches",Rl_CallCompletionMatchesCmd,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_complete_internal",Rl_CompleteInternalCmd,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);

#ifdef WITH_LIBES_DBG
  Tcl_CreateCommand(interp,"rl_dbg",Rl_DbgActiveCmd,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_setDbgInteractor",Rl_SetDbgInteractorCmd,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_setDbgIgnoreFuncs",Rl_SetDbgIgnoreFuncCmd,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,"rl_setDbgArgs",Rl_DbgArgcArgvCmd,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
#endif
  return TCL_OK;
}


/*
 * Rl_ProceedEvent() -- this is event hook for the extension . It 
 * searches for tcl procedure 'rl_event_hook' . If it exists 
 * runs it without parameters .
 *
 */
int Rl_ProceedEvent(count,key)
int count;
int key;
{
  Tcl_CmdInfo info;

  if(!rl_event_hook_name) return -1;

  if(Tcl_GlobalEval(rl_interp,rl_event_hook_name) == TCL_ERROR) {
    return -1;
  }

  return 0;
}

/*
 * Simple interface to readline() function :
 *   readline ?prompt?
 *
 */

int Rl_ReadlineCmd(cdata, interp, argc, argv)
     ClientData cdata;                   /* Client Data */
     Tcl_Interp *interp;                 /* Current interpreter. */
     int argc;                           /* Number of arguments. */
     char **argv;                        /* Argument strings. */
{
 
 char *res;

  if(argc < 1 || argc > 2) {
    Tcl_AppendResult(interp,"wrong # of args , should be \"",argv[0]," ?prompt?\"",
		     (char*)NULL);
    return TCL_ERROR;
  }

  if((res = readline((argc == 2) ? argv[1] : "")) == NULL) {
    Tcl_AppendResult(interp,"readline EOF",(char*)NULL);
    return TCL_ERROR;
  }
  else
    Tcl_SetResult(interp,res,TCL_DYNAMIC);
  
  return TCL_OK;
}




/*
 *  PART I : Mastering Keymaps
 *
 */


/*
 * Rl_RegisterKeymap - obtain keymap and register it in hash table
 *
 */
static
Tcl_HashEntry *Rl_RegisterKeymap(interp,keymaps,map,name)
     Tcl_Interp *interp; 
     Rl_KeymapTable *keymaps;
     Keymap map;
     char *name; 
{
  static char* prefix = "keymap";
  char *tmp,*num;
  Rl_KeymapEntry *rl_KeymapEntry;
  Tcl_HashEntry *newEntry;
  int isOk;

  if(name == NULL) { 
    if(++keymaps->count < 0) {
      Tcl_AppendResult(interp,"Can't allocate id for keymap",(char*)NULL);
      return NULL;
    }
    
    num = ltoa(keymaps->count);
    if((tmp = (char*)ckalloc(strlen(prefix)+strlen(num))+1) == NULL) {
      Tcl_AppendResult(interp,"Can't allocate memory",(char*)NULL);
      return NULL;
    }
    sprintf(tmp,"%s%s",prefix,num);
  }
  else tmp = name;

  if((rl_KeymapEntry = (Rl_KeymapEntry*)ckalloc(sizeof(Rl_KeymapEntry))) == NULL) {
    Tcl_AppendResult(interp,"Can't allocate memory",(char*)NULL);
    return NULL;    
  }
  
  rl_KeymapEntry->map = map;
  Tcl_InitHashTable(&(rl_KeymapEntry->table),TCL_STRING_KEYS);

      
  newEntry = Tcl_CreateHashEntry(&(keymaps->table),tmp,&isOk);
  if(!isOk) {
    Tcl_AppendResult(interp,"Keymap \"",tmp,"\" already exists",(char*)NULL);
    return NULL;
  }

  Tcl_SetHashValue(newEntry,(ClientData)rl_KeymapEntry);
  Tcl_SetResult(interp,tmp,TCL_DYNAMIC);
  return newEntry;  
}

/*
 * Rl_MkBareKeymapCmd() -- interface to rl_make_bare_keymap() . Syntax :
 *  rl_make_bare_keymap ?name?
 * it receives pointer to Rl_KeymapTable structure as cdata
 */
int Rl_MkBareKeymapCmd(cdata, interp, argc, argv)
     ClientData cdata;                   /* Client Data */
     Tcl_Interp *interp;                 /* Current interpreter. */
     int argc;                           /* Number of arguments. */
     char **argv;                        /* Argument strings. */
{
  Rl_KeymapTable *keymaps = (Rl_KeymapTable *)cdata;
  Keymap map;
  char *name = NULL;

  if(argc < 1 || argc > 2) {
    Tcl_AppendResult(interp,"wrong # of args , should be \"",argv[0]," ?name?\"",(char*)NULL);
    return TCL_ERROR;
  }

  if(argc == 2) name = argv[1];

  map = rl_make_bare_keymap();
  if(map == NULL) {
    Tcl_AppendResult(interp,"Can't allocate keymap",(char*)NULL);
    return TCL_ERROR;
  } 
    
  return (Rl_RegisterKeymap(interp,keymaps,map,name) == NULL)?TCL_ERROR:TCL_OK;
}

/*
 * Rl_MkKeymapCmd() -- interface to rl_make_keymap() . Syntax :
 *  rl_make_keymap ?name?
 * it receives pointer to Rl_KeymapTable structure as cdata
 */
int Rl_MkKeymapCmd(cdata, interp, argc, argv)
     ClientData cdata;                   /* Client Data */
     Tcl_Interp *interp;                 /* Current interpreter. */
     int argc;                           /* Number of arguments. */
     char **argv;                        /* Argument strings. */
{
  Rl_KeymapTable *keymaps = (Rl_KeymapTable *)cdata;
  Keymap map;
  char *name = NULL;

  if(argc < 1 || argc > 2) {
    Tcl_AppendResult(interp,"wrong # of args , should be \"",argv[0]," ?name?\"",(char*)NULL);
    return TCL_ERROR;
  }

  if(argc == 2) name = argv[1];

  map = rl_make_keymap();
  if(map == NULL) {
    Tcl_AppendResult(interp,"Can't allocate keymap",(char*)NULL);
    return TCL_ERROR;
  } 
    
  return (Rl_RegisterKeymap(interp,keymaps,map,name) == NULL)?TCL_ERROR:TCL_OK;
}

/*
 * Rl_CopyKeymapCmd() -- interface to rl_copy_keymap() . Syntax :
 *  rl_copy_keymap src_keymap_id ?dst_keymap_id?
 * it receives pointer to Rl_KeymapTable structure as cdata
 */

int Rl_CopyKeymapCmd(cdata, interp, argc, argv)
     ClientData cdata;                   /* Client Data */
     Tcl_Interp *interp;                 /* Current interpreter. */
     int argc;                           /* Number of arguments. */
     char **argv;                        /* Argument strings. */
{
  Rl_KeymapTable *keymaps = (Rl_KeymapTable *)cdata;
  Keymap map;
  Rl_KeymapEntry *src_map,*dst_map;
  Tcl_HashEntry *newEntry,*dst_map_entry;
  Tcl_HashSearch search;
  char *dst_name=NULL;
  int isOk;

  if(argc < 2 || argc > 3) {
    Tcl_AppendResult(interp,"wrong # of args , should be \"",argv[0],
		     " src_keymap_id ?dst_keymap_id?\"",(char*)NULL);
    return TCL_ERROR;
  }
  
  if(argc == 3) dst_name = argv[2];

  if((newEntry = Tcl_FindHashEntry(&(keymaps->table),argv[1])) == NULL) {
    Tcl_AppendResult(interp,argv[1] ," is not valid keymap",(char*)NULL);
    return TCL_ERROR;
  }
  src_map = (Rl_KeymapEntry *)Tcl_GetHashValue(newEntry);
  
  map = rl_copy_keymap(src_map->map);
  if(map == NULL) {
    Tcl_AppendResult(interp,"Can't copy keymap",(char*)NULL);
    return TCL_ERROR;
  } 
  
  if((dst_map_entry = Rl_RegisterKeymap(interp,keymaps,map,dst_name)) == NULL)
    return TCL_ERROR;

  dst_map = (Rl_KeymapEntry*)Tcl_GetHashValue(dst_map_entry);

  for(newEntry = Tcl_FirstHashEntry(&(src_map->table),&search);
      newEntry != NULL ; 
      newEntry = Tcl_NextHashEntry(&search)) { 
    Tcl_HashEntry *dst_entry = 
      Tcl_CreateHashEntry(&(dst_map->table),Tcl_GetHashKey(&(src_map->table),newEntry),&isOk);
    Tcl_SetHashValue(dst_entry,Tcl_GetHashValue(newEntry));
  }

  return TCL_OK;  
}


/*
 * Rl_DelKeymapCmd() -- interface to rl_discard_keymap() . Syntax :
 *  rl_discard_keymap keymap_id
 * it receives pointer to Rl_KeymapTable structure as cdata
 */

int Rl_DelKeymapCmd(cdata, interp, argc, argv)
     ClientData cdata;                   /* Client Data */
     Tcl_Interp *interp;                 /* Current interpreter. */
     int argc;                           /* Number of arguments. */
     char **argv;                        /* Argument strings. */
{
  Rl_KeymapTable *keymaps = (Rl_KeymapTable *)cdata;
  Tcl_HashEntry *newEntry;
  Rl_KeymapEntry *src_map;

  if(argc != 2) {
    Tcl_AppendResult(interp,"wrong # of args , should be \"",argv[0]," keymap_id\"",(char*)NULL);
    return TCL_ERROR;
  }

  if((newEntry = Tcl_FindHashEntry(&(keymaps->table),argv[1])) == NULL) {
    Tcl_AppendResult(interp,argv[1] ," is not valid keymap",(char*)NULL);
    return TCL_ERROR;
  }
  src_map = (Rl_KeymapEntry*)Tcl_GetHashValue(newEntry);
  
  rl_discard_keymap(src_map->map);
  Tcl_DeleteHashTable(&(src_map->table));
  ckfree(src_map);
  Tcl_DeleteHashEntry(newEntry);
  *interp->result = '\0';
  return TCL_OK;
}

/*
 * Find hash table entry by given Keymap
 *
 */
static
Tcl_HashEntry *find_Entry_by_Keymap(table,keymap)
Tcl_HashTable *table;
Keymap keymap;
{
  Tcl_HashEntry *entry;
  Rl_KeymapEntry *map;
  Tcl_HashSearch search;

  for(entry = Tcl_FirstHashEntry(table,&search) ; 
      entry != 0 ; 
      entry = Tcl_NextHashEntry(&search)) {
    map = (Rl_KeymapEntry *)Tcl_GetHashValue(entry);
    if(map->map == keymap) return entry;
  }
    
  return NULL;
}


/*
 * Rl_GetKeymapCmd() -- interface to rl_get_keymap() . Syntax :
 *  rl_get_keymap
 * it receives pointer to Rl_KeymapTable structure as cdata
 */
int Rl_GetKeymapCmd(cdata, interp, argc, argv)
     ClientData cdata;                   /* Client Data */
     Tcl_Interp *interp;                 /* Current interpreter. */
     int argc;                           /* Number of arguments. */
     char **argv;                        /* Argument strings. */
{
  Rl_KeymapTable *keymaps = (Rl_KeymapTable *)cdata;
  Tcl_HashEntry *newEntry;
  Keymap map;

  if(argc != 1) {
    Tcl_AppendResult(interp,"wrong # of args , should be \"",argv[0],"\"",(char*)NULL);
    return TCL_ERROR;
  }
  
  if((map = rl_get_keymap()) == NULL) {
    *interp->result = '\0';
    return TCL_OK;
  }
    
  if((newEntry = find_Entry_by_Keymap(&(keymaps->table),map)) == NULL) {
    Tcl_AppendResult(interp,"returned unregistered keymap , internal bug ? ",(char*)NULL);
    return TCL_ERROR;
  }

  Tcl_SetResult(interp,Tcl_GetHashKey(&(keymaps->table),newEntry),TCL_STATIC);
  return TCL_OK;
}

/*
 * Rl_SetKeymapCmd() -- interface to rl_set_keymap() . Syntax :
 *  rl_set_keymap keymap_id
 * it receives pointer to Rl_KeymapTable structure as cdata
 */

int Rl_SetKeymapCmd(cdata, interp, argc, argv)
     ClientData cdata;                   /* Client Data */
     Tcl_Interp *interp;                 /* Current interpreter. */
     int argc;                           /* Number of arguments. */
     char **argv;                        /* Argument strings. */
{
  Rl_KeymapTable *keymaps = (Rl_KeymapTable *)cdata;
  Tcl_HashEntry *newEntry;
  Rl_KeymapEntry *src_map;

  if(argc != 2) {
    Tcl_AppendResult(interp,"wrong # of args , should be \"",argv[0]," keymap_id\"",(char*)NULL);
    return TCL_ERROR;
  }

  if((newEntry = Tcl_FindHashEntry(&(keymaps->table),argv[1])) == NULL) {
    Tcl_AppendResult(interp,argv[1] ," is not valid keymap",(char*)NULL);
    return TCL_ERROR;
  }
  src_map = (Rl_KeymapEntry*)Tcl_GetHashValue(newEntry);
  rl_set_keymap(src_map->map);
  *interp->result = '\0';
  return TCL_OK;  
}

/*
 * Rl_GetByNameKeymapCmd() -- interface to rl_get_keymap_by_name() . Syntax :
 *   rl_get_keymap_by_name name
 * it receives pointer to Rl_KeymapTable structure as cdata
 */

int Rl_GetByNameKeymapCmd(cdata, interp, argc, argv)
     ClientData cdata;                   /* Client Data */
     Tcl_Interp *interp;                 /* Current interpreter. */
     int argc;                           /* Number of arguments. */
     char **argv;                        /* Argument strings. */
{
  Rl_KeymapTable *keymaps = (Rl_KeymapTable *)cdata;
  Keymap map;
  Tcl_HashEntry *newEntry;

  if(argc != 2) {
    Tcl_AppendResult(interp,"wrong # of args , should be \"",argv[0]," name\"",(char*)NULL);
    return TCL_ERROR;
  }

  map = rl_get_keymap_by_name(argv[1]);
  if(map == NULL) {
    *interp->result = '\0';
    return TCL_OK;
  }
    
  if((newEntry = find_Entry_by_Keymap(&(keymaps->table),map)) == NULL) {
    Tcl_AppendResult(interp,"returned unregistered keymap , internal bug ? ",(char*)NULL);
    return TCL_ERROR;
  }

  Tcl_SetResult(interp,Tcl_GetHashKey(&(keymaps->table),newEntry),TCL_STATIC);
  return TCL_OK;
}  





/*
 * PART II : Manage Key Binding
 *
 */


#ifdef STRONG_BINDINGS
/*
 * Rl_ManageKeyHook -- this is the hook for rl_kbdkmap_hook , it accomulates
 *                     typed key sequence in tcl variable 'rl_typed_keyseq' .
 *                     When 'count' == 2 ( beginning of the new sequence )
 *                     the variable is reset to {} 
 */  
static
int Rl_ManageKeyHook(count,key)
int count;
int key;
{
  char buf[] = " "; 
  if(count == 2) 
    if(Tcl_SetVar(rl_interp,"rl_typed_keyseq","",
		  TCL_LEAVE_ERR_MSG | TCL_GLOBAL_ONLY) == NULL)
      goto error;

  if(key == ESC) {
    if(Tcl_SetVar(rl_interp,"rl_typed_keyseq","\\e",
		  TCL_LEAVE_ERR_MSG | TCL_APPEND_VALUE | TCL_GLOBAL_ONLY) == NULL)
      goto error;
  } else
  if(CTRL_CHAR(key)) {
    if(Tcl_SetVar(rl_interp,"rl_typed_keyseq","\\C-",
		  TCL_LEAVE_ERR_MSG | TCL_APPEND_VALUE | TCL_GLOBAL_ONLY) == NULL)
      goto error;
    *buf = to_lower(UNCTRL(key));
    if(Tcl_SetVar(rl_interp,"rl_typed_keyseq",buf,
		  TCL_LEAVE_ERR_MSG | TCL_APPEND_VALUE | TCL_GLOBAL_ONLY) == NULL)
      goto error;    
  } else {
    *buf = key;
    if(Tcl_SetVar(rl_interp,"rl_typed_keyseq",buf,
		  TCL_LEAVE_ERR_MSG | TCL_APPEND_VALUE | TCL_GLOBAL_ONLY) == NULL)
      goto error;
  }
    
  return key;

 error:
  Tcl_SetVar(rl_interp,"rl_typed_keyseq","",TCL_GLOBAL_ONLY);
  rl_abort();
  return -1;
}

#endif /* STRONG_BINDINGS */

/*
 * Rl_ManageKey -- all keys which are bind with this extension , execute
 *                 this function . It receives key and 
 *
 */
int Rl_ManageKey(count,key)
int count;
int key;
{
  char buf[] = " "; 
  char *execScript = NULL,*keyseq;
  Tcl_HashEntry *entryp,*bindp;
  Rl_KeymapEntry *map_entry;
  Keymap cmap;

  if(key == ESC) {
    if((keyseq = Tcl_SetVar(rl_interp,"rl_typed_keyseq","ESC",
		  TCL_LEAVE_ERR_MSG | TCL_APPEND_VALUE | TCL_GLOBAL_ONLY)) == NULL)
      goto error;
  } else
  if(CTRL_CHAR(key)) {
    if((keyseq = Tcl_SetVar(rl_interp,"rl_typed_keyseq","\\C-",
		  TCL_LEAVE_ERR_MSG | TCL_APPEND_VALUE | TCL_GLOBAL_ONLY)) == NULL)
      goto error;
    *buf = to_lower(UNCTRL(key));
    if((keyseq = Tcl_SetVar(rl_interp,"rl_typed_keyseq",buf,
			    TCL_LEAVE_ERR_MSG | TCL_APPEND_VALUE | TCL_GLOBAL_ONLY)) == NULL)
      goto error;
  } else {
    *buf = key;
    if((keyseq = Tcl_SetVar(rl_interp,"rl_typed_keyseq",buf,
			    TCL_LEAVE_ERR_MSG | TCL_APPEND_VALUE | TCL_GLOBAL_ONLY)) == NULL)
      goto error;
  }

  if((cmap = rl_get_keymap()) == NULL) 
    goto error;
  if((entryp = find_Entry_by_Keymap(&(rl_KeymapTable->table),cmap)) == NULL)
    goto error;
  if((map_entry = (Rl_KeymapEntry*)Tcl_GetHashValue(entryp)) == NULL)
    goto error;
  if((bindp = Tcl_FindHashEntry(&(map_entry->table),keyseq)) == NULL)
    goto error;
  if((execScript = (char*)Tcl_GetHashValue(bindp)) == NULL)
    goto error;

  if(Tcl_VarEval(rl_interp,"uplevel #0 { ",execScript," {",keyseq,"} ",
		 Tcl_GetHashKey(&(rl_KeymapTable->table),entryp)," }",(char*)NULL) == TCL_ERROR) 
    goto error;
  
  Tcl_SetVar(rl_interp,"rl_typed_keyseq","",TCL_GLOBAL_ONLY);
 
  return key;

 error:
  Tcl_SetVar(rl_interp,"rl_typed_keyseq","", TCL_GLOBAL_ONLY);
  rl_abort();
  return -1;
}

/*
 * Rl_BindCmd -- bind some tcl script to keysequence . Sintax :
 *               rl_bind key_seq keymap ?body?
 *               without body , returns the current body or {} if there is no binding .
 *               If body is "" unbinds keyseq .
 *               Always returns body of the binding .
 */

int Rl_BindCmd(cdata, interp, argc, argv)
     ClientData cdata;                   /* Client Data */
     Tcl_Interp *interp;                 /* Current interpreter. */
     int argc;                           /* Number of arguments. */
     char **argv;                        /* Argument strings. */
{
  Rl_KeymapTable *keymaps = (Rl_KeymapTable *)cdata;
  Tcl_HashEntry *entryp,*bindp;
  Rl_KeymapEntry *mapp;
  char *execScript;
  int alreadyBound;

  if(argc < 3 || argc > 4) {
    Tcl_AppendResult(interp,"wrong # of args , should be \"",argv[0],
		     " key_seq keymap ?body?\"",(char*)NULL);
    return TCL_ERROR;
  }

  if((entryp = Tcl_FindHashEntry(&(keymaps->table),argv[2])) == NULL) {
    Tcl_AppendResult(interp,"there is no such keymap : ",argv[2],(char*)NULL);
    return TCL_ERROR;
  }

  if((mapp = (Rl_KeymapEntry *)Tcl_GetHashValue(entryp)) == NULL) {
    Tcl_AppendResult(interp,"keymap entry is NULL . internal error ? ",(char*)NULL);
    return TCL_ERROR;
  }
  
  if(argc == 3 || *argv[3] == '\0') {
    if((bindp = Tcl_FindHashEntry(&(mapp->table),argv[1])) == NULL) {
      Tcl_AppendResult(interp,"there is no registered keyseq \"",argv[1],
		       "\" in keymap ",argv[2],(char*)NULL);
      return TCL_ERROR;
    }
    
    if((execScript = (char*)Tcl_GetHashValue(bindp)) == NULL) {
      Tcl_AppendResult(interp,"exec script entry is NULL . internal error ? ",(char*)NULL);
      return TCL_ERROR;
    }
    
    if(argc == 3) {		/* we shold find the execScript end return */
      Tcl_SetResult(interp,execScript,TCL_STATIC);
      return TCL_OK;
    }
    
    /* now the body present , lets check if we should unbind the sequence */
    if(*argv[3] == '\0') {
      Tcl_DeleteHashEntry(bindp);
      rl_generic_bind(ISFUNC,argv[1],NULL,mapp->map);
      Tcl_SetResult(interp,execScript,TCL_DYNAMIC);
      return TCL_OK;
    }
  }
  
  /* OK , finaly , the body is not empty , so we must to bind the keyseq */
  bindp = Tcl_CreateHashEntry(&(mapp->table),argv[1],&alreadyBound);
  if((execScript = (char*)ckalloc(strlen(argv[3])+1)) == NULL) {
    Tcl_AppendResult(interp,"unable allocate memory",(char*)NULL);
    return TCL_ERROR;
  }
  strcpy(execScript,argv[3]);
  Tcl_SetHashValue(bindp,(ClientData)execScript);
  if(rl_generic_bind(ISFUNC,argv[1],(Function*)Rl_ManageKey,mapp->map) < 0) {
    Tcl_DeleteHashEntry(bindp); /* we must do this if actual bind failed */
    Tcl_AppendResult(interp,"not valid keysequence \"",argv[1],"\"",(char*)NULL);
    return TCL_ERROR;
  }
    
  Tcl_SetResult(interp,execScript,TCL_STATIC);
  return TCL_OK;
}

/*
 * Rl_Parse_n_Bind() -- interface to rl_parse_and_bind() call . Syntax :
 *   rl_parse string
 * returns nothing .
 * This function does nothing with Rl_BindCmd structures . 
 * Since this extension not provides mechanism for providing new named functions 
 * Rl_Parse_n_Bind() is interface for using the precompiled ones .
 */

int Rl_Parse_n_BindCmd(cdata, interp, argc, argv)
     ClientData cdata;                   /* Client Data */
     Tcl_Interp *interp;                 /* Current interpreter. */
     int argc;                           /* Number of arguments. */
     char **argv;                        /* Argument strings. */
{
  if(argc != 2) {
    Tcl_AppendResult(interp,"wrong # of args , should be \"",argv[0], 
		     " string\"",(char*)NULL);
    return TCL_ERROR;
  }
  
  if(rl_parse_and_bind(argv[1]) < 0) {
    Tcl_AppendResult(interp,"could not parse string  : \"",argv[1],"\"",(char*)NULL);
    return TCL_ERROR;
  }
    
  return TCL_OK;
}


/*
 * PART III : Info
 */

/*
 * Rl_HashTableKeys() -- runs over entire hash table and returns tcl list 
 * ( in interp->result ) with all keys in the table ;
 *
 */
static
void  Rl_HashTableKeys(interp,table)
Tcl_Interp *interp;
Tcl_HashTable *table;
{
  Tcl_HashSearch search;
  Tcl_HashEntry *entry;

  Tcl_SetResult(interp,"",TCL_STATIC);
  for(entry = Tcl_FirstHashEntry(table,&search) ;
      entry != NULL ; 
      entry = Tcl_NextHashEntry(&search)) {
    Tcl_AppendElement(interp,Tcl_GetHashKey(table,entry));
  }
}

/*
 * Rl_InfoCmd() -- return various info . Syntax :
 *    rl_info option ?opt1? ?opt2?
 *  where 'option' could be one from :
 *    keymap - if opt1 is ommited return list of all registered keymaps
 *             if opt1 present , it must be valid keymap name . In this 
 *             case all registered keysequences for this colormap are returned .
 *             opt2 is not used .
 *    function - opt1 must not be ommited . It must be keysequence .
 *               If opt2 is ommited , current keymap is used . Otherwize
 *               opt2 must be keymap .
 *               Rteurns type of the function
 *               If the keysequence is not bound {} is returned .
 *               If the keysequence is bound to tcl script , TYPE == SCRIPT
 *               If the keysequence is bound to unnamed function or keymap or macro
 *               TYPE is eiter FUNC KEYMAP MACRO  . 
 *
 */

int Rl_InfoCmd(cdata, interp, argc, argv)
     ClientData cdata;                   /* Client Data */
     Tcl_Interp *interp;                 /* Current interpreter. */
     int argc;                           /* Number of arguments. */
     char **argv;                        /* Argument strings. */
{  
  Rl_KeymapTable *keymaps = (Rl_KeymapTable *)cdata;
  Tcl_HashEntry *newEntry,*bindp;
  Rl_KeymapEntry *map;
  Function *actual_func;
  int func_type,bin_keyseq_len;
  char *func_name;
  char *bin_keyseq; 

  if(argc < 2 || argc > 4) { 
    Tcl_AppendResult(interp,"wrong # of args , should be \"",argv[0],
		     " option ?opt1?\"",(char*)NULL);
    return TCL_ERROR;
  }
  
  if(strcmp(argv[1],"keymap") == 0) {
    if(argc == 3) {
      if((newEntry = Tcl_FindHashEntry(&(keymaps->table),argv[2])) == NULL) {
	Tcl_AppendResult(interp,argv[2] ," is not valid keymap",(char*)NULL);
	return TCL_ERROR;
      }
      if((map = (Rl_KeymapEntry *)Tcl_GetHashValue(newEntry)) == NULL) {
	Tcl_AppendResult(interp,argv[2] ," keymap entry is NULL .internal error ?",(char*)NULL);
	return TCL_ERROR;
      }
      Rl_HashTableKeys(interp,&(map->table));
      return TCL_OK;
    }
    Rl_HashTableKeys(interp,&(keymaps->table));
    return TCL_OK;
  }
  else
  if(strcmp(argv[1],"function") == 0) {
    if(argc == 2) {
      Tcl_AppendResult(interp,"wrong args , should be \"",argv[0],
		       " function keyseq ?keymap?\"",(char*)NULL);
      return TCL_ERROR;
    }
    if(argc == 3) {
      if((newEntry = find_Entry_by_Keymap(&(keymaps->table),rl_get_keymap())) == NULL) {
	Tcl_AppendResult(interp,"returned unregistered keymap , internal bug ? ",(char*)NULL);
	return TCL_ERROR;
      }
    } else {
      if((newEntry = Tcl_FindHashEntry(&(keymaps->table),argv[3])) == NULL) {
	Tcl_AppendResult(interp,argv[3] ," is not valid keymap",(char*)NULL);
	return TCL_ERROR;
      }
    }
    if((map = (Rl_KeymapEntry *)Tcl_GetHashValue(newEntry)) == NULL) {
      Tcl_AppendResult(interp,argv[3] ," keymap entry is NULL .internal error ?",(char*)NULL);
      return TCL_ERROR;
    }
     
    Tcl_SetResult(interp,"",TCL_STATIC);
    if((bin_keyseq = (char*)ckalloc(strlen(argv[2])+1)) == NULL) {
      Tcl_AppendResult(interp,argv[3] ,"unable allocate memory ",(char*)NULL);
      return TCL_ERROR;
    }
    rl_translate_keyseq(argv[2],bin_keyseq,&bin_keyseq_len);
      
    if((actual_func = rl_function_of_keyseq(bin_keyseq,map->map,&func_type)) == 
       (Function*)Rl_ManageKey) {
      ckfree(bin_keyseq);
      if((bindp = Tcl_FindHashEntry(&(map->table),argv[2])) != NULL)
	Tcl_AppendElement(interp,"SCRIPT");
      return TCL_OK;
    }
    else {
      ckfree(bin_keyseq);
      switch(func_type) {
      case ISFUNC : 
	Tcl_AppendElement(interp,"FUNC"); 
	break;
      case ISKMAP : 
	Tcl_AppendElement(interp,"KEYMAP");
	break;
      case ISMACR : 
	Tcl_AppendElement(interp,"MACRO");
	break;
      }
      return TCL_OK;
    }
  }
  else {
    Tcl_AppendResult(interp,"not valid option : ",argv[1],(char*)NULL);
    return TCL_ERROR;
  }
	
  return TCL_OK;
}



/*
 *  APPENDIX A : Functions for accessing readline variables
 */
 

/*
 * return pointer to a integer variable by it's name
 */ 

static
Rl_IntVars *get_int_by_name(name)
char *name;
{
  Rl_IntVars *sv;

  for(sv = rl_IntVars; sv->name != NULL ; sv++)
    if(strcmp(name,sv->name) == 0) return sv;
  return NULL;
}

/*
 * return pointer to a char* variable by it's name
 */ 

static
Rl_StrVars *get_str_by_name(name)
char *name;
{
  Rl_StrVars *sv;

  for(sv = rl_StrVars; sv->name != NULL ; sv++)
    if(strcmp(name,sv->name) == 0) return sv;
  return NULL;
}

/*
 * return pointer to hook name by the hook's name
 */ 

static
Rl_Hooks *get_hook_by_name(name)
char *name;
{
  Rl_Hooks *hook;

  for(hook = rl_hooks; hook->name != NULL ; hook++)
    if(strcmp(name,hook->name) == 0) return hook;
  return NULL;
}

/*
 * Trace integer variable
 */

char * Rl_TraceInt(cdata, interp,name1,name2,flags)
    ClientData cdata;
    Tcl_Interp *interp;
    char *name1,*name2;
    int flags;
{
  Rl_IntVars *p;
  char *val;

  if(name2 != NULL) return " shouldn't be array ";
  if((p = get_int_by_name(name1)) == NULL) return " no such variable ";
  if (flags & TCL_TRACE_READS) 
    Tcl_SetVar(interp,name1,ltoa(*(p->p)),flags&TCL_GLOBAL_ONLY);
  else 
  if(flags & TCL_TRACE_WRITES) {
    if(!p->w) {
      Tcl_SetVar(interp,name1,ltoa(*(p->p)),flags&TCL_GLOBAL_ONLY);
      return "variable is not writable";
    }
    if((val = Tcl_GetVar(interp,name1,flags&TCL_GLOBAL_ONLY)) == NULL) 
      return " no such variable ";
    if (Tcl_GetInt(interp,val,p->p)==TCL_ERROR) return "not a valid int";
  }
  return NULL;
}

/*
 * Trace string variable
 */

char * Rl_TraceStr(cdata, interp,name1,name2,flags)
    ClientData cdata;
    Tcl_Interp *interp;
    char *name1,*name2;
    int flags;
{
  Rl_StrVars *p;
  char *val,*rl_val;
  int len;

  if(name2 != NULL) return " shouldn't be array ";
  if((p = get_str_by_name(name1)) == NULL) return " no such variable ";
  if (flags & TCL_TRACE_READS) 
    Tcl_SetVar(interp,name1,((*(p->p) == NULL) ? "":*(p->p)),flags&TCL_GLOBAL_ONLY);
  else 
  if(flags & TCL_TRACE_WRITES) {
    if(!p->w) {
      Tcl_SetVar(interp,name1,((*(p->p) == NULL) ? "":*(p->p)),flags&TCL_GLOBAL_ONLY);
      return "variable is not writable";
    }    
    if((val = Tcl_GetVar(interp,name1,flags&TCL_GLOBAL_ONLY)) == NULL) 
      return " no such variable ";
    if((rl_val = (char*)ckalloc((len=strlen(val))+1)) == NULL) {
      return " can't allocate memory ";
    }
    strcpy(rl_val,val);
    rl_val[len] = '\0';
    ckfree(*(p->p));
    *(p->p) = rl_val;
  }
  return NULL;
}


/*
 * Trace procedure name variable
 */
char * Rl_TraceHook(cdata, interp,name1,name2,flags)
    ClientData cdata;
    Tcl_Interp *interp;
    char *name1,*name2;
    int flags;
{
  Rl_Hooks *p;
  char *val,*rl_val;
  int len;

  if(name2 != NULL) return " shouldn't be array ";
  if((p = get_hook_by_name(name1)) == NULL) return " no such variable ";
  if(flags & TCL_TRACE_WRITES) {
    
    if((val = Tcl_GetVar(interp,name1,flags&TCL_GLOBAL_ONLY)) == NULL) 
      return " no such variable ";
    if(*(p->name_var)) 
      ckfree(*(p->name_var));
    *(p->name_var) = strdup(val);
    *(p->rl_hook) = p->hook;
  }
  else
  if(flags & TCL_TRACE_UNSETS) {
    if(*(p->name_var)) 
      ckfree(*(p->name_var));
    *(p->name_var) = NULL;
    *(p->rl_hook) = NULL;
    Tcl_TraceVar(interp,name1,TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS,
		 Rl_TraceHook,(ClientData)NULL);
  }
   
  return NULL;
}
