/*
 * colors.c --
 *
 * A source file for Pict images 
 *
 * Copyright (c) 1995 The Regents of the University of California.
 *
 * Author: Pierre-Louis Bossart
 * Date: November 17, 1995
 *
 * Derived from tkImgPhoto.c in the tk4.0b2 distribution 
 * copyrighted as follows:
 *
 *    Copyright (c) 1994 The Australian National University.
 *    Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 *    Author: Paul Mackerras (paulus@cs.anu.edu.au),
 *	      Department of Computer Science,
 *	      Australian National University.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include "tkpict.h"

extern XColor	 lut_colorcell_defs[256];
extern int ncolors;
extern int lut_start;

/*
 *----------------------------------------------------------------------
 *
 * DitherInstance --
 *
 *	This procedure is called to update an area of an instance's
 *	pixmap by dithering the corresponding area of the master.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The instance's pixmap gets updated.
 *
 *----------------------------------------------------------------------
 */

void
DitherInstance(instancePtr, xStart, yStart, width, height)
    PictInstance *instancePtr;	/* The instance to be updated. */
    int xStart, yStart;		/* Coordinates of the top-left pixel in the
				 * block to be dithered. */
    int width, height;		/* Dimensions of the block to be dithered. */
{
    PictMaster *masterPtr;
    XImage *imagePtr;
    int nLines;
    int i, c, x, y;
    int xEnd, yEnd;
    int bitsPerPixel, bytesPerLine, lineLength;
    unsigned char *srcLinePtr, *srcPtr;
    unsigned char *destBytePtr, *dstLinePtr;
    pixel *destLongPtr;
   
 
#ifdef DEBUG
    printf("DitherInstance\n");
#endif

    masterPtr = instancePtr->masterPtr;

    /*
     * First work out how many lines to do at a time,
     * then how many bytes we'll need for pixel storage,
     * and allocate it.
     */

    nLines = (MAX_PIXELS + width - 1) / width;
    if (nLines < 1) {
	nLines = 1;
    }
    if (nLines > height ) {
	nLines = height;
    }

    imagePtr = instancePtr->imagePtr;
    if (imagePtr == NULL) {
	return;			/* we must be really tight on memory */
      }
    bitsPerPixel = imagePtr->bits_per_pixel;
    bytesPerLine = ((bitsPerPixel * width + 31) >> 3) & ~3;
    imagePtr->width = width;
    imagePtr->height = nLines;
    imagePtr->bytes_per_line = bytesPerLine;
    imagePtr->data = (char *) ckalloc((unsigned) (imagePtr->bytes_per_line * nLines));
    if( imagePtr->data == NULL ) {
      (void)fprintf(stderr,"DitherInstance: ckalloc failed \n");
      return;
    }

    lineLength = masterPtr->width;
    srcLinePtr = masterPtr->bytedata + yStart * lineLength + xStart;
    xEnd = xStart + width;

    
    if (bitsPerPixel > 1) {
      /*
     * Loop over the image, doing at most nLines lines before
     * updating the screen image.
     */

      for (; height > 0; height -= nLines) {
	if (nLines > height) {
	  nLines = height;
	}
	dstLinePtr = (unsigned char *) imagePtr->data;
	yEnd = yStart + nLines;
	for (y = yStart; y < yEnd; ++y) {
	  srcPtr = srcLinePtr;
	  destBytePtr = dstLinePtr;
	  destLongPtr = (pixel *) dstLinePtr;
	  	   
	  for (x = xStart; x < xEnd; ++x) {
	    c = srcPtr[0];
	    srcPtr += 1;
	    if (c < 0) {
	      c = 0;
	    } else if (c > 255) {
	      c = 255;
	    }
		
	    i = instancePtr->redValues[c];
	    switch (bitsPerPixel) {
	    case NBBY:
	      *destBytePtr++ = i;
	      break;
	    case NBBY * sizeof(pixel):
	      *destLongPtr++ = i;
	      break;
	    default:
	      XPutPixel(imagePtr, x - xStart, y - yStart,
			(unsigned) i);
	    }
	  }
	
	  srcLinePtr += lineLength;
	  dstLinePtr += bytesPerLine;
      
	}
	/* Update the pixmap for this instance with the block of
	   pixels that we have just computed. */
	XPutImage(instancePtr->display, instancePtr->pixels,
		  instancePtr->gc, imagePtr, 0, 0, xStart, yStart,
		  (unsigned) width, (unsigned) nLines);
	yStart = yEnd;
      }
    }
    ckfree(imagePtr->data);
    imagePtr->data = NULL;
  
} /* end DitherInstance */







void PutOverlayMask(instancePtr, xStart, yStart, width, height,mask)
     PictInstance *instancePtr;	/* The instance to be updated. */
     int xStart, yStart;        /* Coordinates of the top-left pixel in the
				   block to be overlayed. */
     int width, height;		/* Dimensions of the block to be overlayed. */
     unsigned char *mask;       /* pointer to the block */
     
{
    PictMaster *masterPtr;
    static XImage *imagePtr;
    int c, x, y;
    pixel i;
    int xEnd, yEnd;
    static int bitsPerPixel, bytesPerLine, lineLength;
    unsigned char *srcLinePtr, *srcPtr;
    unsigned char *destBytePtr, *dstLinePtr;
    pixel *destLongPtr;
    
#ifdef DEBUG
    printf("PutOverlayMask \n");
#endif
    if(instancePtr->has_overlay == False)
      return;
   
    masterPtr = instancePtr->masterPtr;
    imagePtr = instancePtr->imagePtr;
    if (imagePtr == NULL) {
	return;			/* we must be really tight on memory */
      }
    /* if (imagePtr->data == NULL || 
	imagePtr->width!=width || imagePtr->height!=height) {
      if(imagePtr->data != NULL ) {
	ckfree(imagePtr->data);
	imagePtr->data = NULL;
	}
	*/
    bitsPerPixel = imagePtr->bits_per_pixel;
    bytesPerLine = ((bitsPerPixel * width + 31) >> 3) & ~3;
    imagePtr->width = width;
    imagePtr->height = height;
    imagePtr->bytes_per_line = bytesPerLine;
    imagePtr->data = (char *) ckalloc((unsigned) (imagePtr->bytes_per_line * height));
    if( imagePtr->data == NULL ) {
      (void)fprintf(stderr,"DitherInstance: ckalloc failed \n");
      return;
    }
    
    lineLength = masterPtr->width;
    srcLinePtr = masterPtr->bytedata + yStart * lineLength + xStart;
    xEnd = xStart + width;
    
    dstLinePtr = (unsigned char *) imagePtr->data;
    yEnd = yStart + height;

    if (bitsPerPixel > 1) {
      for (y = yStart; y < yEnd; ++y) {
	srcPtr = srcLinePtr;
	destBytePtr = dstLinePtr;
	destLongPtr = (pixel *) dstLinePtr;
      
	for (x = xStart; x < xEnd; ++x) {
	  c = srcPtr[0];
	  srcPtr += 1;
	  if (c < 0) {
	    c = 0;
	  } else if (c > 255) {
	    c = 255;
	  }
	  
	  i = instancePtr->redValues[c]; 
	  switch (bitsPerPixel) {
	  case NBBY:
	    *destBytePtr++ = mask[x+y*width];
	    break;
	  case NBBY * sizeof(pixel):
	    *destLongPtr++ = mask[x+y*width];
	    break;
	  default:
	    XPutPixel(imagePtr, x - xStart, y - yStart,
		      (unsigned) mask[x+y*width]);
	  }
	}
      
	srcLinePtr += lineLength;
	dstLinePtr += bytesPerLine;
      }
    
      /* Update the pixmap for this instance with the block of
	 pixels that we have just computed. */
      XPutImage(instancePtr->display, instancePtr->pixels,
		instancePtr->overlay_gc, imagePtr, 0, 0, xStart, yStart,
		(unsigned) width, (unsigned) height);
    }
    ckfree(imagePtr->data);
    imagePtr->data = NULL;

} /* end PutOverlayMask */


int GetOverlayMask(PictInstance *instancePtr,Tk_PictImageBlock *blockPtr)
{
#ifdef DEBUG
    printf("GetOverlayMask\n");
#endif
    int width,height;
    int i,j;
    XImage *imagePtr;
    unsigned char *imgPtr;
    long pixel_value;
   
    if( instancePtr == NULL ) {
      printf("no instance \n");
      return 0;
    }
    if( instancePtr->has_overlay == False) {
      (void)fprintf(stderr,"Overlays not enabled on source image\n");
      return 0;
    }

    width = blockPtr->width = instancePtr->width;
    height = blockPtr->height = instancePtr->height;
    blockPtr->pitch = instancePtr->width;
    blockPtr->pixelSize = sizeof(unsigned char);
    blockPtr->datatype = BYTE;
    blockPtr->copy = NO_COPY;
    
    blockPtr->pixelPtr = (unsigned char*)malloc(width*height*sizeof(unsigned char));
    if( blockPtr->pixelPtr == NULL ) {
      (void)fprintf(stderr,"GetOverlayMask: Not enough memory \n");
      return 0;
    }
    imgPtr = blockPtr->pixelPtr;

    imagePtr = instancePtr->imagePtr;
    if (imagePtr == NULL) {
	return 0;			/* we must be really tight on memory */
      }
    if (imagePtr->data != NULL ) {
      free((void*)imagePtr->data);
    }
    imagePtr = XGetImage(instancePtr->display,
			 instancePtr->pixels,
			 0,0,instancePtr->width,instancePtr->height,1L,
			 ZPixmap);

    for(i=0;i<height;i++)
      for(j=0;j<width;j++) {
	pixel_value = XGetPixel(imagePtr,j,i);
	*imgPtr = (pixel_value & 1L)*255;
	imgPtr++;
      }
    if (imagePtr->data != NULL ) {
      free((void*)imagePtr->data);
    }
    return 1;
  } /* end GetOverlayMask */

