/*
 * tkImgPict.c --
 *
 * A source file for Pict images 
 *
 * Copyright (c) 1995 The Regents of the University of California.
 *
 * Author: Pierre-Louis Bossart
 * Date: November 17, 1995
 *
 * Derived from tkImgPhoto.c in the tk4.0b2 distribution 
 * copyrighted as follows:
 *
 *    Copyright (c) 1994 The Australian National University.
 *    Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 *    Author: Paul Mackerras (paulus@cs.anu.edu.au),
 *	      Department of Computer Science,
 *	      Australian National University.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include "tkpict.h"
/* if NDE_SEGMENT is defined, then include the nde include file */
#ifdef NDE_SEGMENT
#include "nde.h"
int has_nde_segment = 1;
#else 
int has_nde_segment = 0;
#endif


static int Default_Screen_Allocated = 0;
static Colormap Default_Screen_Colormap;
static int Default_Screen_ncolors;
static int Default_Screen_lut_start;
static int Default_Private_Allocated = 0;
static Colormap Default_Private_Colormap;
static int Default_Private_ncolors;
static int Default_Private_lut_start;

/*
 * Hash table used to provide access to Pict images from C code.
 */
static Tcl_HashTable imgPictHash;
static int imgPictHashInitialized;	/* set when Tcl_InitHashTable done */

static int		ImgPictCreate _ANSI_ARGS_((Tcl_Interp *interp,
			    char *name, int argc, char **argv,
			    Tk_ImageType *typePtr, Tk_ImageMaster master,
			    ClientData *clientDataPtr));
static ClientData	ImgPictGet _ANSI_ARGS_((Tk_Window tkwin,
			    ClientData clientData));
static void		ImgPictDisplay _ANSI_ARGS_((ClientData clientData,
			    Display *display, Drawable drawable,
			    int imageX, int imageY, int width, int height,
			    int drawableX, int drawableY));
static void		ImgPictFree _ANSI_ARGS_((ClientData clientData,
			    Display *display));
static void		ImgPictDelete _ANSI_ARGS_((ClientData clientData));

Tk_ImageType tkPictImageType = {
    "pict",			/* name */
    ImgPictCreate,		/* createProc */
    ImgPictGet,		        /* getProc */
    ImgPictDisplay,		/* displayProc */
    ImgPictFree,		/* freeProc */
    ImgPictDelete,		/* deleteProc */
    (Tk_ImageType *) NULL	/* nextPtr */
};

/*
 * List of option names.  The order here must match the order of
 * declarations of the OPT_* constants above.
 */ 

static char *optionNames[] = {
    "-format",
    "-from",
    "-shrink",
    "-subsample",
    "-to",
    "-zoom",
    (char *) NULL
};


/*
 * Information used for parsing configuration specifications:
 */
static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_STRING, "-data", (char *) NULL, (char *) NULL,
	 (char *) NULL, Tk_Offset(PictMaster, dataString), TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-format", (char *) NULL, (char *) NULL,
	 (char *) NULL, Tk_Offset(PictMaster, format), TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-file", (char *) NULL, (char *) NULL,
	 (char *) NULL, Tk_Offset(PictMaster, fileString), TK_CONFIG_NULL_OK},
    {TK_CONFIG_INT, "-height", (char *) NULL, (char *) NULL,
	 DEF_Pict_HEIGHT, Tk_Offset(PictMaster, userHeight), 0},
    {TK_CONFIG_INT, "-width", (char *) NULL, (char *) NULL,
	 DEF_Pict_WIDTH, Tk_Offset(PictMaster, userWidth), 0},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	 (char *) NULL, 0, 0}
};



/*
 * Pointer to the first in the list of known Pict image formats.
 */

static Tk_PictImageFormat *formatList = NULL;

/*
 * Forward declarations
 */

static int		ImgPictCmd _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, int argc, char **argv));
static int		ParseSubcommandOptions _ANSI_ARGS_((
			    struct SubcommandOptions *optPtr,
			    Tcl_Interp *interp, int allowedOptions,
			    int *indexPtr, int argc, char **argv));
static void		ImgPictCmdDeletedProc _ANSI_ARGS_((
			    ClientData clientData));
static int		ImgPictConfigureMaster _ANSI_ARGS_((
			    Tcl_Interp *interp, PictMaster *masterPtr,
			    int argc, char **argv, int flags));
static void		ImgPictConfigureInstance _ANSI_ARGS_((
			    PictInstance *instancePtr));
static void		ImgPictSetSize _ANSI_ARGS_((PictMaster *masterPtr,
			    int width, int height));
static void		ImgPictInstanceSetSize _ANSI_ARGS_((
			    PictInstance *instancePtr));

static int		MatchFileFormat _ANSI_ARGS_((Tcl_Interp *interp,
			    FILE *f, char *fileName, char *formatString,
			    Tk_PictImageFormat **imageFormatPtr,
			    int *widthPtr, int *heightPtr));
static int		MatchStringFormat _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, char *formatString,
			    Tk_PictImageFormat **imageFormatPtr,
			    int *widthPtr, int *heightPtr));

static int convert_block_to_byte(void *in, unsigned char *out, int npts,
			  int in_type,double *dispmin, double *dispmax);
static int make_colorbar(Tk_PictHandle handle,
			 int width,int height);
static void normalize_data(PictMaster *masterPtr);


/*
 *----------------------------------------------------------------------
 *
 * Tk_CreatePictImageFormat --
 *
 *	This procedure is invoked by an image file handler to register
 *	a new Pict image format and the procedures that handle the
 *	new format.  The procedure is typically invoked during
 *	Tcl_AppInit.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The new image file format is entered into a table used in the
 *	Pict image "read" and "write" subcommands.
 *
 *----------------------------------------------------------------------
 */

void
Tk_CreatePictImageFormat(formatPtr)
    Tk_PictImageFormat *formatPtr;
				/* Structure describing the format.  All of
				 * the fields except "nextPtr" must be filled
				 * in by caller.  Must not have been passed
				 * to Tk_CreatePictImageFormat previously. */
{
    Tk_PictImageFormat *copyPtr;

#ifdef DEBUG
    printf("Tk_CreatePictImageFormat\n");
#endif

    copyPtr = (Tk_PictImageFormat *) ckalloc(sizeof(Tk_PictImageFormat));
    if(copyPtr == NULL ) {
      (void)fprintf(stderr,"Tk_CreatePictImageFormat: Could not allocate memory\n");
      return;
    }
    *copyPtr = *formatPtr;
    copyPtr->name = (char *) ckalloc((unsigned) (strlen(formatPtr->name) + 1));
    if(copyPtr->name == NULL ) {
      (void)fprintf(stderr,"Tk_CreatePictImageFormat: Could not allocate memory\n");
      return;
    }
    strcpy(copyPtr->name, formatPtr->name);
    copyPtr->nextPtr = formatList;
    formatList = copyPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * ImgPictCreate --
 *
 *	This procedure is called by the Tk image code to create
 *	a new Pict image.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	The data structure for a new Pict image is allocated and
 *	initialized.
 *
 *----------------------------------------------------------------------
 */
static
int ImgPictCreate(interp, name, argc, argv, typePtr, master, clientDataPtr)
    Tcl_Interp *interp;		/* Interpreter for application containing
				 * image. */
    char *name;			/* Name to use for image. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings for options (doesn't
				 * include image name or type). */
    Tk_ImageType *typePtr;	/* Pointer to our type record (not used). */
    Tk_ImageMaster master;	/* Token for image, to be used by us in
				 * later callbacks. */
    ClientData *clientDataPtr;	/* Store manager's token for image here;
				 * it will be returned in later callbacks. */
{
    PictMaster *masterPtr;
    Tcl_HashEntry *entry;
    int isNew;

#ifdef DEBUG
    printf("ImgPictCreate\n");
#endif

    /*
     * Allocate and initialize the Pict image master record.
     */

    masterPtr = (PictMaster *) ckalloc(sizeof(PictMaster));
    if(masterPtr == NULL ) {
      (void)fprintf(stderr,"ImgPictCreate: Could not allocate memory\n");
      return 0;
    }
    memset((void *) masterPtr, 0, sizeof(PictMaster));
    masterPtr->tkMaster = master;
    masterPtr->interp = interp;
    masterPtr->imageCmd = Tcl_CreateCommand(interp, name, ImgPictCmd,
	    (ClientData) masterPtr, ImgPictCmdDeletedProc);
    masterPtr->data = NULL;
    masterPtr->bytedata = NULL;
    masterPtr->instancePtr = NULL;
    masterPtr->validRegion = XCreateRegion();
    masterPtr->dispmax = 0.0;
    masterPtr->dispmin = 0.0;
    masterPtr->user_dispmax = 0.0;
    masterPtr->user_dispmin = 0.0;
    
    /*
     * Process configuration options given in the image create command.
     */

    if (ImgPictConfigureMaster(interp, masterPtr, argc, argv, 0) != TCL_OK) {
	ImgPictDelete((ClientData) masterPtr);
	return TCL_ERROR;
    }

    /*
     * Enter this Pict image in the hash table.
     */

    if (!imgPictHashInitialized) {
	Tcl_InitHashTable(&imgPictHash, TCL_STRING_KEYS);
	imgPictHashInitialized = 1;
    }
    entry = Tcl_CreateHashEntry(&imgPictHash, name, &isNew);
    Tcl_SetHashValue(entry, masterPtr);

    *clientDataPtr = (ClientData) masterPtr;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ImgPictCmd --
 *
 *	This procedure is invoked to process the Tcl command that
 *	corresponds to a Pict image.  See the user documentation
 *	for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

static int
ImgPictCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Information about Pict master. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    PictMaster *masterPtr = (PictMaster *) clientData;
    int c, result, index;
    int x, y, width, height;
    int dataWidth, dataHeight;
    struct SubcommandOptions options;
    int listArgc;
    char **listArgv;
    char **srcArgv;
    unsigned char *pixelPtr;
    Tk_PictImageBlock block;
    Tk_Window tkwin;
    char string[10000];
    char string1[10000];
    XColor color;
    Tk_PictImageFormat *imageFormat;
    int imageWidth, imageHeight;
    int matched;
    FILE *f;
    Tk_PictHandle srcHandle;
    size_t length;
    Tcl_DString buffer;
    char *realFileName;
    int pix_int;
    float pix_float;
    short *shortPtr;
    int *intPtr;
    float *floatPtr;
    int i;
    double x0,y0,dx,dy;

#ifdef DEBUG
    printf("ImgPictCmd\n");
#endif

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);

    if ((c == 'b') && (strncmp(argv[1], "blank", length) == 0)) {
      /*
       * Pict blank command - just call Tk_PictBlank.
       */
      
      if (argc == 2) {
	Tk_PictBlank(masterPtr);
      } else {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " blank\"", (char *) NULL);
	return TCL_ERROR;
      }
    }
    else if (c == 'c') {
      
      if( (strncmp(argv[1], "colorbar", length) == 0)) {
	make_colorbar((Tk_PictHandle)masterPtr,
		      masterPtr->width,masterPtr->height);
      }
      else if( (strncmp(argv[1], "cget", length) == 0)) {
	if (argc != 3) {
	  Tcl_AppendResult(interp, "wrong # args: should be \"",
			   argv[0], " cget option\"",
			   (char *) NULL);
	  return TCL_ERROR;
	}
	result = Tk_ConfigureValue(interp, Tk_MainWindow(interp), configSpecs,
				   (char *) masterPtr, argv[2], 0);
      }
      else if( (strncmp(argv[1], "configure", length) == 0)) {
	/*
	 * Pict configure command - handle this in the standard way.
	 */
	
	if (argc == 2) {
	  return Tk_ConfigureInfo(interp, Tk_MainWindow(interp),
				  configSpecs, (char *) masterPtr, 
				  (char *) NULL, 0);
	}
	if (argc == 3) {
	  return Tk_ConfigureInfo(interp, Tk_MainWindow(interp),
				  configSpecs, (char *) masterPtr, argv[2], 0);
	}
	return ImgPictConfigureMaster(interp, masterPtr, argc-2, argv+2,
				      TK_CONFIG_ARGV_ONLY);
      }
      else if( (strncmp(argv[1], "cmap_stretch", length) == 0)) {
	double x1,y1,x2,y2;
	PictInstance *instancePtr;
	
	if (argc != 6) {
	  Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			   " cmap_stretch x1 y1 x2 y2 \"", (char *) NULL);
	  return TCL_ERROR;
	}
	if( (instancePtr=masterPtr->instancePtr) == NULL )
	  return TCL_ERROR;
	
	if ((Tcl_GetDouble(interp, argv[2], &x1) != TCL_OK) ||
	    (Tcl_GetDouble(interp, argv[3], &y1) != TCL_OK) ||
	    (Tcl_GetDouble(interp, argv[4], &x2) != TCL_OK) ||
	    (Tcl_GetDouble(interp, argv[5], &y2) != TCL_OK)) 
	  return TCL_ERROR;
	
	n1 = floor(x1*instancePtr->ncolors);
	fn1 = floor(y1*instancePtr->ncolors);
	n2 = floor(x2*instancePtr->ncolors);
	fn2 = floor(y2*instancePtr->ncolors);
	
	non_linear_lut(instancePtr->display,
		       instancePtr->colormap,
		       chColor,
		       instancePtr->ncolors,
		       instancePtr->lut_start,
		       instancePtr->has_overlay); 
      }
      
      else if ( (strncmp(argv[1], "colormap", length) == 0)) {
	/*
	 * Pict colormap command - first parse and check parameters.
	 */
	PictInstance *instancePtr;
	
	if (argc != 3) {
	  Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			   " colormap cmap\"", (char *) NULL);
	  return TCL_ERROR;
	}
	instancePtr = masterPtr->instancePtr;
	
	if( instancePtr == NULL )
	  return TCL_OK;
	
	if(strncmp(argv[2], "gray", length) == 0)
	  gray(instancePtr->display,
	       instancePtr->colormap,
	       chColor,
	       instancePtr->ncolors,
	       instancePtr->lut_start,
	       instancePtr->has_overlay); 
	else if(strncmp(argv[2], "ct", length) == 0)
	  ct(instancePtr->display,
	     instancePtr->colormap,
	     chColor,
	     instancePtr->ncolors,
	     instancePtr->lut_start,
	     instancePtr->has_overlay);
      else if(strncmp(argv[2], "hot", length) == 0)
	hot(instancePtr->display,
	    instancePtr->colormap,
	    chColor,
	    instancePtr->ncolors,
	    instancePtr->lut_start,
	    instancePtr->has_overlay); 
      else if(strncmp(argv[2], "cold", length) == 0)
	cold(instancePtr->display,
	     instancePtr->colormap,
	     chColor,
	     instancePtr->ncolors,
	     instancePtr->lut_start,
	     instancePtr->has_overlay);
      else if(strncmp(argv[2], "hls", length) == 0)
	hls(instancePtr->display,
	    instancePtr->colormap,
	    chColor,
	    instancePtr->ncolors,
	    instancePtr->lut_start,
	    instancePtr->has_overlay);
      else if(strncmp(argv[2], "rgb", length) == 0)
	rgb(instancePtr->display,
	    instancePtr->colormap,
	    chColor,
	    instancePtr->ncolors,
	    instancePtr->lut_start,
	    instancePtr->has_overlay);
      else if(strncmp(argv[2], "spectrum", length) == 0)
	spectrum(instancePtr->display,
		 instancePtr->colormap,
		 chColor,
		 instancePtr->ncolors,
		 instancePtr->lut_start,
		 instancePtr->has_overlay);
      else if(strncmp(argv[2], "invert", length) == 0)
	invert(instancePtr->display,
	       instancePtr->colormap,
	       chColor,
	       instancePtr->ncolors,
	       instancePtr->lut_start,
	       instancePtr->has_overlay);
      else if(strncmp(argv[2], "random", length) == 0)
	randwalk_spectrum(instancePtr->display,
			  instancePtr->colormap,
			  chColor,
			  instancePtr->ncolors,
			  instancePtr->lut_start,
			  instancePtr->has_overlay);
      else if(strncmp(argv[2], "bowlerhat", length) == 0)
	bowlerhat(instancePtr->display,
		  instancePtr->colormap,
		  chColor,
		  instancePtr->ncolors,
		  instancePtr->lut_start,
		  instancePtr->has_overlay);
	else if(strncmp(argv[2], "tophat", length) == 0)
	  tophat(instancePtr->display,
		 instancePtr->colormap,
		 chColor,
		 instancePtr->ncolors,
		 instancePtr->lut_start,
		 instancePtr->has_overlay);
	else if(strncmp(argv[2], "hatgray", length) == 0)
	  hatgray(instancePtr->display,
		  instancePtr->colormap,
		  chColor,
		  instancePtr->ncolors,
		  instancePtr->lut_start,
		  instancePtr->has_overlay);
	else if(strncmp(argv[2], "hatct", length) == 0)
	  hatct(instancePtr->display,
		instancePtr->colormap,
		chColor,
		instancePtr->ncolors,
		instancePtr->lut_start,
		instancePtr->has_overlay);
	else return TCL_ERROR;
      }
      else if( (strncmp(argv[1], "copy", length) == 0)) {
	/*
	 * Pict copy command - first parse options.
	 */
	
	index = 2;
	memset((VOID *) &options, 0, sizeof(options));
	options.zoomX = options.zoomY = 1;
	options.subsampleX = options.subsampleY = 1;
	options.name = NULL;
	if (ParseSubcommandOptions(&options, interp,
				   OPT_FROM | OPT_TO | OPT_ZOOM | 
				   OPT_SUBSAMPLE | OPT_SHRINK,
				   &index, argc, argv) != TCL_OK) {
	  return TCL_ERROR;
	}
	if (options.name == NULL || index < argc) {
	  Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			   " copy source-image ?-from x1 y1 x2 y2?",
			   " ?-to x1 y1 x2 y2? ?-zoom x y? ?-subsample x y?",
			   "\"", (char *) NULL);
	  return TCL_ERROR;
	}
	
	/*
	 * Look for the source image and get a pointer to its image data.
	 * Check the values given for the -from option.
	 */
	
	if ((srcHandle = Tk_FindPict(options.name)) == NULL) {
	  Tcl_AppendResult(interp, "image \"", argv[2], "\" doesn't",
			   " exist or is not a Pict image", (char *) NULL);
	  return TCL_ERROR;
	}
	Tk_PictGetImage(srcHandle, &block);
	if ((options.fromX2 > block.width) || (options.fromY2 > block.height)
	    || (options.fromX2 > block.width)
	    || (options.fromY2 > block.height)) {
	  Tcl_AppendResult(interp, "coordinates for -from option extend ",
			   "outside source image", (char *) NULL);
	  return TCL_ERROR;
	}
	
	/*
	 * Fill in default values for unspecified parameters.
	 */
	
	if (((options.options & OPT_FROM) == 0) || (options.fromX2 < 0)) {
	    options.fromX2 = block.width;
	    options.fromY2 = block.height;
	  }
	if (((options.options & OPT_TO) == 0) || (options.toX2 < 0)) {
	    width = options.fromX2 - options.fromX;
	    if (options.subsampleX > 0) {
	      width = (width + options.subsampleX - 1) / options.subsampleX;
	    } else if (options.subsampleX == 0) {
	      width = 0;
	    } else {
	      width = (width - options.subsampleX - 1) / -options.subsampleX;
	    }
	    options.toX2 = options.toX + width * options.zoomX;
	    
	    height = options.fromY2 - options.fromY;
	    if (options.subsampleY > 0) {
	      height = (height + options.subsampleY - 1)
		/ options.subsampleY;
	    } else if (options.subsampleY == 0) {
	      height = 0;
	    } else {
	      height = (height - options.subsampleY - 1)
		/ -options.subsampleY;
	    }
	    options.toY2 = options.toY + height * options.zoomY;
	  }
	
	/*
	 * Set the destination image size if the -shrink option was specified.
	 */
	
	if (options.options & OPT_SHRINK) {
	  ImgPictSetSize(masterPtr, options.toX2, options.toY2);
	}
	
	/*
	 * Copy the image data over using Tk_PictPutZoomedBlock.
	 */
	
	block.pixelPtr += options.fromX * block.pixelSize
	  + options.fromY * block.pitch;
	block.width = options.fromX2 - options.fromX;
	block.height = options.fromY2 - options.fromY;
	Tk_PictPutZoomedBlock((Tk_PictHandle) masterPtr, &block,
			      options.toX, options.toY, 
			      options.toX2 - options.toX,
			      options.toY2 - options.toY, 
			      options.zoomX, options.zoomY,
			      options.subsampleX, options.subsampleY);

      }
    }
    else if (c == 'g') {
      
      if( (strncmp(argv[1], "get", length) == 0)) {
	/*
	 * Pict get command - first parse and check parameters.
	 */
	
	if (argc != 4) {
	  Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			   " get x y\"", (char *) NULL);
	  return TCL_ERROR;
	}
	if ((Tcl_GetInt(interp, argv[2], &x) != TCL_OK)
	    || (Tcl_GetInt(interp, argv[3], &y) != TCL_OK)) {
	  return TCL_ERROR;
	}
	if ((x < 0) || (x >= masterPtr->width)
	    || (y < 0) || (y >= masterPtr->height)) {
	  Tcl_AppendResult(interp, argv[0], " get: ",
			   "coordinates out of range", (char *) NULL);
	  return TCL_ERROR;
	}
	
	/*
	 * Extract the value of the desired pixel and format it as a string.
	 */
	
	switch(masterPtr->datatype)
	  {
	  case BYTE:
	    pix_int = (masterPtr->bytedata)[y * masterPtr->width + x];
	    sprintf(string, "%d", pix_int);
	    break;
	  case WORD:
	    shortPtr = (short*)masterPtr->data;
	    pix_int = shortPtr[y * masterPtr->width + x];
	    sprintf(string, "%d", pix_int);
	    break;
	  case LWORD:
	    intPtr = (int*)masterPtr->data;
	    pix_int = intPtr[y * masterPtr->width + x];
	    sprintf(string, "%d", pix_int);
	    break;
	  case REAL:
	    floatPtr = (float*)masterPtr->data;
	    pix_float = floatPtr[y* masterPtr->width + x];
	    sprintf(string, "%.5g", pix_float);
	    break;
	  default:
	    Tcl_AppendResult(interp, argv[0], " get: ",
		    "unknown data type", (char *) NULL);
	    return TCL_ERROR;
	  }
	Tcl_SetResult(interp,string,TCL_STATIC);
      }
      else if( (strncmp(argv[1], "getline", length) == 0)) {
	/*
	 * Pict get command - first parse and check parameters.
	 */
	
	if (argc != 7) {
	  Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			   " getline x x0 y0 dx dy\"", (char *) NULL);
	  return TCL_ERROR;
	}
	if (Tcl_GetInt(interp, argv[2], &x) != TCL_OK)
	  {
	    return TCL_ERROR;
	  }
	if( (x < 0) || (x >= masterPtr->width))
	  {
	    Tcl_AppendResult(interp, argv[0], " getline: ",
			     "coordinates out of range", (char *) NULL);
	    return TCL_ERROR;
	  }
	if((Tcl_GetDouble(interp, argv[3], &x0) != TCL_OK) ||
	   (Tcl_GetDouble(interp, argv[4], &y0) != TCL_OK) ||
	   (Tcl_GetDouble(interp, argv[5], &dx) != TCL_OK) ||
	   (Tcl_GetDouble(interp, argv[6], &dy) != TCL_OK))
	  {
	    Tcl_AppendResult(interp, argv[0], " getline: ",
			     "parameters out of range", (char *) NULL);
	    return TCL_ERROR;
	  }

	/*
	 * Extract the value of the desired pixel and format it as a string.
	 */
	
	sprintf(string,"");
	for(i=0;i<masterPtr->width;i++) {
	  pix_int = masterPtr->bytedata[x * masterPtr->width + i];
	  sprintf(string1,"%g ",x0+dx*i); 
	  strcat(string,string1);
	  sprintf(string1,"%g ", y0-pix_int*dy);
	  strcat(string,string1);
	}
	Tcl_AppendResult(interp, string, (char *) NULL); 
      }  
      else if( (strncmp(argv[1], "getmin", length) == 0)) {
	sprintf(string,"%.4g",masterPtr->dispmin);
	Tcl_AppendResult(interp, string, (char *) NULL); 
      }
      else if( (strncmp(argv[1], "getmax", length) == 0)) {
	sprintf(string,"%.4g",masterPtr->dispmax);
	Tcl_AppendResult(interp, string, (char *) NULL); 
      }
    }
    else if ((c == 'o') 
	     && (strncmp(argv[1], "overlay", length) == 0)) {
      int nargc;
      PictInstance *instancePtr=masterPtr->instancePtr;
      if( argc < 3 ) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " overlay [enable|disable|line|rectangle|putmask|getmask|setgc] \"", 
			 (char *) NULL);
	return TCL_ERROR;
      }
      
      c = argv[2][0];
      length = strlen(argv[2]);
      nargc = argc -1;
     
      if ((c == 'g')
	  && (strncmp(argv[2], "getmask", length) == 0)) {
	/*
	 * Look for the mask and get a pointer to its image data.
	 */
	
	if ((srcHandle = Tk_FindPict(argv[3])) == NULL) {
	  Tcl_AppendResult(interp, "image \"", argv[3], "\" doesn't",
			   " exist or is not a Pict image", (char *) NULL);
	  return TCL_ERROR;
	}
	if( !GetOverlayMask(((PictMaster*)srcHandle)->instancePtr,&block) ) {
	  Tcl_AppendResult(interp, "Could not get mask from \"", 
			   argv[3], "to \"", argv[0],(char *) NULL);
	  return TCL_ERROR;
	}
	Tk_PictExpand((Tk_PictHandle)masterPtr,block.width,block.height);
	Tk_PictPutBlock((ClientData)masterPtr, &block,
			0,0,masterPtr->width,masterPtr->height);
	block.pixelPtr = NULL; 
	return TCL_OK;
      }
      if( instancePtr == NULL )
	return TCL_OK;
      if ((c == 'e') 
	  && (strncmp(argv[2], "enable", length) == 0)) {
	
	if (nargc == 2) {
	  PictInstance *instancePtr=masterPtr->instancePtr;
	  
	  (void)EnableOverlays(instancePtr); 
	  Tk_ImageChanged(masterPtr->tkMaster, 0, 0,
			  masterPtr->width, masterPtr->height,
			  masterPtr->width, masterPtr->height);
	} else {
	  Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			   "overlay enable\"", (char *) NULL);
	  return TCL_ERROR;
	}
      }
      
      if( instancePtr->has_overlay == False) {
	(void)fprintf(stderr,"Overlays not enabled \n");
	return TCL_OK;
      }
      else if ((c == 'd') 
	       && (strncmp(argv[2], "disable", length) == 0)) {
	
	if (nargc == 2) {
	  (void)DisableOverlays(instancePtr); 
	  Tk_ImageChanged(masterPtr->tkMaster, 0, 0,
			  masterPtr->width, masterPtr->height,
			  masterPtr->width, masterPtr->height);
	} else {
	  Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			   "overlay disable\"", (char *) NULL);
	  return TCL_ERROR;
	}
      }
      else if ((c == 's') 
	       && (strncmp(argv[2], "setgc", length) == 0)) {
	int x;
	
	if (argc == 4) {
	  if( strcmp(argv[3], "xor") == 0)
	    x = GXxor;
	  else if( strcmp(argv[3], "or") == 0)
	    x = GXor;
	  else if( strcmp(argv[3], "copy") == 0)
	    x = GXcopy;
	  else if( strcmp(argv[3], "and") == 0)
	    x = GXand;
	  else if( strcmp(argv[3], "clear") == 0)
	    x = GXclear;
	  else if( strcmp(argv[3], "set") == 0)
	    x = GXset;
	  else return TCL_ERROR;
	  XSetFunction(instancePtr->display,instancePtr->overlay_gc,x); 
	  instancePtr->setgc = x;
	}
	return TCL_OK;

      } 
      else if ((c == 'f') 
	       && (strncmp(argv[2], "fill", length) == 0)) {
	int x,y;
	int minx,maxx,miny,maxy;
	int argstart;
	XPoint *vertex;
	int i=0;
	
	argstart = 3;
	minx = instancePtr->width;
	maxx = 0;
	miny = instancePtr->height;
	maxy = 0;
	vertex = (XPoint*)malloc((nargc-2)*sizeof(XPoint));
	while(argstart < argc) {
	  
	  Tcl_GetInt(interp, argv[argstart],&x);
	  Tcl_GetInt(interp, argv[argstart+1],&y);
	  vertex[i].x = (short) x;
	  vertex[i].y = (short) y;
	  if( x< minx )  minx=x;
	  if( y< miny )  miny=y;
	  if( x> maxx )  maxx=x;
	  if( y> maxy )  maxy=y;
	  argstart+=2;
	  i++;
	}

	XFillPolygon(instancePtr->display,
		     instancePtr->pixels,
		     instancePtr->overlay_gc,
		     vertex,
		     i,Complex,CoordModeOrigin);
	Tk_ImageChanged(masterPtr->tkMaster,minx,miny,
			maxx-minx,maxy-miny,
			masterPtr->width, masterPtr->height);
	XSetFunction(instancePtr->display,instancePtr->overlay_gc,GXcopy);
	free((void*)vertex);
      }
      else if ((c == 'l')
	       && (strncmp(argv[2], "line", length) == 0)) {
	int x,y,x1,y1,x2,y2,temp;
	int minx,maxx,miny,maxy;
		
	if (nargc == 6) {
	  if((Tcl_GetInt(interp, argv[3], &x) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[4], &y) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[5], &x1) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[6], &y1) != TCL_OK))
	    {
	      Tcl_AppendResult(interp, argv[0], " overlay line: ",
			       "parameters out of range", (char *) NULL);
	      return TCL_ERROR;
	    }
	  XDrawLine(instancePtr->display,
		    instancePtr->pixels,
		    instancePtr->overlay_gc,
		    x,y,x1,y1);
	  
	  if(x1<x) {
	    temp=x1;x1=x;x=temp;
	  }
	  if(y1<y) {
	    temp=y1;y1=y;y=temp;
	  }
	  Tk_ImageChanged(masterPtr->tkMaster, x-4, y-4,x1-x+8,y1-y+8,
			  masterPtr->width, masterPtr->height);
	}
	else if (nargc == 8) {
	  if((Tcl_GetInt(interp, argv[3], &x) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[4], &y) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[5], &x1) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[6], &y1) != TCL_OK) || 
	     (Tcl_GetInt(interp, argv[7], &x2) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[8], &y2) != TCL_OK))
	    {
	      Tcl_AppendResult(interp, argv[0], " line: ",
			       "parameters out of range", (char *) NULL);
	      return TCL_ERROR;
	    }
	  XDrawLine(instancePtr->display,
		    instancePtr->pixels,
		    instancePtr->overlay_gc,
		    x,y,x1,y1);
	  XDrawLine(instancePtr->display,
		    instancePtr->pixels,
		    instancePtr->overlay_gc,
		    x,y,x2,y2);
	  
	  minx = x; miny=y;
	  maxx = x; maxy = y;
	  if( x1 < minx )  minx=x1;
	  if( x2 < minx )  minx=x2;
	  if( y1 < miny )  miny=y1;
	  if( y2 < miny )  miny=y2;
	  if( x1 > maxx ) maxx=x1;
	  if( x2 > maxx )  maxx=x2;
	  if( y1 > maxy )  maxy=y1;
	  if( y2 > maxy )  maxy=y2;
	  Tk_ImageChanged(masterPtr->tkMaster,minx-4,miny-4,maxx-minx+8,maxy-miny+8,
			  masterPtr->width, masterPtr->height);
	}
      }
      else if ((c == 'r') 
	       && (strncmp(argv[2], "rectangle", length) == 0)) {
	int x,y,w,h;
		
	if (nargc == 6) {
	  if((Tcl_GetInt(interp, argv[3], &x) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[4], &y) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[5], &w) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[6], &h) != TCL_OK))
	    {
	      Tcl_AppendResult(interp, argv[0], " overlay rectangle: ",
			     "parameters out of range", (char *) NULL);
	      return TCL_ERROR;
	    }
	  XFillRectangle(instancePtr->display,
			 instancePtr->pixels,
			 instancePtr->overlay_gc,
			 x,y,w,h);
	  
	  Tk_ImageChanged(masterPtr->tkMaster, x, y, w, h,
			  masterPtr->width, masterPtr->height);
	}
      }
      else if ((c == 'e') 
	       && (strncmp(argv[2], "ellipse", length) == 0)) {
	int x,y,w,h;
		
	if (nargc == 6) {
	  if((Tcl_GetInt(interp, argv[3], &x) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[4], &y) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[5], &w) != TCL_OK) ||
	     (Tcl_GetInt(interp, argv[6], &h) != TCL_OK))
	    {
	      Tcl_AppendResult(interp, argv[0], " overlay ellipse: ",
			     "parameters out of range", (char *) NULL);
	      return TCL_ERROR;
	    }
	  XFillArc(instancePtr->display,
			 instancePtr->pixels,
			 instancePtr->overlay_gc,
			 x,y,w,h,0,360*64);
	  
	  Tk_ImageChanged(masterPtr->tkMaster, x, y, w, h,
			  masterPtr->width, masterPtr->height);
	}
      }
      else if ((c == 'p')
	       && (strncmp(argv[2], "putmask", length) == 0)) {
	/*
	 * Look for the mask and get a pointer to its image data.
	 */

	if ((srcHandle = Tk_FindPict(argv[3])) == NULL) {
	  Tcl_AppendResult(interp, "image \"", argv[3], "\" doesn't",
			   " exist or is not a Pict image", (char *) NULL);
	  return TCL_ERROR;
	}
	Tk_PictGetImage(srcHandle, &block);
	if( block.width != masterPtr->width ) {
	  (void)fprintf(stderr,"Mask width does not match image width \n");
	  return TCL_ERROR;
	}
	if( block.height != masterPtr->height ) {
	  (void)fprintf(stderr,"Mask height does not match image height \n");
	  return TCL_ERROR;
	}
	if((block.pixelSize != sizeof(unsigned char)) ||
	   (block.datatype != BYTE) ) {
	  (void)fprintf(stderr,"Mask datatype is not of type BYTE \n");
	  return TCL_ERROR;
	}
	PutOverlayMask(instancePtr,0,0,block.width,
		       block.height,block.pixelPtr);
	Tk_ImageChanged(masterPtr->tkMaster, 0,0,
			masterPtr->width, masterPtr->height,
			masterPtr->width, masterPtr->height);
	return TCL_OK;
      }
      else if ( (strncmp(argv[2], "ct_segment", length) == 0)
	       || (strncmp(argv[2], "segment", length) == 0)
	       || (strncmp(argv[2], "erosion", length) == 0) 
	       || (strncmp(argv[2], "dilation", length) == 0) 
	       || (strncmp(argv[2], "close_holes", length) == 0)
	       || (strncmp(argv[2], "seed_fill", length) == 0)) {
#ifdef NDE_SEGMENT
	
	if( (strncmp(argv[2], "ct_segment", length) == 0) ) {
	  unsigned char *g_img;
	  unsigned char *temp_img;
	  short *marker;
	  short *temp_marker;
	  unsigned char *binom_filter;
	  int filter_order;
	  int segment_level;

	  block.height = instancePtr->height;
	  block.width = instancePtr->width;;
	  block.pitch = instancePtr->width;
	  block.pixelSize = sizeof(unsigned char);
	  block.datatype = BYTE;
	  block.copy = NO_COPY;
	  block.pixelPtr = (unsigned char*)malloc((block.width)*
						  (block.height)*
						  sizeof(unsigned char));
	  if( block.pixelPtr == NULL ) {
	    Tcl_AppendResult(interp,argv[0]," overlay ct_segment :Could not allocate memory ", (char *) NULL);
	    return TCL_ERROR;
	  }
	  bcopy(masterPtr->bytedata,block.pixelPtr,(block.width)*
						  (block.height)*
						  sizeof(unsigned char));
	  /* allocate memory for segmentation */
	  filter_order = 7;
	  if( !lallocate_images(&g_img,&temp_img,
				&marker,&temp_marker,
				(block.width)*(block.height),
				&binom_filter,filter_order) )
	    { 
	      Tcl_AppendResult(interp, "Could not allocate memory \"",
			       (char *) NULL);
	      /* free memory */
	      free((void*)block.pixelPtr);
	      block.pixelPtr=NULL;
	      return TCL_ERROR;
	    }
	  
	  /* call to segmentation routine */
	  if( (x=lsegment_slices((unsigned char *)block.pixelPtr,
			       (unsigned char *)g_img,
			       (unsigned char *)temp_img,
			       (short *)marker,(short *)temp_marker,
			       (int)block.height,
			       (int)block.width,(int)1,
			       (unsigned char *)binom_filter,
			       (int)filter_order,80) ) <= 0 )
	    
	    { 
	      Tcl_AppendResult(interp, "lsegment_slices failed \"",
			       (char *) NULL);
	      /* free memory */
	      free((void*)block.pixelPtr);
	      block.pixelPtr=NULL;
	      (void)lfree_images((unsigned char *)g_img,temp_img,
				 (short *)marker,(short *)temp_marker,
				 (unsigned char *)binom_filter);
	      return TCL_ERROR;
	    }
	  (void)lfree_images((unsigned char *)g_img,temp_img,
			     (short *)marker,(short *)temp_marker,
			     (unsigned char *)binom_filter);

	}
	else {
	  /* get the overlays */
	  if( !GetOverlayMask(instancePtr,&block) ) {
	    Tcl_AppendResult(interp, "Could not get mask from \"", 
			     argv[0], (char *) NULL);
	    return TCL_ERROR;
	  }
	  if(block.pixelPtr == NULL ) {
	    Tcl_AppendResult(interp, "Could not get mask from \"", 
			     argv[0], (char *) NULL);
	    return TCL_ERROR;
	  }
	  
	  if( (strncmp(argv[2], "segment", length) == 0) ) {	
	    unsigned char *g_img;
	    unsigned char *temp_img;
	    unsigned char *marker;
	    unsigned char *temp_marker;
	    unsigned char *binom_filter;
	    int filter_order;
	    int segment_level;

	    if( argc != 4 ) {
	      Tcl_AppendResult(interp, "wrong number of parameters. Should be ", argv[0], " overlay segment [0|1|2]",(char *) NULL);
	      /* free memory */
	      free((void*)block.pixelPtr);
	      block.pixelPtr=NULL;
	      return TCL_ERROR;
	    }
	    /* allocate memory for segmentation */
	    filter_order = 7;
	    if( !lallocate_interact_images(&g_img,
					   &marker,&temp_marker,
					   (block.width)*(block.height),
					   &binom_filter,filter_order) )
	      { 
		Tcl_AppendResult(interp, "Could not allocate memory \"",
				 (char *) NULL);
		/* free memory */
		free((void*)block.pixelPtr);
		block.pixelPtr=NULL;
		return TCL_ERROR;
	      }
	    if((Tcl_GetInt(interp, argv[3], &segment_level) != TCL_OK))
	      {
		Tcl_AppendResult(interp, argv[0], " overlay segment: ",
				 "parameters out of range", (char *) NULL);
		/* free memory */
		free((void*)block.pixelPtr);
		block.pixelPtr=NULL;
		return TCL_ERROR;
	      }
	    /* call to segmentation routine */
	    if( (x=lsegment_interact((unsigned char *)masterPtr->bytedata,
				   (unsigned char *)g_img,
				   (unsigned char *)block.pixelPtr,
				   (unsigned char *)marker,(unsigned char *)temp_marker,
				   (int)block.height,
				   (int)block.width,
				   (unsigned char *)binom_filter,
				   (int)filter_order,
				   (int)segment_level)) <=0 )
	      { 
		Tcl_AppendResult(interp, "lsegment_interact failed \"",
				 (char *) NULL);
		/* free memory */
		free((void*)block.pixelPtr);
		block.pixelPtr=NULL;
		(void)lfree_interact_images((unsigned char *)g_img,
				        (unsigned char *)marker,(unsigned char *)temp_marker,
					(unsigned char *)binom_filter);
		return TCL_ERROR;
	      }
	    (void)lfree_interact_images((unsigned char *)g_img,
					(unsigned char *)marker,(unsigned char *)temp_marker,
					(unsigned char *)binom_filter);
	  
	  }
	  else if( (strncmp(argv[2], "erosion", length) == 0)) {
	    int x,y;
	    
	    if( argc != 5 ) {
	      Tcl_AppendResult(interp, "wrong number of parameters. Should be ", argv[0], " overlay erosion x y",(char *) NULL);
	      /* free memory */
	      free((void*)block.pixelPtr);
	      block.pixelPtr=NULL;
	      return TCL_ERROR;
	    }
	    if((Tcl_GetInt(interp, argv[3], &x) != TCL_OK) ||
	       (Tcl_GetInt(interp, argv[4], &y) != TCL_OK) )
	      {
		Tcl_AppendResult(interp, argv[0], " overlay erosion: ",
				 "parameters out of range", (char *) NULL);
		/* free memory */
		free((void*)block.pixelPtr);
		block.pixelPtr=NULL;
		return TCL_ERROR;
	      }
	    if ( !lseparable_erosion_3D(block.pixelPtr,
					block.height,block.width,1,x,y,1) ) {
	      Tcl_AppendResult(interp, "lseparable_erosion_3D failed \"",
			       (char *) NULL);
	      /* free memory */
	      free((void*)block.pixelPtr);
	      block.pixelPtr=NULL;
	      return TCL_ERROR;
	    }
	  }
	  else if( (strncmp(argv[2], "dilation", length) == 0)) {
	    int x,y;
	    
	    if( argc != 5 ) {
	      Tcl_AppendResult(interp, "wrong number of parameters. Should be ", argv[0], " overlay dilation x y", (char *) NULL);
	    /* free memory */
	      free((void*)block.pixelPtr);
	      block.pixelPtr=NULL;
	    return TCL_ERROR;
	    }
	    if((Tcl_GetInt(interp, argv[3], &x) != TCL_OK) ||
	       (Tcl_GetInt(interp, argv[4], &y) != TCL_OK) )
	      {
		Tcl_AppendResult(interp, argv[0], " overlay dilation: ",
				 "parameters out of range", (char *) NULL);
		/* free memory */
		free((void*)block.pixelPtr);
		block.pixelPtr=NULL;
		return TCL_ERROR;
	      }
	    
	    if ( !lseparable_dilation_3D(block.pixelPtr,
					 block.height,block.width,1,x,y,1) ) {
	      Tcl_AppendResult(interp, "lseparable_dilation_3D failed \"",
			       (char *) NULL);
	      /* free memory */
	      free((void*)block.pixelPtr);
	      block.pixelPtr=NULL;
	      return TCL_ERROR;
	    }
	  }
	  else if( (strncmp(argv[2], "close_holes", length) == 0)) {
	    unsigned char *marker;
	    marker = (unsigned char*)malloc((block.height)*
					    (block.width)*sizeof(unsigned char));
	    if( marker == NULL ) {
	      Tcl_AppendResult(interp, "Not enough memory\"",
			       (char *) NULL);
	      /* free memory */
	      free((void*)block.pixelPtr);
	      block.pixelPtr=NULL;
	      return TCL_ERROR;
	    }
	    if ( !lclose_holes_3D(block.pixelPtr,marker,255,
				  block.height,block.width,1,0) ) {
	      Tcl_AppendResult(interp, "lclose_holes failed \"",
			       (char *) NULL);
	      /* free memory */
	      free((void*)block.pixelPtr);
	      block.pixelPtr=NULL;
	      return TCL_ERROR;
	    }
	    free((void*)block.pixelPtr);
	    block.pixelPtr = marker;
	  } 
	  else if( (strncmp(argv[2], "seed_fill", length) == 0)) {
	    int x,y;
	    unsigned char *tempPtr;

	    if( argc != 5 ) {
	      Tcl_AppendResult(interp, "wrong number of parameters. Should be ", argv[0], " overlay seed_fill x y",(char *) NULL);
	      /* free memory */
	      free((void*)block.pixelPtr);
	      block.pixelPtr=NULL;
	      return TCL_ERROR;
	    }
	    if((Tcl_GetInt(interp, argv[3], &x) != TCL_OK) ||
	       (Tcl_GetInt(interp, argv[4], &y) != TCL_OK) )
	      {
		Tcl_AppendResult(interp, argv[0], " overlay seed_fill: ",
				 "parameters out of range", (char *) NULL);
		/* free memory */
		free((void*)block.pixelPtr);
		block.pixelPtr=NULL;
		return TCL_ERROR;
	      }
	    tempPtr = (unsigned char*)malloc(block.height*block.width*sizeof(unsigned char));
	    if( tempPtr == NULL ) {
	      Tcl_AppendResult(interp, "Not enough memory \"",
			       (char *) NULL);
	      /* free memory */
	      free((void*)block.pixelPtr);
	      block.pixelPtr=NULL;
	      return TCL_ERROR;
	    }

	    if ( lfseeded_label_2D(block.pixelPtr,tempPtr,
				    block.height,block.width,x,y) <= 0 ) {
	      Tcl_AppendResult(interp, "lfseeded_label_2D failed \"",
			       (char *) NULL);
	      /* free memory */
	      free((void*)tempPtr);
	      free((void*)block.pixelPtr);
	      block.pixelPtr=NULL;
	      return TCL_ERROR;
	    }
	    free((void*)block.pixelPtr);
	    block.pixelPtr = tempPtr;
	  }

	}
	/* overlay result */
	PutOverlayMask(instancePtr,0,0,block.width,
		       block.height,block.pixelPtr);
	/* update window */
	Tk_ImageChanged(masterPtr->tkMaster, 0,0,
			masterPtr->width, masterPtr->height,
			masterPtr->width, masterPtr->height);
	/* free memory */
	free((void*)block.pixelPtr);
	block.pixelPtr=NULL;
	
	return TCL_OK;
      
#else 
	Tcl_AppendResult(interp, "Undefined command: \"",argv[0]," " , argv[1]," ", argv[2],"\" (Make sure segmentation code is installed", (char *) NULL);
	return TCL_ERROR;
#endif
      }
    }
    else if ((c == 'r') && (strncmp(argv[1], "range", length) == 0)) {
      double dispmax;
      double dispmin;
      PictInstance *instancePtr;

      if (argc != 4) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " range x y\"", (char *) NULL);
	return TCL_ERROR;
      }
      if((Tcl_GetDouble(interp, argv[2], &dispmin) != TCL_OK) 
	 || (Tcl_GetDouble(interp, argv[3], &dispmax) != TCL_OK))
	return TCL_ERROR;
      masterPtr->user_dispmax = dispmax;
      masterPtr->user_dispmin = dispmin;

      normalize_data(masterPtr);
      
      /*
       * Update each instance.
       */
      for(instancePtr = masterPtr->instancePtr; instancePtr != NULL;
	  instancePtr = instancePtr->nextPtr)
	DitherInstance(instancePtr, 0, 0,
		       instancePtr->width,
		       instancePtr->height);

      /*
       * Tell the core image code that this image has changed.
       */

      Tk_ImageChanged(masterPtr->tkMaster,0,0,
		      masterPtr-> width, 
		      masterPtr->height, 
		      masterPtr->width,
		      masterPtr->height);

    }
    else if (c == 'r') {
      
      if( (strncmp(argv[1], "read", length) == 0)) {
	/*
	 * Pict read command - first parse the options specified.
	 */

	index = 2;
	memset((VOID *) &options, 0, sizeof(options));
	options.name = NULL;
	options.format = NULL;
	if (ParseSubcommandOptions(&options, interp,
		OPT_FORMAT | OPT_FROM | OPT_TO | OPT_SHRINK,
		&index, argc, argv) != TCL_OK) {
	    return TCL_ERROR;
	}
	if ((options.name == NULL) || (index < argc)) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " read fileName ?-format format-name?",
		    " ?-from x1 y1 x2 y2? ?-to x y? ?-shrink?\"",
		    (char *) NULL);
	    return TCL_ERROR;
	}

	/*
	 * Open the image file and look for a handler for it.
	 */

	realFileName = Tcl_TildeSubst(interp, options.name, &buffer);
	if (realFileName == NULL) {
	    return TCL_ERROR;
	}
	f = fopen(realFileName, "rb");
	Tcl_DStringFree(&buffer);
	if (f == NULL) {
	    Tcl_AppendResult(interp, "couldn't read image file \"",
		    options.name, "\": ", Tcl_PosixError(interp),
		    (char *) NULL);
	    return TCL_ERROR;
	}
	if (MatchFileFormat(interp, f, options.name, options.format,
		&imageFormat, &imageWidth, &imageHeight) != TCL_OK) {
	    fclose(f);
	    return TCL_ERROR;
	}

	/*
	 * Check the values given for the -from option.
	 */

	if ((options.fromX > imageWidth) || (options.fromY > imageHeight)
		|| (options.fromX2 > imageWidth)
		|| (options.fromY2 > imageHeight)) {
	    Tcl_AppendResult(interp, "coordinates for -from option extend ",
		    "outside source image", (char *) NULL);
	    fclose(f);
	    return TCL_ERROR;
	}
	if (((options.options & OPT_FROM) == 0) || (options.fromX2 < 0)) {
	    width = imageWidth - options.fromX;
	    height = imageHeight - options.fromY;
	} else {
	    width = options.fromX2 - options.fromX;
	    height = options.fromY2 - options.fromY;
	}

	/*
	 * If the -shrink option was specified, set the size of the image.
	 */

	if (options.options & OPT_SHRINK) {
	    ImgPictSetSize(masterPtr, options.toX + width,
		    options.toY + height);
	}

	/*
	 * Call the handler's file read procedure to read the data
	 * into the image.
	 */

	result = (*imageFormat->fileReadProc)(interp, f, options.name,
		options.format, (Tk_PictHandle) masterPtr, options.toX,
		options.toY, width, height, options.fromX, options.fromY);
	if (f != NULL) {
	    fclose(f);
	  }
	return result;
      }
      else if( (strncmp(argv[1], "rdbinary", length) == 0)) {
	/*
	 * Pict read command - first parse the options specified.
	 */
	char *arg;
	int argstart = 2;
	int parsed1 = 1;
	char has_file,has_width,has_height,has_skip,has_type;
	int skip;
	int type;
	int datasize;
	
	if( argc == 2 ) {
	  fprintf(stderr,"Usage: %s rdbinary -file -width -height -skip -nb_slices -skip -type \n",argv[0]);
	  return TCL_ERROR;
	}
	has_file = has_width = has_height = has_type = has_skip = 0;
	
	while ((arg = argv[argstart]) != NULL && parsed1)
	  {
	    parsed1 = 0;
	    
	    if (strcmp(arg,"-file") == 0)
	      {
		has_file = parsed1 = 1;
		++argstart;
		if( argv[argstart] != NULL ) {
		  realFileName = Tcl_TildeSubst(interp,argv[argstart],&buffer);
		  if (realFileName == NULL) {
		    return TCL_ERROR;
		  }
		  ++argstart;
		} else  return TCL_ERROR;
	      }
	    else if (strcmp(arg,"-width") == 0) 
	      {
		has_width = parsed1 = 1;
		++argstart;
		if( argv[argstart] != NULL ) {
		  if( (Tcl_GetInt(interp, argv[argstart], &width) != TCL_OK))
		    return TCL_ERROR;
		} else return TCL_ERROR;
		argstart++;
	      }
	    else if (strcmp(arg,"-height") == 0) 
	      {
		has_height = parsed1 = 1;
		++argstart;
		if( argv[argstart] != NULL ) {
		  if( (Tcl_GetInt(interp, argv[argstart], &height) != TCL_OK))
		    return TCL_ERROR;
		} else return TCL_ERROR;
		argstart++;
	      }
	    else if (strcmp(arg,"-nb_slices") == 0) 
	      {
		parsed1 = 1;
		++argstart;
		if( argv[argstart] != NULL ) {
		  if( (Tcl_GetInt(interp, argv[argstart], &nb_slices) != TCL_OK))
		    return TCL_ERROR;
		} else return TCL_ERROR;
		argstart++;
	      }
	    else if (strcmp(arg,"-skip") == 0) 
	      {
		has_skip = parsed1 = 1;
		++argstart;
		if( argv[argstart] != NULL ) {
		  if( (Tcl_GetInt(interp, argv[argstart], &skip) != TCL_OK))
		    return TCL_ERROR;
		  masterPtr->skip = skip;
		} else return TCL_ERROR;
		argstart++;
	      }
	    else if (strcmp(arg,"-type") == 0) 
	      {
		has_type = parsed1 = 1;
		++argstart;
		if( argv[argstart] != NULL ) {
		  if (strcmp(argv[argstart],"byte") == 0) {
		    type = BYTE;
		    datasize = sizeof(unsigned char);
		  }
		  else if(strcmp(argv[argstart],"short") == 0) {
		    type = WORD;
		    datasize = sizeof(short);
		  }
		  else if(strcmp(argv[argstart],"int") == 0) {
		    type = LWORD;
		    datasize = sizeof(int);
		  }
		  else if(strcmp(argv[argstart],"float") == 0) {
		    type = REAL; 
		    datasize = sizeof(float);
		  }
		} else return TCL_ERROR;
		argstart++;
	      }
	  }
	if (arg != NULL || has_file == 0 )
	  return TCL_ERROR;
	if ( has_width == 0 ) {
	  if(masterPtr->width != 0 )
	    width = masterPtr->width;
	  else
	    return TCL_ERROR;
	}
	if ( has_height == 0 ) {
	  if( masterPtr->height != 0 )
	    height = masterPtr->height;
	  else
	    return TCL_ERROR;
	}
	if( has_skip == 0 )
	  skip = masterPtr->skip;
	if( has_type == 0 ) {
	  type= masterPtr->datatype;
	  datasize = masterPtr->datasize;
	}
	
	/*
	 * Open the image file and look for a handler for it.
	 */
	f = fopen(realFileName, "rb");
	Tcl_DStringFree(&buffer);
	if (f == NULL) {
	  Tcl_AppendResult(interp, "couldn't read image file \"",
			   options.name, "\": ", Tcl_PosixError(interp),
			   (char *) NULL);
	  return TCL_ERROR;
	}
	/*
	 * Call the handler's file read procedure to read the data
	 * into the image.
	 */
     
	result = FileReadRAW(interp,f,realFileName,(Tk_PictHandle)masterPtr,
			     width,height,type,datasize,skip);
	
	if (f != NULL) {
	  fclose(f);
	}
	return result;
      }
      else if( (strncmp(argv[1], "redither", length) == 0)) {
	
	if (argc == 2) {
	  
	  PictInstance *instancePtr;
	  
	  for (instancePtr = masterPtr->instancePtr; instancePtr != NULL;
	       instancePtr = instancePtr->nextPtr) {
	    DitherInstance(instancePtr, 0, 0, 
			   instancePtr->width,instancePtr->height);
	  }
	  i++;
	  Tk_ImageChanged(masterPtr->tkMaster, 0, 0,
			  masterPtr->width, masterPtr->height,
			  masterPtr->width, masterPtr->height);
	} else {
	  Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			   " redither\"", (char *) NULL);
	  return TCL_ERROR;
	}
      }
    }
    else if ((c == 't') && (strncmp(argv[1], "threshold", length) == 0)) {
      double loval,hival;
      double scale;
      
      if (argc != 4) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " threshold x y\"", (char *) NULL);
	return TCL_ERROR;
      }
      if ((Tcl_GetDouble(interp, argv[2], &loval) != TCL_OK)
	  || (Tcl_GetDouble(interp, argv[3], &hival) != TCL_OK)) {
	return TCL_ERROR;
      }
      
      if( masterPtr->instancePtr != NULL ) {
	scale = 255.0/(masterPtr->dispmax - masterPtr->dispmin);
	loval = (loval - masterPtr->dispmin) * scale;
	hival = (hival - masterPtr->dispmin) * scale;
	
	lut_thres(masterPtr->instancePtr->display,
		  masterPtr->instancePtr->colormap,
		  chColor,
		  masterPtr->instancePtr->ncolors,
		  masterPtr->instancePtr->lut_start,
		  masterPtr->instancePtr->has_overlay,
		  floor(loval),floor(hival));
      }
    } 
    else if ((c == 'w') && (strncmp(argv[1], "write", length) == 0)) {
      /*
       * Pict write command - first parse and check any options given.
       */
      
      index = 2;
      memset((VOID *) &options, 0, sizeof(options));
      options.name = NULL;
      options.format = NULL;
      if (ParseSubcommandOptions(&options, interp, OPT_FORMAT | OPT_FROM,
		&index, argc, argv) != TCL_OK) {
	return TCL_ERROR;
      }
      if ((options.name == NULL) || (index < argc)) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " write fileName ?-format format-name?",
		    "?-from x1 y1 x2 y2?\"", (char *) NULL);
	    return TCL_ERROR;
	  }
      if ((options.fromX > masterPtr->width)
	  || (options.fromY > masterPtr->height)
	  || (options.fromX2 > masterPtr->width)
	  || (options.fromY2 > masterPtr->height)) {
	    Tcl_AppendResult(interp, "coordinates for -from option extend ",
			     "outside image", (char *) NULL);
	    return TCL_ERROR;
	}
      
      /*
       * Fill in default values for unspecified parameters.
       */
      
	if (((options.options & OPT_FROM) == 0) || (options.fromX2 < 0)) {
	  options.fromX2 = masterPtr->width;
	  options.fromY2 = masterPtr->height;
	}
      
      /*
       * Search for an appropriate image file format handler,
       * and give an error if none is found.
       */
      
      matched = 0;
      for (imageFormat = formatList; imageFormat != NULL;
	   imageFormat = imageFormat->nextPtr) {
	    if ((options.format == NULL)
		|| (strncasecmp(options.format, imageFormat->name,
				strlen(imageFormat->name)) == 0)) {
	      matched = 1;
	      if (imageFormat->fileWriteProc != NULL) {
		    break;
		  }
	    }
	  }
      if (imageFormat == NULL) {
	if (options.format == NULL) {
	  Tcl_AppendResult(interp, "no available image file format ",
			   "has file writing capability", (char *) NULL);
	} else if (!matched) {
	  Tcl_AppendResult(interp, "image file format \"",
			   options.format, "\" is unknown", (char *) NULL);
	} else {
	  Tcl_AppendResult(interp, "image file format \"",
			   options.format, "\" has no file writing capability",
			   (char *) NULL);
	}
	return TCL_ERROR;
      }

      /*
       * Call the handler's file write procedure to write out
       * the image.
       */
      
      Tk_PictGetImage((Tk_PictHandle) masterPtr, &block);
      block.pixelPtr += options.fromY * block.pitch + options.fromX;
      block.width = options.fromX2 - options.fromX;
      block.height = options.fromY2 - options.fromY;
      return (*imageFormat->fileWriteProc)(interp, options.name,
					   options.format, &block);
    } else {
      Tcl_AppendResult(interp, "bad option \"", argv[1],
		       "\": must be blank,cget,cmap_stretch,colorbar,colormap,configure,copy,get,",
		       "getline,getmin,getmax,overlay,put, range",
		       "read, rdbinary, redither, or write", (char *) NULL);
      return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ParseSubcommandOptions --
 *
 *	This procedure is invoked to process one of the options
 *	which may be specified for the Pict image subcommands,
 *	namely, -from, -to, -zoom, -subsample, -format, and -shrink.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Fields in *optPtr get filled in.
 *
 *----------------------------------------------------------------------
 */

static int
ParseSubcommandOptions(optPtr, interp, allowedOptions, optIndexPtr, argc, argv)
    struct SubcommandOptions *optPtr;
				/* Information about the options specified
				 * and the values given is returned here. */
    Tcl_Interp *interp;		/* Interpreter to use for reporting errors. */
    int allowedOptions;		/* Indicates which options are valid for
				 * the current command. */
    int *optIndexPtr;		/* Points to a variable containing the
				 * current index in argv; this variable is
				 * updated by this procedure. */
    int argc;			/* Number of arguments in argv[]. */
    char **argv;		/* Arguments to be parsed. */
{
    int index, c, bit, currentBit;
    size_t length;
    char *option, **listPtr;
    int values[4];
    int numValues, maxValues, argIndex;

#ifdef DEBUG
    printf("ParseSubcommandOptions\n");
#endif

    for (index = *optIndexPtr; index < argc; *optIndexPtr = ++index) {
	/*
	 * We can have one value specified without an option;
	 * it goes into optPtr->name.
	 */

	option = argv[index];
	if (option[0] != '-') {
	    if (optPtr->name == NULL) {
		optPtr->name = option;
		continue;
	    }
	    break;
	}

	/*
	 * Work out which option this is.
	 */

	length = strlen(option);
	c = option[0];
	bit = 0;
	currentBit = 1;
	for (listPtr = optionNames; *listPtr != NULL; ++listPtr) {
	    if ((c == *listPtr[0])
		    && (strncmp(option, *listPtr, length) == 0)) {
		if (bit != 0) {
		    bit = 0;	/* An ambiguous option. */
		    break;
		}
		bit = currentBit;
	    }
	    currentBit <<= 1;
	}

	/*
	 * If this option is not recognized and allowed, put
	 * an error message in the interpreter and return.
	 */

	if ((allowedOptions & bit) == 0) {
	    Tcl_AppendResult(interp, "unrecognized option \"", argv[index],
		    "\": must be ", (char *)NULL);
	    bit = 1;
	    for (listPtr = optionNames; *listPtr != NULL; ++listPtr) {
		if ((allowedOptions & bit) != 0) {
		    if ((allowedOptions & (bit - 1)) != 0) {
			Tcl_AppendResult(interp, ", ", (char *) NULL);
			if ((allowedOptions & ~((bit << 1) - 1)) == 0) {
			    Tcl_AppendResult(interp, "or ", (char *) NULL);
			}
		    }
		    Tcl_AppendResult(interp, *listPtr, (char *) NULL);
		}
		bit <<= 1;
	    }
	    return TCL_ERROR;
	}

	/*
	 * For the -from, -to, -zoom and -subsample options,
	 * parse the values given.  Report an error if too few
	 * or too many values are given.
	 */

	if ((bit != OPT_SHRINK) && (bit != OPT_FORMAT)) {
	    maxValues = ((bit == OPT_FROM) || (bit == OPT_TO))? 4: 2;
	    argIndex = index + 1;
	    for (numValues = 0; numValues < maxValues; ++numValues) {
		if ((argIndex < argc) && (isdigit(UCHAR(argv[argIndex][0]))
			|| ((argv[argIndex][0] == '-')
			&& (isdigit(UCHAR(argv[argIndex][1])))))) {
		    if (Tcl_GetInt(interp, argv[argIndex], &values[numValues])
			    != TCL_OK) {
			return TCL_ERROR;
		    }
		} else {
		    break;
		}
		++argIndex;
	    }

	    if (numValues == 0) {
		Tcl_AppendResult(interp, "the \"", argv[index], "\" option ",
			 "requires one ", maxValues == 2? "or two": "to four",
			 " integer values", (char *) NULL);
		return TCL_ERROR;
	    }
	    *optIndexPtr = (index += numValues);

	    /*
	     * Y values default to the corresponding X value if not specified.
	     */

	    if (numValues == 1) {
		values[1] = values[0];
	    }
	    if (numValues == 3) {
		values[3] = values[2];
	    }

	    /*
	     * Check the values given and put them in the appropriate
	     * field of the SubcommandOptions structure.
	     */

	    switch (bit) {
		case OPT_FROM:
		    if ((values[0] < 0) || (values[1] < 0) || ((numValues > 2)
			    && ((values[2] < 0) || (values[3] < 0)))) {
			Tcl_AppendResult(interp, "value(s) for the -from",
				" option must be non-negative", (char *) NULL);
			return TCL_ERROR;
		    }
		    if (numValues <= 2) {
			optPtr->fromX = values[0];
			optPtr->fromY = values[1];
			optPtr->fromX2 = -1;
			optPtr->fromY2 = -1;
		    } else {
			optPtr->fromX = MIN(values[0], values[2]);
			optPtr->fromY = MIN(values[1], values[3]);
			optPtr->fromX2 = MAX(values[0], values[2]);
			optPtr->fromY2 = MAX(values[1], values[3]);
		    }
		    break;
		case OPT_SUBSAMPLE:
		    optPtr->subsampleX = values[0];
		    optPtr->subsampleY = values[1];
		    break;
		case OPT_TO:
		    if ((values[0] < 0) || (values[1] < 0) || ((numValues > 2)
			    && ((values[2] < 0) || (values[3] < 0)))) {
			Tcl_AppendResult(interp, "value(s) for the -to",
				" option must be non-negative", (char *) NULL);
			return TCL_ERROR;
		    }
		    if (numValues <= 2) {
			optPtr->toX = values[0];
			optPtr->toY = values[1];
			optPtr->toX2 = -1;
			optPtr->toY2 = -1;
		    } else {
			optPtr->toX = MIN(values[0], values[2]);
			optPtr->toY = MIN(values[1], values[3]);
			optPtr->toX2 = MAX(values[0], values[2]);
			optPtr->toY2 = MAX(values[1], values[3]);
		    }
		    break;
		case OPT_ZOOM:
		    if ((values[0] <= 0) || (values[1] <= 0)) {
			Tcl_AppendResult(interp, "value(s) for the -zoom",
				" option must be positive", (char *) NULL);
			return TCL_ERROR;
		    }
		    optPtr->zoomX = values[0];
		    optPtr->zoomY = values[1];
		    break;
	    }
	} else if (bit == OPT_FORMAT) {
	    /*
	     * The -format option takes a single string value.
	     */

	    if (index + 1 < argc) {
		*optIndexPtr = ++index;
		optPtr->format = argv[index];
	    } else {
		Tcl_AppendResult(interp, "the \"-format\" option ",
			"requires a value", (char *) NULL);
		return TCL_ERROR;
	    }
	}

	/*
	 * Remember that we saw this option.
	 */

	optPtr->options |= bit;
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ImgPictConfigureMaster --
 *
 *	This procedure is called when a Pict image is created or
 *	reconfigured.  It processes configuration options and resets
 *	any instances of the image.
 *
 * Results:
 *	A standard Tcl return value.  If TCL_ERROR is returned then
 *	an error message is left in masterPtr->interp->result.
 *
 * Side effects:
 *	Existing instances of the image will be redisplayed to match
 *	the new configuration options.
 *
 *----------------------------------------------------------------------
 */

static int
ImgPictConfigureMaster(interp, masterPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Interpreter to use for reporting errors. */
    PictMaster *masterPtr;	/* Pointer to data structure describing
				 * overall Pict image to (re)configure. */
    int argc;			/* Number of entries in argv. */
    char **argv;		/* Pairs of configuration options for image. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget,
				 * such as TK_CONFIG_ARGV_ONLY. */
{
    PictInstance *instancePtr;
    char *oldFileString, *oldDataString, *realFileName;
    int result;
    FILE *f;
    Tk_PictImageFormat *imageFormat;
    int imageWidth, imageHeight;
    Tcl_DString buffer;

#ifdef DEBUG
    printf("ImgPictConfigureMaster\n");
#endif

    /*
     * Save the current values for fileString and dataString, so we
     * can tell if the user specifies them anew.
     */

    oldFileString = masterPtr->fileString;
    oldDataString = (oldFileString == NULL)? masterPtr->dataString: NULL;

    /*
     * Process the configuration options specified.
     */

    if (Tk_ConfigureWidget(interp, Tk_MainWindow(interp), configSpecs,
	    argc, argv, (char *) masterPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Regard the empty string for -file, -data or -format as the null
     * value.
     */

    if ((masterPtr->fileString != NULL) && (masterPtr->fileString[0] == 0)) {
	ckfree(masterPtr->fileString);
	masterPtr->fileString = NULL;
    }
    if ((masterPtr->dataString != NULL) && (masterPtr->dataString[0] == 0)) {
	ckfree(masterPtr->dataString);
	masterPtr->dataString = NULL;
    }
    if ((masterPtr->format != NULL) && (masterPtr->format[0] == 0)) {
	ckfree(masterPtr->format);
	masterPtr->format = NULL;
    }

    /*
     * Set the image to the user-requested size, if any,
     * and make sure storage is correctly allocated for this image.
     */

    ImgPictSetSize(masterPtr, masterPtr->width, masterPtr->height);

    /*
     * Read in the image from the file or string if the user has
     * specified the -file or -data option.
     */

    if ((masterPtr->fileString != NULL)
	    && (masterPtr->fileString != oldFileString)) {

	realFileName = Tcl_TildeSubst(interp, masterPtr->fileString, &buffer);
	if (realFileName == NULL) {
	    return TCL_ERROR;
	}
	f = fopen(realFileName, "rb");
	Tcl_DStringFree(&buffer);
	if (f == NULL) {
	    Tcl_AppendResult(interp, "couldn't read image file \"",
		    masterPtr->fileString, "\": ", Tcl_PosixError(interp),
		    (char *) NULL);
	    return TCL_ERROR;
	}
	if (MatchFileFormat(interp, f, masterPtr->fileString,
		masterPtr->format, &imageFormat, &imageWidth,
		&imageHeight) != TCL_OK) {
	    fclose(f);
	    return TCL_ERROR;
	}
	ImgPictSetSize(masterPtr, imageWidth, imageHeight);
	result = (*imageFormat->fileReadProc)(interp, f, masterPtr->fileString,
		masterPtr->format, (Tk_PictHandle) masterPtr, 0, 0,
		imageWidth, imageHeight, 0, 0);
	fclose(f);
	if (result != TCL_OK) {
	    return TCL_ERROR;
	}

	masterPtr->flags |= IMAGE_CHANGED;
    }

    if ((masterPtr->fileString == NULL) && (masterPtr->dataString != NULL)
	    && (masterPtr->dataString != oldDataString)) {

	if (MatchStringFormat(interp, masterPtr->dataString, 
		masterPtr->format, &imageFormat, &imageWidth,
		&imageHeight) != TCL_OK) {
	    return TCL_ERROR;
	}
	ImgPictSetSize(masterPtr, imageWidth, imageHeight);
	if ((*imageFormat->stringReadProc)(interp, masterPtr->dataString,
		masterPtr->format, (Tk_PictHandle) masterPtr,
		0, 0, imageWidth, imageHeight, 0, 0) != TCL_OK) {
	    return TCL_ERROR;
	}

	masterPtr->flags |= IMAGE_CHANGED;
    }

    /*
     * Cycle through all of the instances of this image, regenerating
     * the information for each instance.  Then force the image to be
     * redisplayed everywhere that it is used.
     */

    for (instancePtr = masterPtr->instancePtr; instancePtr != NULL;
	    instancePtr = instancePtr->nextPtr) {
	ImgPictConfigureInstance(instancePtr);
    }

    /*
     * Inform the generic image code that the image
     * has (potentially) changed.
     */

    Tk_ImageChanged(masterPtr->tkMaster, 0, 0, masterPtr->width,
	    masterPtr->height, masterPtr->width, masterPtr->height);
    masterPtr->flags &= ~IMAGE_CHANGED;

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ImgPictConfigureInstance --
 *
 *	This procedure is called to create displaying information for
 *	a Pict image instance based on the configuration information
 *	in the master.  It is invoked both when new instances are
 *	created and when the master is reconfigured.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Generates errors via Tk_BackgroundError if there are problems
 *	in setting up the instance.
 *
 *----------------------------------------------------------------------
 */

static void
ImgPictConfigureInstance(instancePtr)
    PictInstance *instancePtr;	/* Instance to reconfigure. */
{
    PictMaster *masterPtr = instancePtr->masterPtr;
    Display *disp;
    XImage *imagePtr;
    int bitsPerPixel;
    XRectangle validBox;
    int new_image = 0;

#ifdef DEBUG
    printf("ImgPictConfigureInstance\n");
#endif

    /*
     * Create a new XImage structure for sending data to
     * the X server, if necessary.
     */
    disp = instancePtr->display;
    bitsPerPixel = instancePtr->visualInfo.depth;
    
    if ((instancePtr->imagePtr == NULL)
	|| (instancePtr->imagePtr->bits_per_pixel != bitsPerPixel)) {
      new_image = 1;
      if (instancePtr->imagePtr != NULL) {
	XFree((char *) instancePtr->imagePtr);
      }
      imagePtr = XCreateImage(disp,
			      instancePtr->visualInfo.visual,
			      (unsigned) bitsPerPixel,
			      (bitsPerPixel > 1? ZPixmap: XYBitmap), 
			      0, (char *) NULL,
			      1, 1, 32, 0);
      instancePtr->imagePtr = imagePtr;

      /*
       * Determine the endianness of this machine.
       * We create images using the local host's endianness, rather
       * than the endianness of the server; otherwise we would have
       * to byte-swap any 16 or 32 bit values that we store in the
       * image in those situations where the server's endianness
       * is different from ours.
       */
      
      if (imagePtr != NULL) {
	union {
	  int i;
	  char c[sizeof(int)];
	} kludge;
	
	imagePtr->bitmap_unit = sizeof(pixel) * NBBY;
	kludge.i = 0;
	kludge.c[0] = 1;
	imagePtr->byte_order = (kludge.i == 1) ? LSBFirst : MSBFirst;
	_XInitImageFuncPtrs(imagePtr);
      }
    }
  

    /*
     * If the user has specified a width and/or height for the master
     * which is different from our current width/height, set the size
     * to the values specified by the user.  If we have no pixmap, we
     * do this also, since it has the side effect of allocating a
     * pixmap for us.
     */

    if ((instancePtr->pixels == None)
	|| (instancePtr->width != masterPtr->width)
	|| (instancePtr->height != masterPtr->height)) {
      ImgPictInstanceSetSize(instancePtr);
    }

    /*
     * Redither this instance if necessary.
     */
    
    if ((masterPtr->flags & IMAGE_CHANGED)
	|| (new_image == 1)) {
      XClipBox(masterPtr->validRegion, &validBox);
      if ((validBox.width > 0) && (validBox.height > 0)) {
	DitherInstance(instancePtr, validBox.x, validBox.y,
		       validBox.width, validBox.height);
      }
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ImgPictGet --
 *
 *	This procedure is called for each use of a Pict image in a
 *	widget.
 *
 * Results:
 *	The return value is a token for the instance, which is passed
 *	back to us in calls to ImgPictDisplay and ImgPictFree.
 *
 * Side effects:
 *	A data structure is set up for the instance (or, an existing
 *	instance is re-used for the new one).
 *
 *----------------------------------------------------------------------
 */

static ClientData ImgPictGet(tkwin, masterData)
    Tk_Window tkwin;		/* Window in which the instance will be
				 * used. */
    ClientData masterData;	/* Pointer to our master structure for the
				 * image. */
{
    PictMaster *masterPtr = (PictMaster *) masterData;
    PictInstance *instancePtr;
    Display *disp;
    Colormap cmap;
    XVisualInfo *visInfoPtr;
    XRectangle validBox;
    XColor *white, *black;
    XGCValues gcValues;
    int i,r;
    int ncolors;
    int lut_start;
    int color_nb;
    char colormap_level;
    char atom = 0;

#ifdef DEBUG
    printf("ImgPictGet\n");
#endif

    disp = Tk_Display(tkwin);
    visInfoPtr = get_visual(disp);
    colormap_level = Private_Colormap;

    if(colormap_level == READ_SHARED_COLORMAP) {
      if( !init_colors(disp,&cmap,visInfoPtr->visual,
		       &colormap_level,&ncolors,&lut_start,&atom) ) {
	(void)fprintf (stderr,
		 "ERROR: no shared colormap exists.\n");
	(void)fprintf (stderr,
		 "Using the default colormap instead \n");
	colormap_level = DEFAULT_SCREEN_COLORMAP;
      }
    } 
    if(colormap_level == DEFAULT_SCREEN_COLORMAP) {
      if( Default_Screen_Allocated == 0 ) {
	if( !init_colors(disp,&cmap,visInfoPtr->visual,
			 &colormap_level,&ncolors,&lut_start,&atom) ) {
	  (void)fprintf(stderr,"ERROR: not enough colors in screen Default Colormap\n");
	  (void)fprintf(stderr,"Creating a default private colormap instead \n");
	  colormap_level = DEFAULT_PRIVATE_COLORMAP;
	}
	else {
	  Default_Screen_Colormap = cmap;
	  Default_Screen_ncolors = ncolors;
	  Default_Screen_lut_start = lut_start;
	  Default_Screen_Allocated = 1;
	}
      }
      else {
	cmap = Default_Screen_Colormap;
	ncolors = Default_Screen_ncolors;
	lut_start = Default_Screen_lut_start;
      }
    }
    if(colormap_level == DEFAULT_PRIVATE_COLORMAP) {
      if( Default_Private_Allocated == 0 ) {
	if( !init_colors(disp,&cmap,visInfoPtr->visual,
			 &colormap_level,&ncolors,&lut_start,&atom) ) {
	  (void)fprintf(stderr,"init_colors failed \n");
	  exit(0);
	}
	else {
	  Default_Private_Colormap = cmap;
	  Default_Private_ncolors = ncolors;
	  Default_Private_lut_start = lut_start;
	  Default_Private_Allocated = 1;
	}
      }
      else {
	cmap = Default_Private_Colormap;
	ncolors = Default_Private_ncolors;
	lut_start = Default_Private_lut_start;
      }
    }
    if(colormap_level == NEW_PRIVATE_COLORMAP) {
      if( !init_colors(disp,&cmap,visInfoPtr->visual,
		       &colormap_level,&ncolors,&lut_start,&atom) ) {
	(void)fprintf(stderr,"init_colors failed \n");
	exit(0);
      }
    }
    Private_Colormap = DEFAULT_SCREEN_COLORMAP;

    ((Tk_FakeWin *) (tkwin))->atts.colormap = cmap;

    /*
     * Make a new instance of the image.
     */
    instancePtr = (PictInstance *) ckalloc(sizeof(PictInstance));
    if(instancePtr == NULL) {
      (void)fprintf(stderr,"ImgPictGet: Could not allocate memory\n");
      return 0;
    }

    instancePtr->masterPtr = masterPtr;
    instancePtr->display = disp;
    instancePtr->colormap = cmap;
    instancePtr->colormap_level = colormap_level;
    instancePtr->has_overlay = False;
    instancePtr->ncolors = ncolors;
    instancePtr->lut_start = lut_start;
    instancePtr->refCount = 1;
    instancePtr->pixels = None;
    instancePtr->width = 0;
    instancePtr->height = 0;
    instancePtr->imagePtr = 0;
    instancePtr->nextPtr = masterPtr->instancePtr;
    masterPtr->instancePtr = instancePtr;
    instancePtr->visualInfo = *visInfoPtr;
    XFree(visInfoPtr);
    instancePtr->atom = atom;
    color_nb = ncolors-1;

    for (i = 0; i < 256; ++i) {
      r = i*color_nb/255+lut_start;
      instancePtr->redValues[i] = lut_colorcell_defs[r].pixel;
    }
    
    /*
     * Make a GC with background = black and foreground = white.
     */

    white = Tk_GetColor(masterPtr->interp, tkwin, "white");
    black = Tk_GetColor(masterPtr->interp, tkwin, "black");
    gcValues.foreground = (white != NULL)? white->pixel:
	    WhitePixelOfScreen(Tk_Screen(tkwin));
    gcValues.background = (black != NULL)? black->pixel:
	    BlackPixelOfScreen(Tk_Screen(tkwin));
    gcValues.graphics_exposures = False;
    instancePtr->gc = Tk_GetGC(tkwin,
	    GCForeground|GCBackground|GCGraphicsExposures, &gcValues);
    instancePtr->setgc = GXcopy;
    /* Set configuration options and finish the initialization 
       of the instance. */
    ImgPictConfigureInstance(instancePtr);

    /* If this is the first instance, must set the size of the image. */
    if (instancePtr->nextPtr == NULL) {
	Tk_ImageChanged(masterPtr->tkMaster, 0, 0, 0, 0,
		masterPtr->width, masterPtr->height);
    }
    
     /* If we have no pixmap, we do this also, since it has the side
	effect of allocating a pixmap for us.  */
    if (instancePtr->pixels == None) {
      XClipBox(masterPtr->validRegion, &validBox);
      if ((validBox.width > 0) && (validBox.height > 0)) {
	DitherInstance(instancePtr, validBox.x, validBox.y, validBox.width,
		       validBox.height);
      }
    }

    return (ClientData) instancePtr;
}


/*
 *----------------------------------------------------------------------
 *
 * ImgPictDisplay --
 *
 *	This procedure is invoked to draw a Pict image.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A portion of the image gets rendered in a pixmap or window.
 *
 *----------------------------------------------------------------------
 */
static
void ImgPictDisplay(clientData, display, drawable, imageX, imageY, width,
	height, drawableX, drawableY)
    ClientData clientData;	/* Pointer to PictInstance structure for
				 * for instance to be displayed. */
    Display *display;		/* Display on which to draw image. */
    Drawable drawable;		/* Pixmap or window in which to draw image. */
    int imageX, imageY;		/* Upper-left corner of region within image
				 * to draw. */
    int width, height;		/* Dimensions of region within image to draw. */
    int drawableX, drawableY;	/* Coordinates within drawable that
				 * correspond to imageX and imageY. */
{
    PictInstance *instancePtr = (PictInstance *) clientData;

#ifdef DEBUG
    printf("ImgPictDisplay\n");
#endif

    /*
     * If there's no pixmap, it means that an error occurred
     * while creating the image instance so it can't be displayed.
     */

    if (instancePtr->pixels == None) {
	return;
    }

    /*
     * masterPtr->region describes which parts of the image contain
     * valid data.  We set this region as the clip mask for the gc,
     * setting its origin appropriately, and use it when drawing the
     * image.
     */

    XSetRegion(display, instancePtr->gc, instancePtr->masterPtr->validRegion);
    XSetClipOrigin(display, instancePtr->gc, drawableX - imageX,
	    drawableY - imageY);
    XCopyArea(display, instancePtr->pixels, drawable, instancePtr->gc,
	    imageX, imageY, (unsigned) width, (unsigned) height,
	    drawableX, drawableY);
    XSetClipMask(display, instancePtr->gc, None);
    XSetClipOrigin(display, instancePtr->gc, 0, 0);
}

/*
 *----------------------------------------------------------------------
 *
 * ImgPictFree --
 *
 *	This procedure is called when a widget ceases to use a
 *	particular instance of an image.  We don't actually get
 *	rid of the instance until later because we may be about
 *	to get this instance again.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Internal data structures get cleaned up, later.
 *
 *----------------------------------------------------------------------
 */
static
void ImgPictFree(clientData, display)
    ClientData clientData;	/* Pointer to PictInstance structure for
				 * for instance to be displayed. */
    Display *display;		/* Display containing window that used image. */
{
    PictInstance *instancePtr = (PictInstance *) clientData;

#ifdef DEBUG
    printf("ImgPictFree\n");
#endif

    instancePtr->refCount -= 1;
    if (instancePtr->refCount > 0) {
	return;
    }

    /* There are no more uses of the image within this widget.
       free the instance structure. */
    DisposeInstance((ClientData) instancePtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ImgPictDelete --
 *
 *	This procedure is called by the image code to delete the
 *	master structure for an image.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with the image get freed.
 *
 *----------------------------------------------------------------------
 */
static
void ImgPictDelete(masterData)
    ClientData masterData;	/* Pointer to PictMaster structure for
				 * image.  Must not have any more instances. */
{
    PictMaster *masterPtr = (PictMaster *) masterData;
    PictInstance *instancePtr;

#ifdef DEBUG
    printf("ImgPictDelete\n");
#endif

    while ((instancePtr = masterPtr->instancePtr) != NULL) {
	if (instancePtr->refCount > 0) {
	    panic("tried to delete Pict image when instances still exist");
	}
	Tk_CancelIdleCall(DisposeInstance, (ClientData) instancePtr);
	DisposeInstance((ClientData) instancePtr);
    }
    masterPtr->tkMaster = NULL;
    if (masterPtr->imageCmd != NULL) {
	Tcl_DeleteCommand(masterPtr->interp,
		Tcl_GetCommandName(masterPtr->interp, masterPtr->imageCmd));
    }
    if (masterPtr->data != NULL) {
      if( (char*)(masterPtr->data) == (char*)(masterPtr->bytedata) ) {
	ckfree((char *) masterPtr->data);
	masterPtr->data = NULL;
	masterPtr->bytedata = NULL;
      } else {
	ckfree((char *) masterPtr->data);
	masterPtr->data = NULL;
      }
    }
    if (masterPtr->bytedata != NULL) {
	ckfree((char *) masterPtr->bytedata);
    }
    if (masterPtr->validRegion != NULL) {
	XDestroyRegion(masterPtr->validRegion);
    }
    Tk_FreeOptions(configSpecs, (char *) masterPtr, (Display *) NULL, 0);
    ckfree((char *) masterPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ImgPictCmdDeletedProc --
 *
 *	This procedure is invoked when the image command for an image
 *	is deleted.  It deletes the image.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The image is deleted.
 *
 *----------------------------------------------------------------------
 */

static void
ImgPictCmdDeletedProc(clientData)
    ClientData clientData;	/* Pointer to PictMaster structure for
				 * image. */
{
    PictMaster *masterPtr = (PictMaster *) clientData;

#ifdef DEBUG
    printf("ImgPictCmdDeletedProc\n");
#endif

    masterPtr->imageCmd = NULL;
    if (masterPtr->tkMaster != NULL) {
	Tk_DeleteImage(masterPtr->interp, Tk_NameOfImage(masterPtr->tkMaster));
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ImgPictSetSize --
 *
 *	This procedure reallocates the image storage and instance
 *	pixmaps for a Pict image, as necessary, to change the
 *	image's size to `width' x `height' pixels.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Storage gets reallocated, for the master and all its instances.
 *
 *----------------------------------------------------------------------
 */

static void
ImgPictSetSize(masterPtr, width, height)
    PictMaster *masterPtr;
    int width, height;
{
  char *newData;
  int h, offset, pitch;
  char *srcPtr, *destPtr;
  XRectangle validBox, clipBox;
  Region clipRegion;
  PictInstance *instancePtr;

#ifdef DEBUG
    printf("ImgPictSetSize\n");
#endif

    if (masterPtr->userWidth > 0) {
	width = masterPtr->userWidth;
    }
    if (masterPtr->userHeight > 0) {
	height = masterPtr->userHeight;
    }

    /*
     * We have to trim the valid region if it is currently
     * larger than the new image size.
     */

    XClipBox(masterPtr->validRegion, &validBox);
    if ((validBox.x + validBox.width > (unsigned) width)
	    || (validBox.y + validBox.height > (unsigned) height)) {
	clipBox.x = 0;
	clipBox.y = 0;
	clipBox.width = width;
	clipBox.height = height;
	clipRegion = XCreateRegion();
	XUnionRectWithRegion(&clipBox, clipRegion, clipRegion);
	XIntersectRegion(masterPtr->validRegion, clipRegion,
		masterPtr->validRegion);
	XDestroyRegion(clipRegion);
	XClipBox(masterPtr->validRegion, &validBox);
    }

    if ((width != masterPtr->width) || (height != masterPtr->height)) {
      if( masterPtr->data == NULL ) {
	masterPtr->width = width;
	masterPtr->height = height;
      }
      else {
	
	/*
	 * Reallocate storage for the byte image and copy
	 * over valid regions.
	 */
	
	pitch = width;
	newData = (char *) ckalloc((unsigned) (height * pitch*masterPtr->datasize));
	if(newData == NULL) {
	  (void)fprintf(stderr,"ImgPictSetSize: Could not allocate memory\n");
	  return;
	}
	/*
	 * Zero the new array.  The dithering code shouldn't read the
	 * areas outside validBox, but they might be copied to another
	 * Pict image or written to a file.
	 */
	
	if ((masterPtr->data != NULL)
	    && ((width == masterPtr->width) || (width == validBox.width))) {
	  if (validBox.y > 0) {
	    memset((VOID *) newData, 0, (size_t) (validBox.y * pitch*masterPtr->datasize));
	  }
	  h = validBox.y + validBox.height;
	  if (h < height) {
	    memset((VOID *) (newData + h * pitch), 0,
			(size_t) ((height - h) * pitch));
	  }
	} else {
	  memset((VOID *) newData, 0, (size_t) (height * pitch*masterPtr->datasize));
	}
	
	if (masterPtr->data != NULL) {
	  
	  /*
	   * Copy the common area over to the new array array and
	   * free the old array.
	   */
	  
	    if (width == masterPtr->width) {
	      
		/*
		 * The region to be copied is contiguous.
		 */
	      
		offset = validBox.y * pitch;
		memcpy((VOID *) (newData + offset),
		       (VOID *) (masterPtr->data + offset),
		       (size_t) (validBox.height * pitch*masterPtr->datasize));
		
	    } else if ((validBox.width > 0) && (validBox.height > 0)) {
	      
		/*
		 * Area to be copied is not contiguous - copy line by line.
		 */

	      destPtr = newData + (validBox.y * width + validBox.x)*masterPtr->datasize;
	      srcPtr = masterPtr->data + (validBox.y * masterPtr->width
					  + validBox.x)*masterPtr->datasize;
	      for (h = validBox.height; h > 0; h--) {
		memcpy((VOID *) destPtr, (VOID *) srcPtr,
			    (size_t) (validBox.width*masterPtr->datasize));
		destPtr += width*masterPtr->datasize ;
		srcPtr += masterPtr->width*masterPtr->datasize;
	      }
	    }
	    if (masterPtr->data != NULL) {
	      if ((char*)(masterPtr->data) == (char*)(masterPtr->bytedata)) {
		free((void*)masterPtr->data);
		masterPtr->data = NULL;
		masterPtr->bytedata = NULL;
	      } else {
		free((void*)masterPtr->data);
		masterPtr->data = NULL;
		free((void*)masterPtr->bytedata);
		masterPtr->bytedata = NULL;
	      }
	    }
	  }
	masterPtr->data = newData;
	masterPtr->width = width;
	masterPtr->height = height;
	normalize_data(masterPtr);
      } 
    }

  /*
   * Now adjust the sizes of the pixmaps for all of the instances.
   */
  
  for (instancePtr = masterPtr->instancePtr; instancePtr != NULL;
       instancePtr = instancePtr->nextPtr) {
    ImgPictInstanceSetSize(instancePtr);
  }
}

/*
 *----------------------------------------------------------------------
 *
 * ImgPictInstanceSetSize --
 *
 * 	This procedure reallocates the instance pixmap and dithering
 *	error array for a Pict instance, as necessary, to change the
 *	image's size to `width' x `height' pixels.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Storage gets reallocated, here and in the X server.
 *
 *----------------------------------------------------------------------
 */

static void
ImgPictInstanceSetSize(instancePtr)
    PictInstance *instancePtr;		/* Instance whose size is to be
					 * changed. */
{
    PictMaster *masterPtr;
    XRectangle validBox;
    Pixmap newPixmap;

#ifdef DEBUG
    printf("ImgPictInstanceSetSize\n");
#endif

    masterPtr = instancePtr->masterPtr;
    XClipBox(masterPtr->validRegion, &validBox);

    if ((instancePtr->width != masterPtr->width)
	    || (instancePtr->height != masterPtr->height)
	    || (instancePtr->pixels == None)) {
	newPixmap = Tk_GetPixmap(instancePtr->display,
		RootWindow(instancePtr->display,
		    instancePtr->visualInfo.screen),
	(masterPtr->width > 0) ? masterPtr->width: 1,
		(masterPtr->height > 0) ? masterPtr->height: 1,
		instancePtr->visualInfo.depth);

	if (instancePtr->pixels != None) {
	    /*
	     * Copy any common pixels from the old pixmap and free it.
	     */
	    XCopyArea(instancePtr->display, instancePtr->pixels, newPixmap,
		    instancePtr->gc, validBox.x, validBox.y,
		    validBox.width, validBox.height, validBox.x, validBox.y);
	    Tk_FreePixmap(instancePtr->display, instancePtr->pixels);
	}
	instancePtr->pixels = newPixmap;
    }

    instancePtr->width = masterPtr->width;
    instancePtr->height = masterPtr->height;
}

/*
 *----------------------------------------------------------------------
 *
 * DisposeInstance --
 *
 *	This procedure is called to finally free up an instance
 *	of a Pict image which is no longer required.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The instance data structure and the resources it references
 *	are freed.
 *
 *----------------------------------------------------------------------
 */

void
DisposeInstance(clientData)
    ClientData clientData;	/* Pointer to the instance whose resources
				 * are to be released. */
{
    PictInstance *instancePtr = (PictInstance *) clientData;
    PictInstance *prevPtr;

#ifdef DEBUG
    printf("DisposeInstance\n");
#endif
    if(instancePtr->has_overlay) {
      XFreeGC(instancePtr->display,instancePtr->overlay_gc);
      instancePtr->has_overlay = False;
    }
    if(instancePtr->colormap != Default_Screen_Colormap) {
	XFreeColormap(instancePtr->display,
		       instancePtr->colormap);
      }   
 
    if(instancePtr->atom == 1) {
      deinit_disp(instancePtr->display);
    }

    if (instancePtr->pixels != None) {
	Tk_FreePixmap(instancePtr->display, instancePtr->pixels);
    }
    if (instancePtr->gc != None) {
	Tk_FreeGC(instancePtr->display, instancePtr->gc);
    }
    if (instancePtr->imagePtr != NULL) {
	XFree((char *) instancePtr->imagePtr);
    }
    
    if (instancePtr->masterPtr->instancePtr == instancePtr) {
	instancePtr->masterPtr->instancePtr = instancePtr->nextPtr;
    } else {
	for (prevPtr = instancePtr->masterPtr->instancePtr;
		prevPtr->nextPtr != instancePtr; prevPtr = prevPtr->nextPtr) {
	    /* Empty loop body */
	}
	prevPtr->nextPtr = instancePtr->nextPtr;
    }
    ckfree((char *) instancePtr);
}

/*
 *----------------------------------------------------------------------
 *
 * MatchFileFormat --
 *
 *	This procedure is called to find a Pict image file format
 *	handler which can parse the image data in the given file.
 *	If a user-specified format string is provided, only handlers
 *	whose names match a prefix of the format string are tried.
 *
 * Results:
 *	A standard TCL return value.  If the return value is TCL_OK, a
 *	pointer to the image format record is returned in
 *	*imageFormatPtr, and the width and height of the image are
 *	returned in *widthPtr and *heightPtr.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
MatchFileFormat(interp, f, fileName, formatString, imageFormatPtr,
	widthPtr, heightPtr)
    Tcl_Interp *interp;		/* Interpreter to use for reporting errors. */
    FILE *f;			/* The image file, open for reading. */
    char *fileName;		/* The name of the image file. */
    char *formatString;		/* User-specified format string, or NULL. */
    Tk_PictImageFormat **imageFormatPtr;
				/* A pointer to the Pict image format
				 * record is returned here. */
    int *widthPtr, *heightPtr;	/* The dimensions of the image are
				 * returned here. */
{
    int matched;
    Tk_PictImageFormat *formatPtr;

#ifdef DEBUG
    printf("MatchFileFormat\n");
#endif

    /*
     * Scan through the table of file format handlers to find
     * one which can handle the image.
     */

    matched = 0;
    for (formatPtr = formatList; formatPtr != NULL;
	 formatPtr = formatPtr->nextPtr) {
	if ((formatString != NULL)
		&& (strncasecmp(formatString, formatPtr->name,
		strlen(formatPtr->name)) != 0)) {
	    continue;
	}
	matched = 1;
	if (formatPtr->fileMatchProc != NULL) {
	    fseek(f, 0L, SEEK_SET);
	    if ((*formatPtr->fileMatchProc)(f, fileName, formatString,
		    widthPtr, heightPtr)) {
		if (*widthPtr < 1) {
		    *widthPtr = 1;
		}
		if (*heightPtr < 1) {
		    *heightPtr = 1;
		}
		break;
	    }
	}
    }

    if (formatPtr == NULL) {
	if ((formatString != NULL) && !matched) {
	    Tcl_AppendResult(interp, "image file format \"", formatString,
		    "\" is unknown", (char *) NULL);
	} else {
	    Tcl_AppendResult(interp,
		    "couldn't recognize data in image file \"",
		    fileName, "\"", (char *) NULL);
	}
	return TCL_ERROR;
    }

    *imageFormatPtr = formatPtr;
    fseek(f, 0L, SEEK_SET);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * MatchStringFormat --
 *
 *	This procedure is called to find a Pict image file format
 *	handler which can parse the image data in the given string.
 *	If a user-specified format string is provided, only handlers
 *	whose names match a prefix of the format string are tried.
 *
 * Results:
 *	A standard TCL return value.  If the return value is TCL_OK, a
 *	pointer to the image format record is returned in
 *	*imageFormatPtr, and the width and height of the image are
 *	returned in *widthPtr and *heightPtr.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
MatchStringFormat(interp, string, formatString, imageFormatPtr,
	widthPtr, heightPtr)
    Tcl_Interp *interp;		/* Interpreter to use for reporting errors. */
    char *string;		/* String containing the image data. */
    char *formatString;		/* User-specified format string, or NULL. */
    Tk_PictImageFormat **imageFormatPtr;
				/* A pointer to the Pict image format
				 * record is returned here. */
    int *widthPtr, *heightPtr;	/* The dimensions of the image are
				 * returned here. */
{
    int matched;
    Tk_PictImageFormat *formatPtr;

#ifdef DEBUG
    printf("MatchStringFormat\n");
#endif

    /*
     * Scan through the table of file format handlers to find
     * one which can handle the image.
     */
    matched = 0;
    for (formatPtr = formatList; formatPtr != NULL;
	    formatPtr = formatPtr->nextPtr) {
	if ((formatString != NULL) && (strncasecmp(formatString,
		formatPtr->name, strlen(formatPtr->name)) != 0)) {
	    continue;
	}
	matched = 1;
	if ((formatPtr->stringMatchProc != NULL)
		&& (*formatPtr->stringMatchProc)(string, formatString,
		widthPtr, heightPtr)) {
	    break;
	}
    }

    if (formatPtr == NULL) {
	if ((formatString != NULL) && !matched) {
	    Tcl_AppendResult(interp, "image file format \"", formatString,
		    "\" is unknown", (char *) NULL);
	} else {
	    Tcl_AppendResult(interp, "no format found to parse",
		    " image data string", (char *) NULL);
	}
	return TCL_ERROR;
    }

    *imageFormatPtr = formatPtr;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_FindPict --
 *
 *	This procedure is called to get an opaque handle (actually a
 *	PictMaster *) for a given image, which can be used in
 *	subsequent calls to Tk_PictPutBlock, etc.  The `name'
 *	parameter is the name of the image.
 *
 * Results:
 *	The handle for the Pict image, or NULL if there is no
 *	Pict image with the name given.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tk_PictHandle
Tk_FindPict(imageName)
    char *imageName;		/* Name of the desired Pict image. */
{
    Tcl_HashEntry *entry;

#ifdef DEBUG
    printf("Tk_FindPict\n");
#endif

    if (!imgPictHashInitialized) {
	return NULL;
    }
    entry = Tcl_FindHashEntry(&imgPictHash, imageName);
    if (entry == NULL) {
	return NULL;
    }
    return (Tk_PictHandle) Tcl_GetHashValue(entry);
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_PictPutBlock --
 *
 *	This procedure is called to put image data into a Pict image.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The image data is stored.  The image may be expanded.
 *	The Tk image code is informed that the image has changed.
 *
 *---------------------------------------------------------------------- */

void
Tk_PictPutBlock(handle, blockPtr, x, y, width, height)
    Tk_PictHandle handle;	/* Opaque handle for the Pict image
				 * to be updated. */
    register Tk_PictImageBlock *blockPtr;
				/* Pointer to a structure describing the
				 * pixel data to be copied into the image. */
    int x, y;			/* Coordinates of the top-left pixel to
				 * be updated in the image. */
    int width, height;		/* Dimensions of the area of the image
				 * to be updated. */
{
    register PictMaster *masterPtr;
    PictInstance *instancePtr;
    int xEnd, yEnd;    
    XRectangle rect;
  
    int i,j;

#ifdef DEBUG
    printf("Tk_PictPutBlock\n");
#endif

    masterPtr = (PictMaster *) handle;

    if ((masterPtr->userWidth != 0) && ((x + width) > masterPtr->userWidth)) {
	width = masterPtr->userWidth - x;
    }
    if ((masterPtr->userHeight != 0)
	    && ((y + height) > masterPtr->userHeight)) {
	height = masterPtr->userHeight - y;
    }
    if ((width <= 0) || (height <= 0))
	return;
   
    xEnd = x + width;
    yEnd = y + height;

    if ((xEnd > masterPtr->width) || (yEnd > masterPtr->height)) {
	ImgPictSetSize(masterPtr, xEnd, yEnd);
    }
   
    if((x!= 0) || 
       (y!= 0) || 
       (masterPtr->width != blockPtr->width) ||
       (masterPtr->height != blockPtr->height)) {
#ifdef DEBUG
      printf(" needs copy \n");
#endif
      blockPtr->copy = COPY;
    }

   
   
    if( blockPtr->copy == COPY ) 
      {

	if( masterPtr->data == NULL ) {
#ifdef DEBUG
	  printf("needs allocation \n");
#endif
	  masterPtr->datatype = blockPtr->datatype; 
	  masterPtr->datasize = blockPtr->pixelSize; 

	  masterPtr->data = (char*)malloc((size_t)masterPtr->datasize*
					  masterPtr->width*
					  masterPtr->height);
	  if( masterPtr->data == NULL ) {
	    (void)fprintf(stderr,"Could not allocate memory \n");
	    return;
	  }
	} 
	else {
	  if (masterPtr->datatype != blockPtr->datatype ) {
	    (void)fprintf(stderr,"Type mismatch \n");
	    return;
	  }
	}
	if(masterPtr->width == blockPtr->width 
	   && masterPtr->height == blockPtr->height)
	  masterPtr->skip = blockPtr->skip;

	if( blockPtr->datatype == BYTE ) {
	  for(i=0;i<width;i++)
	    for(j=0;j<height;j++)
	      masterPtr->data[i+x+(j+y)*masterPtr->width] =  
		blockPtr->pixelPtr[i+j*blockPtr->pitch];
	}
	else if( blockPtr->datatype == WORD ) {
	  short *srcPtr = (short*)blockPtr->pixelPtr;
	  short *destPtr = (short*)masterPtr->data;
	  for(i=0;i<width;i++)
	    for(j=0;j<height;j++)
	      destPtr[i+x+(j+y)*masterPtr->width] = 
		srcPtr[i+j*blockPtr->pitch];
	}
	else if( blockPtr->datatype == LWORD ) {
	  int *srcPtr = (int*)blockPtr->pixelPtr;
	  int *destPtr = (int*)masterPtr->data;
	  for(i=0;i<width;i++)
	    for(j=0;j<height;j++)
	      destPtr[i+x+(j+y)*masterPtr->width] = 
		srcPtr[i+j*blockPtr->pitch];
	}
	else if( blockPtr->datatype == REAL ) {
	  float *srcPtr = (float*)blockPtr->pixelPtr;
	  float *destPtr = (float*)masterPtr->data;
	  for(i=0;i<width;i++)
	    for(j=0;j<height;j++)
	      destPtr[i+x+(j+y)*masterPtr->width] = 
		srcPtr[i+j*blockPtr->pitch];
	}
      }
    else {
      
      if( masterPtr->bytedata != NULL ) {
	if( (char*)masterPtr->bytedata == (char*)masterPtr->data ) {
	  free((void*)masterPtr->bytedata);
	  masterPtr->bytedata = NULL;
	  masterPtr->data = NULL;
	} else {
	  free((void*)masterPtr->bytedata);
	  masterPtr->bytedata = NULL;
	}
      }
      if( masterPtr->data != NULL ) {
	free((void*)masterPtr->data);
	masterPtr->data = NULL;
      }
      masterPtr->datatype = blockPtr->datatype; 
      masterPtr->datasize = blockPtr->pixelSize; 
      masterPtr->skip = blockPtr->skip;
      
      /* Put the data into our local data array */
      masterPtr->data = (char*)blockPtr->pixelPtr;
    }
     
    normalize_data(masterPtr);
    blockPtr->pixelPtr = NULL;

    /*
     * Add this new block to the region which specifies which data is valid.
     */

    rect.x = x;
    rect.y = y;
    rect.width = width;
    rect.height = height;
    XUnionRectWithRegion(&rect, masterPtr->validRegion,
	    masterPtr->validRegion);

    /*
     * Update each instance.
     */
    for(instancePtr = masterPtr->instancePtr; instancePtr != NULL;
	instancePtr = instancePtr->nextPtr)
      DitherInstance(instancePtr, x, y, width, height);
   
    /*
     * Tell the core image code that this image has changed.
     */

    Tk_ImageChanged(masterPtr->tkMaster, x, y, width, height, masterPtr->width,
	    masterPtr->height);
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_PictPutZoomedBlock --
 *
 *	This procedure is called to put image data into a Pict image,
 *	with possible subsampling and/or zooming of the pixels.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The image data is stored.  The image may be expanded.
 *	The Tk image code is informed that the image has changed.
 *
 *----------------------------------------------------------------------
 */

void
Tk_PictPutZoomedBlock(handle, blockPtr, x, y, width, height, zoomX, zoomY,
	subsampleX, subsampleY)
    Tk_PictHandle handle;	/* Opaque handle for the Pict image
				 * to be updated. */
    register Tk_PictImageBlock *blockPtr;
				/* Pointer to a structure describing the
				 * pixel data to be copied into the image. */
    int x, y;			/* Coordinates of the top-left pixel to
				 * be updated in the image. */
    int width, height;		/* Dimensions of the area of the image
				 * to be updated. */
    int zoomX, zoomY;		/* Zoom factors for the X and Y axes. */
    int subsampleX, subsampleY;	/* Subsampling factors for the X and Y axes. */
{
    register PictMaster *masterPtr;
    PictInstance *instancePtr;
    int xEnd, yEnd;
    int wLeft, hLeft;
    int wCopy, hCopy;
    int blockWid, blockHt;
    unsigned char *srcPtr, *srcLinePtr, *srcOrigPtr;
    unsigned char *destPtr, *destLinePtr;
    int pitch;
    int xRepeat, yRepeat;
    int blockXSkip, blockYSkip;
    XRectangle rect;
    register int il;

#ifdef DEBUG
    printf("Tk_PictPutZoomedBlock\n");
#endif

    if ((zoomX == 1) && (zoomY == 1) && (subsampleX == 1)
	    && (subsampleY == 1)) {
	Tk_PictPutBlock(handle, blockPtr, x, y, width, height);
	return;
    }

    masterPtr = (PictMaster *) handle;

    if ((zoomX <= 0) || (zoomY <= 0))
	return;
    if ((masterPtr->userWidth != 0) && ((x + width) > masterPtr->userWidth)) {
	width = masterPtr->userWidth - x;
    }
    if ((masterPtr->userHeight != 0)
	    && ((y + height) > masterPtr->userHeight)) {
	height = masterPtr->userHeight - y;
    }
    if ((width <= 0) || (height <= 0))
	return;

    xEnd = x + width;
    yEnd = y + height;
    if ((xEnd > masterPtr->width) || (yEnd > masterPtr->height)) {
	ImgPictSetSize(masterPtr, xEnd, yEnd);
    }

  
    if( masterPtr->data == NULL ) {
#ifdef DEBUG
      printf("needs allocation \n");
#endif

      masterPtr->datatype = blockPtr->datatype;
      masterPtr->datasize = blockPtr->pixelSize;

      masterPtr->data = (char*)malloc((size_t)masterPtr->datasize*
				      masterPtr->width*
				      masterPtr->height);
      if( masterPtr->data == NULL ) {
	(void)fprintf(stderr,"Could not allocate memory \n");
	return;
      }


    } else {
      if (masterPtr->datatype != blockPtr->datatype ) {
	(void)fprintf(stderr,"Type mismatch \n");
	return;
      }
    }

    /*
     * Work out what area the pixel data in the block expands to after
     * subsampling and zooming.
     */

    blockXSkip = subsampleX * blockPtr->pixelSize;
    blockYSkip = subsampleY * blockPtr->pitch * blockPtr->pixelSize;
    if (subsampleX > 0)
	blockWid = ((blockPtr->width + subsampleX - 1) / subsampleX) * zoomX;
    else if (subsampleX == 0)
	blockWid = width;
    else
	blockWid = ((blockPtr->width - subsampleX - 1) / -subsampleX) * zoomX;
    if (subsampleY > 0)
	blockHt = ((blockPtr->height + subsampleY - 1) / subsampleY) * zoomY;
    else if (subsampleY == 0)
	blockHt = height;
    else
	blockHt = ((blockPtr->height - subsampleY - 1) / -subsampleY) * zoomY;

         
    destLinePtr = (unsigned char*)(masterPtr->data + 
      (y * masterPtr->width + x)*masterPtr->datasize);
    srcOrigPtr = blockPtr->pixelPtr;
    if (subsampleX < 0) {
      srcOrigPtr += (blockPtr->width - 1) * blockPtr->pixelSize;
    }
    if (subsampleY < 0) {
      srcOrigPtr += (blockPtr->height - 1) * 
	blockPtr->pitch * blockPtr->pixelSize; 
    }
    
    pitch = masterPtr->width*masterPtr->datasize;
    for (hLeft = height; hLeft > 0; ) {
      hCopy = MIN(hLeft, blockHt);
      hLeft -= hCopy;
      yRepeat = zoomY;
      srcLinePtr = srcOrigPtr;
      for (; hCopy > 0; --hCopy) {
	destPtr = destLinePtr;
	for (wLeft = width; wLeft > 0;) {
	  wCopy = MIN(wLeft, blockWid);
	  wLeft -= wCopy;
	  srcPtr = srcLinePtr;
	  for (; wCopy > 0; wCopy -= zoomX) {
	    for (xRepeat = MIN(wCopy, zoomX); xRepeat > 0; xRepeat--) {
	      for(il=0;il<masterPtr->datasize;il++)
		*destPtr++ = srcPtr[il];
	    }
	    srcPtr += blockXSkip;
	  }
	}
	destLinePtr += pitch;
	yRepeat--;
	if (yRepeat <= 0) {
	  srcLinePtr += blockYSkip;
	  yRepeat = zoomY;
	}
      }
    }
       
    normalize_data(masterPtr);
    blockPtr->pixelPtr = NULL;

    /*
     * Add this new block to the region that specifies which data is valid.
     */

    rect.x = x;
    rect.y = y;
    rect.width = width;
    rect.height = height;
    XUnionRectWithRegion(&rect, masterPtr->validRegion,
	    masterPtr->validRegion);

    /*
     * Update each instance.
     */

    for(instancePtr = masterPtr->instancePtr; instancePtr != NULL;
	instancePtr = instancePtr->nextPtr)
      DitherInstance(instancePtr, x, y, width, height);

    /*
     * Tell the core image code that this image has changed.
     */

    Tk_ImageChanged(masterPtr->tkMaster, x, y, width, height, masterPtr->width,
	    masterPtr->height);
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_PictBlank --
 *
 *	This procedure is called to clear an entire Pict image.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The valid region for the image is set to the null region.
 *	The generic image code is notified that the image has changed.
 *
 *----------------------------------------------------------------------
 */

void
Tk_PictBlank(handle)
    Tk_PictHandle handle;	/* Handle for the image to be blanked. */
{
    PictMaster *masterPtr;
   
#ifdef DEBUG
    printf("Tk_PictBlank\n");
#endif

    masterPtr = (PictMaster *) handle;
   
    /*
     * The image has valid data nowhere.
     */

    if (masterPtr->validRegion != NULL) {
	XDestroyRegion(masterPtr->validRegion);
    }
    masterPtr->validRegion = XCreateRegion();

    /*
     * Clear out the data storage array.
     */

    memset((VOID *) masterPtr->data, 0,
	    (size_t) (masterPtr->width * masterPtr->height * masterPtr->datasize));

    /*
     * Tell the core image code that this image has changed.
     */
  
    Tk_ImageChanged(masterPtr->tkMaster, 0, 0, masterPtr->width,
	    masterPtr->height, masterPtr->width, masterPtr->height);

}

/*
 *----------------------------------------------------------------------
 *
 * Tk_PictExpand --
 *
 *	This procedure is called to request that a Pict image be
 *	expanded if necessary to be at least `width' pixels wide and
 *	`height' pixels high.  If the user has declared a definite
 *	image size (using the -width and -height configuration
 *	options) then this call has no effect.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The size of the Pict image may change; if so the generic
 *	image code is informed.
 *
 *----------------------------------------------------------------------
 */

void
Tk_PictExpand(handle, width, height)
    Tk_PictHandle handle;	/* Handle for the image to be expanded. */
    int width, height;		/* Desired minimum dimensions of the image. */
{
    PictMaster *masterPtr;

#ifdef DEBUG
    printf("Tk_PictExpand\n");
#endif

    masterPtr = (PictMaster *) handle;

    /* commenting out the following forces to resize the windows */
    /* if (width <= masterPtr->width) {
	width = masterPtr->width;
    }
    if (height <= masterPtr->height) {
	height = masterPtr->height;
    } */ 

    if ((width != masterPtr->width) || (height != masterPtr->height)) {
	ImgPictSetSize(masterPtr, width, height);
	Tk_ImageChanged(masterPtr->tkMaster, 0, 0, 0, 0, masterPtr->width,
		masterPtr->height);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_PictGetSize --
 *
 *	This procedure is called to obtain the current size of a Pict
 *	image.
 *
 * Results:
 *	The image's width and height are returned in *widthp
 *	and *heightp.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
Tk_PictGetSize(handle, widthPtr, heightPtr)
    Tk_PictHandle handle;	/* Handle for the image whose dimensions
				 * are requested. */
    int *widthPtr, *heightPtr;	/* The dimensions of the image are returned
				 * here. */
{
    PictMaster *masterPtr;

#ifdef DEBUG
    printf("Tk_PictGetSize\n");
#endif

    masterPtr = (PictMaster *) handle;
    *widthPtr = masterPtr->width;
    *heightPtr = masterPtr->height;
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_PictSetSize --
 *
 *	This procedure is called to set size of a Pict image.
 *	This call is equivalent to using the -width and -height
 *	configuration options.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The size of the image may change; if so the generic
 *	image code is informed.
 *
 *----------------------------------------------------------------------
 */

void
Tk_PictSetSize(handle, width, height)
    Tk_PictHandle handle;	/* Handle for the image whose size is to
				 * be set. */
    int width, height;		/* New dimensions for the image. */
{
    PictMaster *masterPtr;

#ifdef DEBUG
    printf("Tk_PictSetSize\n");
#endif

    masterPtr = (PictMaster *) handle;

    masterPtr->userWidth = width;
    masterPtr->userHeight = height;
    ImgPictSetSize(masterPtr, ((width > 0) ? width: masterPtr->width),
	    ((height > 0) ? height: masterPtr->height));
    Tk_ImageChanged(masterPtr->tkMaster, 0, 0, 0, 0,
	    masterPtr->width, masterPtr->height);
}

/*
 *----------------------------------------------------------------------
 *
 * Tk_PictGetImage --
 *
 *	This procedure is called to obtain image data from a Pict
 *	image.  This procedure fills in the Tk_PictImageBlock structure
 *	pointed to by `blockPtr' with details of the address and
 *	layout of the image data in memory.
 *
 * Results:
 *	TRUE (1) indicating that image data is available,
 *	for backwards compatibility with the old Pict widget.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tk_PictGetImage(handle, blockPtr)
    Tk_PictHandle handle;	/* Handle for the Pict image from which
				 * image data is desired. */
    Tk_PictImageBlock *blockPtr;
				/* Information about the address and layout
				 * of the image data is returned here. */
{
    PictMaster *masterPtr;

#ifdef DEBUG
    printf("Tk_PictGetImage\n");
#endif

    masterPtr = (PictMaster *) handle;
    blockPtr->pixelPtr = (unsigned char*)masterPtr->data;
    blockPtr->width = masterPtr->width;
    blockPtr->height = masterPtr->height;
    blockPtr->pitch = masterPtr->width;
    blockPtr->pixelSize = masterPtr->datasize;;
    blockPtr->datatype = masterPtr->datatype;
    blockPtr->copy = COPY;
    return 1;
}


static int make_colorbar(Tk_PictHandle handle,
			 int width, int height)
{
  Tk_PictImageBlock block;
  int i,j;
  unsigned char *pixelPtr;
  int nBytes;
  PictMaster *masterPtr;

#ifdef DEBUG
    printf("make_colorbar \n");
#endif

  masterPtr = (PictMaster *) handle;
  block.datatype = BYTE;
  block.pixelSize = sizeof(unsigned char);
  block.width = width;
  block.height = height;
  block.pitch = block.pixelSize * width;
  
  nBytes = width * height * block.pixelSize;
  pixelPtr = (unsigned char *) ckalloc((unsigned) nBytes);
  if ( pixelPtr == NULL )
    return 0;

  for(j=0;j<height;j++)
    for(i=0;i<width;i++)
      pixelPtr[i+j*width] = i*255/width;

  block.pixelPtr = pixelPtr;

  Tk_PictPutBlock(handle, &block,0,0, width, height);
  return 1;
} /* end make_colorbar */

static int convert_block_to_byte(void *in, unsigned char *out, int npts,
			  int in_type,double *dispmin, double *dispmax)
{
  register int i;
  double scale;
  unsigned char *ptr_out; 

  ptr_out = (unsigned char*)out;
  switch (in_type)
    { 
      
    case WORD:
      {
        unsigned short *ptr = (unsigned short *)in;
	unsigned short max = *ptr++;
	unsigned short min = max;
		
	
	for (i=1;i<npts;++i,++ptr)
	  {
	    if (*ptr > max) max = *ptr;
	    else if (*ptr < min) min = *ptr;
	  }
	if( (*dispmin == 0.0) && (*dispmax == 0.0) ) 
	  {
	    *dispmin = min;
	    *dispmax = max;
	  }
	else {
	  if (*dispmin>(double)min) 
	    *dispmin = min;
	  if (*dispmax<(double)max)
	    *dispmax = max;
	}
	ptr = in;
	if( (*dispmax-*dispmin) == 0.0 ) {
	  fprintf(stderr,"\n Warning: image min and image max are the same \n");
	  scale = 255.0 / pow(2,sizeof(short)*8) * (*ptr);
	  for (i=0;i<npts;++i)
	    *ptr_out++ = scale;
	}
	else {
	  scale = 255.0 / (*dispmax - *dispmin);
	  for (i=0;i<npts;++i)
	    *ptr_out++ = scale * (*ptr++ - *dispmin);
	}
      }
      break;
    case LWORD:
      {
        int *ptr = (int *)in;
	int max = *ptr++;
	int min = max;

	for (i=1;i<npts;++i,++ptr)
	  {
	    if (*ptr > max) max = *ptr;
	    else if (*ptr < min) min = *ptr;
	  }
	if( (*dispmin == 0.0) && (*dispmax == 0.0) ) 
	  {
	    *dispmin = min;
	    *dispmax = max;
	  }
	else {
	  if (*dispmin>(double)min) 
	    *dispmin = min;
	  if (*dispmax<(double)max)
	    *dispmax = max;
	}
	ptr = in; 
	if( (*dispmax-*dispmin) == 0.0 ) {
	  fprintf(stderr,"\n Warning: image min and image max are the same \n");
	  scale = 255.0 / pow(2,sizeof(int)*8) * (*ptr);
	  for (i=0;i<npts;++i)
	    *ptr_out++ = scale;
	}
	else {
	  scale = 255.0 / (*dispmax - *dispmin);
	  for (i=0;i<npts;++i)
	    *ptr_out++ = scale * (*ptr++ - *dispmin);
	}
      }
      break;
    case REAL:
      {
        float *ptr;
	float max;
	float min;

	ptr = (float *)in;
	max = *ptr++;
	min = max;
	
	for (i=1;i<npts;++i,++ptr)
	  {
	    if (*ptr > max) max = *ptr;
	    else if (*ptr < min) min = *ptr;
	  }
	if( (*dispmin == 0.0) && (*dispmax == 0.0) ) 
	  {
	    *dispmin = min;
	    *dispmax = max;
	  }
	else {
	  if (*dispmin>(double)min) 
	    *dispmin = min;
	  if (*dispmax<(double)max)
	    *dispmax = max;
	}
	ptr = (float*)in; 
	if( (*dispmax-*dispmin) == 0.0 ) {
	  fprintf(stderr,"\n Warning: image min and image max are the same \n");
	  scale = 255.0 / pow(2,sizeof(float)*8) * (*ptr);
	  for (i=0;i<npts;++i)
	    *ptr_out++ = scale;
	}
	else {
	  scale = 255.0 / (*dispmax - *dispmin);
	  for (i=0;i<npts;++i)
	    *ptr_out++ = scale * (*ptr++ - *dispmin);
	}
      }
      break;
    default:
      (void)fprintf(stderr,"Unknown data type %d\n",in_type);
      return(0);
    }

  /* (void)fprintf(stderr,"dispmin %g dispmax %g\n",*dispmin,*dispmax); */
  return(1);
} /* end convert_block_to_byte */






static void normalize_data(PictMaster *masterPtr)
{
  unsigned char *out;

#ifdef DEBUG
  printf("normalize_data\n");
#endif

  if( masterPtr->datatype == BYTE ) 
    {
      masterPtr->bytedata = (unsigned char*)masterPtr->data;
      masterPtr->dispmin = 0.0;
      masterPtr->dispmax = 255.0;
    }
  else 
    {
      if( masterPtr->bytedata != NULL ) {
	free((void*)masterPtr->bytedata);
      }
      out=(unsigned char*)ckalloc((size_t)(masterPtr->width*masterPtr->height*sizeof(unsigned char)));
      if( out == NULL ) {
	(void)fprintf(stderr,"Could not allocate memory \n");
	return;
      }
      
      masterPtr->bytedata = out;
      if( masterPtr->user_dispmin != 0.0 || masterPtr->user_dispmax != 0.0 ) {
	masterPtr->dispmin = masterPtr->user_dispmin;
	masterPtr->dispmax = masterPtr->user_dispmax;
      } else {
	masterPtr->dispmin = 0.0;
	masterPtr->dispmax = 0.0;
      }
      convert_block_to_byte((void*)masterPtr->data,
			    (unsigned char*)masterPtr->bytedata,
			    (masterPtr->width)*(masterPtr->height),
			    masterPtr->datatype,
			    &(masterPtr->dispmin),
			    &(masterPtr->dispmax)); 
    }
} /* end normalize_data */
