/*
 * tkPictImgFmtVIFF.c --
 *
 * A Pict image file handler for VIFF files. 
 * At present there is no write function. 
 *
 * Copyright (c) 1995 The Regents of the University of California.
 *
 * Author: Pierre-Louis Bossart
 * Date: November 17, 1995
 *
 * Derived from tkImgFmtPPM.c in the tk4.0b2 distribution 
 * copyrighted as follows:
 * Copyright (c) 1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include "tkpict.h"

#define MAX_MEMORY	1000000		/* don't allocate > 1MB */
#define VIFF 1

/*
 * The format record for the VIFF file format:
 */

static int		FileMatchVIFF (FILE *f, char *fileName,
			    char *formatString, int *widthPtr,
			    int *heightPtr);
static int		FileReadVIFF  (Tcl_Interp *interp,
			    FILE *f, char *fileName, char *formatString,
			    Tk_PictHandle imageHandle, int destX, int destY,
			    int width, int height, int srcX, int srcY);
static int		FileWriteVIFF (Tcl_Interp *interp,
			    char *fileName, char *formatString,
			    Tk_PictImageBlock *blockPtr);

Tk_PictImageFormat tkImgFmtVIFF = {
    "VIFF",			/* name */
    FileMatchVIFF,		/* fileMatchProc */
    NULL,			/* stringMatchProc */
    FileReadVIFF,		/* fileReadProc */
    NULL,			/* stringReadProc */
    FileWriteVIFF,		/* fileWriteProc */
    NULL,			/* stringWriteProc */
};

/*
 * Prototypes for local procedures defined in this file:
 */
static int		ReadVIFFFileHeader (FILE *f, int *widthPtr,
			    int *heightPtr, int *data_size,int *data_type);



/*
 *----------------------------------------------------------------------
 *
 * FileMatchVIFF --
 *
 *	This procedure is invoked by the Pict image type to see if
 *	a file contains image data in VIFF format.
 *
 * Results:
 *	The return value is >0 if the first characters in file "f" look
 *	like VIFF data, and 0 otherwise.
 *
 * Side effects:
 *	The access position in f may change.
 *
 *----------------------------------------------------------------------
 */

static int
FileMatchVIFF(f, fileName, formatString, widthPtr, heightPtr)
    FILE *f;			/* The image file, open for reading. */
    char *fileName;		/* The name of the image file. */
    char *formatString;		/* User-specified format string, or NULL. */
    int *widthPtr, *heightPtr;	/* The dimensions of the image are
				 * returned here if the file is a valid
				 * raw VIFF file. */
{
  int data_size;
  int data_type;

  return ReadVIFFFileHeader(f, widthPtr, heightPtr,&data_size,&data_type);

}

/*
 *----------------------------------------------------------------------
 *
 * FileReadVIFF --
 *
 *	This procedure is called by the Pict image type to read
 *	VIFF format data from a file and write it into a given
 *	Pict image.
 *
 * Results:
 *	A standard TCL completion code.  If TCL_ERROR is returned
 *	then an error message is left in interp->result.
 *
 * Side effects:
 *	The access position in file f is changed, and new data is
 *	added to the image given by imageHandle.
 *
 *----------------------------------------------------------------------
 */

static int
FileReadVIFF(interp, f, fileName, formatString, imageHandle, destX, destY,
	width, height, srcX, srcY)
    Tcl_Interp *interp;		/* Interpreter to use for reporting errors. */
    FILE *f;			/* The image file, open for reading. */
    char *fileName;		/* The name of the image file. */
    char *formatString;		/* User-specified format string, or NULL. */
    Tk_PictHandle imageHandle;	/* The Pict image to write into. */
    int destX, destY;		/* Coordinates of top-left pixel in
				 * Pict image to be written to. */
    int width, height;		/* Dimensions of block of Pict image to
				 * be written to. */
    int srcX, srcY;		/* Coordinates of top-left pixel to be used
				 * in image being read. */
{
    int fileWidth, fileHeight;
    int nBytes, type, count;
    unsigned char *pixelPtr;
    Tk_PictImageBlock block;
    int data_size;
    int data_type;

    type = ReadVIFFFileHeader(f, &fileWidth, &fileHeight,&data_size,&data_type);

    if (type == 0) {
	Tcl_AppendResult(interp, "couldn't read raw VIFF header from file \"",
		fileName, "\"", NULL);
	return TCL_ERROR;
    }
    if ((fileWidth <= 0) || (fileHeight <= 0)) {
	Tcl_AppendResult(interp, "VIFF image file \"", fileName,
		"\" has dimension(s) <= 0", (char *) NULL);
	return TCL_ERROR;
    }

    if ((srcX + width) > fileWidth) {
	width = fileWidth - srcX;
    }
    if ((srcY + height) > fileHeight) {
	height = fileHeight - srcY;
    }
    if ((width <= 0) || (height <= 0)
	|| (srcX >= fileWidth) || (srcY >= fileHeight)) {
	return TCL_OK;
    }

    block.datatype = data_type;
    block.pixelSize = data_size;
    block.width = width;
    block.height = height;
    block.pitch = fileWidth;
    block.skip = 1024;

    Tk_PictExpand(imageHandle, destX + width, destY + height);
    /* goto the correct slice */
    fseek(f, (long) (1024 +data_size*slice_nb*width*height), SEEK_SET); 
   
    if (srcY > 0) {
	fseek(f, (long) (srcY * block.pitch*block.pixelSize), SEEK_CUR);
    }
   
    nBytes = height*block.pitch*block.pixelSize;

    pixelPtr = (unsigned char *) ckalloc((unsigned) nBytes);
    if(pixelPtr == NULL) {
      (void)fprintf(stderr,"FileReadVIFF: Could not allocate memory\n");
      return;
    }

    block.pixelPtr = pixelPtr + srcX * block.pixelSize;
    if ( destX != 0 || destY != 0 || width != fileWidth )
      block.copy = COPY;
    else
      block.copy = NO_COPY; 
  
    count = fread(pixelPtr, 1, (unsigned) nBytes, f);
    if (count != nBytes) {
      Tcl_AppendResult(interp, "error reading VIFF image file \"",
		       fileName, "\": ",
		       feof(f) ? "not enough data" : Tcl_PosixError(interp),
		       (char *) NULL);
      ckfree((char *) pixelPtr);
      return TCL_ERROR;
    }
  
    Tk_PictPutBlock(imageHandle, &block, destX, destY, width, height);
    
    /* free data if necessary */
    if( block.copy == COPY )
	free((void*)pixelPtr);
      
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FileWriteVIFF --
 *
 *	This procedure is invoked to write image data to a file in VIFF
 *	format.
 *
 * Results:
 *	A standard TCL completion code.  If TCL_ERROR is returned
 *	then an error message is left in interp->result.
 *
 * Side effects:
 *	Data is written to the file given by "fileName".
 *
 *----------------------------------------------------------------------
 */

static int
FileWriteVIFF(interp, fileName, formatString, blockPtr)
    Tcl_Interp *interp;
    char *fileName;
    char *formatString;
    Tk_PictImageBlock *blockPtr;
{
    FILE *f;
    int w, h;
    int greenOffset, blueOffset, nBytes;
    unsigned char *pixelPtr, *pixLinePtr;

    return 1;
    
}

/*
 *----------------------------------------------------------------------
 *
 * ReadVIFFFileHeader --
 *
 *	This procedure reads the VIFF header from the beginning of a
 *	VIFF file and returns the dimensions of the image.
 *
 * Results:
 *	The return value is PGM if file "f" appears to start with
 *	a valid PGM header, VIFF if "f" appears to start with a valid
 *      VIFF header, and 0 otherwise.  If the header is valid,
 *	then *widthPtr and *heightPtr are modified to hold the
 *	dimensions of the image.
 *
 * Side effects:
 *	The access position in f advances.
 *
 *----------------------------------------------------------------------
 */

static int
ReadVIFFFileHeader(f, widthPtr, heightPtr, data_size, data_type)
    FILE *f;		/* file  to read the header from */
    int *widthPtr, *heightPtr;  /* The dimensions of the image are
					   * returned here. */
     int *data_size;
     int *data_type;
{
  int n;
  int type;
  int index;
  unsigned char buffer[1024];

  n = fread(buffer,1,1024, f);
  if (n < 1024)
    {
      (void)fprintf(stderr,"\n ERROR: Unable to read data file\n") ;
      return(0) ;
    }

  if((buffer[0] == 0xab ) ||
     (buffer[1] == 0x13 )) {

    *widthPtr = *(unsigned long*)(&buffer[520]);
    *heightPtr = *(unsigned long*)(&buffer[524]);
    index = 520 + 8*sizeof(long) + 2*sizeof(float);
    nb_slices =  *(unsigned long*)(&buffer[index]);
    type = *(unsigned long*)(&buffer[index+sizeof(long)]);

    switch( type ) {
    case 1:
      *data_type = BYTE;
      *data_size = sizeof(char);
      break;
    case 2:
      *data_type = WORD;
      *data_size = sizeof(short);
      break;
    case 4:
      *data_type = LWORD; 
      *data_size = sizeof(int);
      break;
    case 5:
      *data_type = REAL;
      *data_size = sizeof(float);
      break;
    default:
      (void)fprintf(stderr," Error: can only read byte, short, int, float data \n");
      return 0;
    }
    return 1;
  } 
  return(0);
}














