# 
# A trivial file transfer layer on top of DP.
# 
# Assumes a server has been started with "fserver"
# and a client connected to that server via "fconnect"
#
# The user interface:
#
#    fconnect $host $port - open an RPC connection to host:port
#			    and a data connection to host:port-1
#    fget $filename - copy remote file, $filename, to local machine.
#    fput $filename - copy local file, $filename, to remote machine.
#    fdel $filename - delete a remote file named $filename
#    fdir - remote directory listing
#    fpwd - print current remote directory
#    fcd - change remote directory
#
# Problems:
#
#    Single user only since the remote dir is global.
#    Minimal error handling at best.  Need to catch more.
#

package require Dp

#//////////////////////////////////////

proc oops {errMsg} {
    puts stdout "Remote server returned error:"
    puts stdout "$errMsg"
    return
}

#//////////////////////////////////////

proc fget {filename } {
    global f_chan

    dp_RDO $f_chan -onerror oops "set daFile \[open \$filename r\];\
	     dp_send \$dataSC \[dp_recv \$daFile\]"
    set newfile [open $filename {CREAT WRONLY}]
    puts stdout "Getting $filename from remote host."
    while {![eof $dataSock]} {
	dp_send $newfile [dp_recv $dataSock]
	puts stdout "#" nonewline
    }
    dp_RDO $f_chan "close \$daFile"
    close $newfile
}

#//////////////////////////////////////

proc fput {filename} {
    global f_chan

    puts stdout "Putting $filename on remote host. 4096 bytes per #."
    dp_RDO $f_chan "set daFile \[open \$filename w\];\
	    dp_send \$daFile \[dp_recv \$dataSC\]"
    set newfile [open $filename r]
    while {![eof $newfile]} {
	dp_send $f_chan [dp_recv $newfile]
        puts stdout "#" nonewline
    }
    puts stdout ""
    dp_RDO $f_chan "close \$daFile"
    close $newfile
}

#//////////////////////////////////////

proc fpwd {} {
    global f_chan

    dp_RPC $f_chan pwd
}

#//////////////////////////////////////

proc fdir {} {
    global f_chan

# Dang Tcl 7.6 doesn't have 'dir' in Windows.
# We'll emulate it.

    foreach a [dp_RPC $f_chan glob \*] {
	puts $a
    }
}

#//////////////////////////////////////

proc fdel {filename} {
    global f_chan

    dp_RPC $f_chan file delete $filename
}

#//////////////////////////////////////

proc fcd {dir} {
    global f_chan

    dp_RDO $f_chan -onerror oops cd $dir
}

#//////////////////////////////////////

proc fquit {} {
    global f_chan

    dp_RDO $f_chan "close \$dataServ"
    close $dataClient
    close $f_chan
}

#//////////////////////////////////////
# We need to open two ports on each side:
#    the RPC channel - $f_chan
#    the data channel - $dataClient here, $dataSC on the server
# since we can't send normal data over an RPC channel.

proc fconnect {host port} {
    global f_chan

    set f_chan [dp_MakeRPCClient $host $port]
    puts stdout "Connected to $host..."
    set dataPort [incr $port -1]
    if {[catch {dp_RDO $f_chan eval {set dataServ \[dp_connect -server 1\
	    -myport $dataPort\]; set dataSC \[dp_accept \$dataServ\]}} msg]} {
	puts stdout "Error opening dataServer socket: $msg"
	close $f_chan
	return
    }
    if {[catch {set dataClient [dp_connect tcp -host $host -port $dataPort]}
	    msg]} {
    	puts stdout "Error connecting to dataServer socket: $msg"
    	dp_RDO $f_chan "close \$dataServ"
    	close $f_chan
    	return
    }
    
    puts stdout "--------- Directory Listing ----------"
    fdir
    puts stdout "--------- Remote Directory -----------"
    fpwd
}

