/*
 * Extend a tcl command interpreter with commands to query the
 * Internet domain name service.
 *
 * Copyright (c) 1993, 1994
 *                    E. Schoenfelder, J. Schoenwaelder
 *                    TU Braunschweig, Germany
 *                    Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 */

/*
 *
 * internal, common usable interface:
 *
 *	int rc = dns_query (int argc, char *argv[])
 *
 *	with argv as follows:
 *		[-t(imeout) <t>] [-r(etries) <r>]
 *		[-s(erver) <server>]
 *		a | hinfo | ptr | mx | soa  <name_or_ip>
 *
 *	defaults are:
 *		retry-timeout: 2 secs
 *		retries: 2
 *		server: libresolv.a default
 *		query: a (for hostname to ip lookup)
 *
 * 	bugs:  (*) only one query flag is used.
 *	       (*) other default servers are asked too (feature ?).
 *
 *
 *	this is build upon:
 *
 *		int n_ip = dns_a (char *hostname, char *ip[]);
 *		int n_hname = dns_ptr (char *ip, char *hname[]);
 *		int n_hinfo = dns_hinfo (char *hostname, char *hinfo[]);
 *
 * 		return-space is malloc by these routines.
 */


/* comment next line to get a standalone module: */
/* #define STANDALONE		/ * define for standalone version */

#if defined(STANDALONE) && ! defined(DNS)
#define HAVE_DNS
#endif

#ifdef HAVE_DNS

#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <ctype.h>
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/nameser.h>
#include <arpa/inet.h>
#include <resolv.h>
#include <netdb.h>
#ifndef STANDALONE
#include <tcl.h>
#endif

/* debug switch: */
static int do_debug = 0;
#define dprintf		if (do_debug) printf

/* default timeout and retries for resolver-lib: */
static int timeout = 2;
static int retries = 2;

/* first initializion flag: */
static int initialized = 0;

/* default server is here stored: */
static struct in_addr default_server;

/* max # of return hostnames/ip-addrs: */
#define MAXRESULT	10


/*
 * query and reply structure:
 */
typedef struct {
	HEADER qb1;
	char qb2[PACKETSZ];
} querybuf;

/*
 * selfmade reply structure (private use only):
 */
typedef struct {
	int type;		/* T_A, T_SOA, T_HINFO, T_MX */
	int n;			/* # of results stored */
	union {
	  	struct in_addr addr [MAXRESULT]; 
		char str [MAXRESULT][256];
	} u;
} a_res;



/*
 * failsafe malloc:
 */

static void *
xmalloc (n)
int n;
{
	char *nnew;

	for (;;) 
	{
		nnew = malloc ((unsigned) n);
		if (nnew)
			break;
		sleep (1);
	}
	return nnew;
}

static char *
xstrdup (s)
char *s;
{
	char *nnew = xmalloc (strlen (s) + 1);
	strcpy (nnew, s);
	return nnew;
}

#define xfree(s)	free ((void *) s)



/*
 * initialize the resolver; set default retries/timeout
 */

static void
init_stuff ()
{
  int i;

  initialized = 1;
  res_init ();
  _res.options |= RES_RECURSE | RES_DNSRCH | RES_DEFNAMES | RES_AAONLY;

  if (_res.nscount > 0)
    default_server = _res.nsaddr.sin_addr;
  
  for (i = 0; i < MAXDNSRCH + 1 && _res.dnsrch [i]; i++)
    dprintf ("* search-order:  %d: `%s'\n", i, _res.dnsrch [i]);

  _res.retrans = timeout, _res.retry = retries;
}


/*
 * do a single query; mainly called by have_a_query.
 */

static void
do_query (query_string, query_type, query_result)
char *query_string;
int query_type;
a_res *query_result;
{
	querybuf query, answer;
	char buf [512], lbuf [512], auth_buf [512];
	int err, i, qlen, alen, llen, nscount, len, nsents;
	short type, class, rdlen;
	long ttl;
	querybuf *q;
	char *ptr, *eom;

	dprintf ("*** do_query() called:  query_string: `%s'  type: %d\n",
		query_string, query_type);

	query_result->type = -1;
	query_result->n = 0;

	/* if err is set, query_result->u.str [0] contains an 
	 * error-description: 
	 */
	err = 0;

#if 0
	bzero (&query, sizeof (querybuf));
	bzero (&answer, sizeof (answer));
#else
	for (i = 0; i < sizeof (querybuf); i++)
		((char *) &query) [i] = ((char *) &answer) [i] = 0;
#endif

 /**
  ** res_mkquery(op, dname, class, type, data, datalen, newrr, buf, buflen) 
  **/
	
	qlen = res_mkquery (QUERY, query_string, C_IN, query_type, (char *) 0, 0, 0,
			    (char *) &query, sizeof(query));
dprintf ("** res_mkquery() = %d\n", qlen);

	if (qlen <= 0)
	{
		query_result->n = -1;
		strcpy (query_result->u.str [0], "cannot make query");
		return;
	}

 /**
  ** res_send(msg, msglen, answer, anslen)
  **/	

	alen = res_send ((char *) &query, qlen, (char *) &answer, sizeof (answer));

dprintf ("** res_send() = %d\n", alen);

	if (alen <= 0)
	{
		query_result->n = -1;
		strcpy (query_result->u.str [0], "cannot send query");
		return;
	}

	q = &answer;

	/* if there are nameserver entries, only these are for authorative
	   answers of interest: */
	nsents = ntohs (q->qb1.nscount);

dprintf ("** got: 	qents: %d  aents: %d,  nsents: %d  rrents: %d\n",
	 ntohs (q->qb1.qdcount), ntohs (q->qb1.ancount), 
	 nsents, ntohs (q->qb1.arcount));

dprintf ("** got rcode: %d   is response: %d  opcode: %d\n", q->qb1.rcode,
	 q->qb1.qr, q->qb1.opcode);

	if (q->qb1.rcode != 0)
	{
		err = 1;
		if (q->qb1.rcode == 1)
			strcpy (query_result->u.str [0], "format error");
		else if (q->qb1.rcode == 2)
			strcpy (query_result->u.str [0], "server failure");
		else if (q->qb1.rcode == 3)
			strcpy (query_result->u.str [0], "domainname error");
		else if (q->qb1.rcode == 4)
			strcpy (query_result->u.str [0], "not implemented");
		else if (q->qb1.rcode == 5)
			strcpy (query_result->u.str [0], "query refused");
		else
			strcpy (query_result->u.str [0], "unknown error");
		dprintf ("* got error: %s\n", query_result->u.str [0]);

		query_result->type = query_type;
		query_result->n = -1;
		return;
#if 0
		/* dont return; store result only. */
		dprintf ("* failure - return...\n");
		return;
#endif	
	}

	nscount = ntohs (q->qb1.ancount);
	if (! nscount)
		nscount = ntohs (q->qb1.nscount);
	if (! nscount)
		nscount = ntohs (q->qb1.arcount);

	/* give some help (seems to be needed for very ole sun-code... */
	eom = (char *) &answer + alen;
	*eom++ = 0;
	
	ptr = q->qb2;

	/*
	 * skip over question section: 
	 * [ QNAME , QTYPE , QCLASS ]
	 */
	if (q->qb1.qdcount > 0)
	{
		int rc = dn_skipname (ptr, eom);
		dprintf ("** dn_skipname () = %d\n", rc);
		ptr += rc + QFIXEDSZ;
	}

	/*
	 * now we have left a section with:
	 *	Answering RR's
	 *	Authority RR's
	 *	Additional RR's
	 */

	for ( ; nscount; nscount--) 
	{

		/*
		 * every RR looks like:
		 * [ NAME , TYPE , CLASS , TTL , RDLANGTH , RDATA ]
		 */

		/**
		 ** dn_expand(msg, msglen, comp_dn, exp_dn, length)
		 **/

		llen = dn_expand ((char *) q, eom, ptr, lbuf, sizeof (lbuf));

		dprintf ("** dn_expand() = %d\n", llen);

		if (llen < 0)
		{
			/* XXX: what to do ? */
			return;
		}

		if (llen > 0)
			dprintf ("** got `%s'\n", lbuf);

		ptr += llen;

		/* fetch TYPE, CLASS, TTL: */
		GETSHORT (type, (u_char *) ptr);
		GETSHORT (class, (u_char *) ptr);
		GETLONG (ttl, (u_char *) ptr);
		/* fetch RDLENGTH: */
		GETSHORT (rdlen, (u_char *) ptr);

		dprintf ("** type: %d  class: %d  ttl: %ld  rdlen: %d\n", 
			 type, class, ttl, (int) rdlen);

		if (type == T_NS)
		{
			dprintf ("** type T_NS\n");

			len = dn_expand ((char *) q, eom, ptr, buf,
					 sizeof (buf));
			dprintf ("** ns: dn_expand() = %d\n", len);
			if (len < 0)
				return;
			if (len > 0)
				dprintf ("** ns: name got `%s'\n", buf);
			ptr += len;
		}
		else if (type == T_A) 
		{
			unsigned long x;

			dprintf ("** type T_A\n");
			GETLONG (x, (u_char *) ptr);
			dprintf ("** addr: 0x%08lx\n", x);
			
			/* save this address: */
			if (! strcmp (query_string, lbuf)
			    || query_result->type == T_A 
			    || query_result->type == -1)
			{
				query_result->type = T_A;
				query_result->u.addr [query_result->n++].s_addr
					= ntohl (x);
			}
		}
		else if (type == T_SOA) 
		{
			/*
			 * SOA rdata format is:
			 * [ MNAME , RNAME , SERIAL , REFRESH , RETRY
			 *   EXPIRE , MINIMUM ]
			 */
			dprintf ("** type T_SOA\n");

			/* fetch authorative site-name: */
			len = dn_expand ((char *) q, eom, ptr, auth_buf,
					 sizeof (auth_buf));
			dprintf ("** soa: dn_expand() = %d\n", len);
			if (len < 0)
				return;
			if (len > 0)
				dprintf ("** soa: name got `%s'\n", auth_buf);
			ptr += len;
			
			/* fetch/skip over maintainer-name: */
			len = dn_expand ((char *) q, eom, ptr, buf, 
					 sizeof (buf));
			dprintf ("** soa: dn_expand() = %d\n", len);
			if (len < 0)
				return;
			if (len > 0)
				dprintf ("** soa: mname got `%s'\n", buf);
			ptr += len;
			
			/* skip til end of this rr: */
			ptr += 5 * 4;

			if (query_result->type == T_SOA
			    || query_result->type == -1)
			{
				query_result->type = T_SOA;
				strcpy (query_result->u.str[query_result->n++],
					auth_buf);
			}
		}
		else if (type == T_HINFO)
		{
			int i;

			for (i = 0; i < 1; i++)		/* XXX: ??? */
			{
			len = dn_expand ((char *) q, eom, ptr, buf,
					 sizeof (buf));
			dprintf ("** hinfo: dn_expand() = %d\n", len);
			if (len < 0)
				return;
			if (len > 0)
				dprintf ("** hinfo: got `%s'\n", buf);

			ptr += rdlen;

                        if (query_result->type == T_HINFO
                            || query_result->type == -1)
                        {
                                query_result->type = T_HINFO;
                                strcpy (query_result->u.str[query_result->n++],
                                        buf);
			}
			}
		}
		else if (type == T_PTR)
		{
			len = dn_expand ((char *) q, eom, ptr, buf,
					 sizeof (buf));
			dprintf ("** ptr: dn_expand() = %d\n", len);
			if (len < 0)
				return;
			if (len > 0)
				dprintf ("** ptr: got `%s'\n", buf);

			ptr += rdlen;

                        if (query_result->type == T_PTR
                            || query_result->type == -1)
                        {
                                query_result->type = T_PTR;
                                strcpy (query_result->u.str[query_result->n++],
                                        buf);
			}
		}
		else if (type == T_MX)
		{
			unsigned prio;
			GETSHORT (prio, (u_char *) ptr);

			len = dn_expand ((char *) q, eom, ptr, buf,
					 sizeof (buf));
			dprintf ("** mx: dn_expand() = %d\n", len);
			if (len < 0)
				return;
			if (len > 0)
				dprintf ("** mx: got `%s'\n", buf);

			ptr += len;

                        if (query_result->type == T_MX
                            || query_result->type == -1)
                        {
                                query_result->type = T_MX;
                                sprintf (
				query_result->u.str [query_result->n++],
					 "{%s %d}", buf, prio);
			}
		}
		else {
			dprintf ("** unhandled type: %d\n", type);

			ptr += rdlen;
		}
	}
}


/*
 * query_result wil contain the result.
 * if query_result->n < 0, the first string contains the error.
 *
 * (depth was experimental, please ignore...)
 */

static void
have_a_query (query_string, query_type, query_result, depth)
char *query_string;
int query_type;
a_res *query_result;
int depth;
{
	int i;
	a_res res;
	char tmp [256];

	dprintf (
	     "*** have_a_query():  string: `%s'  type: %d  server: 0x%08lx\n",
		 query_string, query_type, (long) _res.nsaddr.sin_addr.s_addr);

	query_result->type = -1;
	query_result->n = 0;

	/* for safety check here too: */
	if (! initialized)
	  init_stuff ();
 

	if (depth > 1)
	{
		dprintf ("** have_a_query(): too deep - aborted.\n");
		return;
	}

	/*
         * loop through every domain suffix: 
	 */
	for (i = -1; i < MAXDNSRCH + 1; i++)
	{
		if (i == -1)
			strcpy (tmp, query_string);
		else if (! _res.dnsrch [i])
			break;
		else
			sprintf (tmp, "%s.%s", query_string, _res.dnsrch [i]);

		dprintf ("** trying: `%s'...\n", tmp);
		do_query (tmp, query_type, &res);
		if (res.type == query_type && res.n > 0)
		{
			*query_result = res;
			return;
		}
		/* check ptr and soa's not recursive: */
		if (query_type == T_SOA || query_type == T_PTR)
		{
			*query_result = res;
			return;
		}
	}

	/*
	 * seems to be unsuccessful: look for any answer:
	 */
	for (i = -1; i < MAXDNSRCH + 1; i++)
	{
		if (i == -1)
			strcpy (tmp, query_string);
		else if (! _res.dnsrch [i])
			break;
		else
			sprintf (tmp, "%s.%s", query_string, _res.dnsrch [i]);

		dprintf ("** trying: `%s'...\n", tmp);
		do_query (tmp, query_type, &res);
		if (res.n > 0)
		{
#if 1
			/* return first answer found: */
			*query_result = res;
                        return;
#else
			query_string = tmp;
			break;
#endif
		}
	}
	
	if (res.n <= 0)
	{
		dprintf ("*** rabaeh: cannot find any answer.\n");
		*query_result = res;
		return;
	}

#if 1
	dprintf ("*** rabaeh: will return anyway.\n");
	return;
#else
	/*
	 * here we could ask recursive other hosts ...
	 * (but still wrong and not used)
	 */

	if (res.type == T_SOA) {
		a_res tmpres;
		do_query (res.u.str [0], T_A, &tmpres);
		if (tmpres.type != T_A || tmpres.n <= 0)
		{
			dprintf ("** have_a_query(): no T_A for T_SOA\n");
			return;
		}
		_res.nsaddr.sin_addr = tmpres.u.addr [0];
		_res.nscount = 1;
		have_a_query (query_string, query_type, query_result, 
			      depth + 1);
		return;
	}
	else 
		dprintf ("** have_a_query(): type is %d ???\n", res.type);
#endif
}



int 
dns_a (hname, ip)
char *hname;
char *ip[];
{
  a_res res;
  int i;

  if (isdigit (hname [0]))
  {
	  if (do_debug)
		  fprintf (stderr, "dns_ptr: returning parameter...\n");
	  ip [0] = xstrdup (hname);
	  res.type = T_A;
	  return res.n = 1;
  }
  
  have_a_query (hname, T_A, &res, 0);
  if (res.n < 0)
  {
	  ip [0] = xstrdup (res.u.str [0]);
	  return -1;
  }
  
  if (res.type != T_A)
	  return 0;				/* XXX: error set ? */
  
  for (i = 0; i < res.n; i++)
    {
      ip [i] = xmalloc (32);
      strcpy (ip [i], inet_ntoa (res.u.addr [i]));	/* XXX */
    }
  return res.n;
}


int
dns_ptr (ip, hname)
char *ip;
char *hname[];
{
  a_res res;
  int i, a, b, c, d;
  char tmp [128];

  if (! isdigit (ip [0]))
  {
	  if (do_debug)
		  fprintf (stderr, "dns_ptr: returning parameter...\n");
	  hname [0] = xstrdup (ip);
	  res.type = T_PTR;
	  return res.n = 1;
  }

  if (4 != sscanf (ip, "%d.%d.%d.%d", &a, &b, &c, &d))
    return 0;

  sprintf (tmp, "%d.%d.%d.%d.in-addr.arpa", d, c, b, a);

  have_a_query (tmp, T_PTR, &res, 0);
  if (res.n < 0)
  {
	  hname [0] = xstrdup (res.u.str [0]);
	  return -1;
  }

  if (res.type != T_PTR)
	  return 0;

  for (i = 0; i < res.n; i++)
      hname [i] = xstrdup (res.u.str [i]);

  return res.n;
}


/*
 * remove '\' from the given string:
 */

static void
hinfo_clean (str)
char *str;
{
	char *ptr;

	while (str && *str)
	{
		if (*str == '\\')
		{
			for (ptr = str; *ptr; ptr++)
				*ptr = *(ptr + 1);
		}
		str++;
	}
}


int
dns_hinfo (hname, hinfo)
char *hname;
char *hinfo[];
{
  a_res res;
  char *beg, *ptr;

  have_a_query (hname, T_HINFO, &res, 0);
  if (res.n < 0)
  {
          hinfo [0] = xstrdup (res.u.str [0]);
          return -1;
  }

  if (res.type != T_HINFO)
	  return 0;

 beg = ptr = res.u.str [0];
  while (*ptr && *ptr != '.')
    {
      if (*ptr == '\\' && *(ptr+1))
	ptr++;
      ptr++;
    }
  
  if (*ptr == '.')
	  *ptr++ = ' ';
  while (*ptr && *ptr != '.')
    {
      if (*ptr == '\\' && *(ptr+1))
	ptr++;
      ptr++;
    }
  
  hinfo [0] = xmalloc (ptr - beg + 1);
  strncpy (hinfo [0], beg, ptr - beg);
  hinfo [0][ptr - beg] = 0;
  hinfo_clean (hinfo [0]);

  return 1;
}


int 
dns_mx (hname, mx)
char *hname;
char *mx[];
{
	a_res res;
	int i;

	have_a_query (hname, T_MX, &res, 0);
	if (res.n < 0)
	{
		mx [0] = xstrdup (res.u.str [0]);
		return -1;
	}

	if (res.type != T_MX)
		return 0;

	for (i = 0; i < res.n; i++)
		mx [i] = xstrdup (res.u.str [i]);

	return res.n;
}


int 
dns_soa (hname, soa)
char *hname;
char *soa[];
{
	a_res res;
	int i;

	have_a_query (hname, T_SOA, &res, 0);
	if (res.n < 0)
	{
		soa [0] = xstrdup (res.u.str [0]);
		return -1;
	}

	if (res.type != T_SOA)
		return 0;

	for (i = 0; i < res.n; i++)
		soa [i] = xstrdup (res.u.str [i]);

	return res.n;
}



/*
 * may be this a tcl-like interface:
 */

int
dns_query (argc, argv, buf)
int argc;
char *argv[];
char *buf;
{
  int query_type = -1;			/* what to do */
  char *query_string = 0;		/* what to pass */
  char *result [MAXRESULT];
  int i, rc;

  dprintf ("* dns_query: called with argc: %d\n", argc);

  /* default values: */
  timeout = 2;
  retries = 2;

  if (! initialized)
      init_stuff ();

  /* reset resolver to default: */
  _res.retry = retries;
  _res.retrans = timeout;
  if (_res.nscount > 0)
    _res.nsaddr.sin_addr = default_server;

  /* parse args: */
  while (argc > 0)
    {
      if (! strcmp (argv [0], "-timeout")
	  || ! strcmp (argv [0], "-t"))
	{
	  int val;
	  argc--, argv++;
	  if (argc <= 0)
	    {
	      if (do_debug)
		fprintf (stderr, "dns_query: missing arg for -timeout\n");
	      strcpy (buf, "missing arg for -timeout");
	      return -1;
	    }
	  val = atoi (argv [0]);
	  if (val < 1)
	  {
	      strcpy (buf, "bad arg for -timeout");
	      return -1;
      	  }
	  _res.retrans = val;
	  dprintf ("* timeout set to %d\n", val);
	  argc--, argv++;
	}
      else if (! strcmp (argv [0], "-retries")
	       || ! strcmp (argv [0], "-r"))
	{
	  int val;
	  argc--, argv++;
	  if (argc <= 0)
	    {
	      if (do_debug)
		fprintf (stderr, "dns_query: missing arg for -retries\n");
	      strcpy (buf, "missing arg for -retries");
	      return -1;
	    }
	  val = atoi (argv [0]);
	  if (val < 1)
	  {
	      strcpy (buf, "bad arg for -retries");
	      return -1;
      	  }
	  _res.retry = val;
	  dprintf ("* retries set to %d\n", val);
	  argc--, argv++;
	}
      else if (! strcmp (argv [0], "-server")
	       || ! strcmp (argv [0], "-s"))
	{
	  char *val;
	  struct hostent *he;

	  argc--, argv++;
	  if (argc <= 0)
	    {
	      if (do_debug)
		fprintf (stderr, "dns_query: missing arg for -server\n");
	      strcpy (buf, "missing arg for -server");
	      return -1;
	    }
	  val = argv [0];
	  if (! (he = gethostbyname (val)))
	    {
	      /* now we should check by-addr and/or call ourself to 
	       * resolve ... */
	      if (do_debug)
		fprintf (stderr, "dns_query: warning: cannot find server\n");
	      sprintf (buf, "cannot find server `%s'", val);
	      return -1;
	    }
	  else {
	    _res.nsaddr.sin_addr = * (struct in_addr *) he->h_addr;
	  }
	  dprintf ("* server set to 0x%08lx\n", 
		   (long) _res.nsaddr.sin_addr.s_addr);
	  argc--, argv++;
	}
      else if (! strcmp (argv [0], "a") && query_type < 0)
	{
	  query_type = T_A;
	  argc--, argv++;	  
	}
      else if ((! strcmp (argv [0], "hinfo")
	       || ! strcmp (argv [0], "h")) && query_type < 0)
	{
	  query_type = T_HINFO;
	  argc--, argv++;	  
	}
      else if ((! strcmp (argv [0], "ptr")
	       || ! strcmp (argv [0], "p")) && query_type < 0)
	{
	  query_type = T_PTR;
	  argc--, argv++;	  
	}
      else if ((! strcmp (argv [0], "mx")
	       || ! strcmp (argv [0], "m")) && query_type < 0)
	{
	  query_type = T_MX;
	  argc--, argv++;	  
	}
      else if ((! strcmp (argv [0], "soa")
	       || ! strcmp (argv [0], "s")) && query_type < 0)
	{
	  query_type = T_SOA;
	  argc--, argv++;  
	}
      else {
	/* now there should be one arg-to-be-queried: */
	if (query_type < 0)
	{
		strcpy (buf, "missing query type");
		return -1;
	}


	if (query_string)
	  {
	    if (do_debug)
	      fprintf (stderr, "dns_query: extra arg `%s'.\n", argv [0]);
	    strcpy (buf, "too many arguments");
	    return -1;
	  }
	else
	  query_string = argv [0];
	argc--, argv++;	  
      }
    }

  /* check for default: */
  if (query_type == -1)
    query_type = T_A;


  if (! query_string)
    {
      if (do_debug)
	fprintf (stderr, "dns_query: no arg to lookup ???\n");
      strcpy (buf, "missing argument");
      return -1;
    }

  /* now do the lookup and prepare the return values: */

  switch (query_type) {
  case T_HINFO:
    rc = dns_hinfo (query_string, result);
    break;
  case T_PTR:
    rc = dns_ptr (query_string, result);
    break;
  case T_MX:
    rc = dns_mx (query_string, result);
    break;
  case T_SOA:
    rc = dns_soa (query_string, result);
    break;
  case T_A:
  default:
    rc = dns_a (query_string, result);
    break;
  }

  if (rc < 0)
  {
	  strcpy (buf, result [0]);
	  return -1;
  }
  else if (! rc)
    {
      sprintf (buf, "<no info>");
    }
  else {
    for (buf [0] = 0, i = 0; i < rc; i++)
    {
	    sprintf (buf + strlen (buf), "%s", result [i]);
	    if (i + 1 < rc)
		    strcat (buf, " ");
	    xfree (result [i]);
    }
  }      

  return rc > 0;
}


#ifndef STANDALONE

/* 
 * tcl called interface:
 */

int
dns_cmd (clientData, interp, argc, argv)
ClientData clientData;
Tcl_Interp* interp;
int argc;
char *argv[];
{
	char answer [1024];
	int n;

	/*
	 * at least:  cmd <type> <string> 
	 */
	if (argc < 3)
	{
		Tcl_AppendResult (interp, "bad # args: ", argv[0], 
				  " ?-timeout t? ?-retries r? ?-s host? ",
				  "a | ptr | hinfo | mx | soa ",
				  " name-or-address", (char *) NULL);
		return TCL_ERROR;
	}
	
	n = dns_query (argc - 1, argv + 1, answer);
	if (! n)
		answer [0] = 0;
	/* if n < 0, answer contains a short error description: */

	Tcl_AppendResult (interp, answer, (char *) NULL);

	return (n >= 0) ? TCL_OK : TCL_ERROR;
}

#else /* STANDALONE */

/*
 * a standalone main() for testing of the query-code:
 */

int
main (argc, argv)
int argc;
char *argv [];
{
	a_res res;
	int rc, n, i;
	static int query_what [2] = { T_A, T_HINFO };
	char tmp [256], *query_string = 0;

	if (argc > 1 && ! strcmp (argv [1], "-d"))
	  {
	    do_debug = 1;
	    argc--, argv++;
	  }

	if (argc <= 2)
	{
 		fputs ("use: dnsfun [<opts>] <hostname>|<ipaddress>\n",
		       stderr);
		exit (1);
	}

	rc = dns_query (argc - 1, argv + 1, tmp);
	printf ("got rc: %d   result: `%s'\n", rc, tmp);
	exit (0);


	/*
	 * initialize resolver:
	 */
	if (! initialized)
	  init_stuff ();
 
#if 1
	for (i = 0; i < MAXDNSRCH + 1 && _res.dnsrch [i]; i++)
		dprintf ("* search-order:  %d: `%s'\n", i, _res.dnsrch [i]);
	_res.retrans = 2, _res.retry = 2;
	dprintf ("* retrans: %d   timeout: %d\n", _res.retrans, _res.retry);
#endif

	{
	  static char hname [256];

	if (isdigit (argv [1][0]))
	{
	  char *res [10];
	  int i, rc = dns_ptr (argv [1], res);
	  printf ("got rc: %d\n", rc);
	  if (! rc)
	    printf ("  no info\n");
	  for (i = 0; i < rc; i++)
	    printf ("  `%s'\n", res [i]);
	  strcpy (hname, res [0]);
	}
	else {
	  char *res [10];
	  int i, rc = dns_a (argv [1], res);
	  printf ("got rc: %d\n", rc);
	  if (! rc)
	    printf ("  no info\n");
	  for (i = 0; i < rc; i++)
	    printf ("  `%s'\n", res [i]);
	  strcpy (hname, argv [1]);
	}
	  
	  if (! hname [0])
	    exit (0);
	  
	  {
	    char *res [10];
	    int i, rc = dns_hinfo (hname, res);
	    printf ("got rc: %d\n", rc);
	  if (! rc)
	    printf ("  no info\n");
	    for (i = 0; i < rc; i++)
	      printf ("  `%s'\n", res [i]);
	    strcpy (hname, argv [1]);
	  }
	  exit (0);
	}
	/* XXX */

	if (isdigit (argv [1][0]))
	{
		int a, b, c, d;
		if (4 == sscanf (argv [1], "%d.%d.%d.%d", &a, &b, &c, &d))
		{
			a_res tmpres;
			sprintf (tmp, "%d.%d.%d.%d.in-addr.arpa",
				 d, c, b, a);
			have_a_query (tmp, T_PTR, &tmpres, 0);
			if (tmpres.type == T_PTR && tmpres.n > 0)
			{
				query_string = xstrdup (tmpres.u.str [0]);
				printf ("got reverse lookup:\n");
				printf ("  `%s'\n", query_string);
			}
		}
	}

	if (! query_string)
		query_string = argv [1];

	for (n = 0; n < 2; n++)
	{
		have_a_query (query_string, query_what [n], &res, 0);

		printf ("query for type %d:\n", query_what [n]);

		if (res.n <= 0)
		{
			printf ("sigh - nothing found.\n");
		}
		else {
			for (i = 0; i < res.n; i++)
			{
				switch (res.type) {
				case T_A:
					printf ("  `%s' (0x%08lx)\n",
						inet_ntoa (res.u.addr [i]),
						res.u.addr[i].s_addr);
					break;
				case T_SOA:
				case T_HINFO:
				case T_PTR:
					printf ("  `%s'\n", res.u.str [i]);
					break;
				default:
					printf ("  ???\n");
				}
			}
		}
	}

	return 0;
}

#endif /* STANDALONE */

#endif /* HAVE_DNS */

/* end of dnsfun.c */
