/*
 * icmp.c
 *
 * Extend a tcl command interpreter an icmp command. This module depends
 * on ntping written by Erik Schoenfelder (schoenfr@ibr.cs.tu-bs.de).
 *
 * Copyright (c) 1993, 1994
 *                    J. Schoenwaelder
 *                    TU Braunschweig, Germany
 *                    Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 */

#ifdef HAVE_ICMP

#include "scotty.h"

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <tcl.h>

#define BUF_INCR 1024

static int xv[2] = {-1, -1};      /* the socket pair used for ntping */
static int pid;                   /* the pid of the ntping process   */
static FILE *nt;                  /* the file to ntping              */

static int timeout = 5;
static int retries = 2;
static int size = 64;
static int ttl = -1;
static int trace = -1;
static int delay = 0;

/* 
 * Extend a tcl command interpreter with an icmp command. The
 * result will be a list of hosts and their round trip times
 * including -1 for unreachable hosts. The icmp command will fail,
 * if there are wrong arguments. When using the time to live (ttl)
 * feature, the command will return the host that answerd the udp
 * packet together with the round trip time.
 */

int
icmp_cmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp* interp;
    int argc;
    char** argv;
{
    int new_timeout = -1;
    int new_retries = -1;
    int new_size = -1;
    int new_ttl = -1;
    int new_trace = -1;
    int new_mask = 0;
    int new_tstamp = 0;
    int new_delay = -1;
    char *p;
    int i;

    char *buffer;
    int buffer_size;

    if (argc == 1) {
	Tcl_AppendResult (interp, "bad # args: ", argv[0],
			  " ?-retries n? ?-size n? ?-timeout n? ?-delay n?",
			  " ?-ttl n? ?-trace n? ?-mask? ?-timestamp? hosts",
			  (char *)NULL);
	return TCL_ERROR;
    }

    if (xv[0] == -1) {                 /* start ntping if not done yet */
	if (socketpair (AF_UNIX, SOCK_STREAM, 0, xv) < 0) {
	    xv[0] = xv[1] = -1;
	    interp->result = "cannot create socketpair";
            return TCL_ERROR;
	}
	pid = fork();
	if (pid < 0) {
	    xv[0] = xv[1] = -1;
	    interp->result = "fork failed";
	    return TCL_ERROR;
	}
	if (pid==0) {                                         /* child */
	    close (xv [0]);
	    dup2  (xv [1], 0);
	    dup2  (xv [1], 1);
	    execl (NTPING, NTPING, "-b", (char *) 0);
	    perror ("scotty: execl ntping");
	    exit (1);
	} else {                                            /* parent */
	    close (xv [1]);
	    nt = fdopen (xv[0], "r+");
	    setbuf (nt, 0);
	}
    }

    /* parse the parameters */
    for (i = 1; i < argc; i++) {
	if (strcmp (argv[i], "-retries") == 0) {
	    if (i == argc-1) break;
	    new_retries = atoi (argv[++i]);
	    continue;
	} else 	if (strcmp (argv[i], "-size") == 0) {
	    if (i == argc-1) break;
            new_size = atoi (argv[++i]);
	    continue;
	} else 	if (strcmp (argv[i], "-timeout") == 0) {
	    if (i == argc-1) break;
            new_timeout = atoi (argv[++i]);
	    continue;
	} else  if (strcmp (argv[i], "-delay") == 0) {
	    if (i == argc-1) break;
	    new_delay = atoi (argv[++i]);
	    continue;
	} else  if (strcmp (argv[i], "-ttl") == 0) {
	    if (i == argc-1) break;
	    new_ttl = atoi (argv[++i]);
	    continue;
	} else  if (strcmp (argv[i], "-trace") == 0) {
	    if (i == argc-1) break;
	    new_trace = atoi (argv[++i]);
	    continue;
	} else  if (strcmp (argv[i], "-mask") == 0) {
	    new_mask++;
	    continue;
	} else  if (strcmp (argv[i], "-timestamp") == 0) {
	    new_tstamp++;
	    continue;
	} else break;
    }

    /* set the static parameters if there are no hosts given */
    if (i == argc) {
	timeout = (new_timeout<0) ? timeout : new_timeout;
	retries = (new_retries<0) ? retries : new_retries;
	size    = ( new_size<0  ) ? size    : new_size;
	delay   = ( new_delay<0 ) ? delay   : new_delay;
	ttl     = (   ttl<0     ) ? ttl     : new_ttl;
	trace   = (  trace<0    ) ? trace   : new_trace;
	return TCL_OK;
    }
    
    /* tell ntping what we are interested in */
    rewind (nt);
    fprintf (nt, "-t %d -r %d -d %d -s %d ", 
	     (new_timeout<0) ? timeout : new_timeout,
	     (new_retries<0) ? retries : new_retries,
	     (new_delay<0)   ? delay   : new_delay,
	     ( new_size<0  ) ? size    : new_size);
    if (ttl > 0 || new_ttl > 0) {
	fprintf (nt, "-ttl %d ", ( ttl > 0 ) ? ttl : new_ttl);
    } else if (trace > 0 || new_trace > 0) {
	fprintf (nt, "-trace %d ", ( trace > 0 ) ? trace : new_trace);
    } if (new_mask > 0) {
	fprintf (nt, "-mask ");
    } else if (new_tstamp > 0) {
	fprintf (nt, "-timestamp ");
    }
    for (; i < argc; i++) fprintf (nt, "%s ", argv[i]);
    fprintf (nt, "\n");
    fflush (nt);

    buffer_size = BUF_INCR;
    buffer = xmalloc (buffer_size);

    rewind(nt);
    p = buffer;
    while (((*p = getc (nt)) != EOF) && (*p != '\n')) {
	p++;
	if (p-buffer == buffer_size) {
	    i = p-buffer;
	    buffer_size += BUF_INCR;
	    buffer = xrealloc (buffer, buffer_size);
	    p = buffer + i;
	}
    }

    if (*p == EOF) {
	Tcl_SetResult (interp, "ntping closed connection", TCL_STATIC);
	return TCL_ERROR;
    }

    *p = '\0';
    Tcl_AppendResult (interp, buffer, (char *) NULL);
    free (buffer);
    return TCL_OK;
}

void
icmp_close()
{
    if (xv[0] != -1) {
	close (xv[0]);
	xv[0] = -1;
	xv[1] = -1;
    }
}

#endif
