/*
 * scotty.c
 *
 * This is scotty, a simple tcl interpreter with some special commands
 * to get information about TCP/IP networks. 
 *
 * Copyright (c) 1993, 1994
 *                    J. Schoenwaelder
 *                    TU Braunschweig, Germany
 *                    Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <stdio.h>
#include <string.h>
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netdb.h>

#include <netinet/in.h>
#include <arpa/inet.h>
#ifdef __osf__
#include <machine/endian.h>
#endif

#include <tcl.h>

#include "scotty.h"

#include "ined.c"

#ifdef HAVE_BONES
extern void Bones_Init();
#endif

static int norc = 0;            /* norc -- dont read the initialization file */
static int ined = 0;            /* dont initialize for tkined */
static int verbose = 0;         /* be verbose - default is off */
static char *progname;          /* the name of the game */
#ifdef HAVE_READLINE
static char *histfilename;      /* name of the current history file */
#endif

static char *prompt = "";       /* the prompt */
static Tcl_Interp *interp;      /* the main tcl interpreter */

char *ScottyInitCmd =
	"if [file exists [info library]/scotty.tcl] {source [info library]/scotty.tcl}";

char *inedMainCmd = "ined_main_loop";

/*
 * A fail-save malloc.
 */

char*
xmalloc (size)
    unsigned size;
{
    char *mem;

    for (;;) 
    {
	mem = (char*) malloc(size);
	if (mem) break;
	sleep (1);
    }
    
    return mem;
}

/*
 * A fail-save realloc.
 */

char*
xrealloc (ptr, size)
    char* ptr;
    unsigned size;
{
    char *mem;
    
    if (!ptr) 
	    mem = (char*) xmalloc(size);
    else
	    for (;;) 
	    {
		mem = (char*) realloc(ptr, size);
		if (mem) break;
		sleep (1);
	    }
    
    return mem;
}

/*
 * We define our own strdup since some systems don't have it.
 */

char* 
xstrdup(s) 
    char *s;
{
    char *ns = xmalloc(strlen(s)+1);
    return strcpy(ns, s);
}

/*
 * sleep -- what else?
 */

static int
sleep_cmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp* interp;
    int argc;
    char** argv;
{
    if (argc != 2) {
	Tcl_AppendResult (interp, "bad # arg: ", argv[0],
			  " time", (char *)NULL);
	return TCL_ERROR;
    }
    
    sleep (atoi(argv[1]));
    
    return TCL_OK;
}

/*
 * Get the current time. Needed to do some performance comparisons.
 */

static int
getclock_cmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp* interp;
    int argc;
    char** argv;
{
    if (argc != 1) {
	Tcl_AppendResult (interp, "bad # arg: ", argv[0], (char *)NULL);
	return TCL_ERROR;
    }
    
    sprintf (interp->result, "%ld", (long) time ((time_t *) NULL));
    return TCL_OK;
}

/*
 * The nslook command is just a wrapper around gethostbyname() and 
 * gethostbyaddr() to do quick name or address lookups. It was motivated
 * by nslookup written by Juergen Nickelsen <nickel@cs.tu-berlin.de>.
 */

static int
nslook_cmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp* interp;
    int argc;
    char** argv;
{
    struct hostent *host;
    unsigned long addr;
    struct in_addr *paddr;
    char buffer[20];
    
    if (argc != 2) {
	Tcl_AppendResult (interp, "bad # arg: ", argv[0],
			  " host-or-address", (char *)NULL);
	return TCL_ERROR;
    }
    
    if ((addr = inet_addr(argv[1])) != -1) {
	if (! (host = gethostbyaddr ((char *) &addr, 4, AF_INET))) {
	    Tcl_AppendResult (interp, "can not lookup ", argv[1], 
			      (char *) NULL);
	    return TCL_ERROR;
	}
	Tcl_AppendResult (interp, host->h_name, (char *)NULL);
	while (*host->h_aliases) {
	    Tcl_AppendResult (interp, " ", *host->h_aliases++, (char *)NULL);
	}
    } else {
	if ((host = gethostbyname (argv[1])) == NULL) {
	    Tcl_AppendResult (interp, "can not lookup ", 
			      argv[1], (char *)NULL);
	    return TCL_ERROR;
	}
	while (*host->h_addr_list) {
	    paddr = (struct in_addr *) *host->h_addr_list++;
            addr = ntohl (paddr->s_addr);
            sprintf (buffer, "%lu.%lu.%lu.%lu ",
                     (addr >> 24) & 0xff, (addr >> 16) & 0xff,
                     (addr >> 8) & 0xff, addr & 0xff);
	    Tcl_AppendResult (interp, buffer, (char *)NULL);
	}
    }
    
    return TCL_OK;
}

#ifdef HAVE_READLINE

/*
 * open and close a history file
 */

static char*
open_history()
{
    char *fn;
    
    if (!(fn = (char*) xmalloc(255))) return NULL;
    strcpy (fn, getenv("HOME"));
    strcat (fn, "/.");
    strcat (fn, progname);
    strcat (fn, "_history");    
    (void) read_history (fn);
    if (verbose) printf ("reading %s\n", fn);
    return fn;
}

static void
close_history()
{
    stifle_history(HISTSIZE);
    
    if (histfilename) {
	if (write_history(histfilename))
		fprintf(stderr, "%s: error writing %s\n", 
			progname, histfilename);
    }
}

/* Generator function for command completion.  STATE lets us know whether
   to start from scratch; without any state (i.e. STATE == 0), then we
   start at the top of the list. */

static char*
command_generator (text, state)
    char* text;
    int state;
{
    static int cmd_list_index;
    static char** cmd_list = NULL;
    static int len;
    char* name;
    int i, result, cmd_list_length;
    char *p;
    
    /* If this is a new word to complete, initialize now.  This 
       includes saving the length of TEXT for efficiency, and 
       initializing the index variable to 0. */
    
    if (!state) {
	cmd_list_index = 0;
	len = strlen (text);
	
	/* get the current command list */
	result = Tcl_Eval (interp, "info commands");
	if (result != TCL_OK) return (char*) NULL;
	
	cmd_list_length = 1;
	p = xstrdup (interp->result);
	while ((p = strchr(++p, ' '))) cmd_list_length++;
	
	if (cmd_list) free (cmd_list);
	cmd_list = (char**) xmalloc((cmd_list_length+1)*sizeof(char*));
	
	i = 0;
	cmd_list[i++] = strtok (interp->result," ");
	while ((cmd_list[i++] = strtok (NULL, " ")));
    }
    
    /* Return the next name which partially matches from 
       the command list. */
    
    while ((name = cmd_list[cmd_list_index])) {
	cmd_list_index++;
	if (strncmp (name, text, len) == 0) return xstrdup(name);
    }
    
    /* If no names matched, then return NULL. */
    return (char*) NULL;
}

/* Attempt to complete on the contents of TEXT.  START and END show the
   region of TEXT that contains the word to complete.  We can use the
   entire line in case we want to do some simple parsing.  Return the
   array of matches, or NULL if there aren't any. */

static char**
scotty_completion (text, start, end)
    char* text;
    int start;
    int end;
{
    /* Try to complete the command or the token */
    return completion_matches (text, command_generator);
}

static void
initialize_readline ()
{
    /* Tell the completer that we want a crack first. */
    rl_attempted_completion_function = scotty_completion;
}

#endif

/*
 * get the commands (using readline) and process them
 */

static void
cmd_loop()
{
    char *line = NULL;
    char *newline;
    char *cmd, *pt;
    int result;
    Tcl_DString buffer;
    Tcl_DStringInit (&buffer);
    
#ifdef HAVE_READLINE
    initialize_readline();
    using_history ();
    histfilename = open_history();
#endif
    
    pt = prompt;
    while (1) {
	if (line != (char *) NULL) free (line);
#ifdef HAVE_READLINE
	line = readline(pt);
#else
	printf("%s", pt);
	fflush(stdout);
	line = (char*) xmalloc(256);
	if (fgets(line, 256, stdin) == NULL) line = NULL;
#endif
	
	if (line && *line) {
	    if (!strcmp (line,"exit")) break;
	    newline = xmalloc(strlen(line)+2);
	    newline = strcat (strcpy(newline, line), "\n");
	    cmd = Tcl_DStringAppend (&buffer, newline, -1);
	    if (! Tcl_CommandComplete (cmd)) {
		if (isatty(0)) pt = "> ";
	    } else {
		result = Tcl_Eval (interp, cmd);
		if (result == TCL_OK) {
		    if (*interp->result != 0) {
			printf ("%s\n", interp->result);
		    }
		} else {
		    printf("Error ");
		    if (*interp->result != 0) {
			printf (": %s\n", interp->result);
		    } else {
			printf ("\n");
		    }
		}
		if (isatty(0)) pt = prompt;
		Tcl_DStringFree (&buffer);
	    }
#ifdef HAVE_READLINE
	    if (isatty(0)) add_history(line);
#endif
	    free (newline);
	}
	
	if (!line) {
	    printf("\n");
	    break;
	}
	
    }
    
    Tcl_DStringFree (&buffer);
    
#ifdef HAVE_READLINE    
    if (isatty(0)) close_history();
#endif
}

/*
 * execute commands in $HOME/.scottyrc
 */

static void
	read_scottyrc()
{
    char *home, *cmd;
    
    if ((home = getenv("HOME")) != NULL) {
	if ((cmd = (char*) xmalloc (strlen(home)+15))) {
	    sprintf (cmd, "%s/.scottyrc", home);
	    if (verbose) printf ("evaluating %s\n", cmd);
	    Tcl_EvalFile (interp, cmd);
	    free (cmd);
	}
    }
}

/*
 * Initialize a new interpreter.
 */

int
Scotty_Init (interp)
    Tcl_Interp* interp;
{
    Tcl_CreateCommand (interp, "sleep", sleep_cmd,
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand (interp, "getclock", getclock_cmd,
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    
    /* The select command from the extended tcl distribution */
    Tcl_CreateCommand (interp, "select", Tcl_SelectCmd,
		       (ClientData) NULL, (void (*)()) NULL);
    
    Tcl_CreateCommand (interp, "nslook", nslook_cmd,
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
#ifdef HAVE_SYSLOG
    Tcl_CreateCommand (interp, "syslog", syslog_cmd,
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
#endif
#ifdef HAVE_ICMP
    Tcl_CreateCommand (interp, "icmp", icmp_cmd,
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
#endif
#ifdef HAVE_RPC
    Tcl_CreateCommand (interp, "rpc", rpc_cmd,
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
#endif
#ifdef HAVE_DNS
    Tcl_CreateCommand (interp, "dns", dns_cmd,
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
#endif
#ifdef HAVE_BONES
    Bones_Init (interp);
#endif
#ifdef HAVE_SNMP
    snmp_init (interp);
#endif
    
    /* We use the raw tcp extensions to tcl written by Tim MacKenzie    */
    /* (tym@dibbler.cs.monash.edu.au) as of Thu Jun 25 15:11:13 EST 1992 */
    
    Tcl_CreateCommand (interp, "accept",  accept_cmd,
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand (interp, "shutdown", shutdown_cmd,
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand (interp, "connect", connect_cmd,
		       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

    if (Tcl_Eval (interp, ScottyInitCmd) == TCL_ERROR) {
	return TCL_ERROR;
    }

    return TCL_OK;
}

/*
 * here we start
 */

int
main(argc, argv)
    int argc;
    char** argv;
{
    char *filename = NULL;
    char *query = NULL;
    char buffer[20];
    Tcl_DString sc_argv;
    int i, c;
    int sc_argc = 0;
    int errflg = 0;
    int res;

    if ((progname = strrchr(argv[0], '/')) == NULL) {
	progname = argv[0];
    } else {
	progname++;
    }
    
    if (isatty(0)) {
	prompt = (char*) xmalloc (strlen(progname)+4);
	sprintf(prompt, "%s > ", progname);
    }

    Tcl_DStringInit (&sc_argv);

    for (i = 1; i < argc; i++) {
        if (argv[i][0] != '-') {
	    Tcl_DStringAppendElement (&sc_argv, argv[i]); 
	    sc_argc++;
	    continue;
	}
	for (c = 1; (i < argc) && (c < strlen(argv[i])); c++) {
	    switch (argv[i][c]) {
	      case 'c' :
		if ( (argv[i][c+1] == '\0') && (++i < argc) )
		    query = argv[i]; else errflg++;
		c = strlen(argv[i]);
		break;
	      case 'f':
		if ( (argv[i][c+1] == '\0') && (++i < argc) ) 
		    filename = argv[i]; else errflg++;
		c = strlen(argv[i]);
		break;
	      case 'n':
		norc++;
		break;
	      case 'i':
		ined++;
		break;
	      case 'v':
		verbose++;
		break;
	      case '?':
		errflg++;
		break;
	      case '-':
		c = strlen(argv[i]);
		for (i++; i < argc; i++) {
		    Tcl_DStringAppendElement (&sc_argv, argv[i]);
		    sc_argc++;
		}
		break;
	      default:
		c = strlen(argv[i]);
		Tcl_DStringAppendElement (&sc_argv, argv[i]);
		sc_argc++;
		break;
	    }
	}
    }
    
    if (errflg) {
	fprintf (stderr,
		 "usage: %s [-c query] [-f file] [-n] [-i] [-v]\n", 
		 progname);
	return 2;
    }

    /* Create and initialize the interpreter */

    interp = Tcl_CreateInterp();
    if (Tcl_Init(interp) == TCL_ERROR) {
        return TCL_ERROR;
    }
    if (Scotty_Init(interp) == TCL_ERROR) {
        return TCL_ERROR;
    }

    /* Set the tcl_interactive variable */

    Tcl_SetVar (interp, "tcl_interactive", "0", TCL_GLOBAL_ONLY);

    /* Set the verbose variable of the interpreter */
    
    sprintf (buffer, "%d", verbose);
    Tcl_SetVar(interp, "verbose", buffer, TCL_GLOBAL_ONLY); 

    /* Set the global variables argc and argv */

    Tcl_SetVar (interp, "argv", Tcl_DStringValue(&sc_argv), TCL_GLOBAL_ONLY);
    Tcl_DStringFree (&sc_argv);
    sprintf (buffer, "%d", sc_argc);
    Tcl_SetVar (interp, "argc", buffer, TCL_GLOBAL_ONLY);

    /* Write the version number to scotty_version. */
    
    Tcl_SetVar(interp, "scotty_version", SCOTTY_VERSION, TCL_GLOBAL_ONLY);
    if (verbose) printf ("scotty version %s\n", SCOTTY_VERSION);

    /* 
     * Adjust the auto_path to take care of $HOME/.tkined,
     * TKINED_PATH and TKINEDPATH. 
     */

    if (ined) {

	Tcl_Eval (interp, ined_tcl);

	Tcl_Eval (interp, 
		  "set auto_path \"/usr/local/lib/tkined [info library]\"");
	Tcl_VarEval (interp, "set auto_path \"", TKINEDLIB, " $auto_path\"",
		     (char *) NULL);
	Tcl_VarEval (interp, "if [info exists env(HOME)] { ",
		     "set auto_path \"$env(HOME)/.tkined $auto_path\"}",
		     (char *) NULL);
	Tcl_VarEval (interp, "if [info exists env(TKINED_PATH)] {",
		     "set auto_path \"$env(TKINED_PATH) $auto_path\"}",
		     (char *) NULL);

	if (Tcl_Eval (interp, "ined size") != TCL_OK) {
	    fprintf (stderr, "%s: can't talk to [tk]ined: %s\n",
		     progname, interp->result);
	    exit (32);
	}
    }

    /* first process the $HOME/.scottyrc */

    if (!norc) read_scottyrc();
    
    /* process a single query */

    if (query) {
	int result = Tcl_Eval (interp, query);
	if (result == TCL_OK) {
	    if (*interp->result != 0) {
		printf ("%s\n", interp->result);
	    }
	} else {
	    fprintf (stderr, "Error: %s\n%s\n", interp->result,
		     Tcl_GetVar (interp, "errorInfo", TCL_GLOBAL_ONLY));
	}
#ifdef HAVE_ICMP
	icmp_close ();
#endif
	return 0;
    }
    
    /* process queries from a file */

    if (filename) {
	res = Tcl_EvalFile (interp, filename);
	if (res != TCL_OK) {
	    fprintf (stderr, "Error: %s\n%s\n", interp->result,
		     Tcl_GetVar (interp, "errorInfo", TCL_GLOBAL_ONLY));
	}

	/* try to eval the ined_command_loop */
	if (ined) {
	    Tcl_Eval (interp, inedMainCmd);
	    fprintf (stderr, "** %s\n", interp->result);
	}

#ifdef HAVE_ICMP
	icmp_close ();
#endif
	return res;
    }
    
    /* ok, we are interactive -- that's where the fun beginns */

    if (isatty(0)) {
	Tcl_SetVar (interp, "tcl_interactive", "1", TCL_GLOBAL_ONLY);
    }
    cmd_loop ();
#ifdef HAVE_ICMP
    icmp_close ();
#endif
    return 0;
}
