/*
 * tcp.c
 *
 * This file contains a simple Tcl "connect" command that returns a
 * standard Tcl File descriptor. You can accept connection and shutdown
 * parts of full duplex connections. This file was heavily motivated
 * by the tclRawTCP extension written by Pekka Nikander 
 * <pnr@innopoli.ajk.tele.fi> and Tim MacKenzie 
 * <tym@dibbler.cs.monash.edu.au) for TCL 6.7 and TK 3.2.
 *
 * Copyright (c) 1993, 1994
 *                    J. Schoenwaelder
 *                    TU Braunschweig, Germany
 *                    Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 * Copyright 1992 Telecom Finland
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  Telecom Finland
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 */

#ifdef HAVE_TCP

#include <assert.h>
#include <errno.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/un.h>

#include <tcl.h>

#include "scotty.h"

#ifdef __osf__
#  include <machine/endian.h>
#endif

static int inet_connect _ANSI_ARGS_((char *host, char *port, int server));
static int unix_connect _ANSI_ARGS_((char *path, int server));

/*
 * Open a socket connection to a given host and service.
 * Set the global variable connect_info(file%d) to the obtained
 * port when setting up server.
 */

int
connect_cmd(notUsed, interp, argc, argv)
    ClientData notUsed;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int fd;
    int server = 0;
    int unicks = 0;
    FILE *f;
    char *fname;
    
    if (argc != 2 && argc != 3 &&
	(argc != 4 || (argc == 4 && strcmp(argv[1], "-server")))) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " [-server] address_spec\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (!strcmp (argv[1], "-server")) server = 1;
    if (argc == 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " [-server] address_spec\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Create the connection
     */

    if (argc - server == 2) {/* Unix domain socket */
	unicks = 1;
	fd = unix_connect (argv[1+server], server);
    } else
	fd = inet_connect (argv[1+server], argv[2+server], server);

    if (fd < 0) {
	/* Tell them why it fell apart */
	if (unicks)
	    if (server)
		Tcl_AppendResult(interp,
		    "Couldn't setup listening socket with path \"",
		    argv[1+server],"\" : ", Tcl_PosixError (interp),
		    (char *) NULL);
	    else
		Tcl_AppendResult(interp,
		    "Couldn't connect to \"",argv[1],"\" : ",
		    Tcl_PosixError (interp), (char *) NULL);
	else
	    if (server)
		Tcl_AppendResult(interp,
		    "couldn't setup listening socket on port ",
		    atoi(argv[3])==0?"any": argv[3]," using address \"",
		    strlen(argv[2])? argv[2] : "anywhere." , "\": ",
		    Tcl_PosixError (interp), (char *)NULL);
	    else
		Tcl_AppendResult(interp, "couldn't open connection to \"",
				 argv[1], "\" port \"", argv[2], "\": ",
				 Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }

    if ((f = fdopen(fd, "r+")) == NULL) {
	Tcl_PosixError (interp);
	return TCL_ERROR;
    }

    /* Make it unbuffered ! */
    setbuf(f, (char *) 0);

    Tcl_EnterFile (interp, f, TCL_FILE_READABLE | TCL_FILE_WRITABLE);

    fname = xstrdup(interp->result);

    if (server && !unicks) {
	char buf[50];
	struct sockaddr_in sockaddr;
	int res,len = sizeof(sockaddr);
	res = getsockname (fd,(struct sockaddr *) &sockaddr, &len);
	if (res < 0) {
	    sprintf (buf, "%d", errno);
	} else 
		sprintf (buf, "%d", (int)ntohs(sockaddr.sin_port));
	Tcl_SetVar2 (interp, "connect_info", fname, buf, TCL_GLOBAL_ONLY);
    }

    Tcl_SetResult (interp, fname, TCL_DYNAMIC);
    return TCL_OK;
}

/*
 * Shutdown a a tcp connection. It is possible to shutdown a socket
 * for reading writing or both, although tcl will not know about it.
 * An attempt to write to a socket that has been closed for writing
 * will not fail. This is ugly but we keep independed from tcl 
 * internals, so I choose this way.
 */

int
shutdown_cmd(notUsed, interp, argc, argv)
    ClientData notUsed;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    FILE *f;
    int fd;
    int perms;

    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " fileid <option>\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    if (Tcl_GetOpenFile (interp, argv[1], 1, 0, &f) != TCL_OK) {
	return TCL_ERROR;
    }
    
    fd = fileno (f);
    perms = Tcl_FilePermissions (f);
    if (perms == -1) {
	interp->result = "can not lookup file access mode";
	return TCL_ERROR;
    }
    
    if (!strcmp (argv[2], "0") || !strcmp (argv[2], "receives") || 
	!strcmp (argv[2], "read")) {
	if (perms & TCL_FILE_READABLE != TCL_FILE_READABLE) {
	    Tcl_AppendResult(interp, "File is not readable",(char *) NULL);
            return TCL_ERROR;
	}
	if (shutdown (fd, 0)) {
	    Tcl_AppendResult (interp, "shutdown: ", Tcl_PosixError (interp),
			      (char *) NULL);
	    return TCL_ERROR;
	}
    } else if (!strcmp (argv[2], "1") || !strcmp (argv[2], "sends") ||
	       !strcmp(argv[2], "write")) {
	if (perms & TCL_FILE_WRITABLE != TCL_FILE_WRITABLE) {
	    Tcl_AppendResult(interp, "File is not writeable",(char *) NULL);
            return TCL_ERROR;
	}
	if (shutdown (fd, 1)) {
	    Tcl_AppendResult (interp, "shutdown: ", Tcl_PosixError (interp),
			      (char *) NULL);
	    return TCL_ERROR;
	}
    } else if (!strcmp(argv[2], "2") || !strcmp(argv[2], "all") ||
	    !strcmp(argv[2], "both")) {
	if (shutdown (fd, 2)) {
	    Tcl_AppendResult (interp, "shutdown: ", Tcl_PosixError (interp),
			      (char *) NULL);
	    return TCL_ERROR;
	}
    } else {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " fileid <option>\"", (char *) NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}
	
/*
 * Accept a connection on a listening socket. Return a new file
 * if a new connection is established. Set the global variable
 * connect_info(file%d) to a list containing the remote address 
 * (host ip, port) of the connector.
 */

int
accept_cmd(notUsed, interp, argc, argv)
    ClientData notUsed;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    struct sockaddr_in sockaddr;
    int len = sizeof sockaddr;
    FILE *f;
    int fd;
    char *fname;
    char buf[100];

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		     " listening_socket\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (Tcl_GetOpenFile (interp, argv[1], 1, 0, &f) != TCL_OK) {
	return TCL_ERROR;
    }
    
    fd = fileno(f);
    
    fd = accept (fd, (struct sockaddr *)&sockaddr, &len);

    if (fd < 0) {
	Tcl_PosixError (interp);
	return TCL_ERROR;
    }

    if ((f = fdopen(fd, "r+")) == NULL) {
	Tcl_PosixError (interp);
	return TCL_ERROR;
    }

    /* Make it unbuffered ! */
    setbuf(f, (char *) 0);

    Tcl_EnterFile (interp, f, TCL_FILE_READABLE | TCL_FILE_WRITABLE);

    fname = xstrdup(interp->result);

    if (sockaddr.sin_family == AF_INET) {
	sprintf (buf,"%s %d",inet_ntoa(sockaddr.sin_addr),
		 ntohs(sockaddr.sin_port));
    } else {
	buf[0] = 0;
    }
    Tcl_SetVar2 (interp, "connect_info", fname, buf, TCL_GLOBAL_ONLY);

    Tcl_SetResult (interp, fname, TCL_DYNAMIC);
    return TCL_OK;
}

/*
 * Create a (unix_domain) fd connection using given rendeavous.
 */

static int
unix_connect(path,server)
    char *path;		/* Path name to create or use */
    int  server;        /* 1->make server, 0->connect to server */
{
    struct sockaddr_un sockaddr;
    int sock, status;
    
    sock = socket(PF_UNIX, SOCK_STREAM, 0);
    if (sock < 0) {
	return -1;
    }
    
    sockaddr.sun_family = AF_UNIX;
    strncpy(sockaddr.sun_path,path,sizeof(sockaddr.sun_path)-1);
    sockaddr.sun_path[sizeof(sockaddr.sun_path)-1] = 0;
    
    if (server)
	status = bind(sock,(struct sockaddr *) &sockaddr, sizeof(sockaddr));
    else
	status = connect(sock, (struct sockaddr *) &sockaddr, sizeof(sockaddr));
    
    if (status < 0) {
	close (sock);
	return -1;
    }

    if (server) {
	listen(sock,5);
	return sock;
    }
    
    return sock;
}

/*
 * Create a (inet domain) fd connection to given host and port.
 */

static int
inet_connect(host, service,server)
    char *host;			/* Host to connect, name or IP address */
    char *service;		/* Port to use, service name or port number */
    int  server;
{
    struct hostent *hostent, _hostent;
    struct servent *servent, _servent;
    struct protoent *protoent;
    struct sockaddr_in sockaddr;
    int sock, status;
    int hostaddr, hostaddrPtr[2];
    int servport;
    extern int errno;
    
    hostent = gethostbyname(host);
    if (hostent == NULL) {
	hostaddr = inet_addr(host);
	if (hostaddr == -1) {
	    if (server && !strlen(host)) 
		hostaddr = INADDR_ANY;
	    else {
		errno = EINVAL;
		return -1;
	    }
	}	
	_hostent.h_addr_list = (char **)hostaddrPtr;
	_hostent.h_addr_list[0] = (char *)&hostaddr;
	_hostent.h_addr_list[1] = NULL;
	_hostent.h_length = sizeof(hostaddr);
	_hostent.h_addrtype = AF_INET;
	hostent = &_hostent;
    }
    servent = getservbyname (service, "tcp");
    if (servent == NULL) {
	servport = htons (atoi(service));
	if (servport == -1) { 
	    errno = EINVAL;
	    return -1;
	}
	_servent.s_port = servport;
	_servent.s_proto = "tcp";
	servent = &_servent;
    }
    protoent = getprotobyname (servent->s_proto);
    if (protoent == NULL) {
	errno = EINVAL;
	return -1;
    }
    
    sock = socket (PF_INET, SOCK_STREAM, protoent->p_proto);
    if (sock < 0) {
	return -1;
    }
    
    sockaddr.sin_family = AF_INET;
    memcpy((char *)&(sockaddr.sin_addr.s_addr),
	   (char *) hostent->h_addr_list[0],
	   (size_t) hostent->h_length);
    sockaddr.sin_port = servent->s_port;

    
    if (server)
	status = bind (sock, (struct sockaddr *) &sockaddr, sizeof(sockaddr));
    else
	status = connect (sock, (struct sockaddr *) &sockaddr, sizeof(sockaddr));
    
    if (status < 0) {
	close (sock);
	return -1;
    }

    if (server) {
	listen(sock,5);
	return sock;
    }
    
    return sock;
}

#endif
