#! /usr/local/bin/scotty -inf
##
## Some utilities for the [TK]INED editor.
##
## Copyright (c) 1993, 1994
##                    J. Schoenwaelder
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

set port 6336

##
## Test if we are at home.
##

proc local {} {
    if {[catch {dns ptr [nslook [exec hostname]]} name]} {
	return 0
    }
    return [string match "*cs.tu-bs.de" $name]
}

##
## Change the label to show the name.
##

proc "label name" {list} {
    foreach comp $list {
	ined -noupdate label [ined_id $comp] name
    }
}

##
## Change the label to show the address.
##

proc "label address" {list} {
    foreach comp $list {
	ined -noupdate label [ined_id $comp] address
    }
}

##
## Change the label to show the text.
##

proc "label text" {list} {
    foreach comp $list {
	ined -noupdate label [ined_id $comp] text
    }
}

##
## Remove the label.
##

proc "label clear" {list} {
    foreach comp $list {
	ined -noupdate label [ined_id $comp] clear
    }
}

##
## Select all objects directly connected to the objects of the current
## selection.
##

proc "select neighbours" {list} {
    foreach comp $list { 
	set id [ined_id $comp]
	set is_selected($id) true
    }
    foreach comp $list {
	foreach link_id [ined_links $comp] {
	    set link [ined retrieve $link_id]
	    set ida [ined_ida $link]
	    set idb [ined_idb $link]
	    if {![info exists is_selected($ida)]} {
		ined -noupdate select $ida
		set is_selected($ida) ""
	    }
	    if {![info exists is_selected($idb)]} {
                ined -noupdate select $idb
                set is_selected($idb) ""
            }
	}
    }
}

##
## Select all members of group objects.
##

proc "select member" {list} {
    foreach comp $list { 
	set id [ined_id $comp]
	set is_selected($id) true
    }
    foreach comp $list {
	if {[ined_type $comp] != "GROUP"} continue
	foreach m [ined_member $comp] {
	    if {![info exists is_selected($m)]} {
		ined -noupdate select $m
		set is_selected($m) ""
	    }
	}
    }
}

##
## Select objects by a regular expression match against their type.
##

proc "select by type" {list} {
    static regex

    if {![info exists regex]} { set regex "" }
    set regex [ined request "Select objects by type." {} \
	       "Enter the regular expression:" "{{} $regex}"]
    if {$regex==""} return

    foreach comp $list {
        set id [ined_id $comp]
        set is_selected($id) true
    }

    foreach object [ined retrieve] {
	set id [ined_id $object]
	set is_sel [info exists is_selected($id)]
	set type [ined_type $object]
	if {[regexp -nocase [string tolower $regex] $type]} {
	    if {!$is_sel} { ined -noupdate select $id }
	} else {
	    if {$is_sel} { ined -noupdate select $id }
	}
    }
}

##
## Select objects by a regular expression match against their name.
##

proc "select by name" {list} {
    static regex

    if {![info exists regex]} { set regex "" }
    set regex [ined request "Select objects by name." {} \
	       "Enter the regular expression:" "{{} $regex}"]
    if {$regex==""} return

    foreach comp $list {
        set id [ined_id $comp]
        set is_selected($id) true
    }

    foreach object [ined retrieve] {
	set id [ined_id $object]
	set is_sel [info exists is_selected($id)]
	set name [ined_name $object]
	if {[regexp -nocase $regex $name]} {
	    if {!$is_sel} { ined -noupdate select $id }
	} else {
	    if {$is_sel} { ined -noupdate select $id }
	}
    }
}

##
## Select objects by a regular expression match against their address.
##

proc "select by address" {list} {
    static regex

    if {![info exists regex]} { set regex "" }
    set regex [ined request "Select objects by name." {} \
	       "Enter the regular expression:" "{{} $regex}"]
    if {$regex==""} return

    foreach comp $list {
        set id [ined_id $comp]
        set is_selected($id) true
    }

    foreach object [ined retrieve] {
	set id [ined_id $object]
	set is_sel [info exists is_selected($id)]
	set address [ined_address $object]
	if {[regexp -nocase $regex $address]} {
	    if {!$is_sel} { ined -noupdate select $id }
	} else {
	    if {$is_sel} { ined -noupdate select $id }
	}
    }
}

##
## Select nodes by number of links. Useful to select gateways.
##

proc "select by # of links" {list} {
    static number

    if {![info exists number]} { set number "" }
    set number [ined request "Select objects by # of links." {} \
	       "Enter the number:" "{{} $number}"]
    if {$number==""} return

    foreach comp $list {
        set id [ined_id $comp]
        set is_selected($id) true
    }

    foreach object [ined retrieve] {
	set id [ined_id $object]
	set is_sel [info exists is_selected($id)]
	set links [ined_links $object]
	if {[llength $links]==$number} {
	    if {!$is_sel} { ined -noupdate select $id }
	} else {
	    if {$is_sel} { ined -noupdate select $id }
	}
    }
}

##
## Let the user type a command and execute it.
##

proc "ined command" {list} {
    static cmd

    if {![info exists cmd]} { set cmd "ined " }
    set cmd [ined request "Execute an ined tcl command." "{Command: {$cmd}}"]
    set cmd [join $cmd]

    if {[string length $cmd]>0} {
        if [catch {eval $cmd} result] {
	    ined acknowledge "Ined command failed:" "$result"
	} else {
	    ined acknowledge "Result of $cmd:" "$result"
	}
    }
}

##
## Read a tcl script from a file and executed all commands.
##

proc "source file" {list} {
    static filename

    if ![info exists filename] { set filename "" }
    set filename [ined request "Read and execute a file." \
		  "{{Filename:} $filename}"]
    if {$filename==""} return

    if [file readable $filename] {
	source $filename
    } else {
	ined acknowledge "File $filename not readable."
    }
}

##
## Become a master. Setup a server socket and wait for an incoming
## request. After connection establishment tell ined to trace all
## necessary information to the speak_proc callback.
##

proc speak_proc {cmd} {
    global socket
    puts $socket "$cmd"
    flush $socket
}

proc "speak to slaves" {list} {
    global socket
    global port
    static control_socket

    ined acknowledge "Speak is still under construction. Use with care!!"

    if {![info exists control_socket]} {
	if {[catch {connect -server "" $port} control_socket]} {
	    ined acknowledge "Can not create serversocket:" $control_socket
	    return
	}
    }
    set socket [accept $control_socket]
    ined trace start speak_proc
    shutdown $control_socket all
    close $control_socket
    return
}

##
## Become a slave. Connect to a server. If succesful read until eof
## and process everything we get. 
##

proc "listen to master" {list} {
    global port

    ined acknowledge "Listen is still under construction. Use with care!!"

    set host [ined request "Hostname of the tkined master:" "{{} localhost}"]
    if {$host==""} return
    if {[catch {connect $host $port} socket]} {
	ined acknowledge "Can not connect to $host:" $socket
	return
    }
    while {![eof $socket]} {
	gets $socket line
	process $line
    }
}

##
## This is the hard part. Here we must parse all commands to eliminate
## simple query commands and to map the ids to the ids used by our slave
## tkined.
##

proc process {cmd} {
    static myid
    if {[llength $cmd]<=2} return
    case [lindex $cmd 1] in {
	{select} {
	    puts "select discarded: $cmd"; flush stdout
	    return
	}
	{create} { 
            set cmd [split $cmd ">"]
	    set id [string trim [lindex $cmd 1]]
	    set cmd [lindex $cmd 0]
	    switch [lindex $cmd 2] {
		LINK {
		    set ida [lindex $cmd 3]
		    if {![info exists myid($ida)]} return
		    set cmd [lreplace $cmd 3 3 $myid($ida)]
		    set idb [lindex $cmd 4]
		    if {![info exists myid($idb)]} return
		    set cmd [lreplace $cmd 4 4 $myid($idb)]
		}
		GROUP {
		    set member [lrange $cmd 3 end]
		    set cmd [lrange $cmd 0 2]
		    foreach guy $member {
			if {![info exists myid($guy)]} return
			lappend cmd $myid($guy)
		    }
		}
	    }
	    if {[catch {eval $cmd} res]} return
	    set myid($id) $res
	    return
	}
	{delete} {
	    set ids [join [lrange $cmd 2 end]]
	    set myids ""
	    foreach id $ids {
		if {![info exists myid($id)]} continue
		lappend myids $myid($id)
	    }
	    puts stdout "[lreplace $cmd 2 end $myids]"; flush stdout
	    catch {eval [lreplace $cmd 2 end $myids]}
	}
	{   retrieve 
	    move 
	    icon 
	    font 
	    color 
	    label 
	    name 
	    address 
	    oid 
	    lower 
	    raise} {
		set id [lindex $cmd 2]
		if {![info exists myid($id)]} return
		catch {eval [lreplace $cmd 2 2 $myid($id)]}
		return
	    }
	
	{page}     { catch {eval $cmd} }

	{stripchart piechart barchart} {
	    set id [lindex $cmd 2]
	    puts stdout "got $id"; flush stdout
	    if {![info exists myid($id)]} return
	    catch {eval [lreplace $cmd 2 2 $myid($id)]}
	    return
	}
	
	{acknowledge confirm request browse list} {
	    return
	}

	{trace status} {
	    return
	}

	{default} {
	    puts stdout "listen to master: unkown command $cmd"; flush stdout
	}
    }
}

##
## Display some help about this tool.
##

proc "help INED Utilities" {list} {
    ined browse "Help about INED Utilities" {
	"label name:" 
	"    Show the name of all selected objects." 
	"" 
	"label address:"
	"    Show the address of all selected objects." 
	""
	"label text:" 
	"    Show the associated text of all selected objects." 
	""   
	"label clear:" 
	"    Don't show any label for all selected objects." 
	"" 
	"select neighbours:" 
	"    Add all objects connected to the selected objects with" 
	"    a direkt link." 
	"" 
	"select member:" 
	"    Add all member of an expanded group to the selected objects."  
	""  
	"select by type:"    
	"    Select all objects which type matches a regular expression." 
	"" 
	"select by name:" 
	"    Select all objects which name matches a regular expression." 
	"" 
	"select by address:" 
	"    Select all objects which address matches a regular expression." 
	"" 
	"select by # of links:" 
	"    Select all objects that have a given number of links." 
	"" 
	"speak to slaves:" 
	"    Set up a TCP connection to forward all changes." 
	"" 
	"listen to master:" 
	"    Set up a TCP connection to receive all changes of another INED." 
	"" 
	"ined command:" 
	"    Enter one of the INED TCL commands and see what happens." 
	"" 
	"ined command:" 
	"    Read and execute a file containing INED TCL commands." 
    } 
}

##
## Delete the TOOL Manager and exit this interpreter.
##

proc "delete INED Utilities" {list} {
    global tools
    foreach id $tools { ined delete $id }
    exit
}

if [local] {
set tools [ ined create TOOL "INED Utilities" \
	    "label name" "label address" "label text" "label clear" "" \
	    "select neighbours" "select member" "select by type" \
	    "select by name" "select by address" "select by # of links" "" \
	    "speak to slaves" "listen to master" "" \
	    "ined command" "source file" "" \
	    "help INED Utilities" "delete INED Utilities" ]
} else {
set tools [ ined create TOOL "INED Utilities" \
	    "label name" "label address" "label text" "label clear" "" \
	    "select neighbours" "select member" "select by type" \
	    "select by name" "select by address" "select by # of links" "" \
	    "ined command" "source file" "" \
	    "help INED Utilities" "delete INED Utilities" ]
}
