#!/bin/sh
# The following line is a comment in Tcl 
# but is visible to the Bourne shell \
	exec wish "$0" ${1+"$@"}

# Define where the source code and include files live.
set tkGrep(SOURCE_DIR) "bin"
set tkGrep(INCLUDE_DIR) "include"

source "$tkGrep(INCLUDE_DIR)/update_index.tcl"
source "$tkGrep(INCLUDE_DIR)/tkGrep_global.tcl"
source "$tkGrep(INCLUDE_DIR)/tkGrep_xresources.tcl"

lappend auto_path $tkGrep(INCLUDE_DIR)
LibraryUpdateIndex $tkGrep(INCLUDE_DIR)

CreateOptionFile $tkGrep(SOURCE_DIR) $tkGrep(USER_OPTION_DIR) tkGrep

# Top frame for Directory, Filename and Regexp.
frame .f1 -width 60 -relief raised

label .f1.fname_label -text Filename -width 8 -anchor e
label .f1.regexp_label -text Regexp -width 8 -anchor e
label .f1.wizard_text -text Wizard -width 8 -anchor e
set tkGrep(ENTRY_WIDGET_FILENAME) [entry .f1.fname_entry -width 45 \
	-textvariable tkGrep(FILENAME)]
set tkGrep(ENTRY_WIDGET_REGEXP) [entry .f1.regexp_entry -width 45 \
	-textvariable tkGrep(REGEXP)]
set tkGrep(LABEL_WIZARD) [label .f1.wizard_label -width 45 -anchor w \
	-background $tkGrep(LABEL_WIZARD_BACKGROUND)]
set tkGrep(ANIMATE_WIDGET) [CreateAnimateWidget .f1 \
	$tkGrep(IMAGE_ICON)]

grid .f1.fname_label .f1.fname_entry -row 1
grid .f1.regexp_label .f1.regexp_entry -row 2
grid .f1.wizard_text .f1.wizard_label -row 3 -pady 3
grid .f1.icon -row 1 -column 2 -rowspan 2 -padx 2

# Bind mouse entry events to appropriate help message in label.
bind .f1.fname_label <Enter> [list $tkGrep(LABEL_WIZARD) configure \
	-text "Grep the files in this entry."]
bind .f1.fname_entry <Enter> [list $tkGrep(LABEL_WIZARD) configure \
	-text "Grep the files in this entry."]
bind .f1.regexp_label <Enter> [list $tkGrep(LABEL_WIZARD) configure \
	-text "Use this Regular Expression."]
bind .f1.regexp_entry <Enter> [list $tkGrep(LABEL_WIZARD) configure \
	-text "Use this Regular Expression."]
bind .f1.wizard_text <Enter> [list $tkGrep(LABEL_WIZARD) configure \
	-text "Wizard Help is displayed here."]
bind .f1.wizard_label <Enter> [list $tkGrep(LABEL_WIZARD) configure \
	-text "Wizard Help is displayed here."]
bind .f1.icon <Enter> [list $tkGrep(LABEL_WIZARD) configure \
	-text "Lighthouse searches while executing command."]

# The options frame is .f2.
CreateOptionFrame .f2 $tkGrep(USER_OPTION_FILE) tkGrep 60 \
	$tkGrep(LABEL_WIZARD)

# Create the Log Window as frame .f4 before the toolbar since it needs
#   to be defined for the Toolbar to configure button commands properly.
set tkGrep(LOG_WINDOW) [CreateLogWindow .f4 60 10 $tkGrep(LABEL_WIZARD)]

# The toolbar is frame .f3.
CreateToolbar .f3 tkGrep $tkGrep(LOG_WINDOW) $tkGrep(IMAGE_DIR) \
	$tkGrep(BROWSER) $tkGrep(HELP_FILE) $tkGrep(LABEL_WIZARD)

# The command display is frame .f5
CreateCommandDisplay .f5 tkGrep $tkGrep(IMAGE_DIR) 49 \
	$tkGrep(LABEL_WIZARD)

# Pack up the parent frame.
pack .f1 .f2 .f3 .f4 .f5 -side top -fill both -ipadx 2 -ipady 2

# Set globals from command line options and determine starting focus.
if {$argc > 1} {
    set i 0
    while {$i < [expr $argc - 1]} {
	append tkGrep(FILENAME)  "[lindex $argv $i] "
	incr i +1
    }
    set tkGrep(REGEXP) [lindex $argv [expr $argc - 1]]
} elseif {$argc == 1} {
    set tkGrep(FILENAME) [lindex $argv 0]
    focus $tkGrep(ENTRY_WIDGET_REGEXP)
} else {
    focus $tkGrep(ENTRY_WIDGET_FILENAME)
}

#<DEF> ---------------------------------------------------------------
#<DEF> tkGrepResetCommandCenter: Clear the command center at the top
#<DEF>       of the application.
#<DEF> 
#<DEF> Parameters:
#<DEF>    None
#<DEF> 
#<DEF> Global Variables:
#<DEF>   tkGrep(FILENAME)
#<DEF>   tkGrep(REGEXP)
#<DEF>   tkGrep(ENTRY_WIDGET_FILENAME)
#<DEF>   tkGrep(ENTRY_WIDGET_REGEXP)
#<DEF>   tkGrep(ENTRY_DEFAULT_BACKGROUND)
#<DEF>   tkGrep(ENTRY_DEFAULT_FOREGROUND)
#<DEF>   tkGrep(OPTION_xxx)
#<DEF>   tkGrep(COMMAND)
#<DEF> 
#<DEF> Notes:
#<DEF>    This procedure clears all the entry widgets in the command 
#<DEF>    center and resets the radio buttons.
#<DEF> 

proc tkGrepResetCommandCenter { } {
    global tkGrep

    # Reset the colors for the entry widgets.
    $tkGrep(ENTRY_WIDGET_REGEXP) configure \
	    -foreground $tkGrep(ENTRY_DEFAULT_FOREGROUND) \
	    -background $tkGrep(ENTRY_DEFAULT_BACKGROUND)
    $tkGrep(ENTRY_WIDGET_FILENAME) configure \
	    -foreground $tkGrep(ENTRY_DEFAULT_FOREGROUND) \
	    -background $tkGrep(ENTRY_DEFAULT_BACKGROUND)
    
    # Null checkbutton values and entry widgets.
    foreach x [array names tkGrep] {
	if [regexp {^OPTION|^FILENAME|^REGEXP|^COMMAND} $x] {
	    set tkGrep($x) ""
	}
    }

    # Set the focus to the FILENAME widget.
    focus $tkGrep(ENTRY_WIDGET_FILENAME)
}

#<DEF> ---------------------------------------------------------------
#<DEF> tkGrepControl: Animate, construct and execute the grep command,
#<DEF>                as well as log the output to the LOG Window.
#<DEF> 
#<DEF> Parameters:
#<DEF>    None
#<DEF> 
#<DEF> Global Variables:
#<DEF>   tkGrep(LOG_WINDOW)
#<DEF>   tkGrep(COLOR_HIGHLIGHT)
#<DEF>   tkGrep(COLOR_BACKGROUND)
#<DEF>   tkGrep(ENTRY_WIDGET_REGEXP)
#<DEF>   tkGrep(ENTRY_WIDGET_FILENAME)
#<DEF>   tkGrep(ENTRY_DEFAULT_FOREGROUND)
#<DEF>   tkGrep(ENTRY_DEFAULT_BACKGROUND)
#<DEF>   tkGrep(ENTRY_ERROR_FOREGROUND)
#<DEF>   tkGrep(ENTRY_ERROR_BACKGROUND)
#<DEF>   tkGrep(ANIMATE_WIDGET)
#<DEF>   tkGrep(IMAGE_ERROR)
#<DEF>   tkGrep(USER_EXAMPLE_FILE)
#<DEF>   tkGrep(ERROR_NOT_VALID_LIST)
#<DEF> 

proc tkGrepControl { } {
    global tkGrep

    set w $tkGrep(LOG_WINDOW)

    # Reset the colors for the entry widgets.
    $tkGrep(ENTRY_WIDGET_REGEXP) configure \
	    -fg $tkGrep(ENTRY_DEFAULT_FOREGROUND) \
	    -bg $tkGrep(ENTRY_DEFAULT_BACKGROUND)
    $tkGrep(ENTRY_WIDGET_FILENAME) configure \
	    -fg $tkGrep(ENTRY_DEFAULT_FOREGROUND) \
	    -bg $tkGrep(ENTRY_DEFAULT_BACKGROUND)
    
    # Clear out all the warnings in the LOG Window.
    set p_list [$w tag nextrange WARNING 0.0]
    if [string length $p_list] {
	$w delete [lindex $p_list 0] [lindex $p_list 1]
    }

    # Create the command line flags.
    set p_flag [CreateCommandLineFlag tkGrep]
    
    # Create the proper regular expression.  Substitute single slashes
    #   with a double because the interpreter will perform substitution
    #   in the tcl shell and command line shell.
    set p_regexp [ValidateRegexp $tkGrep(REGEXP)]
    if {$p_regexp == "-1"} {
	set p_regexp TEST; set tkGrep(REGEXP) TEST
	$tkGrep(ENTRY_WIDGET_REGEXP) configure \
		-fg $tkGrep(ENTRY_ERROR_FOREGROUND) \
		-bg $tkGrep(ENTRY_ERROR_BACKGROUND)
    }

    # Create the filenames to search with the grep command.
    if ![string length $tkGrep(FILENAME)] {
	set tkGrep(FILENAME) $tkGrep(USER_EXAMPLE_FILE)
	$tkGrep(ENTRY_WIDGET_FILENAME) configure \
		-fg $tkGrep(ENTRY_ERROR_FOREGROUND) \
		-bg $tkGrep(ENTRY_ERROR_BACKGROUND)
    }
    set p_file_list [CreateValidFileList $tkGrep(FILENAME)]
    if {$p_file_list == "-1"} {
	LogWarning $w $tkGrep(IMAGE_ERROR) $tkGrep(ERROR_NOT_VALID_LIST)
	$tkGrep(ENTRY_WIDGET_FILENAME) configure \
		-fg $tkGrep(ENTRY_ERROR_FOREGROUND) \
		-bg $tkGrep(ENTRY_ERROR_BACKGROUND)
	return
    }

    # Execute the process pipeline and store the output in the buffer.
    set p_buffer [ExecuteProcessPipeline grep $p_flag \
	    \"$p_regexp\" $p_file_list]
    set tkGrep(COMMAND) "grep $p_flag \"$p_regexp\" $tkGrep(FILENAME)"

    # Die on a system error.
    if [regexp {\*ERROR\*ERROR\*.*\*ERROR\*ERROR\*} $p_buffer] {
	regsub -all {\*ERROR\*ERROR\*} $p_buffer {} p_error
	LogWarning $w $tkGrep(IMAGE_ERROR) $p_error
	return
    }

    # Tell the user nothing was retrieved if the buffer is empty,
    #   and exit routine.
    if ![string length $p_buffer] {
	LogWarning $w $tkGrep(IMAGE_ERROR) \
		"Grep did not find $p_regexp in\n$p_file_list"
	return
    }

    # Indicate the application is doing something.
    AnimateWidget $tkGrep(ANIMATE_WIDGET) 5 tkGrep

    # Remove the start option from the regular expression so that
    #   the pattern still gets highlighted if grep returns extra
    #   characters, like line numbers, etc.
    if [regexp {\^} $p_regexp] {
	regsub {\^} $p_regexp {} p_regexp
    }
    regsub {\(} $p_regexp {\(} p_regexp
    regsub {\)} $p_regexp {\)} p_regexp
    regsub {\[} $p_regexp {\[} p_regexp
    regsub {\]} $p_regexp {\]} p_regexp
    regsub {\<} $p_regexp {\<} p_regexp
    regsub {\>} $p_regexp {\>} p_regexp
    regsub {\}} $p_regexp {\}} p_regexp
    regsub {\{} $p_regexp {\{} p_regexp

    # Display the output buffer in the TEXT_WIDGET_LOG window.
    foreach p_buffer_line [split $p_buffer \n] {
	set p_window_line [lindex [split [$w index insert] .] 0]

	append p_buffer_line \n
	$w insert end $p_buffer_line
	
	# Find the actual string to highlight, then highlight it the
	#   Log Window.
	set p_idx ""; set p_pattern ""
	if [catch {regexp -indices -nocase \
		$p_regexp $p_buffer_line p_idx} p_error] {
	    set p_idx ""
	} 
	if [string length $p_idx] {
	    set p_pattern [string range $p_buffer_line \
		    [lindex $p_idx 0] [lindex $p_idx 1]]
	    TextWidgetHighlightPattern $w $p_buffer_line \
		    $p_pattern $p_window_line 0 \
		    $tkGrep(COLOR_HIGHLIGHT) $tkGrep(COLOR_BACKGROUND)
	}
    } 
}
