#!/bin/sh
# The following line is a comment in Tcl 
# but is visible to the Bourne shell \
	exec wish "$0" ${1+"$@"}

#<DEF> ---------------------------------------------------------------
#<DEF> tkLBCreateWindow
#<DEF>
#<DEF> Description:
#<DEF>   Create a scrollable text window and parse the specified
#<DEF>   HTML file.
#<DEF>

proc tkLBCreateWindow { } {
    global tkLB

    # Define the window manager options.
    wm title . "Viewing Help"
    wm iconname . "tkLB"
    wm iconbitmap . "@$tkLB(IMAGEDIR)/tkLB_icon.xbm"

    # Create the toolbar above the text widget.
    frame .top -borderwidth 1 -relief raised -width 75
    button .top.left -image $tkLB(IMAGE_LEFT) -borderwidth 1 \
	    -command {tkLBOpenPrevNextURL -1}
    button .top.right -image $tkLB(IMAGE_RIGHT) -borderwidth 1 \
	    -command {tkLBOpenPrevNextURL +1}
    button .top.up -image $tkLB(IMAGE_UP) -borderwidth 1 \
	    -command [list .bottom.display.text yview scroll -1 pages]
    button .top.down -image $tkLB(IMAGE_DOWN) -borderwidth 1 \
	    -command [list .bottom.display.text yview scroll +1 pages]
    button .top.exit -image $tkLB(IMAGE_EXIT) -borderwidth 1 \
	    -command exit
    label .top.helplabel -width 20
    set tkLB(ANIMATE_WIDGET) [label .top.icon -relief raised \
	    -borderwidth 1 -image $tkLB(IMAGE_TKLB)]
    pack .top.icon -in .top -side right -padx 10
    pack .top.left .top.right .top.up .top.down .top.exit \
	    .top.helplabel -side left

    # Create trench between toolbar and HTML display.
    frame .middle -borderwidth 1 -relief sunken -width 75 -height 5

    # Create the bottom frame for the text widget.
    frame .bottom -borderwidth 1 -relief raised -width 75

    # Create the text widget to display the HTML.
    set tkLB(HTML_DISPLAY) [ScrolledWidget .bottom.display 1 flat \
	    text .bottom.display.text -setgrid true -wrap word \
	    -relief flat -borderwidth 1 -width 75 -height 30 \
	    -padx 10 -fg $tkLB(FG_COLOR) -bg $tkLB(BG_COLOR) \
	    -font $tkLB(FONT_TXT)]

    # Pack up the help frame.
    pack .top .middle .bottom -side top -fill x

    # Bind mouse entry events to appropriate help message in label.
    foreach b "left right up down exit icon" {
	bind .top.$b <Enter> \
		{.top.helplabel configure -text $tkLB(HELP%W)}
    }
    
    # Configure the text widget attributes.
    bind .bottom.display.text <Enter> \
	    {.top.helplabel configure -text $tkLB(HELP%W); \
	    .bottom.display.text configure -cursor left_ptr}

    # Create and display the HTML buffer.
    if [string length $tkLB(HTML_FILE)] {
	tkLBDisplayBuffer $tkLB(HTML_FILE)
    }
}

proc tkLBOpenPrevNextURL { p_incr } {
    global tkLB

    set p_next_index [expr $tkLB(URL_INDEX) + $p_incr]

    if {$p_next_index < 1} {
	return
    } elseif {$p_next_index > [expr [llength $tkLB(URL_LIST)]/2]} {
	return
    }

    set tkLB(URL_INDEX) $p_next_index

    tkLBDisplayBuffer [lindex $tkLB(URL_LIST) [expr \
	    [lsearch $tkLB(URL_LIST) $tkLB(URL_INDEX)] + 1]]
}


#<DEF> ---------------------------------------------------------------
#<DEF> tkLBDisplayBuffer
#<DEF> 
#<DEF> Description:
#<DEF>   Parse and display the HTML buffer in the supplied window.
#<DEF> 
#<DEF> TEXT FORMATTING ROUTINE DERIVED FROM JOHN HEIDEMANN.
#<DEF> SEE THE FILE PRVCR FOR MORE INFORMATION.
#<DEF>
#<DEF> Parameters:
#<DEF>   p_html_file - This file is necessary to support links to
#<DEF>                 other files, where the command is built
#<DEF>                 during parsing, so re-assigning the global
#<DEF>                 tkLB(HTML_FILE) will not work.
#<DEF>
#<DEF> Notes:
#<DEF>   The tkLB(HTML_DISPLAY) is text widget that uses
#<DEF>   the -wrap option.
#<DEF>

proc tkLBDisplayBuffer { p_html_file } {
    global tkLB

    # Clear out counters and tags.
    set p_level 0; set p_ol_cntr 0; set p_ul_label_cntr 0
    set p_img_image_cntr 0; set p_nesting 0; set p_hr_cntr 0
    set p_tag ""; set p_upper_tag ""; set p_pr_tag ""
    set p_padding ""; set p_tab "        "

    # Define the HTML display window.
    set w $tkLB(HTML_DISPLAY)

    # Reconfigure the text widget to display text, clear it out,
    #   then insert some padding at the top.
    $w configure -state normal; $w delete 0.0 end; $w insert end \n

    # Configure the HTML tags.
    tkLBConfigureTags

    # Strip off the file:/ in p_open_file.
    regsub {^[Ff][Ii][Ll][Ee]:/} $p_html_file {} p_html_file

    # If HTML file is not in URL list, add it.
    set p_url_index [lsearch $tkLB(URL_LIST) $p_html_file]
    if {$p_url_index < 1} {
	incr tkLB(URL_INDEX) +1
	append tkLB(URL_LIST) "$tkLB(URL_INDEX) $p_html_file "
    } else {
	# Account for Tcl list starting with 0, versus URL list
	#   starting with 1.
	set tkLB(URL_INDEX) [expr ($p_url_index + 1)/2]
    }

    # Load the file contents into the buffer.
    if [string length $p_html_file] {
	if [catch {set p_fileID [open $p_html_file r]} p_error] {
	    return
	} else {
	    set p_buffer [read -nonewline $p_fileID]
	}
    } 

    tkLBAnimate

    # To speed up parsing, there are NO comments in this while loop.
    while {[regexp -indices {<([^@>]*)>} $p_buffer mm p_idx] == 1} {
	set p_pr_tag [string toupper $p_tag]
        set p_start [lindex $p_idx 0]; set p_end [lindex $p_idx 1]
        set p_tag [string range $p_buffer $p_start $p_end]
	set p_upper_tag [string toupper $p_tag]
        set p_oldend [$w index end]
	set p_curr_text	[string range $p_buffer 0 [expr $p_start - 2]]
	if {$p_upper_tag != "/PRE"} {
	    set p_curr_text [tkLBRemoveWS $p_curr_text]
	}
        if [regexp {^(UL|OL|DL)} $p_upper_tag] {
	    incr p_level
	    set p_level_tag($p_level) $p_upper_tag
	    if {$p_level == 1} {
		if {$p_upper_tag != "DL"} {
		    append p_curr_text \n\n
		}
	    }
	} elseif [regexp {(/UL|/OL|/DL)} $p_upper_tag] {
	    incr p_level -1
	    if {$p_upper_tag == "/OL"} {set p_ol_cntr 0}
	}
	if {$p_upper_tag == "LI"} {
	    set p_padding [LeftPad "" [expr $p_level * 5]]
	    if {$p_level_tag($p_level) == "OL"} {
		incr p_ol_cntr
		$w insert end "$p_padding$p_ol_cntr. "
	    } elseif {$p_level_tag($p_level) == "UL"} {
		incr p_ul_label_cntr
		switch -exact $p_level {
		    1       {set p_curr_img $tkLB(IMAGE_FILLDOT)}
		    2       {set p_curr_img $tkLB(IMAGE_OPENDOT)}
		    3       {set p_curr_img $tkLB(IMAGE_OPENSQR)}
		    4       {set p_curr_img $tkLB(IMAGE_OPENDMD)}
		    5       {set p_curr_img $tkLB(IMAGE_FILLSQR)}
		    6       {set p_curr_img $tkLB(IMAGE_FILLDMD)}
		    default {set p_curr_img $tkLB(IMAGE_FILLDMD)}
		}
		$w insert end $p_padding
		set p_ul_widget [label $w.ul$p_ul_label_cntr \
			-image $p_curr_img]
		$w window create end -window $p_ul_widget
	    }
	    set p_padding ""
	}
	if {$p_upper_tag == "DT"} {
	    set p_curr_text [LeftPad "" [expr ($p_level - 1) * 5]]
	} elseif {$p_upper_tag == "DD"} {
	    set p_curr_text [LeftPad "" [expr $p_level * 5]]
	}
	if [regexp {(/LI|/DT|/DD)} $p_upper_tag] {
	    append p_curr_text \n
	}
	if {[lsearch $tkLB(NOPRINT_LIST) $p_upper_tag] == -1} {
	    $w insert end $p_curr_text
	}
	tkLBPurgeTag $w $p_oldend insert
	switch -regexp $p_upper_tag {
	    H[1-6]+ {$w insert end \n BR}
	    BR      {$w insert end \n BR}
	}
	if {[string range $p_upper_tag 0 1] == "HR"} {
	    incr p_hr_cntr
	    set p_hr_width [expr 7*[$w cget -width]]
	    set p_hr_widget [frame $w.$p_hr_cntr -width $p_hr_width \
		    -height 4 -borderwidth 10 -relief sunken]
	    $w window create end -window $p_hr_widget
	}
	if {$p_upper_tag == "P"} {
	    $w insert end \n BR
	    if ![regexp {(H[1-6]+|TITLE)} $p_pr_tag] {
		$w insert end \n BR
	    }		
	}
	if {[string range $p_upper_tag 0 0] == "A"} {
	    set p_tag_start [$w index insert]
	    foreach p_arg [split $p_tag] {
		if [regexp -nocase {^name} $p_arg] {
		    set p_name_tag "NAME.[lindex [split $p_arg \"] 1]"
		} elseif [regexp -nocase {href} $p_arg] {
		    set p_href_tag "HREF.[lindex [split $p_arg \"] 1]"
		    $w tag bind $p_href_tag <Enter> \
			    {%W configure -cursor hand2}
		    $w tag bind $p_href_tag <Leave> \
			    {%W configure -cursor left_ptr}
		}
	    }
	    set p_upper_tag A
	} elseif {[string range $p_upper_tag 0 1] =="/A"} {
	    if [info exists p_name_tag] {
		$w tag add $p_name_tag $p_tag_start insert
		set p_href_tag "HREF.#[lindex [split $p_name_tag .] 1]"
		$w tag bind $p_href_tag <Button-1> \
			[list $w yview $p_tag_start]
		$w tag bind $p_href_tag <ButtonRelease> \
			[list $w tag configure $p_href_tag \
			-foreground $tkLB(HREF_SELECT_COLOR)]
		unset p_name_tag; unset p_href_tag
	    } elseif [info exists p_href_tag] {
		$w tag add $p_href_tag $p_tag_start insert
		$w tag configure $p_href_tag -underline 1 \
			-foreground $tkLB(HREF_FG_COLOR)
		regsub {HREF\.} $p_href_tag {} p_href_file
		if [regsub {^[Ff][Ii][Ll][Ee]:/} \
			$p_href_file {} p_href_file] {
		    $w tag bind $p_href_tag <Button-1> \
			    [list tkLBDisplayBuffer $p_href_file]
		}
		unset p_href_tag; unset p_href_file
	    }
	}
	if {[string range $p_upper_tag 0 2] == "IMG"} {
	    incr p_img_image_cntr
	    foreach p_attr [split $p_tag " "] {
		if [regexp -nocase {^src} $p_attr] {
		    set p_img_file_name [lindex [split $p_attr \"] 1]
		    set p_curr_img [image create photo \
			    -file $p_img_file_name]
		    set p_img_widget [label $w.img$p_img_image_cntr \
			    -image $p_curr_img]
		    $w window create end -window $p_img_widget
		}
		if [regexp -nocase {^border} $p_attr] {
		    set p_border [lindex [split $p_attr \"] 1]
		    $p_img_widget configure -borderwidth $p_border
		}
	    }
	    set p_upper_tag IMG
	    set p_image_index [expr [$w index insert] - 1]
	}
	if {[string range $p_upper_tag 0 3] == "BODY"} {
	    foreach p_arg [split $p_upper_tag] {
		if [string match BGCOLOR* $p_arg] {
		    $w configure -bg [lindex [split $p_arg \"] 1]
		} elseif [string match FGCOLOR* $p_arg] {
		    $w configure -fg [lindex [split $p_arg \"] 1]
		} elseif [string match FONT* $p_arg] {
		    $w configure -font [lindex [split $p_arg \"] 1]
		}
	    }
	    set p_upper_tag BODY
	}
	if {[lsearch $tkLB(SINGLE_TAG) $p_upper_tag] == -1} {
	    if {[string range $p_tag 0 0] == "/"} {
		set p_tag [string trimleft $p_upper_tag "/"]
		if ![info exists p_text_index($p_tag,$p_nesting)] {
		    bell; catch {destroy .error}; toplevel .error
		    label .error.msg -text "Missing Begin Tag"
		    button .error.ok -text Ok -command {destroy .error}
		    pack .error.msg .error.ok -fill x
		    return
		}
		$w tag add $p_tag $p_text_index($p_tag,$p_nesting) insert
		if {$p_tag == "CENTER"} {
		    if [info exists p_image_index] {
			$w tag add $p_tag $p_image_index insert
			unset p_image_index
		    }
		}
		unset p_text_index($p_tag,$p_nesting)
		incr p_nesting -1
	    } else {
		incr p_nesting
		if [info exists p_text_index($p_upper_tag,$p_nesting)] {
		    bell; catch {destroy .error}; toplevel .error
		    label .error.msg -text "Missing Ending Tag"
		    button .error.ok -text Ok -command {destroy .error}
		    pack .error.msg .error.ok -fill x
		    return

		}
		set p_text_index($p_upper_tag,$p_nesting) [$w index insert]
	    }
	} 	    
	set p_buffer [string range $p_buffer [expr $p_end + 2] end]
    }

    # Save the old ending in the text widget.
    set p_oldend [$w index end]

    # Display the current buffer and remove the tags from it.
    $w insert end $p_buffer
    tkLBPurgeTag $w $p_oldend insert

    tkLBAnimate

    # Set the state to disabled so the user cannot edit the buffer.
    $w configure -state disabled
}    

#<DEF> ---------------------------------------------------------------
#<DEF> tkLBConfigureTags
#<DEF> 
#<DEF> Description:
#<DEF>   Configure the HTML tags.
#<DEF>

proc tkLBConfigureTags { } {
    global tkLB

    set w $tkLB(HTML_DISPLAY)

    $w tag configure TITLE -font $tkLB(FONT_TITLE)
    $w tag configure H1 -font $tkLB(FONT_H1)
    $w tag configure H2 -font $tkLB(FONT_H2)
    $w tag configure H3 -font $tkLB(FONT_H3)
    $w tag configure H4 -font $tkLB(FONT_H4)
    $w tag configure H5 -font $tkLB(FONT_H5) 
    $w tag configure H6 -font $tkLB(FONT_H6)
    $w tag configure BR -font $tkLB(FONT_BR)
    $w tag configure EM -font $tkLB(FONT_EM)
    $w tag configure CODE -font $tkLB(FONT_CODE)
    $w tag configure SAMP -font $tkLB(FONT_SAMP)
    $w tag configure KBD -font $tkLB(FONT_KBD)
    $w tag configure VAR -font $tkLB(FONT_VAR)
    $w tag configure CITE -font $tkLB(FONT_CITE)    
    $w tag configure STRONG -font $tkLB(FONT_STRONG)
    $w tag configure B -font $tkLB(FONT_B)
    $w tag configure I -font $tkLB(FONT_I) 
    $w tag configure TT -font $tkLB(FONT_IT)
    $w tag configure PRE -font $tkLB(FONT_PRE)

    $w tag configure U -underline 1
    $w tag configure LEFT -justify left
    $w tag configure CENTER -justify center
    $w tag configure RIGHT -justify right
}

#<DEF> ---------------------------------------------------------------
#<DEF> tkLBIncrTag
#<DEF> 
#<DEF> Description:
#<DEF>   Increment the second digit in the tag index.
#<DEF>
#<DEF> Parameters
#<DEF>   p_tag  - Index to encrement.
#<DEF>   p_side - Increment L(left) or (R)right side of the tag.
#<DEF>   p_incr - Amount to increment the index by.
#<DEF>
#<DEF> Example
#<DEF>   set p_tag 12.45
#<DEF>   set p_incr -1
#<DEF>   return 12.44

proc tkLBIncrTag { p_tag p_side p_incr } {
    set p_tag_split [split $p_tag .]
    set p_tag_L [lindex $p_tag_split 0]
    set p_tag_R [lindex $p_tag_split 1]
    if {[string toupper $p_side] == "L"} {
	set p_tag_L [expr $p_tag_L + $p_incr]
    } else {
	set p_tag_R [expr $p_tag_R + $p_incr]
    }
    return "$p_tag_L.$p_tag_R"
}

#<DEF> ---------------------------------------------------------------
#<DEF> tkLBPurgeTag
#<DEF> 
#<DEF> Description:
#<DEF>   Remove all defined tags for a given widget.
#<DEF>
#<DEF> Parameters:
#<DEF>   p_widget - Widget to remove tags from.
#<DEF>   p_start  - Starting tag.
#<DEF>   p_end    - Ending tag.
#<DEF>

proc tkLBPurgeTag {p_widget p_start p_end} {
    foreach p_tag [$p_widget tag names $p_start] {
        $p_widget tag remove $p_tag $p_start $p_end
    }
}

#<DEF> ---------------------------------------------------------------
#<DEF> tkLBRemoveWS
#<DEF> 
#<DEF> Description:
#<DEF>   Substitute special characters and eliminate extra whitespace
#<DEF> in text field.
#<DEF>
#<DEF> Parameters: 
#<DEF>   p_text - Text to perform substitution on.
#<DEF>

proc tkLBRemoveWS { p_text } {
    set p_return_text ""
    set p_ampersand "\\&"

    # Substitute special characters < > & and ".
    regsub -all {&lt} $p_text {<} p_text
    regsub -all {&gt} $p_text {>} p_text
    regsub -all {&amp} $p_text $p_ampersand p_text
    regsub -all {&quot} $p_text {"} p_text
    
    # Eliminate extra whitespace.
    regsub -all "\a" $p_text " " p_text
    regsub -all "\b" $p_text " " p_text
    regsub -all "\f" $p_text " " p_text
    regsub -all "\n" $p_text " " p_text
    regsub -all "\r" $p_text " " p_text
    regsub -all "\t" $p_text " " p_text
    regsub -all "\v" $p_text " " p_text
    regsub -all " +" $p_text " " p_text

    # Trim off extra space on the left side and add it to the right
    #   if it is not already there.
    set p_text [string trimleft $p_text " "]

    if ![regexp " $" $p_text] {append p_text " "}

    return $p_text
}

#<DEF>----------------------------------------------------------------
#<DEF> tkLBAnimate
#<DEF> 
#<DEF> Description:
#<DEF>   Animate file loading while the Browser parses the HTML file
#<DEF>   contents and buffer.
#<DEF> 
#<DEF> Parameters:
#<DEF>   w - set by tkLB(ANIMATE_WIDGET) which stores the label widget
#<DEF>       to use.
#<DEF>
#<DEF> Globals:
#<DEF>   tkLB(IMAGE_ANIMATE_i) 
#<DEF>        - where i = 1,2,3,.. these are the animation GIF files.
#<DEF>

proc tkLBAnimate { } {
    global tkLB

    # Define the widget to configure the images in.
    set w $tkLB(ANIMATE_WIDGET)

    # Go through the 5 animation pictures, pausing after each display.
    foreach i {1 2 3 4 5} {
	$w configure -image $tkLB(IMAGE_ANIMATE_$i)
	update idletasks
	after 200
    }

    # Reset the widget to display the icon.
    $w configure -image $tkLB(IMAGE_TKLB)
}

# Define where the include directory lives.
set tkLB(INCLUDE_DIR) "include"

source "$tkLB(INCLUDE_DIR)/update_index.tcl"
source "$tkLB(INCLUDE_DIR)/tkLB_global.tcl"
source "$tkLB(INCLUDE_DIR)/tkLB_xresources.tcl"

lappend auto_path $tkLB(INCLUDE_DIR)
LibraryUpdateIndex $tkLB(INCLUDE_DIR)

# Parse the command line arguments.
if {$argc == 1} {
    set tkLB(HTML_FILE) [lindex $argv 0]
    regsub {^[Ff][Ii][Ll][Ee]:/} $tkLB(HTML_FILE) {} tkLB(HTML_FILE)
    set tkLB(URL_INDEX) 1
    set tkLB(URL_LIST) "1 $tkLB(HTML_FILE) "   ;#extra space at end.
} else {
    puts stderr "Usage: tkLB file:/htmlfile"
    exit 1
}

tkLBCreateWindow
