#!/bin/sh
# The following line is a comment in Tcl 
# but is visible to the Bourne shell \
        exec wish "$0" ${1+"$@"}

#<DEF>----------------------------------------------------------------
#<DEF> tkREMAddMetaFrame
#<DEF> 
#<DEF> Description:
#<DEF>   Add a frame for meta-characters.
#<DEF>
#<DEF> Parameters:
#<DEF>   args - A list of frames to create and add to the display.
#<DEF>
#<DEF> Globals:
#<DEF>   tkREM(ADD_META_FRAME_ROW)
#<DEF>         - Current row in the FRAME_MODE grid to put the child
#<DEF>           meta-frame.
#<DEF>   tkREM(ADD_META_FRAME_COL)
#<DEF>         - Current col in the FRAME_MODE grid to put the child
#<DEF>           meta-frame.
#<DEF>
#<DEF>   tkREM(LABEL_WIZARD)
#<DEF>         - Display the default meta-frame Wizard help message
#<DEF>           in this label.
#<DEF>
#<DEF> Notes:
#<DEF>   This procedure creates a grid of meta-frames in FRAME_MODE
#<DEF>   The Grid is 2 columns wide, and can be of any depth.  Each
#<DEF>   meta-frame is a grid that is 4 buttons wide, any depth, and
#<DEF>   surrounded by a groove with a label name.
#<DEF>

proc tkREMAddMetaFrame { args } {
    global tkREM

    foreach p_frame $args {
	# Build the physical frame.
	frame $p_frame -relief groove -borderwidth 2

	# Construct the capitolized frame name.
	set p_text [lindex [split $p_frame "."] end]
	set p_text "[string toupper [string range $p_text 0 0]][string \
		range $p_text 1 end]"
	set p_label [label $p_frame.l1 -text $p_text]

	# Place the label outside the frame relative to the upper
	#   left corner.
	place $p_label -x 5 -y -4 -bordermode outside

	# Build the current row and column.
	if ![info exists tkREM(ADD_META_FRAME_ROW)] {
	    set tkREM(ADD_META_FRAME_ROW) 1
	}
	if [info exists tkREM(ADD_META_FRAME_COL)] {
	    incr tkREM(ADD_META_FRAME_COL) +1
	} else {
	    set tkREM(ADD_META_FRAME_COL) 1
	}

	# Divide grid into 2xanydepth.
	if {$tkREM(ADD_META_FRAME_COL) == "3"} {
	    incr tkREM(ADD_META_FRAME_ROW) +1
	    set tkREM(ADD_META_FRAME_COL) 1
	}

	# Put that grid in there baby!
	grid $p_frame -row $tkREM(ADD_META_FRAME_ROW) \
		-column $tkREM(ADD_META_FRAME_COL) \
		-ipadx 5 -ipady 10 -padx 2 -pady 2 -sticky nw

	# Create some default Wizard help.
	bind $p_frame <Enter> [list $tkREM(LABEL_WIZARD) \
		configure -text "$p_text Meta-Frame."]
    }
}

#<DEF>----------------------------------------------------------------
#<DEF> tkREMAddMetaButton
#<DEF>
#<DEF> Description
#<DEF>   Add a button to a meta-frame.
#<DEF>
#<DEF> Parameters:
#<DEF>   p_frame  - Parent meta-frame.
#<DEF>   p_label  - Button's label.
#<DEF>   p_meta   - Meta-characters to insert into the Regexp entry.
#<DEF>   p_wizard - Wizard help message to display when mouse is over
#<DEF>              the button.
#<DEF> Globals:
#<DEF>   tkREM(REGEXP_ENTRY) 
#<DEF>         - Regular Expression entry widget.
#<DEF>   tkREM(ADD_META_BUTTON_INDEX,meta-frame)
#<DEF>         - This index keeps track of the number of buttons in
#<DEF>           meta-frame.
#<DEF>   tkREM(ADD_META_BUTTON_ROW,meta-frame)
#<DEF>         - Row of the button in meta-frame.
#<DEF>   tkREM(ADD_META_BUTTON_COL,meta-frame)
#<DEF>         - Col of the button in meta-frame.
#<DEF>
#<DEF> Notes:
#<DEF>   Each meta-frame has a grid of button.  The globals listed
#<DEF>   keep track of the index, row, and col for each meta-frame.
#<DEF>   The grid is 4 buttons wide and can be any depth.
#<DEF> 

proc tkREMAddMetaButton { p_frame p_label p_meta p_wizard } {
    global tkREM

    # The index for each frame creates a unique button.
    if [info exists tkREM(ADD_META_BUTTON_INDEX,$p_frame)] {
	incr tkREM(ADD_META_BUTTON_INDEX,$p_frame) +1
    } else {
	set tkREM(ADD_META_BUTTON_INDEX,$p_frame) 1
    }

    # Create the command to insert the meta characters in the
    #   Regexp entry
    set p_cmd [list $tkREM(ENTRY_REGEXP) insert insert $p_meta]

    # Create the unique button name with the unique frame index.
    set p_button "$p_frame.button$tkREM(ADD_META_BUTTON_INDEX,$p_frame)"

    # Build the button.
    button $p_button -text $p_label -width 4 -command $p_cmd

    # Create the Wizard help message.
    bind $p_button <Enter> [list $tkREM(LABEL_WIZARD) \
	    configure -text $p_wizard]
    
    # Check the row for the frame exists.
    if ![info exists tkREM(ADD_META_BUTTON_ROW,$p_frame)] {
	set tkREM(ADD_META_BUTTON_ROW,$p_frame) 1
    }

    # Build the current column.
    if [info exists tkREM(ADD_META_BUTTON_COL,$p_frame)] {
	incr tkREM(ADD_META_BUTTON_COL,$p_frame) +1
    } else {
	set tkREM(ADD_META_BUTTON_COL,$p_frame) 1
    }
    
    # Start a new row after seven button (counter at buttons+1).
    if {$tkREM(ADD_META_BUTTON_COL,$p_frame) == 5} {
	incr tkREM(ADD_META_BUTTON_ROW,$p_frame) +1
	set tkREM(ADD_META_BUTTON_COL,$p_frame) 1
    }
    
    # Pack that button up.
    grid $p_button -row $tkREM(ADD_META_BUTTON_ROW,$p_frame) \
	    -column $tkREM(ADD_META_BUTTON_COL,$p_frame)
}

#<DEF>----------------------------------------------------------------
#<DEF> tkREMMetaButtonReset
#<DEF> 
#<DEF> Description
#<DEF>   Reset the index, row and column for each button in the
#<DEF>   specified meta-frame.
#<DEF>
#<DEF> Parameters:
#<DEF>   args - Tcl default list of all the meta-frames in FRAME_MODE
#<DEF>          to reset the index, row, and col.
#<DEF>
#<DEF> Globals:
#<DEF>   tkREM(ADD_META_BUTTON_INDEX,meta-frame)
#<DEF>         - This index keeps track of the number of buttons in
#<DEF>           meta-frame.
#<DEF>   tkREM(ADD_META_BUTTON_ROW,meta-frame)
#<DEF>         - Row of the button in meta-frame.
#<DEF>   tkREM(ADD_META_BUTTON_COL,meta-frame)
#<DEF>         - Col of the button in meta-frame.
#<DEF>

proc tkREMMetaButtonReset { args } {
    global tkREM

    foreach p_frame $args {
	if [info exists tkREM(ADD_META_BUTTON_INDEX,$p_frame)] {
	    unset tkREM(ADD_META_BUTTON_INDEX,$p_frame)
	}
	if [info exists tkREM(ADD_META_BUTTON_ROW,$p_frame)] {
	    unset tkREM(ADD_META_BUTTON_ROW,$p_frame)
	}
	if [info exists tkREM(ADD_META_BUTTON_COL,$p_frame)] {
	    unset tkREM(ADD_META_BUTTON_COL,$p_frame)
	}
    }
}

#<DEF>----------------------------------------------------------------
#<DEF> tkREMBuildMetaMode
#<DEF> 
#<DEF> Description
#<DEF>   Build the tkREM(MODE) interface, in tkREM(FRAME_MODE),
#<DEF>   depending on which button the user presses, and what mode the
#<DEF>   application is currently in.
#<DEF>
#<DEF> Parameters:
#<DEF>   w        - Parent frame.
#<DEF>   h        - Wizard help widget.
#<DEF>   p_mode   - Mode to build.
#<DEF>   p_wizard - Wizard help message.
#<DEF>
#<DEF> Globals:
#<DEF>   tkREM(BUTTON_MODE_button) 
#<DEF>         - Button for each mode defined in the Toolbar.
#<DEF>         - button = grep|perl|tcl|python|lex|awk.
#<DEF>   tkREM(MODE)
#<DEF>         - Current mode application is in.
#<DEF>   tkREM(FRAME_MODE)
#<DEF>         - Frame which is the parent to the smaller mode
#<DEF>           frames.
#<DEF>   tkREM(ADD_META_FRAME_ROW)
#<DEF>         - Current row in FRAME_MODE grid.
#<DEF>   tkREM(ADD_META_FRAME_COL)
#<DEF>         - Current col in FRAME_MODE grid.
#<DEF>

proc tkREMBuildMetaMode { w h p_mode p_wizard } {
    global tkREM

    # Uppercase the mode just in case the caller forgot to.
    set p_mode [string toupper $p_mode]

    # Set the meta-character buttons relief
    foreach b [array names tkREM] {
	if [regexp {^BUTTON_MODE} $b] {
	    if [regexp $p_mode $b] {
		set p_relief sunken
	    } else {
		set p_relief raised
	    }
	    $tkREM($b) configure -relief $p_relief
	}
    }
		
    # If already in selected mode, then return.  If not, then set to
    #   selected mode.
    if [info exists tkREM(MODE)] {
	if {$p_mode == $tkREM(MODE)} {
	    return
	} else {
	    set tkREM(MODE) $p_mode
	}
    } else {
	set tkREM(MODE) $p_mode
    }

    # Now that we are switching to a different mode, get rid of
    #   the last mode's interface meta-frames.
    foreach f [grid slaves $tkREM(FRAME_MODE)] {destroy $f}

    # Reset the FRAME_MODE grid.
    if [info exists tkREM(ADD_META_FRAME_ROW)] {
	unset tkREM(ADD_META_FRAME_ROW) 
    }
    if [info exists tkREM(ADD_META_FRAME_COL)] {
	unset tkREM(ADD_META_FRAME_COL)
    }
    
    # Build the appropriate mode
    switch $p_mode {
	GREP    {tkREMBuildGrepMode}
	PERL    {tkREMBuildPerlMode}
	TCL     {tkREMBuildTclMode}
	PYTHON  {tkREMBuildGrepMode}
        LEX     {tkREMBuildLexMode}
	AWK     {tkREMBuildAwkMode}
    }
}

#<DEF>----------------------------------------------------------------
#<DEF> tkREMBuildGrepMode
#<DEF> 
#<DEF> Description
#<DEF>   Build the interface for the GREP mode.
#<DEF>
#<DEF> Globals:
#<DEF>   tkREM(FRAME_MODE)
#<DEF>         - Frame which is the parent to the smaller mode
#<DEF>           meta-frames.
#<DEF>

proc tkREMBuildGrepMode { } {
    global tkREM

    set f $tkREM(FRAME_MODE)

    tkREMMetaButtonReset $f.classes $f.counting $f.anchors $f.other 

    # Build the meta-frames first
    tkREMAddMetaFrame $f.classes $f.counting $f.anchors $f.other

    # Build the classes buttons
    tkREMAddMetaButton $f.classes "." "." \
	    "Match anything except the newline character."
    tkREMAddMetaButton $f.classes "\[" "\[" \
	    "Start a character class."
    tkREMAddMetaButton $f.classes "\[^" "\[^" \
	    "Start a negated character class."
    tkREMAddMetaButton $f.classes "\]" "\]" \
	    "Close a character class."
    tkREMAddMetaButton $f.classes "-" "-" \
	    "Create a character range like a-h."
    tkREMAddMetaButton $f.classes "a-z" "a-z" \
	    "Match any lowercase letter."
    tkREMAddMetaButton $f.classes "A-Z" "A-Z" \
	    "Match any uppercase letter."
    tkREMAddMetaButton $f.classes "0-9" "0-9" \
	    "Match any digit."

    # Build the counting buttons.
    tkREMAddMetaButton $f.counting "." "." \
	    "Match anything except the newline character."
    tkREMAddMetaButton $f.counting "?" "?" \
	    "One match allowed, but its optional."
    tkREMAddMetaButton $f.counting "*" "*" \
	    "Zero or more matches allowed."
    tkREMAddMetaButton $f.counting "+" "+" \
	    "One match required, additional are optional."
    tkREMAddMetaButton $f.counting "\\\{" "\\\{" \
	    "Start Min required, Max allowed set - \{min,max\}."
    tkREMAddMetaButton $f.counting "," "," \
	    "Seperator for \{min,max\}"
    tkREMAddMetaButton $f.counting "\\\}" "\\\}" \
	    "End Min required, Max allowed set - \{min,max\}."

    # Build the anchors buttons.
    tkREMAddMetaButton $f.anchors "^" "^" \
	    "Insert start of line position marker."
    tkREMAddMetaButton $f.anchors "$" "$" \
	    "Insert end of line position marker."
    tkREMAddMetaButton $f.anchors "^$" "^$" \
	    "Match a blank line."
    tkREMAddMetaButton $f.anchors "^.*$" "^.*$" \
	    "Match an entire line."
    tkREMAddMetaButton $f.anchors "\\<" "\\<" \
	    "Start a word boundary."
    tkREMAddMetaButton $f.anchors "\\>" "\\>" \
	    "End a word boundary."

    # Build the other buttons
    tkREMAddMetaButton $f.other "(" "(" \
	    "Start a group or back reference."
    tkREMAddMetaButton $f.other ")" ")" \
	    "End a group or back reference."
    tkREMAddMetaButton $f.other "|" "|" \
	    "Alternate (or) - a|b|c matches either a or b or c."
    tkREMAddMetaButton $f.other "\\" "\\" \
	    "Used to escape special characters like \\n."
    tkREMAddMetaButton $f.other "\\1" "\\1" \
	    "Text previously matched with 1st of parenthesis."
    tkREMAddMetaButton $f.other "\\2" "\\2" \
	    "Text previously matched with 2nd of parenthesis."
    tkREMAddMetaButton $f.other "\\3" "\\3" \
	    "Text previously matched with 3rd of parenthesis."
}

#<DEF>----------------------------------------------------------------
#<DEF> tkREMBuildPerlMode
#<DEF> 
#<DEF> Description
#<DEF>   Build the interface for the PERL mode.
#<DEF>
#<DEF> Globals:
#<DEF>   tkREM(FRAME_MODE)
#<DEF>         - Frame which is the parent to the smaller mode
#<DEF>           meta-frames.
#<DEF>

proc tkREMBuildPerlMode { } {
    global tkREM

    set f $tkREM(FRAME_MODE)

    tkREMMetaButtonReset $f.grouping $f.quantifiers $f.anchors \
	    $f.modification $f.classes $f.variables $f.operator \
	    $f.shorthand

    # Build the meta-frames first
    tkREMAddMetaFrame $f.grouping $f.quantifiers $f.anchors \
	    $f.modification $f.classes $f.variables $f.operator \
	    $f.shorthand

    # Build the grouping frame.
    tkREMAddMetaButton $f.grouping "(" "(" \
	    "Start a group or back reference."
    tkREMAddMetaButton $f.grouping "(?:" "(?:" \
	    "Pure grouping only."
    tkREMAddMetaButton $f.grouping "(?=" "(?=" \
	    "Grouping with a positive lookahead."
    tkREMAddMetaButton $f.grouping "(?!" "(?!" \
	    "Grouping with a negative lookahead."
    tkREMAddMetaButton $f.grouping "(?i" "(?i" \
	    "Grouping with case-insensitive matching."
    tkREMAddMetaButton $f.grouping "(?m" "(?m" \
	    "Grouping with multi-line matching."
    tkREMAddMetaButton $f.grouping "(?s" "(?s" \
	    "Grouping with single line matching."
    tkREMAddMetaButton $f.grouping "(?x" "(?x" \
	    "Grouping with free formatting."
    tkREMAddMetaButton $f.grouping "(?#" "(?#" \
	    "Grouping comment."
    tkREMAddMetaButton $f.grouping "|" "|" \
	    "Alternate (or) - a|b|c matches either a or b or c."
    tkREMAddMetaButton $f.grouping ")" ")" \
	    "End a group or back reference."

    # Build the quantifiers frame.
    tkREMAddMetaButton $f.quantifiers "\{" "\{" \
	    "Start Min required, Max allowed set."
    tkREMAddMetaButton $f.quantifiers "," "," \
	    "Insert comma - \{min,max\} seperator."
    tkREMAddMetaButton $f.quantifiers "\}" "\}" \
	    "End greedy Min required, Max allowed set."
    tkREMAddMetaButton $f.quantifiers "\}?" "\}?" \
	    "End non-greedy Min required, Max allowed set."
    tkREMAddMetaButton $f.quantifiers "." "." \
	    "Match anything except the newline character."
    tkREMAddMetaButton $f.quantifiers "?" "?" \
	    "Greedy one match allowed, but its optional."
    tkREMAddMetaButton $f.quantifiers "*" "*" \
	    "Greedy zero or more matches allowed."
    tkREMAddMetaButton $f.quantifiers "+" "+" \
	    "Greedy one match required, additional are optional."
    tkREMAddMetaButton $f.quantifiers "??" "??" \
	    "Non-greedy one match allowed, but its optional."
    tkREMAddMetaButton $f.quantifiers "*?" "*?" \
	    "Non-greedy zero or more matches allowed."
    tkREMAddMetaButton $f.quantifiers "+?" "+?" \
	    "Non-greedy one match required, additional are optional."

    # Build the anchors frame.
    tkREMAddMetaButton $f.anchors "^" "^" \
	    "Insert start of line anchor."
    tkREMAddMetaButton $f.anchors "\\b" "\\b" \
	    "Insert a word anchor."
    tkREMAddMetaButton $f.anchors "\\B" "\\B" \
	    "Insert a non-word anchor."
    tkREMAddMetaButton $f.anchors "\\A" "\\A" \
	    "Insert a start of string anchor."
    tkREMAddMetaButton $f.anchors "$" "$" \
	    "Insert end of line anchor."
    tkREMAddMetaButton $f.anchors "\\Z" "\\Z" \
	    "Insert an end of string anchor."
    tkREMAddMetaButton $f.anchors "\\G" "\\G" \
	    "Insert an end of previous match anchor."

    # Build the on the fly text modification frame.
    tkREMAddMetaButton $f.modification "\\l" "\\l" \
	    "Lower the case of the next character."
    tkREMAddMetaButton $f.modification "\\u" "\\u" \
	    "Raise the case of the next character."
    tkREMAddMetaButton $f.modification "\\L" "\\L" \
	    "Lower the case of text until optional \\E."
    tkREMAddMetaButton $f.modification "\\U" "\\U" \
	    "Raise the case of text until optional \\E."
    tkREMAddMetaButton $f.modification "\\Q" "\\Q" \
	    "Add an escape before all non-alpha until \\E."
    tkREMAddMetaButton $f.modification "\\u\\L" "\\u\\L" \
	    "Capitolize first character, lower the rest until \\E."
    tkREMAddMetaButton $f.modification "\\l\\U" "\\l\\U" \
	    "Lowercase first character, raise the rest until \\E."
    tkREMAddMetaButton $f.modification "\\E" "\\E" \
	    "Insert an end mark for text modification."

    # Build the character classes frame.
    tkREMAddMetaButton $f.classes "." "." \
	    "Match anything except the newline character."
    tkREMAddMetaButton $f.classes "\[" "\[" \
	    "Start a character class."
    tkREMAddMetaButton $f.classes "\[^" "\[^" \
	    "Start a negated character class."
    tkREMAddMetaButton $f.classes "\]" "\]" \
	    "Close a character class."
    tkREMAddMetaButton $f.classes "-" "-" \
	    "Create a character range like a-h."
    tkREMAddMetaButton $f.classes "a-z" "a-z" \
	    "Lowercase letter group."
    tkREMAddMetaButton $f.classes "A-Z" "A-Z" \
	    "Uppercase letter group."
    tkREMAddMetaButton $f.classes "0-9" "0-9" \
	    "Match any digit group."

    # Build the variables frame.
    tkREMAddMetaButton $f.variables "\$_" "\$_" \
	    "Insert the default search target."
    tkREMAddMetaButton $f.variables "\$1" "\$1" \
	    "Variable which stores the 1st matched text."
    tkREMAddMetaButton $f.variables "\$2" "\$2" \
	    "Variable which stores the 2nd matched text."
    tkREMAddMetaButton $f.variables "\$3" "\$3" \
	    "Variable which stores the 3rd matched text."
    tkREMAddMetaButton $f.variables "\$+" "\$+" \
	    "Highest filled of \$1, \$2, ..."
    tkREMAddMetaButton $f.variables "\$`" "\$`" \
	    "Text before the match."
    tkREMAddMetaButton $f.variables "\$&" "\$&" \
	    "Text of the match."
    tkREMAddMetaButton $f.variables "\$'" "\$'" \
	    "Text after the match."

    #Build the operator frame.
    tkREMAddMetaButton $f.operator "=~" "=~" \
	    "Insert a target of regular expression operator."
    tkREMAddMetaButton $f.operator "!~" "!~" \
	    "Insert a target of regular expression operator."
    tkREMAddMetaButton $f.operator "m/" "m/" \
	    "Insert the match regular expression operator."
    tkREMAddMetaButton $f.operator "s/" "s/" \
	    "Insert the substitute regular expression operator."
    tkREMAddMetaButton $f.operator "/" "/" \
	    "Used with m/ and s/, or anything you need it for."
    tkREMAddMetaButton $f.operator "split(" "split(" \
	    "Split regular expression operator."
    tkREMAddMetaButton $f.operator ")" ")" \
	    "Close up the split operator."
    tkREMAddMetaButton $f.operator "/x" "/x" \
	    "Remove comments and whitespace from regex."
    tkREMAddMetaButton $f.operator "/o" "/o" \
	    "Compile regex just once."
    tkREMAddMetaButton $f.operator "/s" "/s" \
	    "Single line mode for ^ and $."
    tkREMAddMetaButton $f.operator "/m" "/m" \
	    "Multi-line match mode."
    tkREMAddMetaButton $f.operator "/i" "/i" \
	    "Perform a case insensitive match."
    tkREMAddMetaButton $f.operator "/g" "/g" \
	    "Modify the starting position of the match."
    tkREMAddMetaButton $f.operator "/e" "/e" \
	    "Equivalent to placing eval{..} on substitution s/ / /."
    tkREMAddMetaButton $f.operator "#" "#" \
	    "Insert a comment marker."

    # Build the shorthand frame.
    tkREMAddMetaButton $f.shorthand "\\b" "\\b" \
	    "Match the backspace character."
    tkREMAddMetaButton $f.shorthand "\\t" "\\t" \
	    "Match the tab character."
    tkREMAddMetaButton $f.shorthand "\\n" "\\n" \
	    "Match the newline character."
    tkREMAddMetaButton $f.shorthand "\\r" "\\r" \
	    "Match the carriage return."
    tkREMAddMetaButton $f.shorthand "\\f" "\\f" \
	    "Match the formfeed character."
    tkREMAddMetaButton $f.shorthand "\\a" "\\a" \
	    "Match the alarm (bell) character."
    tkREMAddMetaButton $f.shorthand "\\e" "\\e" \
	    "Match the escape sequence character."
    tkREMAddMetaButton $f.shorthand "\\octal" "\\" \
	    "Match character specified by the octal number."
    tkREMAddMetaButton $f.shorthand "\\x hex" "\\x" \
	    "Match character specified by the hex number."
    tkREMAddMetaButton $f.shorthand "\\c char" "\\c" \
	    "Match the control character specified by char."
    tkREMAddMetaButton $f.shorthand "\\w" "\\w" \
	    "Shorthand for a word - \[a-zA-Z0-9_\]."
    tkREMAddMetaButton $f.shorthand "\\W" "\\W" \
	    "Inverse of \\w."
    tkREMAddMetaButton $f.shorthand "\\s" "\\s" \
	    "Shorthand for whitespace - \[ \\f\\n\\r\\t\]."
    tkREMAddMetaButton $f.shorthand "\\S" "\\S" \
	    "Inverse of \\s."
    tkREMAddMetaButton $f.shorthand "\\d" "\\d" \
	    "Shorthand for digits - \[0-9\]."
    tkREMAddMetaButton $f.shorthand "\\D" "\\D" \
	    "Inverse of \\d."
}

#<DEF>----------------------------------------------------------------
#<DEF> tkREMBuildTclMode
#<DEF> 
#<DEF> Description
#<DEF>   Build the interface for the TCL mode.
#<DEF>
#<DEF> Globals:
#<DEF>   tkREM(FRAME_MODE)
#<DEF>         - Frame which is the parent to the smaller mode
#<DEF>           meta-frames.
#<DEF>

proc tkREMBuildTclMode { } {
    global tkREM

    set f $tkREM(FRAME_MODE)

    tkREMMetaButtonReset $f.commands $f.classes $f.counting \
	    $f.anchors $f.grouping $f.other

    # Build the meta-frames first
    tkREMAddMetaFrame $f.commands $f.classes $f.counting \
	    $f.anchors $f.grouping $f.other

    # Build the commands frame.
    tkREMAddMetaButton $f.commands "regexp" "regexp" \
	    "regexp ?flags? pattern string ?match sub1 sub2...?"
    tkREMAddMetaButton $f.commands "nocase" " -nocase" \
	    "regexp or regsub - ignore case."
    tkREMAddMetaButton $f.commands "indxs" " -indeces" \
	    "regexp - index pair for matching \[sub\]string."
    tkREMAddMetaButton $f.commands "--" " --" \
	    "regexp or regsub - seperate flags from pattern."
    tkREMAddMetaButton $f.commands "regsub" "regsub" \
	    "regsub ?switches? pattern string subspec varname"
    tkREMAddMetaButton $f.commands "-all" " -all" \
	    "regsub - replace all occurences."

    # Build the classes buttons
    tkREMAddMetaButton $f.classes "." "." \
	    "Match anything except the newline character."
    tkREMAddMetaButton $f.classes "\[" "\[" \
	    "Start a character class."
    tkREMAddMetaButton $f.classes "\[^" "\[^" \
	    "Start a negated character class."
    tkREMAddMetaButton $f.classes "\]" "\]" \
	    "Close a character class."
    tkREMAddMetaButton $f.classes "-" "-" \
	    "Create a character range like a-h."
    tkREMAddMetaButton $f.classes "a-z" "a-z" \
	    "Match any lowercase letter."
    tkREMAddMetaButton $f.classes "A-Z" "A-Z" \
	    "Match any uppercase letter."
    tkREMAddMetaButton $f.classes "0-9" "0-9" \
	    "Match any digit."

    # Build the counting buttons.
    tkREMAddMetaButton $f.counting "." "." \
	    "Match anything except the newline character."
    tkREMAddMetaButton $f.counting "?" "?" \
	    "One match allowed, but its optional."
    tkREMAddMetaButton $f.counting "*" "*" \
	    "Zero or more matches allowed."
    tkREMAddMetaButton $f.counting "+" "+" \
	    "One match required, additional are optional."

    # Build the position buttons.
    tkREMAddMetaButton $f.anchors "^" "^" \
	    "Insert start of line position marker."
    tkREMAddMetaButton $f.anchors "$" "$" \
	    "Insert end of line position marker."
    tkREMAddMetaButton $f.anchors "^$" "^$" \
	    "Match a blank line."
    tkREMAddMetaButton $f.anchors "^.*$" "^.*$" \
	    "Match an entire line."

    # Build the grouping frame.
    tkREMAddMetaButton $f.grouping "(" "(" \
	    "Start a group or back reference."
    tkREMAddMetaButton $f.grouping ")" ")" \
	    "End a group or back reference."
    tkREMAddMetaButton $f.grouping "\\(" "\\(" \
	    "Start a subpattern."
    tkREMAddMetaButton $f.grouping "\\)" "\\)" \
	    "End a sbupattern."
    tkREMAddMetaButton $f.grouping "\{" "\{" \
	    "Start protecting regexp from being interpreted."
    tkREMAddMetaButton $f.grouping "\}" "\}" \
	    "End protecting regexp from being interpreted."

    #Build the other frame.
    tkREMAddMetaButton $f.other "&" "&" \
	    "regsub - Replaced with matched pattern."
    tkREMAddMetaButton $f.other "\\1" "\\1" \
	    "Replaced with 1st subpattern in pattern."
    tkREMAddMetaButton $f.other "\\2" "\\2" \
	    "Replaced with 2nd subpattern in pattern."
    tkREMAddMetaButton $f.other "\\3" "\\3" \
	    "Replaced with 3rd subpattern in pattern."
    tkREMAddMetaButton $f.other "\\" "\\" \
	    "Used to escape special characters like \\n."
    tkREMAddMetaButton $f.other "|" "|" \
	    "Alternate (or) - a|b|c matches either a or b or c."
}

#<DEF>----------------------------------------------------------------
#<DEF> tkREMBuildLexMode
#<DEF> 
#<DEF> Description
#<DEF>   Build the interface for the LEX mode.
#<DEF>
#<DEF> Globals:
#<DEF>   tkREM(FRAME_MODE)
#<DEF>         - Frame which is the parent to the smaller mode
#<DEF>           meta-frames.
#<DEF>

proc tkREMBuildLexMode { } {
    global tkREM

    set f $tkREM(FRAME_MODE)

    tkREMMetaButtonReset $f.classes $f.counting $f.anchors $f.other 

    # Build the meta-frames first
    tkREMAddMetaFrame $f.classes $f.counting $f.anchors $f.other

    # Build the classes buttons
    tkREMAddMetaButton $f.classes "." "." \
	    "Match anything except the newline character."
    tkREMAddMetaButton $f.classes "\[" "\[" \
	    "Start a character class."
    tkREMAddMetaButton $f.classes "\[^" "\[^" \
	    "Start a negated character class."
    tkREMAddMetaButton $f.classes "\]" "\]" \
	    "Close a character class."
    tkREMAddMetaButton $f.classes "-" "-" \
	    "Create a character range like a-h."
    tkREMAddMetaButton $f.classes "a-z" "a-z" \
	    "Match any lowercase letter."
    tkREMAddMetaButton $f.classes "A-Z" "A-Z" \
	    "Match any uppercase letter."
    tkREMAddMetaButton $f.classes "0-9" "0-9" \
	    "Match any digit."

    # Build the counting buttons.
    tkREMAddMetaButton $f.counting "." "." \
	    "Match anything except the newline character."
    tkREMAddMetaButton $f.counting "?" "?" \
	    "One match allowed, but its optional."
    tkREMAddMetaButton $f.counting "*" "*" \
	    "Zero or more matches allowed."
    tkREMAddMetaButton $f.counting "+" "+" \
	    "One match required, additional are optional."
    tkREMAddMetaButton $f.counting "\{" "\{" \
	    "Start Min required, Max allowed set - \{min,max\}."
    tkREMAddMetaButton $f.counting "," "," \
	    "Seperator for \{min,max\}"
    tkREMAddMetaButton $f.counting "\}" "\}" \
	    "End Min required, Max allowed set - \{min,max\}."
    tkREMAddMetaButton $f.counting "/" "/" \
	    "Match preceding regexp - 0/1 matches 0 if followed by 1."

    # Build the anchors buttons.
    tkREMAddMetaButton $f.anchors "^" "^" \
	    "Insert start of line position marker."
    tkREMAddMetaButton $f.anchors "$" "$" \
	    "Insert end of line position marker."
    tkREMAddMetaButton $f.anchors "^$" "^$" \
	    "Match a blank line."
    tkREMAddMetaButton $f.anchors "^.*$" "^.*$" \
	    "Match an entire line."

    # Build the other buttons
    tkREMAddMetaButton $f.other "(" "(" \
	    "Start a group or back reference."
    tkREMAddMetaButton $f.other ")" ")" \
	    "End a group or back reference."
    tkREMAddMetaButton $f.other "|" "|" \
	    "Alternate (or) - a|b|c matches either a or b or c."
    tkREMAddMetaButton $f.other "\\" "\\" \
	    "Used to escape special characters like \\n."
}

#<DEF>----------------------------------------------------------------
#<DEF> tkREMBuildAwkMode
#<DEF> 
#<DEF> Description
#<DEF>   Build the interface for the AWK mode.
#<DEF>
#<DEF> Globals:
#<DEF>   tkREM(FRAME_MODE)
#<DEF>         - Frame which is the parent to the smaller mode
#<DEF>           meta-frames.
#<DEF>

proc tkREMBuildAwkMode { } {
    global tkREM

    set f $tkREM(FRAME_MODE)

    tkREMMetaButtonReset $f.sed $f.awk $f.sedOperators $f.awkOperators \
	    $f.counting $f.anchors $f.classes $f.grouping $f.other

    # Build the meta-frames first
    tkREMAddMetaFrame $f.sed $f.awk $f.sedOperators $f.awkOperators \
	    $f.counting $f.anchors $f.classes $f.grouping $f.other

    # Build the sed frame.
    tkREMAddMetaButton $f.sed "sed" "sed" \
	    "Start a sed script."
    tkREMAddMetaButton $f.sed "-e" " -e" \
	    "Expression to evaluate."
    tkREMAddMetaButton $f.sed "-f" " -f" \
	    "Use the following file."
    tkREMAddMetaButton $f.sed "-n" " -n" \
	    "Only print matches once. Avoid double line output."
    tkREMAddMetaButton $f.sed "'" "'" \
	    "Single quote to start sed scipt."
    tkREMAddMetaButton $f.sed "\"" "\"" \
	    "Double quote to invoke shell interpretation of sed script."

    # Build the AWK frame.
    tkREMAddMetaButton $f.awk "awk" "awk" \
	    "Start an Awk command."
    tkREMAddMetaButton $f.awk "-f" "-f" \
	    "Use file specified after option."
    tkREMAddMetaButton $f.awk "-F" "-F" \
	    "Data file field seperator."
    tkREMAddMetaButton $f.awk "-v x=1" "-v" \
	    "Assign a value to variable."
    tkREMAddMetaButton $f.awk "BEGIN" "BEGIN" \
	    "Insert a begin structure."
    tkREMAddMetaButton $f.awk "END" "END" \
	    "Insert an end structure."
    tkREMAddMetaButton $f.awk "'" "'" \
	    "Single quote to start AWK scipt."
    tkREMAddMetaButton $f.awk "\"" "\"" \
	    "Double quote to invoke shell interpretation of AWK script."

    # Build the sed Operators frame.
    tkREMAddMetaButton $f.sedOperators "s/" "s/" \
	    "Substitute regular expression operator."
    tkREMAddMetaButton $f.sedOperators "y/" "y/" \
	    "Transform operator - y/abc/ABC/g."
    tkREMAddMetaButton $f.sedOperators "/" "/" \
	    "Expression seperator - s/<\/html>/\vfill\eject/."
    tkREMAddMetaButton $f.sedOperators "!" "!" \
	    "Alternative exp. seperator - s!</html>!\vfill\eject!."
    tkREMAddMetaButton $f.sedOperators "/g" "/g" \
	    "Global substitution."
    tkREMAddMetaButton $f.sedOperators "/p" "/p" \
	    "Print the contents of the pattern space."
    tkREMAddMetaButton $f.sedOperators "/w" "/w" \
	    "Write pattern space to file."
    tkREMAddMetaButton $f.sedOperators "/d" "/d" \
	    "Delete regexp from pattern space."
    tkREMAddMetaButton $f.sedOperators "/i\\ text" "/i\\" \
	    "Insert text into regexp match."
    tkREMAddMetaButton $f.sedOperators "/a\\ text" "/a\\" \
	    "Append text to regexp match."
    tkREMAddMetaButton $f.sedOperators "/c\\ text" "/c\\" \
	    "Change matched text to text."
    tkREMAddMetaButton $f.sedOperators "/q" "/q" \
	    "Quit script on match."
    tkREMAddMetaButton $f.sedOperators "N" "N" \
	    "Append next line to pattern space."
    tkREMAddMetaButton $f.sedOperators "P" "P" \
	    "Print multiple lines of pattern space."
    tkREMAddMetaButton $f.sedOperators "D" "D" \
	    "Multi-line delete of pattern space."
    tkREMAddMetaButton $f.sedOperators "&" "&" \
	    "Replace with string matched by regexp."

    # Build the AWK Operators frame.
    tkREMAddMetaButton $f.awkOperators "\{" "\{" \
	    "Start an action."
    tkREMAddMetaButton $f.awkOperators "\}" "\}" \
	    "End an action."
    tkREMAddMetaButton $f.awkOperators "~" "~" \
	    "Insert field match operator."
    tkREMAddMetaButton $f.awkOperators "!~" "!~" \
	    "Insert field NOT match operator."
    tkREMAddMetaButton $f.awkOperators "/" "/" \
	    "Start a regular expression match."
    tkREMAddMetaButton $f.awkOperators "print" "print" \
	    "Insert the print command."
    tkREMAddMetaButton $f.awkOperators "\$0" "\$0" \
	    "Reference the entire input line."
    tkREMAddMetaButton $f.awkOperators "\$1" "\$1" \
	    "Reference the 1st column field."
    tkREMAddMetaButton $f.awkOperators "\$2" "\$2" \
	    "Reference the 2nd column field."
    tkREMAddMetaButton $f.awkOperators "\$3" "\$3" \
	    "Reference the 3rd column field."
    tkREMAddMetaButton $f.awkOperators "\$4" "\$4" \
	    "Reference the 4th column field."
    tkREMAddMetaButton $f.awkOperators "\$5" "\$5" \
	    "Reference the 5th column field."
    tkREMAddMetaButton $f.awkOperators "\$6" "\$6" \
	    "Reference the 6th column field."
    tkREMAddMetaButton $f.awkOperators "\$7" "\$7" \
	    "Reference the 7th column field."
    tkREMAddMetaButton $f.awkOperators "\$8" "\$8" \
	    "Reference the 8th column field."
    tkREMAddMetaButton $f.awkOperators "\$9" "\$9" \
	    "Reference the 9th column field."

    # Build the counting buttons.
    tkREMAddMetaButton $f.counting "." "." \
	    "Match anything except the newline character."
    tkREMAddMetaButton $f.counting "?" "?" \
	    "One match allowed, but its optional."
    tkREMAddMetaButton $f.counting "*" "*" \
	    "Zero or more matches allowed."
    tkREMAddMetaButton $f.counting "+" "+" \
	    "One match required, additional are optional."

    # Build the position buttons.
    tkREMAddMetaButton $f.anchors "^" "^" \
	    "Insert start of line position marker."
    tkREMAddMetaButton $f.anchors "$" "$" \
	    "Insert end of line position marker."
    tkREMAddMetaButton $f.anchors "^$" "^$" \
	    "Match a blank line."
    tkREMAddMetaButton $f.anchors "^.*$" "^.*$" \
	    "Match an entire line."

    # Build the classes buttons
    tkREMAddMetaButton $f.classes "." "." \
	    "Match anything except the newline character."
    tkREMAddMetaButton $f.classes "\[" "\[" \
	    "Start a character class."
    tkREMAddMetaButton $f.classes "\[^" "\[^" \
	    "Start a negated character class."
    tkREMAddMetaButton $f.classes "\]" "\]" \
	    "Close a character class."
    tkREMAddMetaButton $f.classes "-" "-" \
	    "Create a character range like a-h."
    tkREMAddMetaButton $f.classes "a-z" "a-z" \
	    "Match any lowercase letter."
    tkREMAddMetaButton $f.classes "A-Z" "A-Z" \
	    "Match any uppercase letter."
    tkREMAddMetaButton $f.classes "0-9" "0-9" \
	    "Match any digit."

    # Build the grouping frame.
    tkREMAddMetaButton $f.grouping "(" "(" \
	    "Start a group or back reference."
    tkREMAddMetaButton $f.grouping ")" ")" \
	    "End a group or back reference."
    tkREMAddMetaButton $f.grouping "\\(" "\\(" \
	    "Start a subpattern."
    tkREMAddMetaButton $f.grouping "\\)" "\\)" \
	    "End a sbupattern."
    tkREMAddMetaButton $f.grouping "\{" "\{" \
	    "Start protecting regexp from being interpreted."
    tkREMAddMetaButton $f.grouping "\}" "\}" \
	    "End protecting regexp from being interpreted."

    #Build the other frame.
    tkREMAddMetaButton $f.other "&" "&" \
	    "regsub - Replaced with matched pattern."
    tkREMAddMetaButton $f.other "\\1" "\\1" \
	    "Replaced with 1st subpattern in pattern."
    tkREMAddMetaButton $f.other "\\2" "\\2" \
	    "Replaced with 2nd subpattern in pattern."
    tkREMAddMetaButton $f.other "\\3" "\\3" \
	    "Replaced with 3rd subpattern in pattern."
    tkREMAddMetaButton $f.other "\\" "\\" \
	    "Used to escape special characters like \\n."
    tkREMAddMetaButton $f.other "|" "|" \
	    "Alternate (or) - a|b|c matches either a or b or c."
}

set tkREM(SOURCE_DIR) "bin"
set tkREM(INCLUDE_DIR) "include"

source "$tkREM(INCLUDE_DIR)/update_index.tcl"
source "$tkREM(INCLUDE_DIR)/tkREM_xresources.tcl"
source "$tkREM(INCLUDE_DIR)/tkREM_global.tcl"

lappend auto_path $tkREM(INCLUDE_DIR)
LibraryUpdateIndex $tkREM(INCLUDE_DIR)

# The top frame contains the regexp entry, help message, and icon.
frame .f1  -relief raised

# Build the entry, help, and icon.
label .f1.regexp -text Regexp -width 7 -anchor e
label .f1.wizard_text -text Wizard -width 7 -anchor e
set tkREM(ENTRY_REGEXP) [text .f1.entry -width 48 -height 1]
set tkREM(LABEL_WIZARD) [label .f1.wizard_label -width 48 -anchor w]
set tkREM(ICON) [label .f1.icon -image $tkREM(IMAGE_ICON) -relief raised]

# Pack up elements of the top frame.
grid .f1.regexp .f1.entry -row 1
grid .f1.wizard_text .f1.wizard_label -row 2
grid .f1.icon -row 1 -column 3 -rowspan 2 -padx 5 -pady 5

# The next frame contains the toolbar, and the regexp mode buttons.
frame .f2 -relief raised

# Define the commands used in the Toolbar.
set xsel_cmd "$tkREM(ENTRY_REGEXP) tag add sel 0.0 insert"
set clear_cmd "$tkREM(ENTRY_REGEXP) delete 0.0 end"
set help_cmd "exec $tkREM(BROWSER) $tkREM(HELP_FILE)"

# Build the standard Toolbar buttons.
button .f2.clipboard -image $tkREM(IMAGE_CLIPBOARD) -command $xsel_cmd
button .f2.clear -image $tkREM(IMAGE_CLEAR) -command $clear_cmd
button .f2.help -image $tkREM(IMAGE_HELP) -command $help_cmd
button .f2.exit -image $tkREM(IMAGE_EXIT) -command exit

grid .f2.clipboard -row 1 -column 1 -rowspan 2
grid .f2.clear -row 1 -column 2 -rowspan 2
grid .f2.help -row 1 -column 3 -rowspan 2
grid .f2.exit -row 1 -column 4 -rowspan 2

# Build the mode buttons alongside the ToolBar Buttons.
set r 1; set c 5
foreach b "grep perl tcl python lex awk" {
    # Set the button mode for the global variable tkREM.
    set buttonmode [string toupper $b]

    # Create the Button Label text.
    set text [string toupper [string range $b 0 0]]
    append text [string range $b 1 end]

    # Build the Wizard help message.
    set wizard "Change to $text mode."
    if {$buttonmode == "AWK"} {set wizard "Change to Sed/Awk mode."}

    # Reset the counters for layered mode buttons.
    if {$buttonmode == "PYTHON"} {set r 2; set c 5}

    # Build the button, bind it's help, and pack it.
    set tkREM(BUTTON_MODE_$buttonmode) [button .f2.$b -text $text -width 4 \
	    -command [list tkREMBuildMetaMode .f3 .f1.wizard_label \
	    $buttonmode $wizard]]

    # Create the Wizard help message.
    bind .f2.$b <Enter> [list .f1.wizard_label configure -text $wizard]

    grid .f2.$b -row $r -column $c -padx 2
    incr c +1
}

# This frame is for the meta-character buttons.
set tkREM(FRAME_MODE) [frame .f3 -relief raised]

# Pack up the application.
pack .f1 .f2 .f3 -side top -fill x -ipadx 2 -ipady 2

# Create the meta-character buttons for the default mode.
tkREMBuildMetaMode .f3 .f1.wizard_label GREP "Change to Grep mode."

# Define the wizard_label messages.
bind .f1.regexp <Enter> {.f1.wizard_label configure \
	-text "Regular Expression Entry."}
bind .f1.entry <Enter> {.f1.wizard_label configure \
	-text "Regular Expression Entry."}
bind .f1.icon <Enter> {.f1.wizard_label configure \
	-text "tkREM Regular Expression Toolbox."}
bind .f1.wizard_text <Enter> {.f1.wizard_label configure \
	-text "Wizard Help is displayed here."}
bind .f1.wizard_label <Enter> {.f1.wizard_label configure \
	-text "Wizard Help is displayed here."}
bind .f2.clipboard <Enter> {.f1.wizard_label configure \
	-text "Copy tkREM Regexp to Clipboard."}
bind .f2.clear <Enter> {.f1.wizard_label configure \
	-text "Clear the tkREM Regexp Entry."}
bind .f2.help <Enter> {.f1.wizard_label configure \
	-text "Read through tkREM HTML Help."}
bind .f2.exit <Enter> {.f1.wizard_label configure \
	-text "Close window and exit tkREM."}

# Start out with the focus on the regexp entry widget.
focus $tkREM(ENTRY_REGEXP)