#!/bin/sh
# The following line is a comment in Tcl 
# but is visible to the Bourne shell \
        exec wish "$0" ${1+"$@"}

#<DEF>----------------------------------------------------------------
#<DEF> tkWinstallCreateScript
#<DEF> 
#<DEF> Description:
#<DEF>   Create the Bourne Shell Script which runs the supplied
#<DEF>   tkWorld program.
#<DEF>
#<DEF> p_parent  - This is the parent directory to the tkWorld
#<DEF>             applications bin directory.
#<DEF>                /usr/local/src/tkWorld-0.03
#<DEF> p_binpath - Directory which contains the Bourne Shell scripts.
#<DEF>                /usr/local/bin
#<DEF> p_file    - Create a Shell script for this application file.
#<DEF>                tkGrep.tcl
#<DEF> p_shpath  - Path for the Bourne Shell.
#<DEF>                /bin/sh
#<DEF>

proc tkWinstallCreateScript { p_parent p_binpath p_file p_shpath } {
    # Open the file handle for the shell script.
    set p_fileId [open "$p_binpath/$p_file" w]

    # Define todays date.
    set p_current_date [exec date +19%y.%m.%d]

    # Write out the Bourne Shell script.
    puts $p_fileId "#!$p_shpath"
    puts $p_fileId "#++"
    puts $p_fileId "#"
    puts $p_fileId "# Name"
    puts $p_fileId "#"
    puts $p_fileId "#   $p_file"
    puts $p_fileId "#"
    puts $p_fileId "# Description"
    puts $p_fileId "#"
    puts $p_fileId "#   Execute tkWorld program $p_file"
    puts $p_fileId "#"
    puts $p_fileId "#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"
    puts $p_fileId "# Modification History"
    puts $p_fileId "#"
    puts $p_fileId "# Name            Date       Comment"
    puts $p_fileId "# ~~~~~~~~~~~~~~~ ~~~~~~~~~~ ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"
    puts $p_fileId "# tkWinstall      $p_current_date Created."
    puts $p_fileId "#"
    puts $p_fileId "#--\n"
    puts $p_fileId "# Applications parent directory."
    puts $p_fileId "cd $p_parent"
    puts $p_fileId ""
    puts $p_fileId "# Execute the application."
    puts $p_fileId "wish bin/$p_file.tcl \"\$\@\""
    puts $p_fileId ""

    # Close the output file.
    catch {close $p_fileId}

    # Change the permissions on the shell script.
    catch {exec chmod 755 $p_binpath/$p_file} p_error
}

#<DEF>----------------------------------------------------------------
#<DEF> tkWinstallCreateUnWinstall
#<DEF>
#<DEF> Description:
#<DEF>   Create the file unWinstall.tcl which will undo the
#<DEF>   installation
#<DEF>
#<DEF> Parameters:
#<DEF>   p_src      - Where the distribution tkWorld-x.xx will live.
#<DEF>                   /usr/local/src
#<DEF>   p_bin      - Where the controlling shell scripts will live.
#<DEF>                   /usr/local/bin
#<DEF>   p_app_list - A Tcl list of all the applications to install.
#<DEF>                   "tkGrep tkREM tkLB tkWinstall"
#<DEF>

proc tkWinstallCreateUnWinstall { p_src p_bin p_app_list } {
    global tkWinstall

    set f [open "$p_src/bin/unWinstall.tcl" w]

    # Define todays date.
    set p_current_date [exec date +19%y.%m.%d]

    # Define global with smaller variables.
    set m tkWinstall(MSG_TEXT)
    set w tkWinstall(ANIMATE_WIDGET)

    # Create the header of the un-install script.
    puts $f "#++"
    puts $f "#"
    puts $f "# Name"
    puts $f "#"
    puts $f "#   unWinstall.tcl"
    puts $f "#"
    puts $f "# Description"
    puts $f "#"
    puts $f "#   This script contains the Tcl commands which un-install the"
    puts $f "#   tkWorld distribution."
    puts $f "#"
    puts $f "#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"
    puts $f "# Modification History"
    puts $f "#"
    puts $f "# Name            Date       Comment"
    puts $f "# ~~~~~~~~~~~~~~~ ~~~~~~~~~~ ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~"
    puts $f "# tkWinstall      $p_current_date Created."
    puts $f "#"
    puts $f "#--\n"
    
    puts $f "proc tkWinstallUninstall \{ \} \{"
    puts $f "    global tkWinstall\n"
    puts $f "    # Remove the distributions source directory."
    puts $f "    if \[catch \{exec rm -rf $p_src\} p_error\] \{"
    puts $f "        set tkWinstall(MSG_TEXT) \$p_error"
    puts $f "        \$tkWinstall(BUTTON_UNINSTALL) configure -command \{\}"
    puts $f "        return"
    puts $f "    \} else \{"
    puts $f "        set tkWinstall(MSG_TEXT) \"$p_src un-installed.\""
    puts $f "    \}\n"
    puts $f "    AnimateWidget \$tkWinstall(ANIMATE_WIDGET) 6 tkWinstall\n"
	       
    # Remove all the shells scripts.
    puts $f "    # Remove the individual Bourne Shell scripts."
    foreach p_app $p_app_list {
	regsub {.tcl} $p_app {} p_app
	puts $f "    if \[catch \{file delete $p_bin/$p_app\} p_error\] \{"
	puts $f "        set tkWinstall(MSG_TEXT) \$p_error"
	puts $f "        \$tkWinstall(BUTTON_UNINSTALL) configure -command \{\}"
	puts $f "        return"
	puts $f "    \} else \{"
	puts $f "        set tkWinstall(MSG_TEXT) \"$p_app un-installed.\""
        puts $f "        AnimateWidget \$tkWinstall(ANIMATE_WIDGET) 6 tkWinstall"
	puts $f "    \}\n"
    }

    puts $f "    AnimateWidget \$tkWinstall(ANIMATE_WIDGET) 6 tkWinstall\n"
    puts $f "    \$tkWinstall(BUTTON_UNINSTALL) configure -command \{\}\n"
    puts $f "    bell; set tkWinstall(MSG_TEXT) \"Un-install Complete.\""
    puts $f "\}\n"

    catch {close $f}
}

#<DEF>----------------------------------------------------------------
#<DEF> tkWinstallPathError
#<DEF>
#<DEF> Description:
#<DEF>   Check if the path exists and is a directory.
#<DEF>
#<DEF> Parameters:
#<DEF>   p_path  - Path to check.
#<DEF>   p_entry - Entry widget to highlight if error occurs.
#<DEF>
#<DEF> Globals:
#<DEF>   MSG_TEXT
#<DEF>         This is the global variable which controls the text
#<DEF>         which gets displayed on the bottom of the install
#<DEF>         application.  Changing this variable changes the
#<DEF>         messages which get displayed to the user.
#<DEF>
#<DEF>   ENTRY_ERROR_FOREGROUND
#<DEF>         Foreground color of the widget to highlight.
#<DEF>
#<DEF> Return:
#<DEF>   0 - No errors.
#<DEF>   1 - Path either does not exist, or it is not a directory.
#<DEF>

proc tkWinstallPathError { p_path p_entry } {
    global tkWinstall

    # If no bin directory exists, send a message to the user and
    #   return to install mode of operation.
    if ![file exists $p_path] {
	set tkWinstall(MSG_TEXT) "$p_path does not exist.  Either \
		create it using mkdir, or specify a different path."
	bell; focus $p_entry; $p_entry configure \
		-background $tkWinstall(ENTRY_ERROR_BACKGROUND)
	return 1
    }

    # If the bin file is not a directory, then send a message to
    #   the user and return to install mode of operation.
    if ![file isdirectory $p_path] {
	set tkWinstall(MSG_TEXT) "$p_path is not a directory. \
		Please specify another path."
	bell; focus $p_entry; $p_entry configure \
		-background $tkWinstall(ENTRY_ERROR_BACKGROUND)
	return 1
    }

    return 0
}


#<DEF>----------------------------------------------------------------
#<DEF> tkWinstallDist
#<DEF>   Install the distribution.
#<DEF>
#<DEF> Description:
#<DEF>   Based on which user, either do a single-user install, or
#<DEF>   perform a ROOT mode install.  The difference being in root
#<DEF>   mode, all the source files get moved to other locations in
#<DEF>   the filesystem, and the source files are removed from the
#<DEF>   default TAR directory.
#<DEF>
#<DEF> Globals:
#<DEF>   SOURCE_DIR
#<DEF>         This is where the source files are going to live.
#<DEF>         /usr/local/src/tkWorld-0.03
#<DEF>   BIN_DIR
#<DEF>         This is where the shell scripts which run each
#<DEF>         application live.
#<DEF>         /usr/local/bin
#<DEF>   DISTRIBUTION
#<DEF>         The tkWorld release.
#<DEF>   USER
#<DEF>         Person running the script.
#<DEF>   MSG_TEXT
#<DEF>         This is the global variable which controls the text
#<DEF>         which gets displayed on the bottom of the install
#<DEF>         application.  Changing this variable changes the
#<DEF>         messages which get displayed to the user.
#<DEF>

proc tkWinstallDist { } {
    global tkWinstall

    # Setup the entry colors.
    $tkWinstall(ENTRY_BIN) configure \
	    -background $tkWinstall(ENTRY_DEFAULT_BACKGROUND)
    $tkWinstall(ENTRY_SH) configure \
	    -background $tkWinstall(ENTRY_DEFAULT_BACKGROUND)
    if [info exists tkWinstall(ENTRY_SRC)] {
	$tkWinstall(ENTRY_SRC) configure \
		-background $tkWinstall(ENTRY_DEFAULT_BACKGROUND)
    }

    set tkWinstall(MSG_TEXT) "Installation started..."

    AnimateWidget $tkWinstall(ANIMATE_WIDGET) 6 tkWinstall

    # Define all the files.
    set p_file_list [glob -nocomplain bin/.* bin/* images/* include/* doc/*]

    # Define the application list.
    set p_app_list [glob -nocomplain bin/*.tcl]
    regsub -all {bin/} $p_app_list {} p_app_list
    regsub -all {\.tcl} $p_app_list {} p_app_list
    set tkWinstall(MSG_TEXT) "All applications defined."

    # Get the current directory.
    set p_wd [exec pwd]

    AnimateWidget $tkWinstall(ANIMATE_WIDGET) 6 tkWinstall

    # Exit if somehow this script gets started without being in
    #   the correct directory.
    if {[file tail $p_wd] != "tkWorld-$tkWinstall(DISTRIBUTION)"} {
	bell
	set tkWinstall(MSG_TEXT) \
		"Install Error: You are not in the distribution\
		directory so there are no source files to move into\
		place. Please switch to\
		$tkWinstall(HOME)/tkWorld-$tkWinstall(DISTRIBUTION)\
		directory and run this application again."
	$tkWinstall(BUTTON_INSTALL) configure -command {}
	return
    }

    AnimateWidget $tkWinstall(ANIMATE_WIDGET) 6 tkWinstall

    # Change the owner of the distribution files and directories.
    set p_user [string tolower $tkWinstall(USER)]
    foreach p_file $p_file_list {
        if [catch {exec chown -R $p_user:$p_user $p_file} p_error] {
	    set tkWinstall(MSG_TEXT) "Install Error: $p_error"
	    $tkWinstall(BUTTON_INSTALL) configure -command {}
	    return
	}
    }

    set tkWinstall(MSG_TEXT) \
	    "[string tolower $tkWinstall(USER)] now owns the\
	    tkWorld-$tkWinstall(DISTRIBUTION) distribution."

    AnimateWidget $tkWinstall(ANIMATE_WIDGET) 6 tkWinstall

    # Change the permissions on the distribution files to reflect a
    #   global environment.
    foreach p_file $p_file_list {
        if [file isdirectory $p_file] {
            if [catch {exec chmod 755 $p_file} p_error] {
		set tkWinstall(MSG_TEXT) $p_error
		$tkWinstall(BUTTON_INSTALL) configure -command {}
		return
	    }
        }
        if [file isfile $p_file] {
	    if [catch {exec chmod 644 $p_file} p_error] {
		set tkWinstall(MSG_TEXT) "Install Error: $p_error"
		$tkWinstall(BUTTON_INSTALL) configure -command {}
		return
	    }
	}
    }

    set tkWinstall(MSG_TEXT) \
	    "File permissions setup for the distribution."

    AnimateWidget $tkWinstall(ANIMATE_WIDGET) 6 tkWinstall

    if [tkWinstallPathError $tkWinstall(BIN_DIR) \
	    $tkWinstall(ENTRY_BIN)] {return}

    # Build the shell script files which run each application in
    #   tkWorld. 
    foreach p_app $p_app_list {
	tkWinstallCreateScript $tkWinstall(SOURCE_DIR) \
		$tkWinstall(BIN_DIR) $p_app $tkWinstall(SH_PATH)
	AnimateWidget $tkWinstall(ANIMATE_WIDGET) 6 tkWinstall
	set tkWinstall(MSG_TEXT) "$p_app deployed and operational."
    }
	
    AnimateWidget $tkWinstall(ANIMATE_WIDGET) 6 tkWinstall

    # ROOT installation moves the source code to user entered 
    #   directory and claims tkWinstall.
    if {$tkWinstall(USER) == "ROOT"} {
	if [catch {exec chmod 700 \
		$tkWinstall(BIN_DIR)/tkWinstall} p_error] {
	    bell; set tkWinstall(MSG_TEXT) "Install Error: $p_error"
	    return
	}
	if [tkWinstallPathError [file dirname $tkWinstall(SOURCE_DIR)] \
		$tkWinstall(ENTRY_SRC)] {return}
	cd ..
	file copy "tkWorld-$tkWinstall(DISTRIBUTION)" \
		$tkWinstall(SOURCE_DIR)
	cd $p_wd
	AnimateWidget $tkWinstall(ANIMATE_WIDGET) 6 tkWinstall
    }

    # Create the UnWinstall.tcl script
    tkWinstallCreateUnWinstall $tkWinstall(SOURCE_DIR) \
	    $tkWinstall(BIN_DIR) $p_app_list

    # Indicate that the installation process is complete.
    bell; $tkWinstall(BUTTON_INSTALL) configure -command {}

    AnimateWidget $tkWinstall(ANIMATE_WIDGET) 6 tkWinstall

    # Display the last message to the user.
    set tkWinstall(MSG_TEXT) "Installation Complete.  To un-install\
	    tkWorld-$tkWinstall(DISTRIBUTION), just type tkWinstall\
	    at the prompt.  Example:\n\n\t prompt> tkWinstall\n\nThe\
	    application will have a different interface, and only\
	    display the Un-install, and Exit, buttons."
}

source "include/standard_library.tcl"
source "include/tkWinstall_global.tcl"
source "include/tkWinstall_xresources.tcl"

# Check if we are in install or un-install mode.
if [file exists "bin/unWinstall.tcl"] {
    set tkWinstall(USER_INSTALL_MODE) 0
    set tkWinstall(MSG_TEXT) "Click on the Un-Install button."
    source "bin/unWinstall.tcl"
} else {
    set tkWinstall(USER_INSTALL_MODE) 1
    set tkWinstall(MSG_TEXT) "Click on the Install button."
}

# Build the title frame with the tkWorld Icon.
frame .f1 -relief raised
if $tkWinstall(USER_INSTALL_MODE) {
    label .f1.title -font -*-helvetica-*-r-*-*-24-*-*-*-*-*-*-* \
	    -text "tkWorld-$tkWinstall(DISTRIBUTION) Installation"
} else {
    label .f1.title -font -*-helvetica-*-r-*-*-24-*-*-*-*-*-*-* \
	    -text "tkWorld-$tkWinstall(DISTRIBUTION) Un-install"
}
label .f1.mode -text $tkWinstall(USER_MODE)
set tkWinstall(ANIMATE_WIDGET) [label .f1.icon -relief raised \
	-image $tkWinstall(IMAGE_ICON)]
grid .f1.title -row 0 -column 0 -padx 5 -pady 5
grid .f1.icon -row 0 -column 1 -rowspan 2 -padx 5 -pady 5
grid .f1.mode -row 1 -column 0 -padx 5 -pady 5 -sticky w

# Build the entry frame based on install or un-install, and on the
#   user.  Root mode determines where to put the source code, whereas
#   single user does not. 
if $tkWinstall(USER_INSTALL_MODE) {
    frame .f2 -relief raised
    if {$tkWinstall(USER) == "ROOT"} {
	message .f2.message_src -width 350 -anchor w -text \
		"Enter the parent directory of \
		tkWorld-$tkWinstall(DISTRIBUTION)."
	set tkWinstall(ENTRY_SRC) [entry .f2.entry_src -width 50 \
		-textvariable tkWinstall(SOURCE_DIR)]
	pack .f2.message_src .f2.entry_src -padx 5 -pady 5
    }
    message .f2.message_bin -width 350 -anchor w -text \
	    "Input the bin path. This is where the Bourne Shell scripts\
	    which run each application in\
	    tkWorld-$tkWinstall(DISTRIBUTION) live."
    set tkWinstall(ENTRY_BIN) [entry .f2.entry_bin -width 50 \
	    -textvariable tkWinstall(BIN_DIR)]
    message .f2.message_sh -width 350 -anchor w -text \
	    "Verify the path for the Bourne Shell by typing: which sh\
	    at your prompt."
    set tkWinstall(ENTRY_SH) [entry .f2.entry_sh -width 50 \
	    -textvariable tkWinstall(SH_PATH)]
    pack .f2.message_bin .f2.entry_bin .f2.message_sh \
	    .f2.entry_sh -padx 5 -pady 5
}

# Create the toolbar depending on whether we install or un-install.
frame .f3 -relief raised
if $tkWinstall(USER_INSTALL_MODE) {
    set tkWinstall(BUTTON_INSTALL) [button .f3.install \
	    -image $tkWinstall(IMAGE_INSTALL) \
	    -command tkWinstallDist]
    grid .f3.install -row 0 -column 0 -padx 5 -pady 5
} else {
    set tkWinstall(BUTTON_UNINSTALL) [button .f3.uninstall \
	    -image $tkWinstall(IMAGE_UNINSTALL) \
	    -command tkWinstallUninstall]
    grid .f3.uninstall -row 0 -column 0 -padx 5 -pady 5
}
button .f3.exit -image $tkWinstall(IMAGE_EXIT) -command exit
grid .f3.exit -row 0 -column 1 -padx 5 -pady 5

# Build the frame and message box.  Use this box to display information
#   to the user during the install process.
frame .f4 -relief raised
message .f4.message -width 350 -anchor w \
	-textvariable tkWinstall(MSG_TEXT)
pack .f4.message -padx 5 -pady 5

pack .f1 
if $tkWinstall(USER_INSTALL_MODE) {
    pack .f2 -side top -fill both
    # Determine the opening focus and move the cursor to the end in
    #   all the entry widgets.
    if {$tkWinstall(USER) == "ROOT"} {
	focus .f2.entry_src
	.f2.entry_src icursor end
    } else {
	focus .f2.entry_bin
    }
    .f2.entry_bin icursor end
    .f2.entry_sh icursor end
}
pack .f3 .f4 -side top -fill both

