# Define where the image files live.
set tkStdLib(IMAGE_DIR) "images"

#<DEF> ---------------------------------------------------------------
#<DEF> CloseWindow
#<DEF> 
#<DEF> Description:
#<DEF>   Destroy the supplied window.
#<DEF>
#<DEF> Parameters: 
#<DEF>   p_window - Window to destroy.
#<DEF>

proc CloseWindow {p_window} {destroy $p_window}

#<DEF> ---------------------------------------------------------------
#<DEF> ClearTextWidget
#<DEF> 
#<DEF> Description:
#<DEF>   Clear the supplied text widget.
#<DEF>
#<DEF> Parameters: 
#<DEF>   p_widget - Text widget to clear.
#<DEF>

proc ClearTextWidget {p_widget} {$p_widget delete 0.0 end}

#<DEF> ---------------------------------------------------------------
#<DEF> ClearEntryWidget
#<DEF> 
#<DEF> Description:
#<DEF>   Clear the supplied entry widget.
#<DEF>
#<DEF> Parameters: 
#<DEF>   p_widget - Entry widget to clear.
#<DEF>

proc ClearEntryWidget {p_widget} {$p_widget delete 0 end}

#<DEF> ---------------------------------------------------------------
#<DEF> ScrolledWidget
#<DEF> 
#<DEF> Description:
#<DEF>   Put scrollbars around a given widget.
#<DEF> 
#<DEF> Parameters: 
#<DEF>   p_frame        - Parent frame to place the widget in.
#<DEF>   p_frame_bw     - Parent frame borderwidth.
#<DEF>   p_frame_relief - Parent frame relief.
#<DEF>   p_widget       - Type of widget to put in the parent frame.
#<DEF>   args           - List arguments which characterize the widget.
#<DEF> 
#<DEF> Notes: 
#<DEF>   ScrolledWidget .main 10 raised text .main.text \
#<DEF>            -width 70 -height 20
#<DEF>                  

proc ScrolledWidget {p_frame p_frame_bw p_frame_relief p_widget \
	p_widget_frame args} {
    frame $p_frame -borderwidth $p_frame_bw -relief $p_frame_relief

    # Create the main widget to add scrollbars too.
    eval $p_widget $p_widget_frame $args \
	    {-xscrollcommand [list $p_frame.xscroll set] \
	    -yscrollcommand [list $p_frame.yscroll set]}
 
    # Create the y scrollbar.
    scrollbar $p_frame.yscroll -orient vertical -width 12 \
	    -borderwidth $p_frame_bw \
	    -command [list $p_widget_frame yview]

    # Create a frame for the padding and x scrollbar.
    frame $p_frame.bottom
    scrollbar $p_frame.xscroll -orient horizontal -width 12 \
	    -borderwidth $p_frame_bw \
	    -command [list $p_widget_frame xview]

    # Determine the size of the padding.
    set pad [expr [$p_frame.yscroll cget -width] + \
	    2*([$p_frame.yscroll cget -bd] + \
	       [$p_frame.yscroll cget -highlightthickness])]

    # Create a frame for the padding to be packed in $p_frame.bottom
    frame $p_frame.pad -width $pad -height $pad

    # Pack up the bottom of the window.
    pack $p_frame.bottom -side bottom -fill x 
    pack $p_frame.pad -in $p_frame.bottom -side right 
    pack $p_frame.xscroll -in $p_frame.bottom -side bottom -fill x 

    pack $p_frame.yscroll -side right -fill y 
    pack $p_widget_frame -side left -fill both -expand true
    pack $p_frame -side top -fill both -expand true

    return $p_frame.$p_widget
}

#<DEF> ---------------------------------------------------------------
#<DEF> RightPad
#<DEF> 
#<DEF> Description:
#<DEF>   Makes a string a certain length by adding a certain set of
#<DEF>   characters (default is space) to the right.
#<DEF> 
#<DEF> Parameters:
#<DEF>   p_string  - String to make certain length.
#<DEF>   p_length  - Length to make the string.
#<DEF>   p_padding - Set of characters to add to the string.
#<DEF> 

proc RightPad {p_string p_pad_length {p_padding " "}} {
    # Get the current length of string.
    set p_curr_length [string length $p_string]

    # return p_string if smaller than p_pad_length.
    if {$p_curr_length > $p_pad_length} {
	return $p_string
    }

    # Create the padding string.
    set tmp_string ""
    for {set i $p_curr_length} {$i < $p_pad_length} {incr i +1} {
        append tmp_string $p_padding
    }
    return $p_string$tmp_string
}

#<DEF> ---------------------------------------------------------------
#<DEF> LeftPad
#<DEF> 
#<DEF> Description:
#<DEF>   Makes a string a certain length by adding a certain set of
#<DEF>   characters (default is space) to the left.
#<DEF> 
#<DEF> Parameters:
#<DEF>   p_string  - String to make certain length.
#<DEF>   p_length  - Length to make the string.
#<DEF>   p_padding - Set of characters to add to the string.
#<DEF> 

proc LeftPad {p_string p_pad_length {p_padding " "}} {
    # Get the current length of string.
    set p_curr_length [string length $p_string]

    # return p_string if smaller than p_pad_length.
    if {$p_curr_length > $p_pad_length} {
	return $p_string
    }

    # Create the padding string.
    set tmp_string ""
    for {set i $p_curr_length} {$i < $p_pad_length} {incr i +1} {
        append tmp_string $p_padding
    }
    return $tmp_string$p_string
}

#<DEF> ---------------------------------------------------------------
#<DEF> TrimQuotes
#<DEF> 
#<DEF> Description:
#<DEF>   Strip off starting and ending quotes from string.
#<DEF> 
#<DEF> Parameters:
#<DEF>   p_string - string to strip the quotes off of.
#<DEF> 
#<DEF> Notes:
#<DEF>   Escape backslashes (\) prevent stripping.
#<DEF> 

proc TrimQuotes { p_string } {
    # Define single and double quotes
    set p_single \'; set p_double \"

    # Indeces of the last and 2nd to last characters in the string.
    set p_last [expr [string length $p_string]-1]
    set p_prev [expr $p_last-1]

    # Indicator for string starting with a quote.
    set p_start 0		;# False.

    # Strip off the first quote and decrement the indeces.
    if {[string index $p_string 0] == $p_single} {
	set p_string [string range $p_string 1 $p_last]
	set p_start 1; incr p_last -1; incr p_prev -1
    } elseif {[string index $p_string 0] == $p_double} {
	set p_string [string range $p_string 1 $p_last]
	set p_start 1; incr p_last -1; incr p_prev -1
    } 

    # Strip off the last quote unless the previous character is
    #   an escape backslash.
    if $p_start {
	if {[string index $p_string $p_last] == $p_single} {
	    if {[string range $p_string $p_prev $p_prev] != "\\"} {
		return [string range $p_string 0 $p_prev]
	    }
	} elseif {[string index $p_string $p_last] == $p_double} {
	    if {[string range $p_string $p_prev $p_prev] != "\\"} {
		return [string range $p_string 0 $p_prev]
	    }
	}
    }
    return $p_string
}

#<DEF> ---------------------------------------------------------------
#<DEF> TextWidgetHighlightPattern
#<DEF> 
#<DEF> Description:
#<DEF>   Highlight the pattern in the supplied widget.
#<DEF>
#<DEF> Parameters:
#<DEF>   p_widget     - Widget placing text in (Log Window).
#<DEF>   p_txt        - Text being searching.
#<DEF>   p_pattern    - Pattern to highlight in the Log Window.
#<DEF>   p_line       - Current line in p_widget.
#<DEF>   p_col        - Current column in p_widget.
#<DEF>   p_foreground - Foreground color of the HIGHLIGHTED text.
#<DEF>   p_background - Background color of the HIGHLIGHTED text.
#<DEF>

proc TextWidgetHighlightPattern { p_widget p_txt p_pattern p_line \
	p_col p_foreground p_background} {
    global TEXTWIDGETHIGHLIGHTPATTERN__COUNTER

    if ![info exists TEXTWIDGETHIGHLIGHTPATTERN__COUNTER] {
	set TEXTWIDGETHIGHLIGHTPATTERN__COUNTER 1
	$p_widget tag configure HIGHLIGHT \
		-foreground $p_foreground \
		-background $p_background
    }

    # Define the start and ending locations within the line.
    set p_start [string first $p_pattern $p_txt]
    if {$p_start == -1} {return}
    set p_end [expr $p_start + [string length $p_pattern]]

    # Define the start and end window locations of the first
    #   matched regexp, then add the tag.
    set p_w_start [expr $p_col + $p_start]
    set p_w_end [expr $p_col + $p_end]
	
    # Add the tag to the text in the Log Window.
    $p_widget tag add HIGHLIGHT $p_line.$p_w_start $p_line.$p_w_end
    
    # Set the txt line to after the matched pattern and search the new
    #   text for more matches.
    set p_txt [string range $p_txt $p_end end]
    if [string length $p_txt] {
	TextWidgetHighlightPattern $p_widget $p_txt $p_pattern \
		$p_line $p_w_end $p_foreground $p_background
    }
}

#<DEF>----------------------------------------------------------------
#<DEF> AnimateWidget
#<DEF> 
#<DEF> Description:
#<DEF>   Animate the widget by running through the animation GIF
#<DEF>   images.
#<DEF> 
#<DEF> Parameters:
#<DEF>   p_icon - Widget the icon is in.
#<DEF>   p_image_cnt - The number of images used in the animation.
#<DEF>   p_global_varname - Name of the array variable containing the
#<DEF>                      global definitions.
#<DEF>
#<DEF> Globals:
#<DEF>   X(ANIMATE_ICON)
#<DEF>         - GIF file containing the application's Icon.
#<DEF>   X(IMAGE_ANIMATE_i)
#<DEF>         - i = 1,2,3,..., these are the animation GIF files.
#<DEF>

proc AnimateWidget { p_widget p_image_cnt p_array_varname } {
    upvar $p_array_varname X

    # Go through the animation images, pausing after each display.
    for {set i 1} {$i <= $p_image_cnt} {incr i +1} {
	$p_widget configure -image $X(IMAGE_ANIMATE_$i)
	update idletasks
	after 200
    }

    # Reset the widget to display the original icon.
    $p_widget configure -image $X(IMAGE_ICON)
}

#<DEF>----------------------------------------------------------------
#<DEF> LogWarning
#<DEF> 
#<DEF> Description:
#<DEF>   Display a warning message in supplied widget.
#<DEF>
#<DEF> Parameters:
#<DEF>   p_widget      - Widget to display the error in (Text
#<DEF>                   or Canvas).
#<DEF>   p_image_error - Image to display in front of the error
#<DEF>                   message
#<DEF>   p_error_msg   - Error message to display.
#<DEF>
#<DEF> Notes:
#<DEF>   This works best with a text widget with -width 70.
#<DEF> 

proc LogWarning {p_widget p_image_error p_error_msg } {
    global LOGWARNING__CNTR
    
    # Get the width of the widget and define the widths of the
    #   message box and underline frame.
    set p_width [$p_widget cget -width]
    set p_msg_width [expr 5*$p_width + 20]
    set p_frame_width [expr 6*$p_width + 20]

    # Store the beginning spot for the WARNING tag.
    set p_start [$p_widget index insert]

    # Create a unique ID for the label widgets.
    if ![info exists LOGWARNING__CNTR] {
	set LOGWARNING__CNTR 0
    } else {
	incr LOGWARNING__CNTR +1 
    }

    # Display the error image.
    set p_label [label $p_widget.label_$LOGWARNING__CNTR \
	    -image $p_image_error]
    $p_widget window create end -window $p_label
    
    # Display the error message in a message box for uniform width.
    set w_message [message $p_widget.message_$LOGWARNING__CNTR \
	    -width $p_msg_width -justify left -text $p_error_msg]
    $p_widget window create end -window $w_message
    $p_widget insert end \n

    set w_canvas [canvas $p_widget.canvas_$LOGWARNING__CNTR \
	    -width $p_frame_width -height 0 -relief sunken \
	    -borderwidth 1]
    $p_widget window create end -window $w_canvas
    $p_widget insert end \n

    # Add the WARNING tag to the inserted material.
    $p_widget tag add WARNING $p_start [$p_widget index insert]
}

#<DEF>----------------------------------------------------------------
#<DEF> CreateCommandLineFlag
#<DEF> 
#<DEF> Description:
#<DEF>   Create the command line flags from the defined options in the
#<DEF>   global variable.
#<DEF> 
#<DEF> Parameters:
#<DEF>   p_array_varname - Name of the array variable containing the
#<DEF>                     global OPTION definitions.
#<DEF>
#<DEF> Globals:
#<DEF>   X(OPTION_XXX)
#<DEF>         - Command line option in the form -o.
#<DEF>

proc CreateCommandLineFlag { p_array_varname } {
    upvar $p_array_varname X

    set p_flag ""

    # Create the command line flags.
    foreach p_name [array names X] {
	if [regexp {^OPTION} $p_name] {
	    append p_flag "$X($p_name)"
	}
    }
    regsub -all { |\-} $p_flag {} p_flag
    if [string length $p_flag] {
	set p_flag "-$p_flag"
    }
    
    return $p_flag
}

#<DEF>----------------------------------------------------------------
#<DEF> ValidateRegexp
#<DEF> 
#<DEF> Description:
#<DEF>   Put correct amount of \ in the regexp.  Return a -1 error
#<DEF>   code if regexp is null.
#<DEF>
#<DEF> Parameters:
#<DEF>   p_regexp - regular expression to validate.
#<DEF>

proc ValidateRegexp { p_regexp } {

    set p_regexp [TrimQuotes $p_regexp]
    if ![string length $p_regexp] {
	return -1
    }

    if [regexp {^\\$} $p_regexp] {
	set p_regexp {\\\\}
    }

    regsub {\(} $p_regexp {\(} p_regexp
    regsub {\)} $p_regexp {\)} p_regexp
    regsub {\[} $p_regexp {\[} p_regexp
    regsub {\]} $p_regexp {\]} p_regexp
    regsub {\<} $p_regexp {\<} p_regexp
    regsub {\>} $p_regexp {\>} p_regexp
    regsub {\}} $p_regexp {\}} p_regexp
    regsub {\{} $p_regexp {\{} p_regexp
    
    return $p_regexp
}

#<DEF>----------------------------------------------------------------
#<DEF> CreateValidFileList
#<DEF> 
#<DEF> Description:
#<DEF>   Create a list of valid filenames.  
#<DEF>
#<DEF> Parameters:
#<DEF>   p_file - List of files to validate.
#<DEF>
#<DEF> Notes:
#<DEF>   Return a -1 error code if an error occurs trying to expand a
#<DEF>   wildcard or if a list element is not a file.
#<DEF>

proc CreateValidFileList { p_file } {
    set p_file_list ""; set p_error ""; set p_tmp ""

    # Create the initial list.
    foreach p_tmp $p_file {
	if [catch {append p_file_list "[glob $p_tmp] "} p_error] {
	    return -1
	}
    }

    # Return error code is null list exists.
    if ![string length $p_file_list] {
	return -1
    }

    # Return an error code if any element of the list is not a file.
    foreach p_file $p_file_list {
	if ![file isfile $p_file] {
	    return -1
	}
    }
    
    return $p_file_list
}

#<DEF>----------------------------------------------------------------
#<DEF> ExecuteProcessPipeline
#<DEF> 
#<DEF> Description:
#<DEF>   Execute a system process pipeline.
#<DEF>
#<DEF> Parameters:
#<DEF>   args - each not null arguments is added to the p_pipe
#<DEF>          Tcl list.
#<DEF>
#<DEF> Notes:
#<DEF>   When an error occurs, the routine returns a somewhat unique
#<DEF>   string of the form 
#<DEF>   
#<DEF>           *ERROR*ERROR*command error here*ERROR*ERROR*
#<DEF>   
#<DEF>   which can be caught and parsed, so that the command's error
#<DEF>   message can be sent to the user.
#<DEF>

proc ExecuteProcessPipeline { args } {
    set p_buffer ""

    # Build the process pipeline by appending not null elements to
    #   form a Tcl list.
    set p_pipe "|"
    foreach x $args {
	set x [string trimright [string trimleft $x]]
	if [string length $x] {
	    append p_pipe "$x "
	}
    }

    # Open the process pipeline.
    if [catch {set p_fileId [open $p_pipe]} p_error] {
	return "*ERROR*ERROR*$p_error*ERROR*ERROR*"
    }

    # Read the contents of the pipeline into the buffer.
    if [catch {set p_buffer [read $p_fileId]} p_error] {
	return "*ERROR*ERROR*$p_error*ERROR*ERROR*"
    }

    # Swallow child process exited abnormally error because
    #   this comes back if command returns null output.
    if [catch {close $p_fileId} p_error] {
	if ![regexp -nocase {child.process.exit} $p_error] {
	    return "*ERROR*ERROR*$p_error*ERROR*ERROR*"
	}
    }

    return $p_buffer
}

#<DEF>----------------------------------------------------------------
#<DEF> CreateLogWindow
#<DEF> 
#<DEF> Description:
#<DEF>   Uses parent frame .f4 to create the widget.
#<DEF>
#<DEF> Parameters:
#<DEF>   none
#<DEF>
#<DEF> Notes:
#<DEF>   Requires standard_library.tcl procedure ScrolledWidget.
#<DEF>

proc CreateLogWindow { p_frame p_width p_height p_wizard} {
    frame $p_frame -borderwidth 1 -relief raised
    ScrolledWidget $p_frame.output 1 flat text $p_frame.output.text \
	    -width $p_width -height $p_height -setgrid true -wrap none
    pack $p_frame.output -side bottom

    bind $p_frame.output.text <Enter> [list $p_wizard configure \
	    -text "Log Window displays output from command"]

    return $p_frame.output.text
}

#<DEF>----------------------------------------------------------------
#<DEF> CreateToolbar
#<DEF> 
#<DEF> Description:
#<DEF>   Creates the standard Toolbar with help label.
#<DEF>
#<DEF> Parameters:
#<DEF>   p_frame   - Name of the parent frame.
#<DEF>   p_app     - Name of the parent application (like tkSort).
#<DEF>   p_log     - Class for the log window (.f4.output.text)
#<DEF>   p_img_dir - Location of the image files.
#<DEF>   p_browser - Browser to use (tkLB).
#<DEF>   p_help    - Help file that browser uses.
#<DEF>
#<DEF> Requires standard_library.tcl procedures:
#<DEF>   ScrolledWidget
#<DEF>   ClearTextWidget
#<DEF>

proc CreateToolbar { p_frame p_app p_log p_img_dir p_browser p_help \
	p_wizard } {
    # Define the gifs for the toolbar.
    set p_img_run [image create photo \
	    -file $p_img_dir/tkWorld_run.gif]
    set p_img_exit [image create photo \
	    -file $p_img_dir/tkWorld_exit.gif]
    set p_img_clipboard [image create photo \
	    -file $p_img_dir/tkWorld_clipboard.gif]
    set p_img_reset [image create photo \
	    -file $p_img_dir/tkWorld_reset.gif]
    #set p_img_clear [image create photo \
	    -file $p_img_dir/tkWorld_clear.gif]
    set p_img_clear [image create photo \
	    -file $p_img_dir/tkWorld_clear.gif]
    set p_img_help [image create photo \
	    -file $p_img_dir/tkWorld_button_help.gif]
    set p_img_exit [image create photo \
	    -file $p_img_dir/tkWorld_exit.gif]

    # Create the appControl and appResetCommandCenter commands.
    set p_control_cmd $p_app; append p_control_cmd Control
    set p_reset_cmd $p_app; append p_reset_cmd ResetCommandCenter
    set p_xsel_cmd "$p_log tag add sel 0.0 end"

    # Build the Toolbar.
    frame $p_frame -width 60 -borderwidth 1 -relief raised
    button $p_frame.run -image $p_img_run -borderwidth 1 \
	    -command $p_control_cmd
    button $p_frame.reset -image $p_img_reset -borderwidth 1 \
	    -command $p_reset_cmd
    button $p_frame.clear -image $p_img_clear -borderwidth 1 \
	    -command [list ClearTextWidget $p_log]
    button $p_frame.help -image $p_img_help -borderwidth 1 \
	    -command [list exec $p_browser $p_help &]
    button $p_frame.exit -image $p_img_exit -borderwidth 1 \
	    -command exit
    button $p_frame.clipboard -image $p_img_clipboard -borderwidth 1 \
	    -command $p_xsel_cmd
    label $p_frame.helplabel -width 24 -height 1 -text ""
    grid $p_frame.run $p_frame.reset $p_frame.clear \
	    $p_frame.clipboard $p_frame.help $p_frame.exit -padx 3

    # Now bind the help messages to entry events.
    bind $p_frame.run <Enter> [list $p_wizard configure \
	    -text "Execute $p_app command."]
    bind $p_frame.reset <Enter> [list $p_wizard configure \
	    -text "Reset $p_app Command Center entries."]
    bind $p_frame.clear <Enter> [list $p_wizard configure \
	    -text "Clear the $p_app Log Window."]
    bind $p_frame.clipboard <Enter> [list $p_wizard configure \
	    -text "Copy $p_app Log Window to clipboard."]
    bind $p_frame.help <Enter> [list $p_wizard configure \
	    -text "Read through $p_app HTML help."]
    bind $p_frame.exit <Enter> [list $p_wizard configure \
	    -text "Close window and exit $p_app."]
}

#<DEF>----------------------------------------------------------------
#<DEF> CreateOptionFrame
#<DEF> 
#<DEF> Description:
#<DEF>   Parses options file and display user preferred options.
#<DEF>  
#<DEF> Parameters:
#<DEF>   p_frame       - Name of the parent frame.
#<DEF>   p_option_file - Name of the options file to parse.
#<DEF>   p_app         - Name of the calling application (ex: tkSort).
#<DEF> 
#<DEF> Requires the procedures below be defined in the application:
#<DEF>
#<DEF>   appControl
#<DEF>   appResetCommandCenter
#<DEF>
#<DEF> Where app is the applications name, like tkGrep.
#<DEF> 
#<DEF> Notes:
#<DEF>   Read the buffer and build the labels, checkbuttons and global
#<DEF>   option variables.  The options file is built with lines that
#<DEF>   have the following format:
#<DEF>  
#<DEF>           label string:grep command line flag
#<DEF>  
#<DEF>   Here are some example lines from the distribution .tkGreprc
#<DEF>   file:
#<DEF>  
#<DEF>           Byte Offset:-b
#<DEF>           Line Number in File:-n
#<DEF>  
#<DEF>   The first element is the label which gets displayed, followed
#<DEF>   by the actual command line flag.  The global option variable
#<DEF>   gets defined as $p_app(OPTION_LABEL_STRING)
#<DEF>  
#<DEF>   Comments are allowed and indicated by the pound(#) character
#<DEF>   as the first character in the line.  Option file lines which
#<DEF>   are commented do not appear in the interface.  In the
#<DEF>   following example, the first line is a comment.
#<DEF>  
#<DEF>           #Byte Offset:-b
#<DEF>           Line Number in File:-n
#<DEF>   
#<DEF>   Comments give the user the power to define which options
#<DEF>   appear in the interface and also write reminders in the
#<DEF>   options file for what each command line option does.
#<DEF>  

proc CreateOptionFrame { p_frame p_option_file p_app p_width \
	p_wizard} {

    # Create the options in the Command Center.
    frame $p_frame -width $p_width -borderwidth 1 -relief raised

    # Load the users .tkApprc file into the buffer.
    set p_buffer [read [open $p_option_file r]]

    # Initialize counters.
    set p_max_col 8; set p_row 0; set p_col 0

    foreach line [split $p_buffer \n] {
	if [regexp {^#|^ *$} $line] {continue}
	# Define the command line flag and wizard help message.
	set p_list [split $line :]
	set p_flag [lindex $p_list 0]
	regsub { } $p_flag {} p_flag
	set p_wizardhelp [lindex $p_list 1]

	# Set the global variable which will add command line option
	#   to the applications command.
	set p_variable $p_app; append p_variable "(OPTION_$p_flag)"
	
	# Increment the row, build the checkbuttons, and create
	#   wizard help
	incr p_col +1
	set b [checkbutton $p_frame.$p_row$p_col -text $p_flag \
		-onvalue $p_flag -offvalue "" -variable $p_variable \
		-borderwidth 1 -activebackground #fffff7098f9d \
		-activeforeground black -width 4 -anchor w]
	bind $b <Enter> [list $p_wizard configure -text $p_wizardhelp]
	grid $b -row $p_row -column $p_col
	
	# If at max columns, begin new row.
	if {$p_col == $p_max_col} {
	    incr p_row +1
	    set p_col 0
	}
    }
}

#<DEF>----------------------------------------------------------------
#<DEF> CreateOptionFile - Creates the user option files and directory.
#<DEF>  
#<DEF> Parameters:
#<DEF>   p_src_dir  - Directory to copy distribution files from.
#<DEF>   p_dest_dir - The users options directory.
#<DEF>   p_app      - Name of the calling application (ex: tkSort).
#<DEF> 
#<DEF> Files which get created are (for p_app = tkGrep):
#<DEF>   .tkGreprc
#<DEF>   .tkGreprc.old
#<DEF>   tkGrep.ex
#<DEF>

proc CreateOptionFile { p_src_dir p_dest_dir p_app } {

    set p_tkrc ".$p_app"; append p_tkrc "rc"
    set p_tkrc_old "$p_tkrc.old"
    set p_tk_ex "$p_app.ex"

    # Create the users options directory and file from the
    #   distribution default if they do not already exists.
    if ![file exists $p_dest_dir] {
	exec mkdir $p_dest_dir
	exec cp $p_src_dir/$p_tkrc $p_dest_dir/$p_tkrc
	exec cp $p_src_dir/$p_tkrc $p_dest_dir/$p_tkrc_old
	if [file exists $p_src_dir/$p_tk_ex] {
	    exec cp $p_src_dir/$p_tk_ex $p_dest_dir/$p_tk_ex
	}
    } elseif ![file exists $p_dest_dir/$p_tkrc] {
	if [file exists $p_dest_dir/$p_tkrc_old] {
	    exec cp $p_dest_dir/$p_tkrc_old $p_dest_dir/$p_tkrc
	} else {
	    exec cp $p_src_dir/$p_tkrc $p_dest_dir/$p_tkrc
	    exec cp $p_src_dir/$p_tkrc $p_dest_dir/$p_tkrc_old
	}
    } elseif [file exists $p_dest_dir/$p_tkrc] {
	if ![file exists $p_dest_dir/$p_tkrc_old] {
	    exec cp $p_dest_dir/$p_tkrc $p_dest_dir/$p_tkrc_old
	}
    } 
    if {![file exists $p_dest_dir/p_tk_ex] && \
	    [file exists $p_src_dir/$p_tk_ex]} {
	exec cp $p_src_dir/$p_tk_ex $p_dest_dir/$p_tk_ex
    }
}

#<DEF>----------------------------------------------------------------
#<DEF> CreateAnimateWidget - Build the Animation widget.
#<DEF>  
#<DEF> Parameters:
#<DEF>   p_frame - parent widget
#<DEF>   p_img   - Image to animate in label.
#<DEF> 
#<DEF>

proc CreateAnimateWidget { p_frame p_img } {
    return [label $p_frame.icon -relief raised \
	-borderwidth 1 -image $p_img]
}

#<DEF>----------------------------------------------------------------
#<DEF> CreateCommandDisplay - Entry which shows command
#<DEF>
#<DEF> Parameters:
#<DEF>   p_frame       - Parent widget.
#<DEF>   p_array_name  - Array which contains globals.
#<DEF>   p_img_dir     - Image directory.
#<DEF>   p_width       - width of the entry.
#<DEF>   p_wizardlabel - bind help messages to this widget.
#<DEF> 
#<DEF> Globals:
#<DEF>   COMMAND - This must be defined.  The variable 
#<DEF>             p_array_name(COMMAND) is the global which stores 
#<DEF>             the command to display.
#<DEF> 

proc CreateCommandDisplay { p_frame p_array_name p_img_dir p_width \
	p_wizardlabel } {
    upvar $p_array_name X

    set p_img_clip [image create photo \
	    -file $p_img_dir/tkWorld_smallclipboard.gif]

    # Define the name of the global variable which contains the
    #   command string.
    set p_textvarname "$p_array_name"; append p_textvarname {(COMMAND)}
    
    # Define the selection command for the entry widget.
    set p_cmd "$p_frame.display_entry select from 0"
    append p_cmd {;} "$p_frame.display_entry select to end"

    frame $p_frame -borderwidth 1 -relief raised

    label $p_frame.display_text -width 8 -text Command -anchor e
    entry $p_frame.display_entry -width $p_width -bg white \
	    -textvariable $p_textvarname
    button $p_frame.clipboard -borderwidth 1 \
	    -image $p_img_clip -command $p_cmd
    pack $p_frame.display_text $p_frame.display_entry \
	    $p_frame.clipboard -side left -padx 1

    set p_cmd "$p_frame.command.entry selection clear"
    selection own -command $p_cmd

    bind $p_frame.display_text <Enter> [list $p_wizardlabel configure \
	    -text "Command Display"]
    bind $p_frame.display_entry <Enter> [list $p_wizardlabel configure \
	    -text "Command Display"]
    bind $p_frame.clipboard <Enter> [list $p_wizardlabel configure \
	    -text "Copy Command to Clipboard"]

    return $p_frame.display_entry
}