/* 
 * tixGrid.c --
 *
 *	This module implements "tixGrid" widgets.
 *
 * Copyright (c) 1996, Expert Interface Technologies
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include <tkInt.h>
#include <tixInt.h>
#include <tixDef.h>
#include <tixGrid.h>

/*
 * Information used for argv parsing.
 */
static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_COLOR, "-background", "background", "Background",
       DEF_GRID_BG_COLOR, Tk_Offset(WidgetRecord, normalBg),
       TK_CONFIG_COLOR_ONLY},

    {TK_CONFIG_COLOR, "-background", "background", "Background",
       DEF_GRID_BG_MONO, Tk_Offset(WidgetRecord, normalBg),
       TK_CONFIG_MONO_ONLY},

    {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *) NULL,
       (char *) NULL, 0, 0},

    {TK_CONFIG_SYNONYM, "-bg", "background", (char *) NULL,
       (char *) NULL, 0, 0},

    {TK_CONFIG_BORDER, "-highlightbackground", "highlightBackground",
       "HighlightBackground",
       DEF_GRID_BG_COLOR, Tk_Offset(WidgetRecord, border),
       TK_CONFIG_COLOR_ONLY},

    {TK_CONFIG_BORDER, "-highlightbackground", "highlightBackground",
       "HighlightBackground",
       DEF_GRID_BG_MONO, Tk_Offset(WidgetRecord, border),
       TK_CONFIG_MONO_ONLY},

    {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
       DEF_GRID_BORDER_WIDTH, Tk_Offset(WidgetRecord, borderWidth), 0},

    {TK_CONFIG_STRING, "-browsecmd", "browseCmd", "BrowseCmd",
	DEF_GRID_BROWSE_COMMAND, Tk_Offset(WidgetRecord, browseCmd),
	TK_CONFIG_NULL_OK},

    {TK_CONFIG_STRING, "-command", "command", "Command",
       DEF_GRID_COMMAND, Tk_Offset(WidgetRecord, command),
       TK_CONFIG_NULL_OK},

    {TK_CONFIG_ACTIVE_CURSOR, "-cursor", "cursor", "Cursor",
       DEF_GRID_CURSOR, Tk_Offset(WidgetRecord, cursor),
       TK_CONFIG_NULL_OK},

    {TK_CONFIG_SYNONYM, "-fg", "foreground", (char *) NULL,
       (char *) NULL, 0, 0},

    {TK_CONFIG_BOOLEAN, "-floatingcols", "floatingCols", "FloatingCols",
	DEF_GRID_FLOATING_COLS, Tk_Offset(WidgetRecord, floatRange[1]), 0},

    {TK_CONFIG_BOOLEAN, "-floatingrows", "floatingRows", "FloatingRows",
	DEF_GRID_FLOATING_ROWS, Tk_Offset(WidgetRecord, floatRange[0]), 0},


    {TK_CONFIG_FONT, "-font", "font", "Font",
       DEF_GRID_FONT, Tk_Offset(WidgetRecord, fontPtr), 0},

    {TK_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
       DEF_GRID_FG_COLOR, Tk_Offset(WidgetRecord, normalFg),
       TK_CONFIG_COLOR_ONLY},

    {TK_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
       DEF_GRID_FG_MONO, Tk_Offset(WidgetRecord, normalFg),
       TK_CONFIG_MONO_ONLY},

    {TK_CONFIG_STRING, "-formatcmd", "formatCmd", "FormatCmd",
       DEF_GRID_FORMAT_COMMAND, Tk_Offset(WidgetRecord, formatCmd),
       TK_CONFIG_NULL_OK},

    {TK_CONFIG_PIXELS, "-height", "height", "Height",
       DEF_GRID_HEIGHT, Tk_Offset(WidgetRecord, height), 0},

    {TK_CONFIG_COLOR, "-highlightcolor", "highlightColor", "HighlightColor",
       DEF_GRID_HIGHLIGHT_COLOR, Tk_Offset(WidgetRecord, highlightColorPtr),
       TK_CONFIG_COLOR_ONLY},

    {TK_CONFIG_COLOR, "-highlightcolor", "highlightColor", "HighlightColor",
       DEF_GRID_HIGHLIGHT_MONO, Tk_Offset(WidgetRecord, highlightColorPtr),
       TK_CONFIG_MONO_ONLY},

    {TK_CONFIG_PIXELS, "-highlightthickness", "highlightThickness",
       "HighlightThickness",
       DEF_GRID_HIGHLIGHT_WIDTH, Tk_Offset(WidgetRecord, highlightWidth), 0},

    {TK_CONFIG_INT, "-leftmargin", "leftMargin", "LeftMargin",
       DEF_GRID_LEFT_MARGIN, Tk_Offset(WidgetRecord, hdrSize[1]), 0},

    {TK_CONFIG_CUSTOM, "-itemtype", "itemType", "ItemType",
       DEF_GRID_ITEM_TYPE, Tk_Offset(WidgetRecord, diTypePtr),
       0, &tixConfigItemType},

    {TK_CONFIG_PIXELS, "-padx", "padX", "Pad",
	DEF_GRID_PADX, Tk_Offset(WidgetRecord, padX), 0},

    {TK_CONFIG_PIXELS, "-pady", "padY", "Pad",
	DEF_GRID_PADY, Tk_Offset(WidgetRecord, padY), 0},

    {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
       DEF_GRID_RELIEF, Tk_Offset(WidgetRecord, relief), 0},

    {TK_CONFIG_BORDER, "-selectbackground", "selectBackground", "Foreground",
	DEF_GRID_SELECT_BG_COLOR, Tk_Offset(WidgetRecord, selectBorder),
	TK_CONFIG_COLOR_ONLY},

    {TK_CONFIG_BORDER, "-selectbackground", "selectBackground", "Foreground",
	DEF_GRID_SELECT_BG_MONO, Tk_Offset(WidgetRecord, selectBorder),
	TK_CONFIG_MONO_ONLY},

    {TK_CONFIG_PIXELS, "-selectborderwidth", "selectBorderWidth","BorderWidth",
       DEF_GRID_SELECT_BORDERWIDTH,Tk_Offset(WidgetRecord, selBorderWidth),0},

    {TK_CONFIG_COLOR, "-selectforeground", "selectForeground", "Background",
       DEF_GRID_SELECT_FG_COLOR, Tk_Offset(WidgetRecord, selectFg),
       TK_CONFIG_COLOR_ONLY},

    {TK_CONFIG_COLOR, "-selectforeground", "selectForeground", "Background",
       DEF_GRID_SELECT_FG_MONO, Tk_Offset(WidgetRecord, selectFg),
       TK_CONFIG_MONO_ONLY},

    {TK_CONFIG_UID, "-selectmode", "selectMode", "SelectMode",
	DEF_GRID_SELECT_MODE, Tk_Offset(WidgetRecord, selectMode), 0},

    {TK_CONFIG_UID, "-selectunit", "selectUnit", "SelectUnit",
	DEF_GRID_SELECT_UNIT, Tk_Offset(WidgetRecord, selectUnit), 0},

    {TK_CONFIG_STRING, "-sizecmd", "sizeCmd", "SizeCmd",
       DEF_GRID_SIZE_COMMAND, Tk_Offset(WidgetRecord, sizeCmd),
       TK_CONFIG_NULL_OK},
 
    {TK_CONFIG_STRING, "-takefocus", "takeFocus", "TakeFocus",
	DEF_GRID_TAKE_FOCUS, Tk_Offset(WidgetRecord, takeFocus),
	TK_CONFIG_NULL_OK},

    {TK_CONFIG_INT, "-topmargin", "topMargin", "TopMargin",
       DEF_GRID_TOP_MARGIN, Tk_Offset(WidgetRecord, hdrSize[0]), 0},

    {TK_CONFIG_PIXELS, "-width", "width", "Width",
	DEF_GRID_WIDTH, Tk_Offset(WidgetRecord, width), 0},

    {TK_CONFIG_STRING, "-xscrollcommand", "xScrollCommand", "ScrollCommand",
       DEF_GRID_X_SCROLL_COMMAND,
       Tk_Offset(WidgetRecord, scrollInfo[0].command),
       TK_CONFIG_NULL_OK},

    {TK_CONFIG_STRING, "-yscrollcommand", "yScrollCommand", "ScrollCommand",
       DEF_GRID_Y_SCROLL_COMMAND,
       Tk_Offset(WidgetRecord, scrollInfo[1].command),
       TK_CONFIG_NULL_OK},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
       (char *) NULL, 0, 0}
};	

static Tk_ConfigSpec entryConfigSpecs[] = {
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
       (char *) NULL, 0, 0}
};


/*
 * Forward declarations for procedures defined later in this file:
 */

	/* These are standard procedures for TK widgets
	 * implemeted in C
	 */

static void		WidgetCmdDeletedProc _ANSI_ARGS_((
			    ClientData clientData));
static int		WidgetConfigure _ANSI_ARGS_((Tcl_Interp *interp,
			    WidgetPtr wPtr, int argc, char **argv,
			    int flags));
static void		WidgetDestroy _ANSI_ARGS_((ClientData clientData));
static void		WidgetEventProc _ANSI_ARGS_((ClientData clientData,
			    XEvent *eventPtr));
static int		WidgetCommand _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *, int argc, char **argv));
static void		WidgetDisplay _ANSI_ARGS_((ClientData clientData));
static void		WidgetComputeGeometry _ANSI_ARGS_((
			    ClientData clientData));
static void		IdleHandler _ANSI_ARGS_((
			    ClientData clientData));
	/* Extra procedures for this widget
	 */
static void 		ComputeElementGeometry _ANSI_ARGS_((WidgetPtr wPtr,
			    GridElem *chPtr));
static void 		ComputeOneElementGeometry _ANSI_ARGS_((WidgetPtr wPtr,
			    GridElem *chPtr));
static int		ConfigElement _ANSI_ARGS_((WidgetPtr wPtr,
			    GridElem *chPtr, int argc, char ** argv,
			    int flags, int forced));
static int 		CurSelection _ANSI_ARGS_((Tcl_Interp * interp,
			    WidgetPtr wPtr, GridElem * chPtr));
static GridElem * 	NewElement _ANSI_ARGS_((Tcl_Interp *interp,
			    WidgetPtr wPtr, int argc, char ** argv,
			    char * pathName, char * defParentName));
static void		Tix_GrDrawMainBody _ANSI_ARGS_((WidgetPtr wPtr,
			    RenderInfo * riPtr,Drawable drawable));
static void		UpdateScrollBars _ANSI_ARGS_((WidgetPtr wPtr,
			    int sizeChanged));
static void		GetScrollFractions _ANSI_ARGS_((
			    WidgetPtr wPtr, Tix_GridScrollInfo *siPtr,
			    double * first_ret, double * last_ret));
static void		Tix_GrDItemSizeChanged _ANSI_ARGS_((
			    Tix_DItem *iPtr));
static int		Tix_GrGetPosn _ANSI_ARGS_((
			    Tcl_Interp * interp, char * tag,
			    int * x_ret, int * y_ret));
static GridElem *	Tix_GrGetItem _ANSI_ARGS_((Tcl_Interp * interp,
			    WidgetPtr wPtr, int x, int y));
static void		Tix_GrFreeEntry _ANSI_ARGS_((GridElem * entPtr));
static void		Tix_GrPropagateSize _ANSI_ARGS_((
			    WidgetPtr wPtr, GridElem * chPtr));
static RenderBlock *	Tix_GrAllocateRenderBlock _ANSI_ARGS_((
			    WidgetPtr wPtr, int winW, int winH,
			    int *exactW, int *exactH));
static void		Tix_GrFreeRenderBlock _ANSI_ARGS_((
			    WidgetPtr wPtr, RenderBlock * rbPtr));
static void		Tix_GrComputeSelection _ANSI_ARGS_((
			    WidgetPtr wPtr));
static TIX_DECLARE_SUBCMD(Tix_GrSet);
static TIX_DECLARE_SUBCMD(Tix_GrCGet);
static TIX_DECLARE_SUBCMD(Tix_GrConfig);
static TIX_DECLARE_SUBCMD(Tix_GrDelete);
static TIX_DECLARE_SUBCMD(Tix_GrDebugInfo);
static TIX_DECLARE_SUBCMD(Tix_GrDropSite);
static TIX_DECLARE_SUBCMD(Tix_GrEntryCget);
static TIX_DECLARE_SUBCMD(Tix_GrEntryConfig);
EXTERN TIX_DECLARE_SUBCMD(Tix_GrFormat);
static TIX_DECLARE_SUBCMD(Tix_GrGeometryInfo);
static TIX_DECLARE_SUBCMD(Tix_GrHide);
static TIX_DECLARE_SUBCMD(Tix_GrInfo);
static TIX_DECLARE_SUBCMD(Tix_GrNearest);
static TIX_DECLARE_SUBCMD(Tix_GrSee);
EXTERN TIX_DECLARE_SUBCMD(Tix_GrSelection);
EXTERN TIX_DECLARE_SUBCMD(Tix_GrSetSize);
static TIX_DECLARE_SUBCMD(Tix_GrSetSite);
static TIX_DECLARE_SUBCMD(Tix_GrShow);
static TIX_DECLARE_SUBCMD(Tix_GrView);

/*
 *--------------------------------------------------------------
 *
 * Tix_GridCmd --
 *
 *	This procedure is invoked to process the "tixGrid" Tcl
 *	command.  It creates a new "TixGrid" widget.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	A new widget is created and configured.
 *
 *--------------------------------------------------------------
 */
int
Tix_GridCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    Tk_Window main = (Tk_Window) clientData;
    WidgetPtr wPtr;
    Tk_Window tkwin;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " pathName ?options?\"", (char *) NULL);
	return TCL_ERROR;
    }

    tkwin = Tk_CreateWindowFromPath(interp, main, argv[1], (char *) NULL);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }

    Tk_SetClass(tkwin, "TixGrid");

    /*
     * Allocate and initialize the widget record.
     */
    wPtr = (WidgetPtr) ckalloc(sizeof(WidgetRecord));

    wPtr->dispData.tkwin 	= tkwin;
    wPtr->dispData.display 	= Tk_Display(tkwin);
    wPtr->dispData.interp 	= interp;
    wPtr->dispData.sizeChangedProc = Tix_GrDItemSizeChanged;
    wPtr->fontPtr		= NULL;
    wPtr->normalBg 		= NULL;
    wPtr->normalFg		= NULL;
    wPtr->command 		= NULL;
    wPtr->border 		= NULL;
    wPtr->borderWidth 		= 0;
    wPtr->selectBorder 		= NULL;
    wPtr->selBorderWidth 	= 0;
    wPtr->selectFg		= NULL;
    wPtr->backgroundGC		= None;
    wPtr->selectGC		= None;
    wPtr->anchorGC		= None;
    wPtr->highlightWidth	= 0;
    wPtr->highlightColorPtr	= NULL;
    wPtr->highlightGC		= None;
    wPtr->relief 		= TK_RELIEF_FLAT;
    wPtr->cursor 		= None;
    wPtr->idleEvent 		= 0;
    wPtr->toRedraw 		= 0;
    wPtr->toResize 		= 0;
    wPtr->toComputeSel 		= 0;
    wPtr->hasFocus 		= 0;
    wPtr->selectMode		= NULL;
    wPtr->selectUnit		= NULL;
    wPtr->anchor 		= 0;
    wPtr->dropSite 		= 0;
    wPtr->browseCmd		= 0;
    wPtr->formatCmd		= 0;
    wPtr->sizeCmd		= 0;
    wPtr->takeFocus		= NULL;
    wPtr->serial		= 0;
    wPtr->fontChanged		= 0;
    wPtr->mainRB		= (RenderBlock*)NULL;
    wPtr->hdrSize[0]		= 1;
    wPtr->hdrSize[1]		= 1;
    wPtr->expArea.x1 		= 10000;
    wPtr->expArea.y1 		= 10000;
    wPtr->expArea.x2 		= 0;
    wPtr->expArea.y2 		= 0;
    wPtr->dataSet		= TixGridDataSetInit();
    wPtr->renderInfo		= NULL;
    wPtr->defSize[0].sizeType	= TIX_GR_DEFINED_CHAR;
    wPtr->defSize[0].charValue	= 10.0;
    wPtr->defSize[0].pad0	= 2;
    wPtr->defSize[0].pad1	= 2;
    wPtr->defSize[1].sizeType	= TIX_GR_DEFINED_CHAR;
    wPtr->defSize[1].charValue	= 1.2;
    wPtr->defSize[1].pad0	= 2;
    wPtr->defSize[1].pad1	= 2;
    wPtr->gridSize[0]		= 0;
    wPtr->gridSize[1]		= 0;

    wPtr->scrollInfo[0].command  = NULL;
    wPtr->scrollInfo[1].command  = NULL;

    wPtr->scrollInfo[0].max  = 1;
    wPtr->scrollInfo[0].unit   = 1;
    wPtr->scrollInfo[0].offset = 0;
    wPtr->scrollInfo[0].window = 1.0;
    wPtr->scrollInfo[1].max  = 1;
    wPtr->scrollInfo[1].unit   = 1;
    wPtr->scrollInfo[1].offset = 0;
    wPtr->scrollInfo[1].window = 1.0;

    Tix_SimpleListInit(&wPtr->colorInfo);
    Tix_SimpleListInit(&wPtr->selList);
    Tix_SimpleListInit(&wPtr->mappedWindows);

    Tk_CreateEventHandler(wPtr->dispData.tkwin,
	ExposureMask|StructureNotifyMask|FocusChangeMask,
	WidgetEventProc, (ClientData) wPtr);
    wPtr->widgetCmd = Tcl_CreateCommand(interp,
	Tk_PathName(wPtr->dispData.tkwin), WidgetCommand, (ClientData) wPtr,
	WidgetCmdDeletedProc);

    if (WidgetConfigure(interp, wPtr, argc-2, argv+2, 0) != TCL_OK) {
	Tk_DestroyWindow(wPtr->dispData.tkwin);
	return TCL_ERROR;
    }

    interp->result = Tk_PathName(wPtr->dispData.tkwin);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * WidgetConfigure --
 *
 *	This procedure is called to process an argv/argc list in
 *	conjunction with the Tk option database to configure (or
 *	reconfigure) a List widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as colors, border width,
 *	etc. get set for wPtr;  old resources get freed,
 *	if there were any.
 *
 *----------------------------------------------------------------------
 */
static int
WidgetConfigure(interp, wPtr, argc, argv, flags)
    Tcl_Interp *interp;			/* Used for error reporting. */
    WidgetPtr wPtr;			/* Information about widget. */
    int argc;				/* Number of valid entries in argv. */
    char **argv;			/* Arguments. */
    int flags;				/* Flags to pass to
					 * Tk_ConfigureWidget. */
{
    XGCValues gcValues;
    GC newGC;
    XFontStruct *oldFontPtr;
    size_t length;
    Tix_StyleTemplate stTmpl;

    oldFontPtr = wPtr->fontPtr;

    if (Tk_ConfigureWidget(interp, wPtr->dispData.tkwin, configSpecs,
	    argc, argv, (char *) wPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    if (oldFontPtr != wPtr->fontPtr) {
	int i;

	/* Font has been changed (initialized) */
	wPtr->fontChanged = 1;

	TkComputeTextGeometry(wPtr->fontPtr, "0", 1,
	    0, &wPtr->fontSize[0], &wPtr->fontSize[1]);

	for (i=0; i<2; i++) {
	    switch (wPtr->defSize[i].sizeType) {
	      case TIX_GR_DEFINED_CHAR:
		wPtr->defSize[i].pixels =
		    wPtr->defSize[i].charValue * wPtr->fontSize[i];
		break;
	      case TIX_GR_AUTO:
		if (i==0) {
		    wPtr->defSize[i].pixels = 10 * wPtr->fontSize[0];
		}
		if (i==1) {
		    wPtr->defSize[i].pixels =  1 * wPtr->fontSize[1];
		}
		break;
	    }
	}
    }

    /*
     * A few options need special processing, such as setting the
     * background from a 3-D border, or filling in complicated
     * defaults that couldn't be specified to Tk_ConfigureWidget.
     */

    Tk_SetBackgroundFromBorder(wPtr->dispData.tkwin, wPtr->border);

    /*
     * Note: GraphicsExpose events are disabled in normalGC because it's
     * used to copy stuff from an off-screen pixmap onto the screen (we know
     * that there's no problem with obscured areas).
     */

    /* The background GC */
    gcValues.foreground 	= wPtr->normalBg->pixel;
    gcValues.graphics_exposures = False;

    newGC = Tk_GetGC(wPtr->dispData.tkwin,
	GCForeground|GCGraphicsExposures, &gcValues);
    if (wPtr->backgroundGC != None) {
	Tk_FreeGC(wPtr->dispData.display, wPtr->backgroundGC);
    }
    wPtr->backgroundGC = newGC;

    /* The selected text GC */
    gcValues.font 		= wPtr->fontPtr->fid;
    gcValues.foreground 	= wPtr->selectFg->pixel;
    gcValues.background 	= Tk_3DBorderColor(wPtr->selectBorder)->pixel;
    gcValues.graphics_exposures = False;

    newGC = Tk_GetGC(wPtr->dispData.tkwin,
	GCForeground|GCBackground|GCFont|GCGraphicsExposures, &gcValues);
    if (wPtr->selectGC != None) {
	Tk_FreeGC(wPtr->dispData.display, wPtr->selectGC);
    }
    wPtr->selectGC = newGC;

    /* The dotted anchor lines */
    gcValues.foreground 	= wPtr->normalFg->pixel;
    gcValues.background 	= wPtr->normalBg->pixel;
    gcValues.graphics_exposures = False;
    gcValues.line_style         = LineDoubleDash;
    gcValues.dashes 		= 2;
    gcValues.subwindow_mode	= IncludeInferiors;

    newGC = Tk_GetGC(wPtr->dispData.tkwin,
	GCForeground|GCBackground|GCGraphicsExposures|GCLineStyle|GCDashList|
	GCSubwindowMode, &gcValues);
    if (wPtr->anchorGC != None) {
	Tk_FreeGC(wPtr->dispData.display, wPtr->anchorGC);
    }
    wPtr->anchorGC = newGC;

    /* The highlight border */
    gcValues.background 	= wPtr->selectFg->pixel;
    gcValues.foreground 	= wPtr->highlightColorPtr->pixel;
    gcValues.graphics_exposures = False;

    newGC = Tk_GetGC(wPtr->dispData.tkwin,
	GCForeground|GCBackground|GCGraphicsExposures, &gcValues);
    if (wPtr->highlightGC != None) {
	Tk_FreeGC(wPtr->dispData.display, wPtr->highlightGC);
    }
    wPtr->highlightGC = newGC;

    /* We must set the options of the default styles so that
     * -- the default styles will change according to what is in
     *    stTmpl
     */
    stTmpl.fontPtr 			= wPtr->fontPtr;
    stTmpl.pad[0]  			= wPtr->padX;
    stTmpl.pad[1]  			= wPtr->padY;
    stTmpl.colors[TIX_DITEM_NORMAL].fg  = wPtr->normalFg;
    stTmpl.colors[TIX_DITEM_NORMAL].bg  = wPtr->normalBg;
    stTmpl.colors[TIX_DITEM_SELECTED].fg= wPtr->selectFg;
    stTmpl.colors[TIX_DITEM_SELECTED].bg= Tk_3DBorderColor(wPtr->selectBorder);
    stTmpl.flags = TIX_DITEM_FONT|TIX_DITEM_NORMAL_BG|
	TIX_DITEM_SELECTED_BG|TIX_DITEM_NORMAL_FG|TIX_DITEM_SELECTED_FG |
	TIX_DITEM_PADX|TIX_DITEM_PADY;

    Tix_SetDefaultStyleTemplate(wPtr->dispData.tkwin, &stTmpl);

    Tix_GrDoWhenIdle(wPtr, TIX_GR_RESIZE);

    return TCL_OK;
}
/*
 *--------------------------------------------------------------
 *
 * WidgetCommand --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
WidgetCommand(clientData, interp, argc, argv)
    ClientData clientData;		/* Information about the widget. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int code;

    static Tix_SubCmdInfo subCmdInfo[] = {
	{TIX_DEFAULT_LEN, "anchor", 1, 2, Tix_GrSetSite,
	   "option ?index?"},
	{TIX_DEFAULT_LEN, "cget", 1, 1, Tix_GrCGet,
	   "option"},
	{TIX_DEFAULT_LEN, "configure", 0, TIX_VAR_ARGS, Tix_GrConfig,
	   "?option? ?value? ?option value ... ?"},
	{TIX_DEFAULT_LEN, "delete", 1, 2, Tix_GrDelete,
	   "from ?to?"},
	{TIX_DEFAULT_LEN, "debuginfo", 1, TIX_VAR_ARGS, Tix_GrDebugInfo,
	   "option ?args ...?"},
#if 0
	{TIX_DEFAULT_LEN, "dropsite", 1, 2, Tix_GrDropSite,
	   "option ?entryPath?"},
	{TIX_DEFAULT_LEN, "entrycget", 2, 2, Tix_GrEntryCget,
	   "entryPath option"},
#endif
	{TIX_DEFAULT_LEN, "entryconfigure", 1, TIX_VAR_ARGS, Tix_GrEntryConfig,
	   "index ?option? ?value? ?option value ... ?"},
	{TIX_DEFAULT_LEN, "format", 1, TIX_VAR_ARGS, Tix_GrFormat,
	   "option ?args ...?"},
	{TIX_DEFAULT_LEN, "geometryinfo", 0, 2, Tix_GrGeometryInfo,
	   "?width height?"},
#if 0
	{TIX_DEFAULT_LEN, "hide", 2, 2, Tix_GrHide,
	   "option entryPath"},
	{TIX_DEFAULT_LEN, "info", 1, TIX_VAR_ARGS, Tix_GrInfo,
	   "option ?args ...?"},
#endif
	{TIX_DEFAULT_LEN, "nearest", 2, 2, Tix_GrNearest,
	   "x y"},
#if 0
	{TIX_DEFAULT_LEN, "see", 1, 1, Tix_GrSee,
	   "entryPath"},
#endif
	{TIX_DEFAULT_LEN, "selection", 3, 5, Tix_GrSelection,
	   "option x1 y1 ?x2 y2?"},
	{TIX_DEFAULT_LEN, "set", 2, TIX_VAR_ARGS, Tix_GrSet,
	   "x y ?option value ...?"},
	{TIX_DEFAULT_LEN, "size", 1, TIX_VAR_ARGS, Tix_GrSetSize,
	   "option ?args ...?"},
#if 0
	{TIX_DEFAULT_LEN, "show", 2, 2, Tix_GrShow,
	   "option entryPath"},
#endif
	{TIX_DEFAULT_LEN, "xview", 0, 3, Tix_GrView,
	   "args"},
	{TIX_DEFAULT_LEN, "yview", 0, 3, Tix_GrView,
	   "args"},
    };

    static Tix_CmdInfo cmdInfo = {
	Tix_ArraySize(subCmdInfo), 1, TIX_VAR_ARGS, "?option? arg ?arg ...?",
    };

    Tk_Preserve(clientData);
    code = Tix_HandleSubCmds(&cmdInfo, subCmdInfo, clientData,
	interp, argc, argv);
    Tk_Release(clientData);

    return code;
}

/*
 *--------------------------------------------------------------
 *
 * WidgetEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on Lists.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */
static void
WidgetEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    XEvent *eventPtr;		/* Information about event. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;
    int x2, y2;

    switch (eventPtr->type) {
      case DestroyNotify:
	if (wPtr->dispData.tkwin != NULL) {
	    wPtr->dispData.tkwin = NULL;
	    Tcl_DeleteCommand(wPtr->dispData.interp, 
	        Tcl_GetCommandName(wPtr->dispData.interp, wPtr->widgetCmd));
	}
	Tix_GrCancelDoWhenIdle(wPtr, TIX_GR_RESIZE);
	Tix_GrCancelDoWhenIdle(wPtr, TIX_GR_COMPUTE_SEL);
	Tix_GrCancelDoWhenIdle(wPtr, TIX_GR_REDRAW);
	Tk_EventuallyFree((ClientData) wPtr, (Tix_FreeProc*)WidgetDestroy);
	break;

      case ConfigureNotify:
	wPtr->expArea.x1 = 0;
	wPtr->expArea.y1 = 0;
	wPtr->expArea.x2 = Tk_Width (wPtr->dispData.tkwin) - 1;
	wPtr->expArea.y2 = Tk_Height(wPtr->dispData.tkwin) - 1;
	Tix_GrDoWhenIdle(wPtr, TIX_GR_RESIZE);
	break;

      case Expose:
	if (wPtr->expArea.x1 > eventPtr->xexpose.x) {
	    wPtr->expArea.x1 = eventPtr->xexpose.x;
	}
	if (wPtr->expArea.y1 > eventPtr->xexpose.y) {
	    wPtr->expArea.y1 = eventPtr->xexpose.y;
	}
	x2 = eventPtr->xexpose.x + eventPtr->xexpose.width  - 1;
	y2 = eventPtr->xexpose.y + eventPtr->xexpose.height - 1;

	if (wPtr->expArea.x2 < x2) {
	    wPtr->expArea.x2 = x2;
	}
	if (wPtr->expArea.y2 < y2) {
	    wPtr->expArea.y2 = y2;
	}
	Tix_GrDoWhenIdle(wPtr, TIX_GR_REDRAW);
	break;

      case FocusIn:
	wPtr->hasFocus = 1;
	wPtr->redrawBorder = 1;
	Tix_GrDoWhenIdle(wPtr, TIX_GR_REDRAW);
	break;

      case FocusOut:
	wPtr->hasFocus = 0;
	wPtr->redrawBorder = 1;
	Tix_GrDoWhenIdle(wPtr, TIX_GR_REDRAW);
	break;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * WidgetDestroy --
 *
 *	This procedure is invoked by Tk_EventuallyFree or Tk_Release
 *	to clean up the internal structure of a List at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the List is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
WidgetDestroy(clientData)
    ClientData clientData;	/* Info about my widget. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;


    if (wPtr->backgroundGC != None) {
	Tk_FreeGC(wPtr->dispData.display, wPtr->backgroundGC);
    }
    if (wPtr->selectGC != None) {
	Tk_FreeGC(wPtr->dispData.display, wPtr->selectGC);
    }
    if (wPtr->anchorGC != None) {
	Tk_FreeGC(wPtr->dispData.display, wPtr->anchorGC);
    }
    if (wPtr->highlightGC != None) {
	Tk_FreeGC(wPtr->dispData.display, wPtr->highlightGC);
    }

    if (wPtr->mainRB) {
	Tix_GrFreeRenderBlock(wPtr, wPtr->mainRB);
    }

    Tix_GrFreeUnusedColors(wPtr, 1);

    if (!Tix_IsLinkListEmpty(wPtr->mappedWindows)) {
	/*
	 * All mapped windows should have been unmapped when the
	 * the entries were deleted
	 */
	panic("tixGrid: mappedWindows not NULL");
    }

    Tk_FreeOptions(configSpecs, (char *) wPtr, wPtr->dispData.display, 0);
    ckfree((char *) wPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * WidgetCmdDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted.  If
 *	the widget isn't already in the process of being destroyed,
 *	this command destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */
static void
WidgetCmdDeletedProc(clientData)
    ClientData clientData;	/* Pointer to widget record for widget. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;

    /*
     * This procedure could be invoked either because the window was
     * destroyed and the command was then deleted (in which case tkwin
     * is NULL) or because the command was deleted, and then this procedure
     * destroys the widget.
     */
    if (wPtr->dispData.tkwin != NULL) {
	Tk_Window tkwin = wPtr->dispData.tkwin;
	wPtr->dispData.tkwin = NULL;
	Tk_DestroyWindow(tkwin);
    }
}

static void RecalScrollRegion(wPtr, winW, winH)
    WidgetPtr wPtr;
    int winW;
    int winH;
{
    int gridSize[2];
    int winSize[2];
    int i, k;
    int pixels;
    int count;
    int visibleSize;
    int totalSize;
    int pad0, pad1;

    winSize[0] = winW;
    winSize[1] = winH;
    
    Tix_GridDataGetGridSize(wPtr->dataSet, &gridSize[0],
	&gridSize[1]);

    for (i=0; i<2; i++) {
	for (k=0; k<wPtr->hdrSize[i] && k<gridSize[i]; k++) {
	    winSize[i] -= Tix_GridDataGetRowColSize(wPtr, wPtr->dataSet, i,
		k, &wPtr->defSize[i], &pad0, &pad1);
	    winSize[i] -= pad0 + pad1;
	}
	if (winSize[i] <= 0) {
	    wPtr->scrollInfo[i].max    = 0;
	    wPtr->scrollInfo[i].window = 1.0;
	    continue;
	}
	visibleSize = winSize[i];

	for (count=0,k=gridSize[i]-1; k>=wPtr->hdrSize[i]&&k>=0; count++,k--) {
	    winSize[i] -= Tix_GridDataGetRowColSize(wPtr, wPtr->dataSet, i,
		k, &wPtr->defSize[i], &pad0, &pad1);
	    winSize[i] -= pad0 + pad1;

	    if (winSize[i] == 0) {
		++ count;
		break;
	    }
	    else if (winSize[i] < 0) {
		break;
	    }
	}

	if (count == 0) {
	    /* the last element is only bigger than the size of the window*/
	    count = 1;		/* treat it as if the last element is just
				 * about the size as the window */
	}

	wPtr->scrollInfo[i].max  = (gridSize[i]-wPtr->hdrSize[i]) - count;

	/* calculate the total pixel size  (%%SLOOOOOOW)*/

	for (totalSize=0,k=wPtr->hdrSize[i];k<gridSize[i];k++) {
	    totalSize += Tix_GridDataGetRowColSize(wPtr, wPtr->dataSet, i,
		k, &wPtr->defSize[i], &pad0, &pad1);
	    totalSize += pad0 + pad1;
	}

	/* we may need some left over spaces after the last element */
	totalSize += (-winSize[i]);

	wPtr->scrollInfo[i].window =
	    (double)(visibleSize) / (double)totalSize;
    }
}


/*
 *--------------------------------------------------------------
 *
 * WidgetComputeGeometry --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	none
 *
 *--------------------------------------------------------------
 */
static void
WidgetComputeGeometry(clientData)
    ClientData clientData;
{
    WidgetPtr wPtr = (WidgetPtr)clientData;
    int winW, winH, exactW, exactH;
    Tk_Window tkwin = wPtr->dispData.tkwin;

    winW = Tk_Width (tkwin) - 2*wPtr->highlightWidth - 2*wPtr->borderWidth;
    winH = Tk_Height(tkwin) - 2*wPtr->highlightWidth - 2*wPtr->borderWidth;

    RecalScrollRegion(wPtr, winW, winH);
    printf("(%d %f %d) (%d %f %d)\n", 
        wPtr->scrollInfo[0].max,
        wPtr->scrollInfo[0].window,
        wPtr->scrollInfo[0].offset,
        wPtr->scrollInfo[1].max,
        wPtr->scrollInfo[1].window,
        wPtr->scrollInfo[1].offset);

    UpdateScrollBars(wPtr, 1);

    if (wPtr->mainRB) {
	Tix_GrFreeRenderBlock(wPtr, wPtr->mainRB);
    }
    wPtr->mainRB = Tix_GrAllocateRenderBlock(wPtr, winW, winH,&exactW,&exactH);

    wPtr->expArea.x1 = 0;
    wPtr->expArea.y1 = 0;
    wPtr->expArea.x2 = Tk_Width (wPtr->dispData.tkwin) - 1;
    wPtr->expArea.y2 = Tk_Height(wPtr->dispData.tkwin) - 1;

    Tix_GrDoWhenIdle(wPtr, TIX_GR_COMPUTE_SEL);
    Tix_GrDoWhenIdle(wPtr, TIX_GR_REDRAW);
}

/*----------------------------------------------------------------------
 * DItemSizeChanged --
 *
 *	This is called whenever the size of one of the HList's items
 *	changes its size.
 *----------------------------------------------------------------------
 */
static void
Tix_GrDItemSizeChanged(iPtr)
    Tix_DItem *iPtr;
{
    WidgetPtr wPtr = (WidgetPtr)iPtr->base.clientData;

    if (wPtr) {
	/* double-check: perhaps we haven't set the clientData yet! */
	Tix_GrDoWhenIdle(wPtr, TIX_GR_RESIZE);
    }
}

/*
 *----------------------------------------------------------------------
 * Tix_GrDoWhenIdle --
 *----------------------------------------------------------------------
 */
void
Tix_GrDoWhenIdle(wPtr, type)
    WidgetPtr wPtr;
    int type;
{
    switch (type) {
      case TIX_GR_RESIZE:
	wPtr->toResize = 1;
	break;
      case TIX_GR_COMPUTE_SEL:
	wPtr->toComputeSel = 1;
	break;
      case TIX_GR_REDRAW:
	wPtr->toRedraw = 1;
	break;
    }

    if (!wPtr->idleEvent) {
	wPtr->idleEvent = 1;
	Tk_DoWhenIdle(IdleHandler, (ClientData)wPtr);
    }
}

/*
 *----------------------------------------------------------------------
 * Tix_GrCancelDoWhenIdle --
 *----------------------------------------------------------------------
 */
void
Tix_GrCancelDoWhenIdle(wPtr, type)
    WidgetPtr wPtr;
    int type;
{
    switch (type) {
      case TIX_GR_RESIZE:
	wPtr->toResize = 0;
	break;
      case TIX_GR_COMPUTE_SEL:
	wPtr->toComputeSel = 0;
	break;
      case TIX_GR_REDRAW:
	wPtr->toRedraw = 0;
	break;
    }

    if (wPtr->idleEvent) {
	if (!wPtr->toResize && !wPtr->toComputeSel && !wPtr->toRedraw) {
	    Tk_CancelIdleCall(IdleHandler, (ClientData)wPtr);
	}
	wPtr->idleEvent = 0;
    }
}

static void
IdleHandler(clientData)
    ClientData clientData;	/* Info about my widget. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;

    if (!wPtr->idleEvent) {	/* sanity check */
	return;
    }

    if (wPtr->toResize) {
	wPtr->toResize = 0;
	WidgetComputeGeometry(clientData);
    }
    if (wPtr->toComputeSel) {
	wPtr->toComputeSel = 0;
	Tix_GrComputeSelection(wPtr);
    }
    if (wPtr->toRedraw) {
	wPtr->toRedraw = 0;
	WidgetDisplay(clientData);
    }

    wPtr->idleEvent = 0;    
}


/*
 *----------------------------------------------------------------------
 *
 * WidgetDisplay --
 *
 *	Draw the widget to the screen.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *
 *----------------------------------------------------------------------
 */
static void Tix_GrDisplayMainBody(wPtr, buffer, winW, winH)
    WidgetPtr wPtr;
    Drawable buffer;
    int winW;
    int winH;
{
    Tk_Window tkwin = wPtr->dispData.tkwin;
    RenderInfo mainRI;			/* render info for main body */
    Rect a, main;
    int visible;

    if (buffer == Tk_WindowId(tkwin)) {
	/* rendering directly into the window */
	mainRI.origin[0] = wPtr->highlightWidth + wPtr->borderWidth;
	mainRI.origin[1] = wPtr->highlightWidth + wPtr->borderWidth;

    } else {
	/* rendering into a pixmap */
	mainRI.origin[0] = wPtr->highlightWidth + wPtr->borderWidth
	    - wPtr->expArea.x1;
	mainRI.origin[1] = wPtr->highlightWidth + wPtr->borderWidth
	    - wPtr->expArea.y1;
    }

    mainRI.drawable = buffer;
    wPtr->renderInfo = &mainRI;
    wPtr->colorInfoCounter ++;
    Tix_GrDrawMainBody(wPtr, &mainRI, buffer);
    wPtr->renderInfo = NULL;

    /* free the unwanted colors */
    Tix_GrFreeUnusedColors(wPtr, 0);
}

static void
WidgetDisplay(clientData)
    ClientData clientData;	/* Info about my widget. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;
    Drawable buffer;
    Tk_Window tkwin = wPtr->dispData.tkwin;
    GridElem * chPtr;
    int x, y;
    int winH, winW, expW, expH, hdrW, hdrH;

    if (!Tk_IsMapped(tkwin)) {
	return;
    }

    wPtr->serial ++;

    if (wPtr->redrawBorder && wPtr->highlightWidth>0) {
	wPtr->expArea.x1 = 0;
	wPtr->expArea.y1 = 0;
	wPtr->expArea.x2 = Tk_Width (tkwin) - 1;
	wPtr->expArea.y2 = Tk_Height(tkwin) - 1;
    }

    /* Draw the main body */
    expW = wPtr->expArea.x2 - wPtr->expArea.x1 + 1;
    expH = wPtr->expArea.y2 - wPtr->expArea.y1 + 1;

    if (expW > 0 && expH <= 0) {
	goto done;
    }

    winW = Tk_Width(tkwin)  - 2*wPtr->highlightWidth - 2*wPtr->borderWidth;
    winH = Tk_Height(tkwin) - 2*wPtr->highlightWidth - 2*wPtr->borderWidth;

    buffer = Tix_GetRenderBuffer(wPtr->dispData.display, Tk_WindowId(tkwin),
	expW, expH, Tk_Depth(tkwin));

    XFillRectangle(wPtr->dispData.display, buffer, wPtr->backgroundGC,
	0, 0, expW, expH);

    if (wPtr->mainRB) {
	Tix_GrDisplayMainBody(wPtr, buffer, winW, winH);
    }

    /* Draw the border */
    Tk_Draw3DRectangle(wPtr->dispData.tkwin, buffer, wPtr->border,
	wPtr->highlightWidth - wPtr->expArea.x1,
	wPtr->highlightWidth - wPtr->expArea.y1,
	Tk_Width(tkwin)  - 2*wPtr->highlightWidth, 
	Tk_Height(tkwin) - 2*wPtr->highlightWidth, wPtr->borderWidth,
	wPtr->relief);
  
    /* Draw the highlight */
    if (wPtr->highlightWidth > 0) {
	GC gc;

	if (wPtr->hasFocus) {
	    gc = wPtr->highlightGC;
	} else {
	    gc = Tk_3DBorderGC(wPtr->dispData.tkwin, wPtr->border, 
		TK_3D_FLAT_GC);
	}
	Tk_DrawFocusHighlight(tkwin, gc, wPtr->highlightWidth, buffer);
    }

    /*
     * Copy the information from the off-screen pixmap onto the screen,
     * then delete the pixmap.
     */
    if (buffer != Tk_WindowId(tkwin)) {
	XCopyArea(wPtr->dispData.display, buffer, Tk_WindowId(tkwin),
	    wPtr->backgroundGC, 0, 0, expW, expH,
	    wPtr->expArea.x1, wPtr->expArea.y1);
	XFreePixmap(wPtr->dispData.display, buffer);
    }

  done:
    wPtr->expArea.x1 = 10000;
    wPtr->expArea.y1 = 10000;
    wPtr->expArea.x2 = 0;
    wPtr->expArea.y2 = 0;
}

/*----------------------------------------------------------------------
 * "debuginfo" sub command -- 
 *
 *	Sets the item at the position on the grid. This either creates
 *	a new element or modifies the existing element. (if you don't want
 *	to change the -itemtype of the existing element, it will be more
 *	efficient to call the "itemconfigure" command).
 *----------------------------------------------------------------------
 */
static int
Tix_GrDebugInfo(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;
    size_t len = strlen(argv[0]);
    char c = argv[0][0];
    char buff[100];

#if 0
    if (c == 'd' && strncmp("depth", argv[0], len)==0) {
	sprintf(buff, "%d %d",
	    Tix_GrTreeDepth(wPtr->roots[0].r.rootNode),
	    Tix_GrTreeDepth(wPtr->roots[1].r.rootNode));
	Tcl_AppendResult(interp, buff, NULL);
	return TCL_OK;
    }
    else if (c == 'n' && strncmp("numleaf", argv[0], len)==0) {
	sprintf(buff, "%d %d",
	    wPtr->roots[0].r.rootNode->numLeaf,
	    wPtr->roots[1].r.rootNode->numLeaf);
	Tcl_AppendResult(interp, buff, NULL);
	return TCL_OK;
    } else {
	Tcl_AppendResult(interp, "unknown option \"", argv[0], "\"", NULL);
	return TCL_ERROR;
    }
#else
    return TCL_OK;
#endif

}
/*----------------------------------------------------------------------
 * "set" sub command -- 
 *
 *	Sets the item at the position on the grid. This either creates
 *	a new element or modifies the existing element. (if you don't want
 *	to change the -itemtype of the existing element, it will be more
 *	efficient to call the "itemconfigure" command).
 *----------------------------------------------------------------------
 */
static int
Tix_GrSet(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;
    GridElem * chPtr = NULL;
    Tix_DItem * iPtr;
    int x, y;
    char * ditemType;
    char buff[40];
    int code = TCL_OK;

    /*------------------------------------------------------------
     * (0) We need to find out where you want to set
     *------------------------------------------------------------
     */
    if (Tcl_GetInt(interp, argv[0], &x) != TCL_OK) {
	return TCL_ERROR;
    }
    if (Tcl_GetInt(interp, argv[1], &y) != TCL_OK) {
	return TCL_ERROR;
    }

    /*------------------------------------------------------------
     * (1) We need to determine the option: -itemtype.
     *------------------------------------------------------------
     */
    /* (1.0) Find out the -itemtype, if specified */
    ditemType = wPtr->diTypePtr->name;   		 /* default value */
    if (argc > 2) {
	size_t len;
	int i;
	if (argc %2 != 0) {
	    Tcl_AppendResult(interp, "value for \"", argv[argc-1],
		"\" missing", NULL);
	    code = TCL_ERROR;
	    goto done;
	}
	for (i=2; i<argc; i+=2) {
	    len = strlen(argv[i]);
	    if (strncmp(argv[i], "-itemtype", len) == 0) {
		ditemType = argv[i+1];
	    }
	}
    }

    if (Tix_GetDItemType(interp, ditemType) == NULL) {
	code = TCL_ERROR;
	goto done;
    }

    /*
     * (2) Get this item (a new item will be allocated if it does not exist
     *     yet)
     */
    chPtr = Tix_GrGetItem(interp, wPtr, x, y);

    /* (2.1) The Display item data */
    if ((iPtr = Tix_DItemCreate(&wPtr->dispData, ditemType)) == NULL) {
	code = TCL_ERROR;
	goto done;
    }
    iPtr->base.clientData = (ClientData)wPtr;     /* %%%% */

    if (chPtr->iPtr) {
	Tix_DItemFree(chPtr->iPtr);
    }
    chPtr->iPtr = iPtr;

    if (ConfigElement(wPtr, chPtr, argc-2, argv+2, 0, 1) != TCL_OK) {
	code = TCL_ERROR; goto done;
    }
    Tix_GrPropagateSize(wPtr, chPtr);

  done:
    if (code == TCL_ERROR) {
	/* ? */
    } else {
	sprintf(buff, "@%d,%d", x, y);
	Tcl_AppendResult(interp, buff, NULL);
	Tix_GrDoWhenIdle(wPtr, TIX_GR_RESIZE);
    }

    return code;
}

/*----------------------------------------------------------------------
 * "cget" sub command --
 *----------------------------------------------------------------------
 */
static int
Tix_GrCGet(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;

    return Tk_ConfigureValue(interp, wPtr->dispData.tkwin, configSpecs,
	(char *)wPtr, argv[0], 0);
}

/*----------------------------------------------------------------------
 * "configure" sub command
 *----------------------------------------------------------------------
 */
static int
Tix_GrConfig(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;

    if (argc == 0) {
	return Tk_ConfigureInfo(interp, wPtr->dispData.tkwin, configSpecs,
	    (char *) wPtr, (char *) NULL, 0);
    } else if (argc == 1) {
	return Tk_ConfigureInfo(interp, wPtr->dispData.tkwin, configSpecs,
	    (char *) wPtr, argv[0], 0);
    } else {
	return WidgetConfigure(interp, wPtr, argc, argv,
	    TK_CONFIG_ARGV_ONLY);
    }
}

/*----------------------------------------------------------------------
 * "geometryinfo" sub command
 *----------------------------------------------------------------------
 */
static int
Tix_GrGeometryInfo(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;
    int qSize[2];
    double first[2], last[2];
    char string[80], i;

    if (argc == 2) {
	if (Tcl_GetInt(interp, argv[0], &qSize[0]) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Tcl_GetInt(interp, argv[1], &qSize[1]) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	qSize[0] = Tk_Width (wPtr->dispData.tkwin);
	qSize[1] = Tk_Height(wPtr->dispData.tkwin);
    }
    qSize[0] -= 2*wPtr->borderWidth + 2*wPtr->highlightWidth;
    qSize[1] -= 2*wPtr->borderWidth + 2*wPtr->highlightWidth;

    RecalScrollRegion(wPtr, qSize[0], qSize[1]);

    for (i=0; i<2; i++) {
	qSize[i] -= 2*wPtr->borderWidth + 2*wPtr->highlightWidth;
	GetScrollFractions(wPtr, &wPtr->scrollInfo[i],
	    &first[i], &last[i]);
    }

    sprintf(string, "{%f %f} {%f %f}", first[0], last[0], first[1], last[1]);
    Tcl_AppendResult(interp, string, NULL);

    return TCL_OK;
}

/*----------------------------------------------------------------------
 * "delete" sub command
 *----------------------------------------------------------------------
 */
static int
Tix_GrDelete(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{

  done:
    return TCL_OK;
}

/*----------------------------------------------------------------------
 * "entryconfigure" sub command
 *----------------------------------------------------------------------
 */
static int
Tix_GrEntryConfig(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;
    GridElem * chPtr, * dummy;

#if 0
    if (Tix_GrGetFromTo(interp, wPtr, 1, argv, &chPtr, &dummy)
		!= TCL_OK) {
	return TCL_ERROR;
    }
#endif

    if (chPtr == NULL) {
	Tcl_AppendResult(interp, "list entry \"", argv[1],
	    "\" does not exist", NULL);
	return TCL_ERROR;
    }

    if (argc == 1) {
	return Tix_ConfigureInfo2(interp, wPtr->dispData.tkwin,
	    (char*)chPtr, entryConfigSpecs, chPtr->iPtr, (char *) NULL, 0);
    } else if (argc == 2) {
	return Tix_ConfigureInfo2(interp, wPtr->dispData.tkwin,
	    (char*)chPtr, entryConfigSpecs, chPtr->iPtr, (char *) argv[1], 0);
    } else {
	return ConfigElement(wPtr, chPtr, argc-1, argv+1,
	    TK_CONFIG_ARGV_ONLY, 0);
    }
}

/*----------------------------------------------------------------------
 * "nearest" sub command
 *----------------------------------------------------------------------
 */
static int
Tix_GrNearest(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;
    GridElem * chPtr;
    int posn[2], i, j, curRow;
    int index;
    char buff[100];

    return TCL_OK;
}

/*----------------------------------------------------------------------
 * "anchor", "dragsite" and "dropsire" sub commands --
 *
 *	Set/remove the anchor element
 *----------------------------------------------------------------------
 */
static int
Tix_GrSetSite(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    int changed = 0;
    WidgetPtr wPtr = (WidgetPtr) clientData;
    GridElem * fromPtr;
    GridElem * toPtr;			/* unused */
    GridElem ** changePtr;
    size_t len;

    /* Determine which site should be changed (the last else clause
     * doesn't need to check the string because HandleSubCommand
     * already ensures that only the valid options can be specified.
     **/
    len = strlen(argv[-1]);
    if (strncmp(argv[-1], "anchor", len)==0) {
	changePtr = &wPtr->anchor;
    }
    else if (strncmp(argv[-1], "dragsite", len)==0) {
	changePtr = &wPtr->dragSite;
    }
    else {
	changePtr = &wPtr->dropSite;
    }

    len = strlen(argv[0]);
    if (strncmp(argv[0], "set", len)==0) {
	if (argc == 2) {

	} else {
	    Tcl_AppendResult(interp, "wrong # of arguments, must be: ",
		Tk_PathName(wPtr->dispData.tkwin), " ", argv[-1],
		" set index", NULL);
	    return TCL_ERROR;
	}
    }
    else if (strncmp(argv[0], "clear", len)==0) {
	if (*changePtr != NULL) {
	    *changePtr = NULL;
	    changed = 1;
	}
    }
    else {
	Tcl_AppendResult(interp, "wrong option \"", argv[0], "\", ",
	    "must be clear or set", NULL);
	return TCL_ERROR;
    }

    if (changed) {
	/* %% recal the exposed area here */
	Tix_GrDoWhenIdle(wPtr, TIX_GR_REDRAW);
    }

    return TCL_OK;
}

void Tix_GrScrollPage(wPtr, count, axis)
    WidgetPtr wPtr;
    int count;
    int axis;
{
    int offset, k, i = axis;
    int winSize, sz, start, num;
    int pad0, pad1; 

    Tix_GridScrollInfo * siPtr = &wPtr->scrollInfo[axis];
    int gridSize[2];

    if (count == 0) {
	return;
    }

    Tix_GridDataGetGridSize(wPtr->dataSet, &gridSize[0],
	&gridSize[1]);

    if (gridSize[i] < wPtr->hdrSize[i]) {	 /* no much data to work on */
	return;
    }

    if (axis == 0) {
	winSize = Tk_Width(wPtr->dispData.tkwin);
    } else {
	winSize = Tk_Height(wPtr->dispData.tkwin);
    }
    winSize -= 2*wPtr->highlightWidth + 2*wPtr->borderWidth;

    for (k=0; k<wPtr->hdrSize[i] && k<gridSize[i]; k++) {
	winSize -= Tix_GridDataGetRowColSize(wPtr, wPtr->dataSet, i,
		k, &wPtr->defSize[i], &pad0, &pad1);
	winSize -= pad0 + pad1;
    }

    if (winSize <= 0) {
	return;
    }

    if (count > 0) {
	start = siPtr->offset + wPtr->hdrSize[i];
	for (; count > 0; count--) {
	    sz = winSize;

	    for (num=0,k=start; k<gridSize[i]; k++,num++) {
		sz -= Tix_GridDataGetRowColSize(wPtr, wPtr->dataSet, i,
		    k, &wPtr->defSize[i], &pad0, &pad1);
		sz -= pad0 + pad1;
		if (sz == 0) {
		    num++;
		    break;
		}
		if (sz < 0) {
		    break;
		}
	    }
	    if (num==0) {
		num++;
	    }
	    start += num;
	}
	siPtr->offset = start - wPtr->hdrSize[i];
    }
    else {
	start = siPtr->offset + wPtr->hdrSize[i];

	for (; count < 0; count++) {
	    sz = winSize;

	    for (num=0,k=start-1; k>=wPtr->hdrSize[i]; k--,num++) {
		sz -= Tix_GridDataGetRowColSize(wPtr, wPtr->dataSet, i,
		    k, &wPtr->defSize[i], &pad0, &pad1);
		sz -= pad0 + pad1;
		if (sz == 0) {
		    num++;
		    break;
		}
		if (sz < 0) {
		    break;
		}
	    }
	    if (num==0) {
		num++;
	    }
	    start -= num;
	}
	siPtr->offset = start - wPtr->hdrSize[i];
    }
}

/*----------------------------------------------------------------------
 * "xview" and "yview" sub command
 *----------------------------------------------------------------------
 */
static int
Tix_GrView(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    WidgetPtr wPtr = (WidgetPtr) clientData;
    int axis, oldXOff, oldYOff;
    Tix_GridScrollInfo * siPtr;

    if (argv[-1][0] == 'x') {
	axis = 0;
    } else {
	axis = 1;
    }

    oldXOff = wPtr->scrollInfo[0].offset;
    oldYOff = wPtr->scrollInfo[1].offset;

    if (argc == 0) {
	char string[20];

	sprintf(string, "%f", wPtr->scrollInfo[axis].offset);
	Tcl_AppendResult(interp, string, NULL);
	return TCL_OK;
    }
    else {
	int offset;
	siPtr = &wPtr->scrollInfo[axis];

	if (Tcl_GetInt(interp, argv[0], &offset) == TCL_OK) {
	    /* backward-compatible mode */
	    siPtr->offset = offset;
	} else {
	    int type, count;
	    double fraction;

	    Tcl_ResetResult(interp);

	    /* Tk_GetScrollInfo () wants strange argc,argv combinations .. */
	    type = Tk_GetScrollInfo(interp, argc+2, argv-2, &fraction, &count);

	    switch (type) {
	      case TK_SCROLL_ERROR:
		return TCL_ERROR;

	      case TK_SCROLL_MOVETO:
		if (siPtr->window < 1.0) {
		    fraction /= (1.0 - siPtr->window);
		}

		siPtr->offset = (int)(fraction * (siPtr->max+1));
		break;

	      case TK_SCROLL_PAGES:
		Tix_GrScrollPage(wPtr, count, axis);
		break;

	      case TK_SCROLL_UNITS:
		siPtr->offset += count * siPtr->unit;
		break;
	    }
	}
    }


    UpdateScrollBars(wPtr, 0);

    if (oldXOff != wPtr->scrollInfo[0].offset ||
	oldYOff != wPtr->scrollInfo[1].offset) {
	Tix_GrDoWhenIdle(wPtr, TIX_GR_RESIZE);
    }
    return TCL_OK;
}
/*----------------------------------------------------------------------
 *
 *
 * 			Memory Management Section
 *
 *
 *----------------------------------------------------------------------
 */
static int
ConfigElement(wPtr, chPtr, argc, argv, flags, forced)
    WidgetPtr wPtr;
    GridElem *chPtr;
    int argc;
    char ** argv;
    int flags;
    int forced;
{
    int sizeChanged;

    if (Tix_WidgetConfigure2(wPtr->dispData.interp, wPtr->dispData.tkwin,
	(char*)chPtr, entryConfigSpecs, chPtr->iPtr, argc, argv, flags,
	forced, &sizeChanged) != TCL_OK) {
	return TCL_ERROR;
    }

    if (sizeChanged) {
#if 0
	chPtr->size[0] = chPtr->iPtr->base.size[0];
	chPtr->size[1] = chPtr->iPtr->base.size[1];
#endif
	Tix_GrDoWhenIdle(wPtr, TIX_GR_RESIZE);
    } else {
	Tix_GrDoWhenIdle(wPtr, TIX_GR_REDRAW);
    }
    return TCL_OK;
}

static char * areaNames[4] = {
    "s-margin",
    "x-margin",
    "y-margin",
    "main"
};

static int Tix_GrCallFormatCmd(wPtr, which)
    WidgetPtr wPtr;
    int which;
{
    int size;
    int code;
    char * p;
    char preAlloc[1000];

    size = strlen(wPtr->formatCmd) + 20*4;
    if (size > 1000) {
	p = (char*)ckalloc(size);
    } else {
	p = preAlloc;
    }

    wPtr->renderInfo->fmt.whichArea = which;
    sprintf(p, "%s %s %d %d %d %d", wPtr->formatCmd, areaNames[which],
	wPtr->renderInfo->fmt.x1, 
	wPtr->renderInfo->fmt.y1, 
	wPtr->renderInfo->fmt.x2, 
	wPtr->renderInfo->fmt.y2);
    code = Tcl_GlobalEval(wPtr->dispData.interp, p);

    if (code != TCL_OK) {
	Tcl_AddErrorInfo(wPtr->dispData.interp,
	    "\n    (format command executed by tixGrid)");
	Tk_BackgroundError(wPtr->dispData.interp);
    }

    if (p != preAlloc) {
	free((char*)p);
    }

    return code;
}


static void Tix_GrFormatBackground(wPtr, riPtr, drawable)
    WidgetPtr wPtr;
    RenderInfo * riPtr;
    Drawable drawable;
{
    int x1, y1, x2, y2;

    if (wPtr->formatCmd == NULL) {
	return;
    }

    /* the horizontal margin */
    wPtr->renderInfo->fmt.x1 =
      wPtr->scrollInfo[0].offset + wPtr->hdrSize[0];
    wPtr->renderInfo->fmt.x2 =
      wPtr->scrollInfo[0].offset + wPtr->mainRB->size[0] - 1;
    wPtr->renderInfo->fmt.y1 = 0;
    wPtr->renderInfo->fmt.y2 = wPtr->hdrSize[1] - 1;
    
    Tix_GrCallFormatCmd(wPtr, TIX_X_MARGIN);

    /* the vertical margin */
    wPtr->renderInfo->fmt.x1 = 0;
    wPtr->renderInfo->fmt.x2 = wPtr->hdrSize[0] - 1;
    wPtr->renderInfo->fmt.y1 = 
      wPtr->scrollInfo[1].offset + wPtr->hdrSize[1];
    wPtr->renderInfo->fmt.y2 =
      wPtr->scrollInfo[1].offset + wPtr->mainRB->size[1] - 1;
    
    Tix_GrCallFormatCmd(wPtr, TIX_Y_MARGIN);

    /* the main area */

    wPtr->renderInfo->fmt.x1 =
      wPtr->scrollInfo[0].offset + wPtr->hdrSize[0];
    wPtr->renderInfo->fmt.x2 =
      wPtr->scrollInfo[0].offset + wPtr->mainRB->size[0] - 1;
    wPtr->renderInfo->fmt.y1 =
      wPtr->scrollInfo[1].offset + wPtr->hdrSize[1];
    wPtr->renderInfo->fmt.y2 =
      wPtr->scrollInfo[1].offset + wPtr->mainRB->size[1] - 1;
    
    Tix_GrCallFormatCmd(wPtr, TIX_MAIN);


    /* the stationary part of the margin */
    wPtr->renderInfo->fmt.x1 = 0;
    wPtr->renderInfo->fmt.x2 = wPtr->hdrSize[0] - 1;
    wPtr->renderInfo->fmt.y1 = 0;
    wPtr->renderInfo->fmt.y2 = wPtr->hdrSize[1] - 1;
    
    Tix_GrCallFormatCmd(wPtr, TIX_S_MARGIN);

}

static void Tix_GrComputeSubSelection(wPtr, rect, offs)
    WidgetPtr wPtr;
    int rect[2][2];
    int offs[2];
{
    int iMin, iMax, jMin, jMax;
    Tix_ListIterator li;
    SelectBlock * sbPtr;
    int i, j, x, y;

    Tix_SimpleListIteratorInit(&li);
    for (Tix_SimpleListStart(&wPtr->selList, &li);
	 !Tix_SimpleListDone(&li);
	 Tix_SimpleListNext (&wPtr->selList, &li)) {

	sbPtr = (SelectBlock *)li.curr;
	if (rect[0][0] > sbPtr->range[0][0]) {
	    iMin = rect[0][0];
	} else {
	    iMin = sbPtr->range[0][0];
	}
	if (rect[1][0] > sbPtr->range[1][0]) {
	    jMin = rect[1][0];
	} else {
	    jMin = sbPtr->range[1][0];
	}

	if (rect[0][1] < sbPtr->range[0][1] ||
	    	sbPtr->range[0][1] == TIX_GR_MAX) {
	    iMax = rect[0][1];
	} else {
	    iMax = sbPtr->range[0][1];
	}
	if (rect[1][1] < sbPtr->range[1][1] ||
	    	sbPtr->range[1][1] == TIX_GR_MAX) {
	    jMax = rect[1][1];
	} else {
	    jMax = sbPtr->range[1][1];
	}

	for (x=offs[0],i=iMin; i<=iMax; i++,x++) {
	    for (y=offs[1],j=jMin; j<=jMax; j++,y++) {

		switch (sbPtr->type) {
		  case TIX_GR_CLEAR:
		    wPtr->mainRB->elms[x][y].selected = 0;
		    break;
		  case TIX_GR_SET:
		    wPtr->mainRB->elms[x][y].selected = 1;
		    break;
		  case TIX_GR_TOGGLE:
		    wPtr->mainRB->elms[x][y].selected = 
		      !wPtr->mainRB->elms[x][y].selected;
		    break;

		}
	    }
	}
    }
}

static void Tix_GrComputeSelection(wPtr)
    WidgetPtr wPtr;
{
    int rect[2][2], offs[2];
    int i, j;

    for (i=0; i<wPtr->mainRB->size[0]; i++) {
	for (j=0; j<wPtr->mainRB->size[1]; j++) {
	    wPtr->mainRB->elms[i][j].selected = 0;
	}
    }

    /* the stationary part of the margin */
    rect[0][0] = 0;
    rect[0][1] = wPtr->hdrSize[0] - 1;
    rect[1][0] = 0;
    rect[1][1] = wPtr->hdrSize[1] - 1;
    offs[0]    = 0;
    offs[1]    = 0;

    Tix_GrComputeSubSelection(wPtr, rect, offs);

    /* the horizontal margin */
    rect[0][0] = wPtr->scrollInfo[0].offset + wPtr->hdrSize[0];
    rect[0][1] = wPtr->scrollInfo[0].offset + wPtr->mainRB->size[0] - 1;
    rect[1][0] = 0;
    rect[1][1] = wPtr->hdrSize[1] - 1;
    offs[0]    = wPtr->hdrSize[0];
    offs[1]    = 0;
    
    Tix_GrComputeSubSelection(wPtr, rect, offs);

    /* the vertical margin */
    rect[0][0] = 0;
    rect[0][1] = wPtr->hdrSize[0] - 1;
    rect[1][0] = wPtr->scrollInfo[1].offset + wPtr->hdrSize[1];
    rect[1][1] = wPtr->scrollInfo[1].offset + wPtr->mainRB->size[1] - 1;
    offs[0]    = 0;
    offs[1]    = wPtr->hdrSize[1];

    Tix_GrComputeSubSelection(wPtr, rect, offs);

    /* the main area */

    rect[0][0] = wPtr->scrollInfo[0].offset + wPtr->hdrSize[0];
    rect[0][1] = wPtr->scrollInfo[0].offset + wPtr->mainRB->size[0] - 1;
    rect[1][0] = wPtr->scrollInfo[1].offset + wPtr->hdrSize[1];
    rect[1][1] = wPtr->scrollInfo[1].offset + wPtr->mainRB->size[1] - 1;
    offs[0]    = wPtr->hdrSize[0];
    offs[1]    = wPtr->hdrSize[1];

    Tix_GrComputeSubSelection(wPtr, rect, offs);

    /* force a redraw %% smart guys will compute only the area that
     * has changed */
    wPtr->expArea.x1 = 0;
    wPtr->expArea.y1 = 0;
    wPtr->expArea.x2 = Tk_Width (wPtr->dispData.tkwin) - 1;
    wPtr->expArea.y2 = Tk_Height(wPtr->dispData.tkwin) - 1;
    Tix_GrDoWhenIdle(wPtr, TIX_GR_REDRAW);
}

/*----------------------------------------------------------------------
 * Tix_GrDrawMainBody --
 *
 *	Redraws the main body of the Grid
 *----------------------------------------------------------------------
 */
static void Tix_GrDrawMainBody(wPtr, riPtr, drawable)
    WidgetPtr wPtr;
    RenderInfo * riPtr;
    Drawable drawable;
{
    int x, y, i, j;
    int x1, y1, x2, y2;
    GridElem * chPtr;
    int margin = wPtr->borderWidth + wPtr->highlightWidth;

    fprintf(stderr, "ex1=%d, ex2=%d, ey1=%d, ey2=%d\n",
	wPtr->expArea.x1, wPtr->expArea.x2,
	wPtr->expArea.y1, wPtr->expArea.y2);

    for (i=0; i<wPtr->mainRB->size[0]; i++) {
	for (j=0; j<wPtr->mainRB->size[1]; j++) {
	    wPtr->mainRB->elms[i][j].borderW[0][0] = 0;
	    wPtr->mainRB->elms[i][j].borderW[1][0] = 0;
	    wPtr->mainRB->elms[i][j].borderW[0][1] = 0;
	    wPtr->mainRB->elms[i][j].borderW[1][1] = 0;
	    wPtr->mainRB->elms[i][j].filled        = 0;
	}
    }

    Tix_GrFormatBackground(wPtr, riPtr, drawable);

    for (x=0,i=0; i<wPtr->mainRB->size[0]; i++) {
	x1 = x  + margin;
	x2 = x1 - 1 + wPtr->mainRB->dispSize[0][i].total;

	if (x1 > wPtr->expArea.x2) {
	    goto nextCol;
	}
	if (x2 < wPtr->expArea.x1) {
	    goto nextCol;
	}
	/*
	 * iterate over the columns
	 */
	for (y=0,j=0; j<wPtr->mainRB->size[1]; j++) {
	    /*
	     * iterate over each item in the column, from top
	     * to bottom
	     */
	    y1 = y  + margin;
	    y2 = y1 - 1 + wPtr->mainRB->dispSize[1][j].total;

	    if (y1 > wPtr->expArea.y2) {
		goto nextRow;
	    }
	    if (y2 < wPtr->expArea.y1) {
		goto nextRow;
	    }
	    if (!wPtr->mainRB->elms[i][j].filled) {
		if (wPtr->mainRB->elms[i][j].selected) {

		    Tk_Fill3DRectangle(wPtr->dispData.tkwin,
		    	drawable, wPtr->selectBorder,
			x+riPtr->origin[0]+
			    wPtr->mainRB->elms[i][j].borderW[0][0],
			y+riPtr->origin[1]+
			    wPtr->mainRB->elms[i][j].borderW[1][0],
			wPtr->mainRB->dispSize[0][i].total - 
			    wPtr->mainRB->elms[i][j].borderW[0][0] -
			    wPtr->mainRB->elms[i][j].borderW[0][1],
			wPtr->mainRB->dispSize[1][j].total - 
			    wPtr->mainRB->elms[i][j].borderW[1][0] -
			    wPtr->mainRB->elms[i][j].borderW[1][1],
		    	0, TK_RELIEF_FLAT);
		}
	    }

	    chPtr = wPtr->mainRB->elms[i][j].chPtr;
	    if (chPtr != NULL) {
		if (Tix_DItemType(chPtr->iPtr) == TIX_DITEM_WINDOW) {
		    Tix_DItemDisplay(Tk_WindowId(wPtr->dispData.tkwin), None,
			chPtr->iPtr, x1, y1, 
			wPtr->mainRB->dispSize[0][i].size,
	       		wPtr->mainRB->dispSize[1][j].size,
		        TIX_DITEM_NORMAL_FG);
		} else {
		    int drawX, drawY;
		    drawX = x + riPtr->origin[0] +
		        wPtr->mainRB->dispSize[0][i].preBorder;
		    drawY = y + riPtr->origin[1] +
		        wPtr->mainRB->dispSize[1][j].preBorder;

		    Tix_DItemDisplay(drawable, None, chPtr->iPtr,
		    	drawX, drawY,
			wPtr->mainRB->dispSize[0][i].size,
			wPtr->mainRB->dispSize[1][j].size,
			TIX_DITEM_NORMAL_FG);
		}
#if 0
		fprintf(stderr, "(%d %d) ", j, i);
#endif
	    } else {
#if 0
		fprintf(stderr, "(   ) ");
#endif
	    }
	  nextRow:
	    y+= wPtr->mainRB->dispSize[1][j].total;
	}
      nextCol:
#if 0
	fprintf(stderr, "\n");
#endif
	x+= wPtr->mainRB->dispSize[0][i].total;
    }
 
    for (i=0; i<wPtr->mainRB->size[0]; i++) {
	for (j=0; j<wPtr->mainRB->size[1]; j++) {
	    chPtr = wPtr->mainRB->elms[i][j].chPtr;
	    if (chPtr != NULL) {
		if (Tix_DItemType(chPtr->iPtr) == TIX_DITEM_WINDOW) {

		    Tix_SetWindowItemSerial(&wPtr->mappedWindows,
			chPtr->iPtr, wPtr->serial);
		}
	    }
	}
    }

    /* unmap those windows we mapped the last time */
    Tix_UnmapInvisibleWindowItems(&wPtr->mappedWindows, wPtr->serial);
}


/*----------------------------------------------------------------------
 *  UpdateScrollBars
 *----------------------------------------------------------------------
 */
static void GetScrollFractions(wPtr, siPtr, first_ret, last_ret)
    WidgetPtr wPtr;
    Tix_GridScrollInfo *siPtr;
    double * first_ret;
    double * last_ret;
{
    double first, last;
    double usuable;

    /* check ... */
    if (siPtr->offset < 0) {
	siPtr->offset  = 0;
    }
    if (siPtr->offset > siPtr->max) {
	siPtr->offset = siPtr->max;
    }

    usuable = 1.0 - siPtr->window;
    
    if (siPtr->max > 0) {
	first = usuable * (double)(siPtr->offset) / (double)(siPtr->max);
	last  = first + siPtr->window;
    } else {
	first = 0.0;
	last  = 1.0;
    }

    *first_ret = first;
    *last_ret  = last;
}

static void UpdateScrollBars(wPtr, sizeChanged)
    WidgetPtr wPtr;
    int sizeChanged;
{
    int i;
    Tix_GridScrollInfo *siPtr;
    Tcl_Interp * interp = wPtr->dispData.interp;

    for (i=0; i<2; i++) {
	double first, last;
	double usuable;

	siPtr = &wPtr->scrollInfo[i];

	/* check ... */
	if (siPtr->offset < 0) {
	    siPtr->offset  = 0;
	}
	if (siPtr->offset > siPtr->max) {
	    siPtr->offset = siPtr->max;
	}

	usuable = 1.0 - siPtr->window;

	if (siPtr->max > 0) {
	    first = usuable * (double)(siPtr->offset) / (double)(siPtr->max);
	    last  = first + siPtr->window;
	} else {
	    first = 0.0;
	    last  = 1.0;
	}

	if (siPtr->command) {
	    char buff[60];

	    sprintf(buff, " %f %f", first, last);
	    if (Tcl_VarEval(interp, siPtr->command, buff, 
	        (char *) NULL) != TCL_OK) {
		Tcl_AddErrorInfo(interp,
		    "\n    (scrolling command executed by tixGrid)");
		Tk_BackgroundError(interp);
	    }
	}
    }


    if (wPtr->sizeCmd && sizeChanged) {
	if (Tcl_GlobalEval(wPtr->dispData.interp, wPtr->sizeCmd) != TCL_OK) {
	    Tcl_AddErrorInfo(wPtr->dispData.interp,
		"\n    (size command executed by tixGrid)");
	    Tk_BackgroundError(wPtr->dispData.interp);
	}
    }
}

/* Translate a position token into (x,y) indices. For the time begin,
 * supports only @x,y incides (direct grid address). In the future may
 * support aliasing, etc
 */
static int Tix_GrGetPosn(interp, tag, x_ret, y_ret)
    Tcl_Interp * interp;
    char * tag;
    int * x_ret;
    int * y_ret;
{
    int code = TCL_OK;

    if (*tag == '@') {
	char * ys;
	++tag;
	if ((ys = (char*)strchr(tag, ',')) == NULL) {
	    code = TCL_ERROR;
	    goto done;
	}
	*ys++ = 0;
	if (Tcl_GetInt(interp, tag, x_ret) != TCL_OK) {
	    code = TCL_ERROR;
	    goto done;
	}
	if (Tcl_GetInt(interp, ys, y_ret) != TCL_OK) {
	    code = TCL_ERROR;
	    goto done;
	}

	if (*x_ret < 0 || *y_ret < 0) {
	    Tcl_AppendResult(interp, "grid tag \"", tag,
		"\" out of range. Must be non-negative.", NULL);
	    code = TCL_ERROR;
	} else {
	    /* Change it from 0 based to 1 based */
	    *x_ret;
	    *y_ret;
	}
	goto done;
    } else {
	Tcl_AppendResult(interp, "unknown grid tag \"", tag, "\"", NULL);
	code = TCL_ERROR;
    }

  done:
    return code;
}

/*----------------------------------------------------------------------
 * Tix_GrGetItem --
 *
 *	Returns the element. If it doesn't exist, create a new one
 *	and return it.
 *----------------------------------------------------------------------
 */
static GridElem *
Tix_GrGetItem(interp, wPtr, x, y)
    Tcl_Interp * interp;
    WidgetPtr wPtr;
    int x;
    int y;
{

    static GridElem * defaultEntry = NULL;
    GridElem * chPtr;

    if (defaultEntry == NULL) {
	defaultEntry = (GridElem*)ckalloc(sizeof(GridElem));
	defaultEntry->iPtr     = NULL;
    }

    chPtr = (GridElem*)TixGridDataCreateEntry(wPtr->dataSet, x, y,
	(ClientData)defaultEntry);

    if (chPtr == defaultEntry) {
	defaultEntry = NULL;
    }

    return chPtr;
}

static void
Tix_GrPropagateSize(wPtr, chPtr)
    WidgetPtr wPtr;
    GridElem * chPtr;
{
#if 0
    int i;

    for (i=0; i<2; i++) {
	TreeListRoot * rPtr;
	GridHeader * hdr;

	rPtr = chPtr->nodes[i].root;
	hdr = (GridHeader*) rPtr->data;

	if (hdr->size < chPtr->size[i]) {
	    hdr->size = chPtr->size[i];
	    hdr->recalSize = 0;
	}
    }
#endif
}

static RenderBlock * Tix_GrAllocateRenderBlock(wPtr, winW, winH, exactW,exactH)
    WidgetPtr wPtr;
    int winW;
    int winH;
    int * exactW;
    int * exactH;
{
    RenderBlock * rbPtr;
    int i, j, k;
    int hdrSize[2];
    int offset[2];		/* how much the entries were scrolled */
    int winSize[2];
    int exactSize[2];		/* BOOL: are all the visible coloums and rows
				 * displayed in whole */
    int pad0, pad1; 
    hdrSize[0] = 1;
    hdrSize[1] = 1;

    offset[0] = wPtr->scrollInfo[0].offset + hdrSize[0];
    offset[1] = wPtr->scrollInfo[1].offset + hdrSize[1];

    winSize[0] = winW;
    winSize[1] = winH;

    rbPtr = (RenderBlock*)ckalloc(sizeof(RenderBlock));

    rbPtr->size[0]=0;
    rbPtr->size[1]=0;

    /* (1) find out the size requirement of each row and column.
     *     The results are stored in rbPtr->size[i] and
     *     rbPtr->dispSize[i][0 .. (rbPtr->size[i]-1)]
     */
    for (i=0; i<2; i++) {
	/* i=0 : handle the column sizes;
	 * i=1 : handle the row    sizes;
	 */
	int index;
	int pixelSize = 0;

	/* The margins */
	for (index=0; index<hdrSize[i] && pixelSize<winSize[i]; index++) {
	    pixelSize += Tix_GridDataGetRowColSize(wPtr, wPtr->dataSet, i,
		index, &wPtr->defSize[i], &pad0, &pad1);
	    pixelSize += pad0 + pad1;
	    rbPtr->size[i] ++;
	}

	for (index=offset[i]; pixelSize<winSize[i]; index++) {
	    pixelSize += Tix_GridDataGetRowColSize(wPtr, wPtr->dataSet, i,
		index, &wPtr->defSize[i], &pad0, &pad1);
	    pixelSize += pad0 + pad1;
	    rbPtr->size[i] ++;
	}
	if (pixelSize == winSize[i]) {
	    exactSize[i] = 1;
	} else {
	    exactSize[i] = 0;
	}
    }

    /* return values */

    *exactW = exactSize[0];
    *exactH = exactSize[1];

    rbPtr->dispSize[0] = (ElmDispSize*)
	ckalloc(sizeof(ElmDispSize)*rbPtr->size[0]);
    rbPtr->dispSize[1] = (ElmDispSize*)
	ckalloc(sizeof(ElmDispSize)*rbPtr->size[1]);

    /* (2) fill the size info of all the visible rows and cols into
     *     the dispSize arrays;
     */

    for (i=0; i<2; i++) {
	/* i=0 : handle the column sizes;
	 * i=1 : handle the row    sizes;
	 */
	int index;

	for (k=0; k<rbPtr->size[i]; k++) {
	    if (k < hdrSize[i]) {			/* The margins */
		index = k;
	    } else {
		index = k + offset[i] - hdrSize[i];
	    }

	    rbPtr->dispSize[i][k].size = Tix_GridDataGetRowColSize(wPtr, 
		wPtr->dataSet, i, index, &wPtr->defSize[i], &pad0, &pad1);
	    rbPtr->dispSize[i][k].preBorder  = pad0;
	    rbPtr->dispSize[i][k].postBorder = pad1;
	}
    }

    /* (3) Put the visible elements into the render block array,
     *     rbPtr->elms[*][*].
     */
    rbPtr->elms = (RenderBlockElem**)
        ckalloc(sizeof(RenderBlockElem*)*rbPtr->size[0]);

    for (i=0; i<rbPtr->size[0]; i++) {
	rbPtr->elms[i] = (RenderBlockElem*)
	    ckalloc(sizeof(RenderBlockElem) * rbPtr->size[1]);
	for (j=0; j<rbPtr->size[1]; j++) {
	    rbPtr->elms[i][j].chPtr = NULL;
	    rbPtr->elms[i][j].selected = 0;
	}
    }

    for (i=0; i<rbPtr->size[0]; i++) {
	for (j=0; j<rbPtr->size[1]; j++) {
	    int x, y;

	    if (i<hdrSize[0]) {
		x = i;
	    } else {
		x = i+offset[0]-hdrSize[0];
	    }

	    if (j<hdrSize[1]) {
		y = j;
	    } else {
		y = j+offset[1]-hdrSize[1];
	    }

	    rbPtr->elms[i][j].chPtr = (GridElem*) TixGridDataFindEntry(
		wPtr->dataSet, x, y);
	}
    }

#if 1
    fprintf(stderr, "%d %d\n", rbPtr->size[0], rbPtr->size[1]);
#endif

    for (k=0; k<2; k++) {
	for (i=0; i<rbPtr->size[k]; i++) {
	    rbPtr->dispSize[k][i].total = 
	        + rbPtr->dispSize[k][i].preBorder
	        + rbPtr->dispSize[k][i].size
	        + rbPtr->dispSize[k][i].postBorder;
#if 1
	    fprintf(stderr, "%d ", rbPtr->dispSize[k][i].preBorder);
	    fprintf(stderr, "%d ", rbPtr->dispSize[k][i].size);
	    fprintf(stderr, "%d ", rbPtr->dispSize[k][i].postBorder);
#endif	

	}
#if 1
	fprintf(stderr, "\n");
#endif
    }

    return rbPtr;
}

static void Tix_GrFreeRenderBlock(wPtr, rbPtr)
    WidgetPtr wPtr;
    RenderBlock * rbPtr;
{
    int i;

    for (i=0; i<rbPtr->size[0]; i++) {
	ckfree((char*)rbPtr->elms[i]);
    }
    ckfree((char*)rbPtr->elms);
    ckfree((char*)rbPtr->dispSize[0]);
    ckfree((char*)rbPtr->dispSize[1]);
    ckfree((char*)rbPtr);
}
