###################################################################
# CompressMotionEvents
###################################################################
proc CompressMotionEvents {} {
    global ReplayData
    MsgToUser "Compressing events in script.  Please wait ..."
    set newScript {}
    set lastItem ""
    set lastEvent NotMotion
    foreach thisItem $ReplayData(Script) {
        set thisEvent NotMotion
        if {[regexp {Bind,([^,]*),(.*)} \
                [lindex $thisItem 2] what which event]} {
            if {[string first Motion $event] >= 0} {
                set thisEvent Motion
            }
        }
        if {($lastItem!="") && \
                (($lastEvent!="Motion") || ($thisEvent!="Motion"))} {
            lappend newScript $lastItem
        }
        set lastItem $thisItem
        set lastEvent $thisEvent
    }
    lappend newScript $lastItem
    # reconstruct the list of events
    EraseScript all
    foreach item $newScript {
        eval InsertAction $item
    }
    set ReplayData(ScriptIndex) 0
    SelectActionListItem 0 see
    MsgToUser "Motion events have been compressed."
}
###################################################################
# InsertAction: insert an action in the replay script.
#     delay: the time between this event and the last event
#     app: the application that the action occurred in
#     subscript: the subscript of the event which records everything
#         we need to know about it, in particular, where to find
#         the script of the original binding
#     replaceList: the details of the event so we can replay it exactly
#          as it happened.
###################################################################
proc InsertAction {delay app subscript replaceList} {
    global ReplayData
    set item [list $delay $app $subscript $replaceList]
    set index $ReplayData(ScriptIndex)
    set ReplayData(Script) [linsert $ReplayData(Script) $index $item]
    if $ReplayData(showEventsRecording) {
        .replay.events.list insert $index \
            [format {%4.1f %s %s} [expr $delay/10.0] $subscript \
                [rrlookup W $replaceList]]
    }
    incr ReplayData(ScriptIndex)
    SelectActionListItem $ReplayData(ScriptIndex) see
}
###################################################################
# MoveEvent: move an event around in the event list.
#   First it does some checking to prevent removing of the end markers.
#   Then it does the move by deleting the event and inserting it
#   somewhere else.
###################################################################
proc MoveEvent {places} {
    global ReplayData
    set index $ReplayData(ScriptIndex)
    set newPlace [expr $index + $places]
    set max [expr [llength $ReplayData(Script)] - 2]
    if {$index == 0} {
        MsgToUser "You cannot move the beginning-of-script marker" low
        return
    }
    if {$newPlace < 1} {
        MsgToUser "You cannot move anything before the beginning-of-script\
            marker" low
        return
    }
    if {$index > $max} {
        MsgToUser "You cannot move the end-of-script marker" low
        return
    }
    if {$newPlace > $max} {
        MsgToUser "You cannot move anything after the end-of-script\
            marker" low
        return
    }
    set action [lindex $ReplayData(Script) $index]
    DeleteEvent
    incr ReplayData(ScriptIndex) $places
    eval InsertAction $action
    incr ReplayData(ScriptIndex) -1
    SelectActionListItem $ReplayData(ScriptIndex) see
    MsgToUser "One script action was moved" info
}
###################################################################
# DeleteEvent: removes an event from the event list.
###################################################################
proc DeleteEvent {} {
    global ReplayData
    set index $ReplayData(ScriptIndex)
    set max [expr [llength $ReplayData(Script)] - 2]
    if {$index == 0} {
        MsgToUser "You cannot delete the beginning-of-script marker" low
        return
    }
    if {$index > $max} {
        MsgToUser "You cannot delete the end-of-script marker" low
        return
    }
    set ReplayData(Script) [lreplace $ReplayData(Script) $index $index]
    .replay.events.list delete $index
    SelectActionListItem $ReplayData(ScriptIndex) see
    MsgToUser "One script action was deleted" info
}
###################################################################
# EditEvent: allows you to change the delay of an event.
###################################################################
proc EditEvent {} {
    global ReplayData
    set w .editBox
    if [winfo exists $w] {
        wm deiconify $w
    } else {
        toplevel $w
        wm title $w "Edit event delay"
        label $w.l1 -text ""
        pack $w.l1 -side top
        frame $w.fr
        pack $w.fr -side top
        label $w.fr.label -text "New delay:"
        pack $w.fr.label -side left
        entry $w.fr.entry -bd 2 -relief sunken
        pack $w.fr.entry -side left -fill x -expand yes
        bind $w.fr.entry <Return> "ChangeDelay $w"
        button $w.exit -text "Dismiss" -command "wm withdraw $w"
        pack $w.exit -side top
    }
    set index $ReplayData(ScriptIndex)
    set action [lindex $ReplayData(Script) $index]
    set delay [lindex $action 0]
    $w.l1 configure -text [format {Current delay for this event is %.1f.
Type in a new value and press Return.} [expr $delay/10.0]]
}
###################################################################
# ChangeDelay -- callback for EditEvent.
###################################################################
proc ChangeDelay {w} {
    global ReplayData
    set index $ReplayData(ScriptIndex)
    set action [lindex $ReplayData(Script) $index]
    set newDelay [expr int([$w.fr.entry get] * 10)]
    set newAction [lreplace $action 0 0 $newDelay]
    DeleteEvent
    InsertAction $newDelay [lindex $action 1] [lindex $action 2] \
        [lindex $action 3]
    wm withdraw $w
    MsgToUser "Delay was changed" info
}
###################################################################
# Rewind: makes the first event the current event.
###################################################################
proc Rewind {} {
    global ReplayData
    set ReplayData(ScriptIndex) 0
    .replay.events.list yview 0
    SelectActionListItem 0 see
    MsgToUser "Script rewound to beginning" info
}
###################################################################
# AddPause: adds a Pause event.
###################################################################
proc AddPause {} {
    global ReplayData
    InsertAction 0 ThisApp Pause ""
}
###################################################################
# EraseScript: erases all the events in the event list.
###################################################################
proc EraseScript {{how inside}} {
    global ReplayData
    MsgToUser "Erase the entire script"
    set ReplayData(Script) {}
    .replay.events.list delete 0 end
    if {$how == "inside"} {
        set ReplayData(ScriptIndex) 0
        InsertAction 0 ThisApp {*** End of script ***} ""
        set ReplayData(ScriptIndex) 0
        InsertAction 0 ThisApp {*** Beginning of script ***} ""
    }
}
###################################################################
# StartRecording
###################################################################
proc StartRecording {} {
    global ReplayData
    MsgToUser "Start recording a script"
    set ReplayData(RecordingOn) 1
    set ReplayData(LastEventAt) 0
    # turn on the timer
    set ReplayData(Timer) 0
    ReplayTimerTick
}
###################################################################
# StopRecording
###################################################################
proc StopRecording {} {
    global ReplayData
    MsgToUser "Stop recording a script"
    set ReplayData(RecordingOn) 0
    set ReplayData(PlayingOn) 0
    # turn off the timer
    set ReplayData(Timer) -1
}
###################################################################
# Replay
###################################################################
proc Replay {} {
    global ReplayData
    MsgToUser "Start replaying the script"
    set ReplayData(PlayingOn) 1
    set ReplayData(UseDelays) 1
    ReplayActions
    set ReplayData(PlayingOn) 0
    MsgToUser "Finished replaying the script"
}
###################################################################
# ReplayFast: replays the events with no time delays.
###################################################################
proc ReplayFast {} {
    global ReplayData
    MsgToUser "Start replaying a script (no delays)"
    set ReplayData(PlayingOn) 1
    set ReplayData(UseDelays) 0
    ReplayActions
    set ReplayData(PlayingOn) 0
    MsgToUser "Finished replaying the script"
}
###################################################################
# Save: saves the event list in the file.
###################################################################
proc Save {} {
    global ReplayData
    MsgToUser "Saving file. Please wait ..." info
    WriteFile $ReplayData(ScriptFileName)
}
###################################################################
# WriteFile: writes the script list to a file.
###################################################################
proc WriteFile {filename} {
    global ReplayData
    set fid [open $filename w]
    # remove the dummy begin and end events
    set rdScript $ReplayData(Script)
    set script [lrange $rdScript 1 [expr [llength $rdScript]-2]]
    foreach event $script {
        puts $fid [format "InsertAction {%s} {%s} {%s} {%s}" \
            [lindex $event 0] [lindex $event 1] \
            [lindex $event 2] [lindex $event 3]]
    }
    close $fid
    MsgToUser "Saved script to \"$filename\""
}
###################################################################
# SaveAs: saves the event list in a new file.
###################################################################
proc SaveAs {} {
    global ReplayData
    MsgToUser "Select a name to save the script to"
    set filename [FSBox "Select a name to save the script to"]
    if {$filename == ""} {
        MsgToUser "Save cancelled"
       return
    }
    if [file exists $filename] {
        set ret [tk_dialog .askwrite "Overwrite file?" \
               "The file $filename already exists, overwrite it?" \
               "" 0 "Overwrite file" "Cancel save"]
       if {$ret == 1} {
            MsgToUser "Save cancelled"
            return
       }
    }
    set ReplayData(ScriptFileName) $filename
    SetScriptFileName
    MsgToUser "Saving file as $filename. Please wait ..." info
    WriteFile $filename
}
###################################################################
# Load: inserts a file into the event list.
###################################################################
proc Load {} {
    global ReplayData
    MsgToUser "Select a file to load the script from"
    set filename [FSBox "Select a file to load the script from"]
    if {$filename == ""} {
       MsgToUser "Load cancelled"
       return
    }
    LoadScript $filename
}
###################################################################
# LoadScript
###################################################################
proc LoadScript {filename} {
    global ReplayData
    MsgToUser "Loading script \"$filename\".  Please wait ..."
    if {[file readable $filename]} {
        source $filename
        Rewind
        set ReplayData(ScriptFileName) $filename
        SetScriptFileName
        MsgToUser "Loaded script \"$filename\"" info
        return 1
    } else {
        MsgToUser "Cannot read \"$filename\", not loaded" medium
        return 0
    }
    Rewind
}
###################################################################
# MakeLoadAppBox: request and application name to load and connect to.
###################################################################
proc MakeLoadAppBox {} {
    global ReplayData
    MsgToUser "Select an application to load"
    set scriptName [FSBox "Select an application to load"]
    if {$scriptName == ""} {
       MsgToUser "Load app was cancelled"
       return
    }
    LoadAndConnect $scriptName
}
###################################################################
# LoadAndConnect: load application and then connect to it.
###################################################################
proc LoadAndConnect {scriptName} {
    MsgToUser "Load and connect to \"$scriptName\""
    set appCmd [list wish $scriptName]
    if {![LoadApp $appCmd]} {
        # if load fails do not try to connect
        return
    }
    # wait 2 seconds for it start up
    after 2000
    ConnectToApp $scriptName
}
###################################################################
# LoadApp: load an application and connect to it.
###################################################################
proc LoadApp {appCmd} {
    global ReplayData
    if {$ReplayData(RecordingOn)} {
        InsertAction 0 ThisApp LoadApp $appCmd
    }
    if {[catch [list eval exec $appCmd &] ret]} {
        MsgToUser "Load of \"$appCmd\" failed" low
        return 0
    } else {
        MsgToUser "Load of \"$appCmd\" succeeded"
        return 1
    }
}
