###################################################################
# This file is sourced by the application we are recording.
###################################################################
#
# if we are already connected then skip everything
#
global db__rd
if {[info exists db__rd(RebindingComplete)]} {
   RemoteMsgToUser "Already bound to $db__rd(ThisApp)" low
} else {
#
# List of widget commands.  Add you widget to this list if you want to
#    be able to record its events
#
set WidgetCreateCommands {button canvas checkbutton entry frame label \
    listbox menubutton menu message radiobutton scale scrollbar text \
    toplevel}
set db__rd(Current) ""
set db__rd(Replaying) 0
#
# NOTE: we must rename "toplevel" with another name beginning with "t"
# or else the call will be interpreted as a call to "frame"
#
#
# Rename all the commands we need to intercept, that is, bind and all
#   of the widget creation commands.  This way we will know when a new
#   widget is created and when a binding is changed in an existing widget.
#
rename bind bind__rd
#
foreach cmd $WidgetCreateCommands {
    rename $cmd [format {%s__rd} $cmd]
}
#
# Redefine all the commands we just renamed.
#
# redefine all the widget creation commands -- each one will create the
#   widget and then call our proc to redefine the bindings of the new
#   widget.
#
foreach cmd $WidgetCreateCommands {
    eval [format {
        proc %s {w args} {
            global db__rd
            # create the object first
            uplevel 1 %s__rd $w $args
            RebindOneWidget $w
        }
    } $cmd $cmd]
}
#
###################################################################
# bind -- when an event is bound to a tcl script we change it to
#   a call to our own callback which then calls the tcl script.
###################################################################
proc bind {tag args} {
    global db__rd
    switch [llength $args] \
    0 {
        # Return all the events that are bound.
        # Let the real bind do that.
        return [uplevel 1 bind__rd $tag]
    } \
    1 {
        # return the binding for this event ($args = $event)
        set index [format {Bind,%s,%s} $tag $args]
        if [info exists db__rd($index)] {
            return $db__rd($index)
        } else {
            return ""
        }
    } \
    2 {
        # bind to an event
        set event [FixEvent [lindex $args 0]]
        set binding [lindex $args 1]
        RebindOneEvent $tag $event $binding
    } \
    default {
        puts "ERROR: wrong number of args: bind $w $args"
    }
}
###################################################################
# FixEvent: put an event name into a standard form.  We let Tk do the
#    conversion to its standard form by binding a dummy widget.
#
#   We need to do this because some events can be specified in several
#   different ways.  For example <1>, <Button-1> and <ButtonPress-1>
#   are all ways to specify the same events.  The standard form
#   (that Tk returns) is <Button-1>.  Since we store event bindings
#   by the name of the event we have to use the same event name all
#   the time.
###################################################################
proc FixEvent {ev1} {
    label .dummy__rd
    bind__rd .dummy__rd $ev1 {#}
    set ev2 [bind__rd .dummy__rd]
    if [catch "destroy .dummy__rd" ret] {
        puts "destroy .dummy__rd failed: $ret"
    }
    return $ev2
}
###################################################################
# RebindWidgetAndChildren: rebind one widget and then all its children.
#   This proc is called with "." when we connect (the call is at the end
#   of this file) to rebind all the existing widgets.  Our control
#   of the bind command and the widget creation command ensures that
#   we redefine all new bindings that are created after the call
#   to "RebindWidgetAndChildren ."
###################################################################
proc RebindWidgetAndChildren {w} {
    global db__rd
    RebindOneWidget $w
    foreach child [winfo children $w] {
        RebindWidgetAndChildren $child
    }
}
###################################################################
# RebindOneWidget: we want to redefine all new bindings.  Bindings are
#   not attached directly to widgets but to tags which are then attached
#   to widgets.  So we go through all the tags associated with this widget.
#   For each event bound to that tag we define the binding with
#   RebindOneEvent.
#
#   We check for tags we have already redefined and do
#   not do them over.  This is mainly be class tags (Button, Text, etc.)
#
#   We also all a CatchClicks tag to the bindtag list for each widget.
#   This allows us to get control back during replay.
###################################################################
proc RebindOneWidget {w} {
    global db__rd
    #
    # Handle the special cases first
    #
    HandleSpecialCases $w
    #
    # find all tags that must be rebound for this widget
    #
    set bt [bindtags $w]
    foreach tag $bt {
        #
        # rebind each event defined for this class or widget
        #
        foreach event [bind__rd $tag] {
            set binding [bind__rd $tag $event]
            RebindOneEvent $tag $event $binding
        }
    }
    # so we can stop replays with a mouse button click
    bindtags $w [linsert $bt 0 CatchClicks]
}
###################################################################
# RebindOneEvent:  we define one event of one tag.
#   We save the actual tcl script that the event is bound to in th
#   db__rd table (tcl associative array).
#
#   We scan the tcl script to find all the
#   %-field tags it uses.  We do this because our replacement script
#   has to capture all those percent fields so we can insert them
#   into the actual tcl script when we are recording and when we are
#   replaying.  We make sure to always capture the x and y (mouse
#   position) fields so we can move the mouse smoothly during replay.
#
#   The rebinding is done by calling the real Tk bind command, bind__rd.
###################################################################
proc RebindOneEvent {tag event binding} {
    global db__rd
    #
    # do not remap again if it has already been remapped
    #
    if {[string compare cb__rd [lindex $binding 0]] != 0} {
        set db__rd(Bind,$tag,$event) $binding
        #
        # find out what %-fields it requires
        # (GetPercentFields will always include %W because we use it)
        #
        set percentFields [GetPercentFields $binding]
        # make sure we pick up the mouse coordinates
        lappend percentFields [list x {%x}] [list y {%y}]
        # do the rebinding
        bind__rd $tag $event \
            [format {cb__rd {%s} {%s} {%s}} $tag $event $percentFields]
    }
}
###################################################################
# RedefineTextCommand: redefine the widget command for an individual
#   text widget.  We must do this so we can detect changes in the
#   internal text widget bindings after the text widget is created.
#
#   We check each subcommand to see if it is the
#   "tag bind $tag $sequence $binding"
#   subcommand.  We handle that one ourselves and all the other subcommands
#   are passed to the real text widget command.
#
#   We create the "proc" command with a format so we can insert the
#   right things but keep everything else from being evaluated too
#   early.  Then we eval the proc command at the global level.
###################################################################
proc RedefineTextCommand {w} {
    global db__rd
    # rename the widget command
    set neww [format {%s__rd} $w]
    rename $w $neww
    set cmd [format {
        proc %s {subcmd args} {
            if {([string compare $subcmd "tag"]==0)
             && ([string compare [lindex $args 0] "bind"]==0)
             && ([llength $args]==4)} {
                set tag [lindex $args 1]
                set event [FixEvent [lindex $args 2]]
                set binding [lindex $args 3]
                RebindTextBinding %s $tag $event $binding
            } else {
                return [eval %s $subcmd $args]
            }
        }
    } $w $w $neww]
    uplevel #0 $cmd
}
###################################################################
# RedefineCanvasCommand: redefine the widget command for an individual
#   canvas widget.  We must do this so we can detect changes in the
#   internal canvas widget bindings after the canvas widget is created.
#
#   We check each subcommand to see if it is the
#   "bind $tagOrId $sequence $binding"
#   subcommand.  We handle that one ourselves and all the other subcommands
#   are passed to the real text widget command.
#
#   We create the "proc" command with a format so we can insert the
#   right things but keep everything else from being evaluated too
#   early.  Then we eval the proc command at the global level.
###################################################################
proc RedefineCanvasCommand {w} {
    global db__rd
    # rename the widget command
    set neww [format {%s__rd} $w]
    rename $w $neww
    set cmd [format {
        proc %s {subcmd args} {
            global db__rd
            set current $db__rd(Current)
            if {$db__rd(Replaying) && ($current != "")} {
                set newArgs {}
                foreach arg $args {
                    if {[string compare $arg "current"] == 0} {
                        lappend newArgs $current
                    } else {
                        lappend newArgs $arg
                    }
                }
                set args $newArgs
            }
            if {([string compare $subcmd "bind"]==0)
             && ([llength $args]==3)} {
                set tag [lindex $args 0]
                set event [FixEvent [lindex $args 1]]
                set binding [lindex $args 2]
                RebindCanvasBinding %s $tag $event $binding
            } else {
                return [eval %s $subcmd $args]
            }
        }
    } $w $w $neww]
    uplevel #0 $cmd
}
###################################################################
# RebindTextBinding: redefine a text tag binding.
#   First we check to see if the binding has already be redefined.
#   If so we do not redefine it again.
#
#   We redefine it in the usual way.  Tag binding also use %-fields
#   so we capture those and make sure that W, x, and y are captured.
#   We also make sure the percent fields identify this as a Text
#   widget binding.
###################################################################
proc RebindTextBinding {w tag event binding} {
    global db__rd
    if {[string compare "cb__rd" [lindex $binding 0]] != 0} {
        set longTag [format {Text,%s,%s} $w $tag]
        set db__rd(Bind,$longTag,$event) $binding
        set alist [GetPercentFields $binding]
        set alist [lreplace $alist 0 0 \
            [list W {%W}] [list Widget Text] \
            [list x {%x}] [list y {%y}]]
        set cb [format {cb__rd {%s} {%s} {%s}} $longTag $event $alist]
        eval [list ${w}__rd tag bind $tag $event $cb]
    }
}
###################################################################
# RebindCanvasBinding: redefine a canvas binding.
#   First we check to see if the binding has already be redefined.
#   If so we do not redefine it again.
#
#   We redefine it in the usual way.  Canvas binding also use %-fields
#   so we capture those and make sure that W, x, and y are captured.
#   We also make sure the percent fields identify this as a Canvas
#   widget binding.
###################################################################
proc RebindCanvasBinding {w tag event binding} {
    global db__rd
    if {[string compare "cb__rd" [lindex $binding 0]] != 0} {
        set longTag [format {Canvas,%s,%s} $w $tag]
        set db__rd(Bind,$longTag,$event) $binding
        set alist [GetPercentFields $binding]
        set alist [lreplace $alist 0 0 \
            [list W {%W}] [list Widget Canvas] \
            [list x {%x}] [list y {%y}]]
        set cb [format {cb__rd {%s} {%s} {%s}} $longTag $event $alist]
        eval [list ${w}__rd bind $tag $event $cb]
    }
}
###################################################################
# HandleSpecialCases: handle the widgets that need special processing.
#   These are widgets that have their own internal binding system.
#   
#   For Text widgets, we redefine the command itself and then go through
#   all the tags and all events bound to that tag and redefine the
#   binding.
#
#   For Canvas widgets we have to do a ltitle more work.  There is no
#   easy way to get all the existing tags but we can go through each
#   item on the canvas.  So, first, we go through each item and do two
#   things with each item.  First we find its tags and, if we haven't
#   seen the tag before we add it to our list of all tags on the canvas.
#   Second we rebind the bindings associated with that item.
#
#   Now we have a list of all the tags of the canvas.  (Note: the
#   RemoveDuplicates is not really necessary since we did it as we
#   collected the tags).  For each tag we redefine its bindings.
###################################################################
proc HandleSpecialCases {w} {
    global db__rd
    set class [winfo class $w]
    switch $class \
        "Text" {
            RedefineTextCommand $w
            foreach tag [$w tag names] {
                foreach event [$w tag bind $tag] {
                    set binding [$w tag bind $tag $event]
                    RebindTextBinding $w $tag $event $binding
                }
            }
        } \
        "Canvas" {
            RedefineCanvasCommand $w
            # go through each item on the canvas
            set tags {}
            foreach item [$w find all] {
                foreach tag [$w gettags $item] {
                    if {[lsearch $tags $tag] < 0} {
                        lappend tags $tag
                    }
                }
                # go through each event bound for that item
                foreach event [$w bind $item] {
                    set binding [$w bind $item $event]
                    RebindCanvasBinding $w $item $event $binding
                }
            }
            foreach tag [RemoveDuplicates $tags] {
                foreach event [$w bind $tag] {
                    set binding [$w bind $tag $event]
                    RebindCanvasBinding $w $tag $event $binding
                }
            }
        }
}
###################################################################
# GetPercentFields: this proc scans a tcl script and finds all the
#   strings of the form "%?" where "?" is some letter.  These are
#   the "percent fields" that need to be replaced with fields from
#   the X event structure before the binding is evaluated.
#
#   The return value is a list of pairs where each pair has the form:
#   {W %W}, {$x x}, etc.  This makes an association list that is easy
#   for the code that does the replacements to use.
###################################################################
proc GetPercentFields {action} {
    set fields [list [list W {%W}]]
    set len [string length $action]
    #
    # go through each character of the script looking for %s
    #
    for {set i 0} {$i < $len} {incr i} {
        set ch [string index $action $i]
        if {[string compare $ch "%"]==0} {
            # we found a %, see what the letter is
            incr i
            set ch [string index $action $i]
            # skip %% which is an escaped %
            if {[string compare $ch "%"]!=0} {
                # we found a %? field, see if we already have it
                if {[lookup $ch $fields]==""} {
                    set x [format "%s {%%%s}" $ch $ch]
                    lappend fields $x
                }
            }
        }
    }
    return $fields
}
###################################################################
# RemoveDuplicates: converts a list with duplicates to a list without
#   duplicates.
###################################################################
proc RemoveDuplicates {bag} {
    set nodups {}
    set index 1
    foreach item $bag {
        set nextone [lsearch -exact [lrange $bag $index end] $item]
        if {$nextone < 0} {
            lappend nodups $item
        }
        incr index
    }
    return $nodups
}
###################################################################
# cb__rd: general callback for all bindings.
#
#   First it makes some adjustments to the %-field replacement list.
#
#   Second it sends a message to TkReplay to record the event.
#   TkReplay will ignore the event if it is not in recording mode.
#
#   Third, it executes the action using DoAction.
###################################################################
proc cb__rd {tag event percentFields args} {
    global db__rd
    # only do the callback when we are recording
    if $db__rd(Replaying) {
        return
    }
    #
    # add some percent fields that will be needed later
    #
    lappend percentFields [list Args "$args"]
    set w [lookup W $percentFields]
    if {[winfo class $w] == "Canvas"} {
        lappend percentFields [list Widget Canvas] \
            [list Current [$w find withtag current]]
    }
    #
    # record the event with TkReplay
    #
    set cmd [format {RecordAction {%s} {%s} {%s} {%s}} \
        $db__rd(ThisApp) $tag $event $percentFields]
    # use catch in case TkReplay has exited
    catch [list send $db__rd(ReplayApp) $cmd]
    #
    # Execute the real binding
    #
    return [DoAction $db__rd(Bind,$tag,$event) $percentFields]
}
###################################################################
# ReplayAction: this is a little complicated because of the possibility
#   that a bindings might not return for a long time.  For example, it
#   might have a tkwait in it.  So we execute the binding with an
#   "after 1" so this procedure will not be delayed indefinitely.
#   This proc must return because it is called by a "send" and if it
#   does not return it will hang the "send".  On the other hand,
#   we do not want to return immediately because the next event might
#   assume that this event has completed.  So we delay a maximum
#   timeout period or until the binding completes, whichever is sooner.
#   This is accomplished by setting up an "after" timeout to send a
#   completion message after the timeout period.  The DoActionAndReply
#   procedure will call the action and then send a reply message.
#   The record and replay application will always get two replies but it
#   will act on the one it gets first and ignore the other one.  This
#   requires each action to have a unique id (which is assinged by the
#   record and replay application).
###################################################################
proc ReplayAction {actionID timeout subscript replaceList} {
    global db__rd
    after $timeout [list \
        send $db__rd(ReplayApp) [list ActionEnd $actionID timeout]]
    if {[info exists db__rd($subscript)]} {
        set action $db__rd($subscript)
    } else {
        # This may happen as events are unbound (happens in Tk4.0b3)
        set action {#}
        ###puts "ERROR: db__rd($subscript) is not defined"
    }
    after 1 [list DoActionAndReply $actionID $action $replaceList]
}
###################################################################
# DoActionAndReply: do the action and set the variable to signal completion
#   of the action.
###################################################################
proc DoActionAndReply {actionID action replaceList} {
    global db__rd
    set ret [DoAction $action $replaceList]
    send  $db__rd(ReplayApp) [list ActionEnd $actionID completion]
}
###################################################################
# DoAction: this procedure executes the original binding for an action.
#    It is used while recording events to execute the action requested and
#        it is used when replaying events to replay the action.
#    Notes:
#    (1) we must do the %-substitutions just like the event binding
#        expects.  This is done by SubInAction.
#    (2) we put in the special case for canvas because we have to handle
#        the "current" tag correctly.  We capture the "current" object
#        when we make the recording and substitute it for the string
#        "current" in the binding.
#    (3) we have to execute the binding at the global scope, that is
#        why we use "uplevel #0"
###################################################################
proc DoAction {action replaceList} {
    global db__rd
    if {$action == {}} {
        return 0
    }
    # This is only important for canvases.
    # For other widgets db__rd(Current) will be set to the empty string.
    set db__rd(Current) [lookup Current $replaceList]
    set ret [uplevel #0 [SubInAction $action $replaceList]]
    return $ret
}
###################################################################
# SubInAction: this procedure takes an action script and does
#     all the %-replacements.  The parameters "replaceList" contains
#     all the necessary replacement strings.
###################################################################
proc SubInAction {action replaceList} {
    set len [string length $action]
    set subbedAction ""
    #
    # go through each character of the script looking for %s
    #
    for {set i 0} {$i < $len} {incr i} {
        set ch [string index $action $i]
        if {[string compare $ch "%"]==0} {
            # we found a %, see what to substitute
            incr i
            set ch [string index $action $i]
            if {[string compare $ch "%"]==0} {
                # a %% is replaced by a %
                append subbedAction "%"
            } else {
                # replace the %? which the appropriate string
                append subbedAction [lookup $ch $replaceList]
            }
        } else {
            # other characters are just copied over
            append subbedAction $ch
        }
    }
    return $subbedAction
}
###################################################################
# SubForCurrent: this procedure takes an action script and 
#    replaces all instances of "current" as a token with the value
#    passed for current
###################################################################
proc SubForCurrent {action current} {
    regsub -all -- {([^a-zA-Z])current([^a-zA-Z])} \
        $action [format {\1%s\2} $current] subbedAction
    return $subbedAction
}
###################################################################
# lookup: lookup a %-character in the association list.
#     The format of "alist" is:
#         { {A a-string} {B b-string} ...}
###################################################################
proc lookup {item alist} {
    foreach pair $alist {
        if {[string compare $item [lindex $pair 0]]==0} {
            return [lindex $pair 1]
        }
    }
    return ""
}
###################################################################
# RemoteMsgToUser: send comments to TkReplay to display.
###################################################################
proc RemoteMsgToUser {msg {level info}} {
    global db__rd
    set cmd [format {MsgToUser {%s} {%s}} $msg $level]
    # use catch in case TkReplay is exited
    catch [list send $db__rd(ReplayApp) $cmd]
}
#
# initialization code
#
RebindWidgetAndChildren .
set db__rd(RebindingComplete) 1
}
