###################################################################
# RecordAction: records an action in the replay list (if recording is on).
###################################################################
proc RecordAction {app tag event replaceList} {
    global ReplayData
    #
    # Only record events when the recording switch is on.
    #
    if {!$ReplayData(RecordingOn)} {
        return
    }
    #
    # Compute the delay between this event and the last event
    # and call InsertAction to insert it into the replay list.
    #
    set currentTime $ReplayData(Timer)
    set delay [expr $currentTime-$ReplayData(LastEventAt)]
    set ReplayData(LastEventAt) $currentTime
    set subscript "Bind,$tag,$event"
    InsertAction $delay $app $subscript $replaceList
}
###################################################################
# ReplayTimerTick: this is called every 100ms while recording is on.
#     It is used to record the delay between events so we can replay
#     them at the same speed they were recorded at.
###################################################################
proc ReplayTimerTick {} {
    global ReplayData
    # has the timer been turned off?
    if {$ReplayData(Timer) < 0} {
        return
    }
    incr ReplayData(Timer)
    if {$ReplayData(RecordingOn)} {
        after 100 ReplayTimerTick
    }
}
###################################################################
# MovePointerTo: this proc moves the mouse pointer to a specified
#   location.  The simplest implementation would be a single
#   WarpPointer command.  Instead we move it in a smooth motion
#   to its new location.  This makes it easier for people to follow
#   the mouse cursor.
#
#   We move a little at a time, moving only in 45 degree angles or
#   up and down.  There is an option to move faster.
###################################################################
proc MovePointerTo {x y app w} {
    global ReplayData
    # get the X id of thew window so we can send it to WarpPointer
    set wid [SendToApp $app [list winfo id $w]]
    if {$wid == ""} {
        # the window does not exists. It was probably destroyed by a
        # previous callback.
        return
    }
    if {$ReplayData(MouseSpeed) != 999} {
        # get the root x and y coordinates of the (upper left corner of)
        # the widget
        set wx  [SendToApp $app [list winfo rootx $w]]
        set wy  [SendToApp $app [list winfo rooty $w]]
        # compute the x,y coordinates to move to
        set tox [expr $wx+$x]
        set toy [expr $wy+$y]
        # find out where the point pointer is now
        set curx [winfo pointerx .]
        set cury [winfo pointery .]
        # compute the deltas and the distance
       set dx [expr $tox-$curx]
        set dy [expr $toy-$cury]
        set distance [expr round(sqrt($dx*$dx + $dy*$dy))]
        # adjust things so a single loop will go either direction
        if {$dx >= 0} {
            set xinc 1
        } else {
            set xinc -1
        }
        if {$dy >= 0} {
            set yinc 1
        } else {
            set yinc -1
        }
        # deal with the speeup factor
        set xinc [expr $ReplayData(MouseSpeed) * $xinc]
        set yinc [expr $ReplayData(MouseSpeed) * $yinc]
        # move the pointer incrementally in a loop
        while {($dx != 0) || ($dy != 0)} {
            # figure out how much to move in the x and y directions
            if {$dx != 0} {
                # do not go too far
                if {abs($xinc) > abs($dx)} {
                    set xinc $dx
                }
                set relx $xinc
                set dx [expr $dx - $xinc]
            } else {
                set relx 0
            }
            if {$dy != 0} {
                if {abs($yinc) > abs($dy)} {
                    set yinc $dy
                }
                set rely $yinc
                set dy [expr $dy - $yinc]
            } else {
                set rely 0
            }
            WarpPointer $relx $rely 0
            update
        }
    }
    WarpPointer $x $y $wid
}
###################################################################
# MovePointerTo: this proc moves the mouse pointer to a specified
#   location.  The simplest implementation would be a single
#   WarpPointer command.  Instead we move it in a smooth motion
#   to its new location.  This makes it easier for people to follow
#   the mouse cursor.
#
#   We move a little at a time, moving in a diagonal line from the
#   starting point to the stopping point.
#   There is an option to move faster.
###################################################################
proc NewMovePointerTo {x y app w} {
#
# Doesn't work yet.  Finish it later.
#
    global ReplayData
    # get the X id of thew window so we can send it to WarpPointer
    set wid [SendToApp $app [list winfo id $w]]
    # get the root x and y coordinates of the (upper left corner of) the widget
    set wx  [SendToApp $app [list winfo rootx $w]]
    set wy  [SendToApp $app [list winfo rooty $w]]
    # compute the x,y coordinates to move to
    set tox [expr $wx+$x]
    set toy [expr $wy+$y]
    # find out where the point pointer is now
    set fromx [winfo pointerx .]
    set fromy [winfo pointery .]
    # compute the deltas, the distance and the slope of the line between them
    set dx [expr $tox-$fromx]
    set dy [expr $toy-$fromy]
    set distance [expr round(sqrt(double($dx*$dx + $dy*$dy)))]
    set slope [expr double($dx)/double($dy)]
    # figure out the sign of the increment
    if {$dx < 0} {
        set xinc -1
    } else {
        set xinc 1
    }
    # handle the speedup factor
    set speedup 5
    if {$ReplayData(moveMouseQuickly)} {
        set xinc [expr $speedup * $xinc]
    }
    set curx $fromx
    set cury $fromy
    for {set i 0} {$i < $dx} {incr i} {
        set x [expr $fromx + $i*$xinc]
        set relx [expr $x - $curx]
        set curx $x
        set y [expr round($fromy + double($i)/$slope)]
        set rely [expr $y - $cury]
        set cury $y
        WarpPointer $relx $rely 0
        update
    }
    WarpPointer $x $y $wid
}
###################################################################
# ReplayActions: this proc plays the recorded actions starting at the
#   current action.  This means that you must rewind to play all
#   the actions.  Or you can start anywhere in the recording.
###################################################################
proc ReplayActions {} {
    global ReplayData
    #
    # If we are pointing with a window, make the window visiable
    #
    if {$ReplayData(pointWithArrow)} {
       GetReplayPointer
    }
    #
    # Make the binding for CatchClicks in each connected application.
    # This will allow the user to stop the replay in the middle with
    # a mouse button click or by pressing Escape.  Since the replay takes
    # control of the mouse, this may be the only way to stop a replay.
    #
    foreach app $ReplayData(ConnectedApps) {
        SendToApp $app [list bind__rd CatchClicks <1> \
            {send $db__rd(ReplayApp) RemoteInput Button1}]
        SendToApp $app [list bind__rd CatchClicks <Escape> \
            {send $db__rd(ReplayApp) RemoteInput Escape}]
        SendToApp $app [list set db__rd(Replaying) 1]
    }
    set maxIndex [expr [llength $ReplayData(Script)]-1]
    #
    # Loop through the events and replay them.
    #
    while 1 {
        #
        # If user has pressed Stop (or clicked) then quit playing
        #
        if !$ReplayData(PlayingOn) {
            break
        }
        set index $ReplayData(ScriptIndex)
        #
        # move to the next action to replay
        #
        if {$index < $maxIndex} {
            incr ReplayData(ScriptIndex)
        }
        set action [lindex $ReplayData(Script) $index]
        #
        # Select the event we are playing (if that option is turned on)
        # so that it is highlighted in the event list.
        #
        if $ReplayData(showEventsPlaying) {
            SelectActionListItem $index middle
        }
        #
        # Wait for the delay period (if delay is greater than 0).
        #
        set delay [lindex $action 0]
        if {$delay > 0} {
            if $ReplayData(UseDelays) {
                after [expr 100*$delay]
            }
        }
        #
        # Get the data on the event
        #
        set app [lindex $action 1]
        set subscript [lindex $action 2]
        set replaceList [lindex $action 3]
        #
        # Check for the commands handled here (not remotely)
        #
        switch $subscript \
            Pause {
                if $ReplayData(showEventsPlaying) {
                    SelectActionListItem $ReplayData(ScriptIndex) see
                }
                MsgToUser "Script replay was paused"
                break
            } \
            "*** Beginning of script ***" {
                MsgToUser "Start script"
                continue
            } \
            "*** End of script ***" {
                MsgToUser "End of script"
                break
            } \
            LoadApp {
                LoadApp $replaceList
                continue
            } \
            ConnectToApp {
                ConnectToApp [lindex $replaceList 0]
                continue
            } \
            BeginComment {
                MakeComment [lindex $replaceList 0] [lindex $replaceList 1] \
                    [lindex $replaceList 2]
                continue
            } \
            EndComment {
                catch [list destroy [format {.comment%s} $replaceList]]
                continue
            }
        #
        # parse the subscript to get the widget and event
        #
        set event ""
        regexp {Bind,([^,]*),(.*)} $subscript what which event
        #
        # Get the widget name
        #
        set w [rrlookup W $replaceList]
        #
        # see if we should move the mouse or pointing window to this event
        #
        set pointToWidget 1
        if {!$ReplayData(showFocusEvents)} {
            switch $event \
                "<FocusIn>"   - \
                "<FocusOut>" {
                    set pointToWidget 0
                }
        }
        if {$pointToWidget && !$ReplayData(showEnterEvents)} {
            switch $event \
                "<Enter>" - \
                "<Any-Enter>" - \
                "<Leave>" - \
                "<Any-Leave>" {
                    set pointToWidget 0
                }
        }
        #
        # Move the mouse pointer (if that option is on)
        #
        if {$ReplayData(HasWarpPointer)
         && $ReplayData(movePointer)
         && $pointToWidget} {
            set x [rrlookup x $replaceList]
            if {$x == ""} {
                 set x 5
            }
            set y [rrlookup y $replaceList]
            if {$y == ""} {
                 set y 5
            }
            MovePointerTo $x $y $app $w
        }
        #
        # Move the pointer window (if that option is on)
        #
        if {$pointToWidget && $ReplayData(pointWithArrow)} {
            set rootx [SendToApp $app [list winfo rootx $w]]
            set rooty [SendToApp $app [list winfo rooty $w]]
            set x [expr $rootx-56]; if {$x<0} {set x 0}
            set y [expr $rooty-25]; if {$y<0} {set y 0}
            wm geometry .replaypointer "+$x+$y"
            raise .replaypointer
            update
        }
        #
        # Assign a unique identifier to the action.
        #
        set uid [GetUniqueID]
        #
        # Figure out the maximum time to wait for the event handler to
        # complete.  Some event handlers call tkwait and so may not
        # return for a long time.
        set nextDelay [lindex [lindex $ReplayData(Script) [expr $index+1]] 0]
        if {$nextDelay == ""} {
            set nextDelay 0
        }
        #
        # give a little extra time for the timeout, 1 second
        #
        set timeout [expr $nextDelay+1000]
        #
        # Send the command to the remote application to execute
        # the binding for this event.  The tcl scripts to execute
        # are stored in the remote application.  We only store the
        # widget name and the event name.
        #
        SendToApp $app \
            [format {ReplayAction {%s} {%s} {%s} {%s}} \
                $uid $timeout $subscript $replaceList]
        set ReplayData(PendingCommandUID) $uid
        tkwait variable ReplayData(CommandDone)
        update
    }
    foreach app $ReplayData(ConnectedApps) {
        SendToApp $app [list bind__rd CatchClicks <1> {#}]
        SendToApp $app [list bind__rd CatchClicks <Escape> {#}]
        SendToApp $app [list set db__rd(Replaying) 0]
    }
    if {$ReplayData(pointWithArrow)} {
        wm withdraw .replaypointer
    }
}
###################################################################
# SelectActionListItem: select an item in the list of actions and
#  make sure that it can be seen.
###################################################################
proc SelectActionListItem {index {how see}} {
    set lb .replay.events.list
    if {$how == "see"} {
        $lb see $index
    } else { # place it in the middle of the list box
        set height [$lb cget -height]
        set top [expr $index - ($height/2)]
        if {$top < 0} {
            set top 0
        }
        $lb yview $top
    }
    $lb selection clear 0 end
    $lb selection set $index
}
###################################################################
# SelectAndSetItem: callback for a select click in the list box
#   that show the actions.
###################################################################
proc SelectAndSetItem {listbox x y} {
    global ReplayData
    set index [$listbox index @$x,$y]
    set ReplayData(ScriptIndex) $index
    SelectActionListItem $index
}
###################################################################
# GetUniqueID: generate a unique identifier.
###################################################################
proc GetUniqueID {} {
    global ReplayData
    set uid $ReplayData(UIDCounter)
    incr ReplayData(UIDCounter)
    return $uid
}
###################################################################
# ActionEnd: called only from a "send" from the remote application.
#   It is called when the action is completed or it times out.
#   We get two replies from each action, one from the timeout
#   and one from the action command completion.  We ignore any
#   old action completions that some and and wait for the completion
#   (or timeout) of the current action.
###################################################################
proc ActionEnd {uid how} {
    global ReplayData
    if {$uid == $ReplayData(PendingCommandUID)} {
        set ReplayData(CommandDone) 1
    }
}
###################################################################
# RemoteInput: an input event from the remote application
###################################################################
proc RemoteInput {input} {
    global ReplayData
    switch $input \
        Escape - \
        Button1 {
            set ReplayData(PlayingOn) 0
            MsgToUser "Script replay terminated by Escape or button click" low
        } \
        default {
            MsgToUser "Input $input ignored during replay" info
        }
}
###################################################################
# rrlookup: lookup a %-character in the association list.
#     The format of "alist" is:
#         { {A a-string} {B b-string} ...}
###################################################################
proc rrlookup {item alist} {
    foreach pair $alist {
        if {[string compare $item [lindex $pair 0]]==0} {
            return [lindex $pair 1]
        }
    }
    return ""
}
###################################################################
# GetReplayPointer: creates or deiconifies the big pointer we use to
#     point out the widget where the event is happening.
###################################################################
proc GetReplayPointer {} {
    set w .replaypointer
    if [winfo exists $w] {
        wm deiconify $w
    } else {
        toplevel $w
        wm geometry $w "+0+0"
        wm transient $w
        canvas $w.canvas -width 50 -height 50 -background white
        $w.canvas create line 0 25 50 25 -width 20 -arrow last \
            -fill red -arrowshape {25 40 15}
        pack $w.canvas
    }
}
