
/*
 * bConvert.c --
 *
 *	Implementation of procedures converting binary data to
 *	strings and vice versa.
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: bConvert.c,v 1.3 1996/01/13 21:44:51 aku Exp $
 */

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include "blobInt.h"


/*
 * declaration of internal procedures.
 */

static int
CvtEsc2Bin _ANSI_ARGS_ ((Tcl_Interp* interp,
			 Blob        b,
			 int*        length,
			 char*       inout_string));
static int
CvtBin2Esc _ANSI_ARGS_ ((Tcl_Interp* interp,
			 Blob        b,
			 int         length,
			 char*       input,
			 char**      result));
static int
CvtHex2Bin _ANSI_ARGS_ ((Tcl_Interp* interp,
			 Blob        b,
			 int*        length,
			 char*       inout_string));
static int
CvtBin2Hex _ANSI_ARGS_ ((Tcl_Interp* interp,
			 Blob        b,
			 int         length,
			 char*       input,
			 char**      result));
static int
CvtNone2Bin _ANSI_ARGS_ ((Tcl_Interp* interp,
			  Blob        b,
			  int*        length,
			  char*       inout_string));
static int
CvtBin2None _ANSI_ARGS_ ((Tcl_Interp* interp,
			  Blob        b,
			  int         length,
			  char*       input,
			  char**      result));
static int
CvtUUencode2Bin _ANSI_ARGS_ ((Tcl_Interp* interp,
			      Blob        b,
			      int*        length,
			      char*       inout_string));
static int
CvtBin2UUencode _ANSI_ARGS_ ((Tcl_Interp* interp,
			      Blob        b,
			      int         length,
			      char*       input,
			      char**      result));
static int
CvtBase642Bin _ANSI_ARGS_ ((Tcl_Interp* interp,
			    Blob        b,
			    int*        length,
			    char*       inout_string));
static int
CvtBin2Base64 _ANSI_ARGS_ ((Tcl_Interp* interp,
			    Blob        b,
			    int         length,
			    char*       input,
			    char**      result));

static int
Hex2Bin _ANSI_ARGS_ ((CONST char* in, char* out));

static void
Bin2Hex _ANSI_ARGS_ ((CONST char* in, char* out));

static int
ScaleUp3to4 _ANSI_ARGS_ ((int length));

static void
Split3to4 _ANSI_ARGS_ ((char*       target,
			int         length,
			CONST char* source));

static void
Split _ANSI_ARGS_ ((CONST char* in, char* out));

static void
Unsplit _ANSI_ARGS_ ((CONST char* in, char* out));

static void
ApplyEncoding _ANSI_ARGS_ ((char*       target,
			    int         length,
			    CONST char* map));

static int
Encode _ANSI_ARGS_ ((Tcl_Interp* interp,
		     int         length,
		     char*       input,
		     CONST char* map,
		     char**      result));

static int
Decode _ANSI_ARGS_ ((Tcl_Interp* interp,
		     int*        length,
		     char*       inout_string,
		     CONST char* map,
		     int         padChar));

/*
 * Character mappings for uuencoding, base64-encoding (bin -> ascii)
 *
 * Index this array by a 6-bit value to obtain the corresponding
 * 8-bit character.
 *                                                                                            |
 *                                      1         2         3         4         5         6   6
 *                            01234567890123456789012345678901234567890123456789012345678901234 */
static CONST char* uuMap   = "`!\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_~";
static CONST char* baseMap = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";

/*
 * Character mappings for uuencoding, base64-encoding (ascii -> bin)
 *
 * Index this array by a 8 bit value to get the 6-bit binary field
 * corresponding to that value.  Any illegal characters return high bit set.
 */

static CONST char uuMapAsc [] = {
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0001,0002,0003,0004,0005,0006,0007,
	0010,0011,0012,0013,0014,0015,0016,0017,
	0020,0021,0022,0023,0024,0025,0026,0027,
	0030,0031,0032,0033,0034,0035,0036,0037,
	0040,0041,0042,0043,0044,0045,0046,0047,
	0050,0051,0052,0053,0054,0055,0056,0057,
	0060,0061,0062,0063,0064,0065,0066,0067,
	0070,0071,0072,0073,0074,0075,0076,0077,
	0000,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	/* */
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200
};

static CONST char baseMapAsc [] = {
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0076,0200,0200,0200,0077,
	0064,0065,0066,0067,0070,0071,0072,0073,
	0074,0075,0200,0200,0200,0200,0200,0200,
	0200,0000,0001,0002,0003,0004,0005,0006,
	0007,0010,0011,0012,0013,0014,0015,0016,
	0017,0020,0021,0022,0023,0024,0025,0026,
	0027,0030,0031,0200,0200,0200,0200,0200,
	0200,0032,0033,0034,0035,0036,0037,0040,
	0041,0042,0043,0044,0045,0046,0047,0050,
	0051,0052,0053,0054,0055,0056,0057,0060,
	0061,0062,0063,0200,0200,0200,0200,0200,
	/* */
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200,
	0200,0200,0200,0200,0200,0200,0200,0200
};


/*
 *------------------------------------------------------*
 *
 *	BlobFindConversion --
 *
 *	------------------------------------------------*
 *	Retrieves the conversion procedures associated to
 *	a conversion option.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a standard TCL error code
 *		the found conversion procedures
 *		(in case of TCL_OK)
 *
 *------------------------------------------------------*
 */

EXTERN int
BlobFindConversion (interp, code, s2bFun, b2sFun)
Tcl_Interp*         interp;
CONST char*         code;
Blob_CvtString2Bin* s2bFun;
Blob_CvtBin2String* b2sFun;
{
  BlobInterpData* iData;
  Tcl_HashEntry*  hPtr;

  assert (code);

  iData = Tcl_GetAssocData (interp, ASSOC_KEY, NULL);
  hPtr  = Tcl_FindHashEntry (&iData->convSet, (char*) code);

  if (! hPtr)
    {
      Tcl_AppendResult (interp,
			"unknown conversion option \"-", code, "\"",
			0);
      return TCL_ERROR;
    }
  else
    {
      BlobConv* conv = (BlobConv*) Tcl_GetHashValue (hPtr);

      if (s2bFun)
	*s2bFun = conv->s2bFun;

      if (b2sFun)
	*b2sFun = conv->b2sFun;
    }

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	BlobAddStdConversionMethods --
 *
 *	------------------------------------------------*
 *	Registers the conversion methods initially
 *	available (no, escape, hex)
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		As of 'Blob_AddConversion'	
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

int
BlobAddStdConversionMethods (interp)
Tcl_Interp* interp;
{
  int res;

  res = Blob_AddConversion (interp, "no", CvtNone2Bin, CvtBin2None);

  if (res != TCL_OK)
    return res;

  res = Blob_AddConversion (interp, "esc", CvtEsc2Bin, CvtBin2Esc);

  if (res != TCL_OK)
    return res;

  res = Blob_AddConversion (interp, "hex", CvtHex2Bin, CvtBin2Hex);

  if (res != TCL_OK)
    return res;

  res = Blob_AddConversion (interp, "uu", CvtUUencode2Bin, CvtBin2UUencode);

  if (res != TCL_OK)
    return res;

  res = Blob_AddConversion (interp, "base64", CvtBase642Bin, CvtBin2Base64);

  return res;
}

/*
 *------------------------------------------------------*
 *
 *	CvtEsc2Bin --
 *
 *	------------------------------------------------*
 *	The given string is converted in place into its
 *	equivalent binary representation.  The string
 *	is assumed to be in ESCAPE-representation.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		'inout_string' and 'length' are changed.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
CvtEsc2Bin (interp, b, length, string)
Tcl_Interp* interp;	/* interpreter we are working in */
Blob        b;         	/* blob whose data to convert. */
int*        length;    	/* size of data to convert (in byte) */
char*       string;	/* IN/OUT reference to data to convert */
{
  int i, j, len;
  int escape = Blob_GetEscape (b);

  if (escape == BLOB_ERROR)
    {
      Tcl_AppendResult(interp, " string input : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }

  /*
   * 'i' refers to the byte position being translated.
   * 'j' refers to the character position the translation
   * shall be placed in.
   */

  len = strlen (string);

  for (i=j=0; i < len; i++, j++)
    {
      if (escape == string [i])
	{
	  /* escape sequence detected, translate by second character */
	  switch (string [i+1])
	    {
	    case '0':
	      /* @0 => \0 */
	      /* SKIP next character of escape sequence !! */

	      string [j] = '\0';
	      i++;
	      break;

	    case '@':
	      /* @@ => @ */
	      /* SKIP next character of escape sequence !! */

	      string [j] = '@';
	      i++;
	      break;

	    default:
	      /* @..., unknown escape sequence, transform @ by identity */

	      string [j] = string [i];
	      break;
	    }
	}
      else
	{
	  /*
	   * translate all other characters via identity
	   * I don't care about 'i == j' here
	   */
	  string [j] = string [i];
	}
    }

  *length = j;

  /*
   * It is not necessary to close 'string' with \0.
   * Binary data is now stored therein and the used
   * part of memory is determined through '*length'.
   */

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	CvtBin2Esc --
 *
 *	------------------------------------------------*
 *	the given blob-data is converted in place into
 *	its equivalent (escaped) string representation.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Allocates memory
 *
 *	Result:
 *		a standard TCL error code
 *		and an allocated string in 'result'.
 *
 *------------------------------------------------------*
 */

static int
CvtBin2Esc (interp, b, length, input, result)
Tcl_Interp* interp;	/* interpreter we are working in */
Blob        b;          /* blob whose data to convert. */
int         length;	/* size of data to convert (in byte) */
char*       input;      /* reference to binary data to convert */
char**      result;	/* reference to conversion result */
{
  int i, j;
  int numberOfEscapes = 0;

  char* conversionTarget;

  int escape = Blob_GetEscape (b);

  if (escape == BLOB_ERROR)
    {
      Tcl_AppendResult(interp, " string input : ",
		       Blob_LastError (b), 0);
      return TCL_ERROR;
    }

  /*
   * scan through binary data and count all bytes requiring
   * an escape translation and use this information to allocate
   * a buffer large enough to hold the conversion result.
   */

  for (i=0; i < length; i++)
    {
      if ((escape == input [i]) ||
	  ('\0'   == input [i]))
	{
	  numberOfEscapes ++;
	}
    }

  conversionTarget =
    ckalloc ((length+numberOfEscapes+1) * sizeof (char));

  if (conversionTarget == 0)
    {
      char buf [30], buf2 [30];

      sprintf (buf,  "%d", length + numberOfEscapes);
      sprintf (buf2, "%d", length);

      Tcl_AppendResult (interp, 
		"could not allocate place for string result of conversion: ",
			buf, " characters for ", buf2, " bytes",
			0);

      return TCL_ERROR;
    }

  for (i=j=0; i < length; i++, j++)
    {
      if (escape == input [i])
	{
	  /*
	   * @ ---> @@
	   */

	  conversionTarget [j] = escape; j++;
	  conversionTarget [j] = escape;
	}
      else if ('\0' == input [i])
	{
	  /*
	   * \0 ---> @0
	   */

	  conversionTarget [j] = escape; j++;
	  conversionTarget [j] = '0';
	}
      else
	{
	  /*
	   * identity for all other bytes.
	   */

	  conversionTarget [j] = input [i];
	}
    }

  conversionTarget [j] = '\0';

  *result = conversionTarget;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	CvtHex2Bin --
 *
 *	------------------------------------------------*
 *	The given string is converted in place into its
 *	equivalent binary representation.  The string
 *	is assumed to be in HEX-representation.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		'inout_string' and 'length' are changed.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
CvtHex2Bin (interp, b, length, string)
Tcl_Interp* interp;	/* interpreter we are working in */
Blob        b;         	/* blob whose data to convert. */
int*        length;    	/* size of data to convert (in byte) */
char*       string;	/* IN/OUT reference to data to convert */
{
  int i, j, len;

  /*
   * 'i' refers to the byte position being translated.
   * 'j' refers to the character position the translation
   * shall be placed in.
   */

  len = strlen (string);

  for (i=j=0; i < len; i+=2, j++)
    {
      if (TCL_OK != Hex2Bin (&string [i], &string [j]))
	{
	  char hex [3];
	  char buf [30];
	  sprintf (buf, "%d", i);
	  hex [0] = string [i];
	  hex [1] = string [i+1];
	  hex [2] = '\0';
	  Tcl_AppendResult (interp,
			    "illegal HEX-code '", hex, "' at position ",
			    buf, " of input", 0);
	  return TCL_ERROR;
	}
    }

  *length = j;

  /*
   * It is not necessary to close 'string' with \0.
   * Binary data is now stored therein and the used
   * part of memory is determined through '*length'.
   */

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	CvtBin2Hex --
 *
 *	------------------------------------------------*
 *	the given blob-data is converted in place into
 *	its equivalent (hex) string representation.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Allocates memory
 *
 *	Result:
 *		a standard TCL error code
 *		and an allocated string in 'result'.
 *
 *------------------------------------------------------*
 */

static int
CvtBin2Hex (interp, b, length, input, result)
Tcl_Interp* interp;	/* interpreter we are working in */
Blob        b;          /* blob whose data to convert. */
int         length;	/* size of data to convert (in byte) */
char*       input;      /* reference to binary data to convert */
char**      result;	/* reference to conversion result */
{
  int i, j;
  char *conversionTarget;

  conversionTarget = ckalloc ((2*length+1) * sizeof (char));

  if (conversionTarget == 0)
    {
      char buf [30], buf2 [30];

      sprintf (buf,  "%d", 2*length+1);
      sprintf (buf2, "%d", length);

      Tcl_AppendResult (interp, 
		"could not allocate place for string result of conversion: ",
			buf, " characters for ", buf2, " bytes",
			0);

      return TCL_ERROR;
    }

  for (i=j=0; i < length; i++, j+=2)
    {
      Bin2Hex (&input [i], &conversionTarget [j]);
    }

  conversionTarget [j] = '\0';

  *result = conversionTarget;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	Hex2Bin --
 *
 *	------------------------------------------------*
 *	takes 2 characters from 'in' and converts them
 *	into a single byte. Assumes HEX representation
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a standard TCL error code.
 *
 *------------------------------------------------------*
 */

static int
Hex2Bin (in, out)
CONST char* in;
char*       out;
{
  char high = in [0];
  char low  = in [1];

#define RANGE(low,x,high) (((low) <= (x)) && ((x) <= (high)))

  if (RANGE ('0',high,'9'))
    high -= '0';
  else if (RANGE ('a',high,'f'))
    high -= 'a'-10;
  else if (RANGE ('A',high,'F'))
    high -= 'A'-10;
  else
    return TCL_ERROR;


  if (RANGE ('0',low,'9'))
    low -= '0';
  else if (RANGE ('a',low,'f'))
    low -= 'a'-10;
  else if (RANGE ('A',low,'F'))
    low -= 'A'-10;
  else
    return TCL_ERROR;

  *out = 0xFF & ((high << 4) | low);

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	Bin2Hex
 *
 *	------------------------------------------------*
 *	takes a single byte from 'in' and generates the
 *	equivalent HEX representation (2 characters)
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The generated characters are written to
 *		'out'.
 *
 *	Result:
 *		See above.
 *
 *------------------------------------------------------*
 */

static void
Bin2Hex (in, out)
CONST char* in;
char*       out;
{
  char high = ((*in) >> 4) & 0x0F;
  char low  =  (*in)       & 0x0F;

  high += ((high < 10) ? '0' : 'A'-10);
  low  += ((low  < 10) ? '0' : 'A'-10);

  out [0] = high;
  out [1] = low;
}

/*
 *------------------------------------------------------*
 *
 *	CvtNone2Bin --
 *
 *	------------------------------------------------*
 *	The given string is converted in place into its
 *	equivalent binary representation.  Actually NO
 *	conversion takes place !.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
CvtNone2Bin (interp, b, length, string)
Tcl_Interp* interp;	/* interpreter we are working in */
Blob        b;         	/* blob whose data to convert. */
int*        length;    	/* size of data to convert (in byte) */
char*       string;	/* IN/OUT reference to data to convert */
{
  /* nothing to do !! */
  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	CvtBin2None --
 *
 *	------------------------------------------------*
 *	the given blob-data is converted in place into
 *	its equivalent string representation. Actually
 *	NO conversion takes place
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Allocates memory
 *
 *	Result:
 *		a standard TCL error code
 *		and an allocated string in 'result'.
 *
 *------------------------------------------------------*
 */

static int
CvtBin2None (interp, b, length, input, result)
Tcl_Interp* interp;	/* interpreter we are working in */
Blob        b;          /* blob whose data to convert. */
int         length;	/* size of data to convert (in byte) */
char*       input;      /* reference to binary data to convert */
char**      result;	/* reference to conversion result */
{
  char *conversionTarget;

  conversionTarget = ckalloc ((length+1) * sizeof (char));

  if (conversionTarget == 0)
    {
      char buf [30], buf2 [30];

      sprintf (buf,  "%d", length+1);
      sprintf (buf2, "%d", length);

      Tcl_AppendResult (interp, 
		"could not allocate place for string result of conversion: ",
			buf, " characters for ", buf2, " bytes",
			0);

      return TCL_ERROR;
    }

  memcpy ((VOID*) conversionTarget,
	  (VOID*) input,
	  length * sizeof (char));

  conversionTarget [length] = '\0';

  *result = conversionTarget;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	CvtUUencode2Bin --
 *
 *	------------------------------------------------*
 *	The given string is converted in place into its
 *	equivalent binary representation. The procedure
 *	assummes the string to uuencoded.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
CvtUUencode2Bin (interp, b, length, string)
Tcl_Interp* interp;	/* interpreter we are working in */
Blob        b;         	/* blob whose data is to be converted. */
int*        length;    	/* size of data to convert (in byte) */
char*       string;	/* IN/OUT reference to data to convert */
{
  return Decode (interp, length, string, uuMapAsc, '~');
}

/*
 *------------------------------------------------------*
 *
 *	CvtBin2UUencode --
 *
 *	------------------------------------------------*
 *	the given blob-data is converted in place into
 *	its equivalent string representation.  This string
 *	is based on uuencoding.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		see 'Encode'
 *
 *	Result:
 *		see 'Encode'
 *
 *------------------------------------------------------*
 */

static int
CvtBin2UUencode (interp, b, length, input, result)
Tcl_Interp* interp;	/* interpreter we are working in */
Blob        b;          /* blob whose data is to be converted. */
int         length;	/* size of data to convert (in byte) */
char*       input;      /* reference to binary data to convert */
char**      result;	/* reference to conversion result */
{
  return Encode (interp, length, input, uuMap, result);
}

/*
 *------------------------------------------------------*
 *
 *	CvtBase642Bin --
 *
 *	------------------------------------------------*
 *	The given string is converted in place into its
 *	equivalent binary representation. The procedure
 *	assummes the string to be in base64-encoding.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
CvtBase642Bin (interp, b, length, string)
Tcl_Interp* interp;	/* interpreter we are working in */
Blob        b;         	/* blob whose data is to be convert. */
int*        length;    	/* size of data to convert (in byte) */
char*       string;	/* IN/OUT reference to data to convert */
{
  return Decode (interp, length, string, baseMapAsc, '=');
}

/*
 *------------------------------------------------------*
 *
 *	CvtBin2Base64 --
 *
 *	------------------------------------------------*
 *	the given blob-data is converted in place into
 *	its equivalent string representation.  This string
 *	is based on base64-encoding.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		see 'Encode'.
 *
 *	Result:
 *		see 'Encode'.
 *
 *------------------------------------------------------*
 */

static int
CvtBin2Base64 (interp, b, length, input, result)
Tcl_Interp* interp;	/* interpreter we are working in */
Blob        b;          /* blob whose data is to be converted. */
int         length;	/* size of data to convert (in byte) */
char*       input;      /* reference to binary data to convert */
char**      result;	/* reference to conversion result */
{
  return Encode (interp, length, input, baseMap, result);
}

/*
 *------------------------------------------------------*
 *
 *	ScaleUp3to4 --
 *
 *	------------------------------------------------*
 *	Compute the length of the string representation
 *	for uuencoding / base64-encoding of binary data.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		see above.
 *
 *------------------------------------------------------*
 */

static int
ScaleUp3to4 (length)
int length;
{
  /* uuencoded or base64-encoded data
   * requires 4 bytes for every 3 bytes of original
   */

  int offset;

  /* pad length to make full triples first */
  offset = length % 3;
  if (offset)
    length += 3 - offset;

  /* now calc resulting space */
  return (length / 3) * 4;
}

/*
 *------------------------------------------------------*
 *
 *	Split3to4 --
 *
 *	------------------------------------------------*
 *	Split every 3 bytes of input into 4 bytes,
 *	actually 6-bit values and place them in the
 *	target.  Padding at the end is done with a value
 *	of '64' (6 bit -> values in range 0..63).
 *	This feature is used by ApplyEncoding.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		see above.
 *
 *------------------------------------------------------*
 */

static void
Split3to4 (target, length, input)
char*       target;
int         length;
CONST char* input;
{
  int offset, i , j;
  offset = length % 3;

  for (i=0, j=0; i < length; i+= 3, j+= 4)
    {
      Split (&input [i], &target [j]);
    }

  /* convert possible incomplete triple at end of data,
   * use pad values to indicate incompleteness.
   */

  if (offset)
    {
      char buf [3];

      /* correct indices */
      i -= 3;
      j -= 4;

      memset (buf, '\0', 3*sizeof (char));
      memcpy (buf, &input [i], (3-offset)*sizeof (char));

      Split (&target [j], buf);

      switch (offset)
	{
	case 1:
	  target [j+2] = 64;
	  target [j+3] = 64;
	  break;

	case 2:
	  target [j+3] = 64;
	  break;

	case 0:
	default:
	  /* should not happen */
	  assert (0);
	}
    }
}

/*
 *------------------------------------------------------*
 *
 *	Split --
 *
 *	------------------------------------------------*
 *	takes 3 bytes from 'in', splits them into
 *	4 6-bit values and places them then into 'out'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The generated characters are written to
 *		'out'.
 *
 *	Results:
 *		See above.
 *
 *------------------------------------------------------*
 */

static void
Split (in, out)
CONST char* in;
char*       out;
{
  out [0] = (077 &   (in [0] >> 2));
  out [1] = (077 & (((in [0] << 4) & 060) | ((in [1] >> 4) & 017)));
  out [2] = (077 & (((in [1] << 2) & 074) | ((in [2] >> 6) &  03)));
  out [3] = (077 &   (in [2] & 077));
}

/*
 *------------------------------------------------------*
 *
 *	Unsplit --
 *
 *	------------------------------------------------*
 *	takes 4 bytes from 'in' (6-bit values) and
 *	merges them into 3 bytes (8-bit values).
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The generated bytes are written to 'out'.
 *
 *	Results:
 *		See above.
 *
 *------------------------------------------------------*
 */

static void
Unsplit (in, out)
CONST char* in;
char*       out;
{
#define	MASK(i,by,mask) ((in [i] by) & mask)

  /* use temp storage in case of 'in', 'out'
   * overlapping each other.
   */

  char o1, o2, o3;

  o1 = MASK (0, << 2, 0374) | MASK (1, >> 4, 003);
  o2 = MASK (1, << 4, 0360) | MASK (2, >> 2, 017);
  o3 = MASK (2, << 6, 0300) | MASK (3,     , 077);

  out [0] = o1;
  out [1] = o2;
  out [2] = o3;

#undef MASK
}

/*
 *------------------------------------------------------*
 *
 *	ApplyEncoding
 *
 *	------------------------------------------------*
 *	transform 6-bit values into real characters
 *	according to the specified character-mapping.
 *	The map HAS TO contain at least 65 characters,
 *	the last one being the PAD character to use.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The characters are read from and written
 *		to 'target'.
 *
 *	Results:
 *		See above.
 *
 *------------------------------------------------------*
 */

static void
ApplyEncoding (target, length, map)
char*       target;
int         length;
CONST char* map;
{
  int i;

  for (i=0; i < length; i++)
    {
      target [i] = map [target [i]];
    }
}

/*
 *------------------------------------------------------*
 *
 *	Encode --
 *
 *	------------------------------------------------*
 *	the given blob-data is converted in place into
 *	its equivalent string representation.
 *	This procedure can be used for all encodings
 *	based on 3->4 byte conversion and a character-
 *	mapping.  Examples are uuencode, base64.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Allocates memory
 *
 *	Result:
 *		a standard TCL error code
 *		and an allocated string in 'result'.
 *
 *------------------------------------------------------*
 */

static int
Encode (interp, length, input, map, result)
Tcl_Interp* interp;	/* interpreter we are working in */
int         length;	/* size of data to convert (in byte) */
char*       input;      /* reference to binary data to convert */
CONST char* map;        /* the map to use to transform 6-bit values */
char**      result;	/* reference to conversion result */
{
  char *conversionTarget;
  int   binLength;

  binLength = ScaleUp3to4 (length);
  conversionTarget = ckalloc ((binLength+1) * sizeof (char));

  if (conversionTarget == 0)
    {
      char buf [30], buf2 [30];

      sprintf (buf,  "%d", binLength+1);
      sprintf (buf2, "%d", length);

      Tcl_AppendResult (interp, 
		"could not allocate place for string result of conversion: ",
			buf, " characters for ", buf2, " bytes",
			0);

      return TCL_ERROR;
    }

  Split3to4     (conversionTarget, length, input);
  ApplyEncoding (conversionTarget, binLength, map);

  conversionTarget [binLength] = '\0';

  *result = conversionTarget;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	Decode --
 *
 *	------------------------------------------------*
 *	The given string is converted in place into its
 *	equivalent binary representation.  The procedure
 *	assummes the string to be encoded with a 3->4
 *	byte scheme (such as uuencding, base64).
 *
 *	The map has to contain at least 256 characters.
 *      It is indexed by a 8 bit value to get the 6-bit
 *	binary field corresponding to that value.  Any
 *	illegal characters have high bit set.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
Decode (interp, length, string, map, padChar)
Tcl_Interp* interp;	/* interpreter we are working in */
int*        length;    	/* size of data to convert (in byte) */
char*       string;	/* IN/OUT reference to data to convert */
CONST char* map;        /* transformation map to use during decoding */
int         padChar;    /* character used for padding */
{
  int i, j, pad, maplen, len;

  /* check arguments first */

  len = *length;

  if (0 != (len % 4))
    {
      Tcl_AppendResult (interp,
			"length of input is no multiple of 4", 0);
      return TCL_ERROR;
    }

  /* convert and count padding */

  for (i=len-1, pad=0;
       (i >= 0) && (padChar == string [i]);
       pad++, i--)
    string [i] = '\0';

  if (pad > 2)
    {
      Tcl_AppendResult (interp,
			"padding exceeds 2 characters", 0);
      return TCL_ERROR;
    }

  maplen = i+1;

  /* convert characters to 6-bit values */

  for (i=0; i < maplen; i++)
    {
      char tmp = map [string [i]];

      if (tmp & 0x80) /* high-bit set? */
	{
	  char buf [50];
	  sprintf (buf, "%d", i);
	  Tcl_AppendResult (interp,
			    "illegal character at position ",
			    buf, " of input", 0);
	  return TCL_ERROR;
	}

      string [i] = tmp;
    }

  /* merge quadruples of 6-bit values into 3 bytes each */

  for (i=0, j=0; i < len; i+=4, j+=3)
    Unsplit (&string [i], &string [j]);

  /* calc length of binary data, correct for possible padding */

  len = (len / 4) * 3;
  if (pad)
    {
      len -= pad;
    }

  *length = len;

  return TCL_OK;
}

