
/*
 * bInput.c --
 *
 *	Implementation of procedures processing command arguments to
 *	generate the input to act on by a command.
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: bInput.c,v 1.4 1996/01/13 21:44:52 aku Exp $
 */

#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include "blobInt.h"

/*
 * declarations of internal procedures.
 */

static int
GetFromString _ANSI_ARGS_ ((Tcl_Interp* interp,
			    int         argc,
			    char**      argv,
			    Blob        b,
			    int*        length,
			    char**      data,
			    int*        dataIsAllocated,
			    int*        skipped));
static int
GetFromBlob _ANSI_ARGS_ ((Tcl_Interp* interp,
			  int         argc,
			  char**      argv,
			  Blob        b,
			  int*        length,
			  char**      data,
			  int*        dataIsAllocated,
			  int*        skipped));
static int
GetFromFile _ANSI_ARGS_ ((Tcl_Interp* interp,
			  int         argc,
			  char**      argv,
			  int*        length,
			  char**      data,
			  int*        dataIsAllocated,
			  int*        skipped));
/*
 * 'BlobGetFromPack' is not defined here due to its complexity.
 * See 'bPack.c'.
 */

/*
 *------------------------------------------------------*
 *
 *	Blob_GetInput --
 *
 *	------------------------------------------------*
 *	Determines data to act on from given arguments.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		as of subroutines GetFrom.. ()
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

int
Blob_GetInput (interp, argc, argv,
	       b, blobName,
	       length, data, dataIsAllocated, skipped)
Tcl_Interp* interp;		/* interpreter we are working in/with */
int         argc;		/* number of arguments to process */
char**      argv;		/* reference to arguments to process */
Blob        b;			/* handle of whose input is computed */
CONST char* blobName;		/* name of same thing */
int*        length;		/* OUT: length of generated data (#bytes) */
char**      data;		/* OUT: reference to generated data */
int*        dataIsAllocated;	/* OUT: flag, indicating, wether 'data'
				 * was malloc'd on the heap or not. */
int*        skipped;            /* OUT: number of arguments processed. */
{
  /*
   * Accepted argument syntax:
   *
   * [] string ?-option? <string> ?from ?to??
   * [] blob   <name>                   ?from ?to??
   * [] file   <filehandle>             ?length?
   * [] pack   <format> <value>...
   *
   * with 'option' being the name of a registered conversion method.
   * standard methods are: no, esc, hex, uu, base64
   */

  char *type;    /* reference to selector (argv [0]) */
  int   len;     /* length of selector */
  char  c;       /* first character of selector */
  int   num = 0; /* number of processed arguments */
  int   res;

  if (argc < 2)
    {
      Tcl_AppendResult (interp,
			"wrong # args for determination of input: not enough",
			0);
      return TCL_ERROR;
    }

  type = argv [0];
  c    = type [0];
  len  = strlen (type);

  /*
   * Skip over selector and determine procedure to use.
   */

  argc --;
  argv ++;
  num  ++;

  switch (c)
    {
    case 's':
      if (0 == strncmp ("string", type, len))
	{
	  res = GetFromString (interp, argc, argv, b,
			       length, data, dataIsAllocated, &num);
	}
      break;

    case 'b':
      if (0 == strncmp ("blob", type, len))
	{
	  res = GetFromBlob (interp, argc, argv, b,
			     length, data, dataIsAllocated, &num);
	}
      break;

    case 'f':
      if (0 == strncmp ("file", type, len))
	{
	  res = GetFromFile (interp, argc, argv,
			     length, data, dataIsAllocated, &num);
	}
      break;

    case 'p':
      if (0 == strncmp ("pack", type, len))
	{
	  res = BlobGetFromPack (interp, argc, argv,
				 length, data, dataIsAllocated, &num);
	}
      break;

    default:
      /* invalid selector, raise error */

      Tcl_AppendResult (interp, 
			"unknown input type \"", type, "\"", 0);
      return TCL_ERROR;
    }

  if ((res == TCL_OK) && (! skipped) && (num <= argc))
    {
      Tcl_AppendResult (interp,
			"wrong # args for determination of input: too much",
			0);
      return TCL_ERROR;
    }

  if (skipped)
    *skipped = num;

  return res;
}

/*
 *------------------------------------------------------*
 *
 *	GetFromString --
 *
 *	------------------------------------------------*
 *	Take string argument and convert to binary data
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
GetFromString (interp, argc, argv, b,
	       length, data, dataIsAllocated, skipped)

Tcl_Interp* interp;		/* interpreter we are working in/with */
int         argc;		/* number of arguments to process */
char**      argv;		/* reference to arguments to process */
Blob        b;		        /* handle of blob acting on generated input */
int*        length;		/* OUT: length of generated data (#bytes) */
char**      data;		/* OUT: reference to generated data */
int*        dataIsAllocated;	/* OUT: flag, indicating, wether 'data'
				 * was malloc'd on the heap or not. */
int*        skipped;            /* OUT: number of processed arguments */
{
  /*
   * Accepted argument syntax:
   *
   * []      <string> ?from ?to??
   * [] -opt <string> ?from ?to??
   *
   * with 'opt' being in the set of strings accepted by 'BlobFindConversion'
   *
   * without -xxx argument default to conversion -esc
   */

  int   result;
  char *source;
  int   from;
  int   to;
  int   len;
  int   edges;

  Blob_CvtString2Bin cvtFun;

  /* conversion option specified ? */
  if ('-' == argv [0][0])
    {
      char* option = argv [0];

      /* skip over '-' */
      option ++;

      if (TCL_OK != BlobFindConversion (interp, option, &cvtFun, NULL))
	return TCL_ERROR;

      /* skip over legal option */
      argv ++;
      argc --;
      (*skipped) ++;
    }
  else if (TCL_OK != BlobFindConversion (interp, "esc", &cvtFun, NULL))
    return TCL_ERROR;


  if (argc < 1)
    {
      Tcl_AppendResult (interp,
			"wrong # args for determination of input: not enough",
			0);
      return TCL_ERROR;
    }

  source = argv [0];
  len    = strlen (source);

  /*
   * Skip over input string
   */

  argc --;
  argv ++;
  (*skipped) ++;

  if (TCL_OK != BlobGetIntervalSpecInside (interp, argc, argv,
					   len, &from, &to, &edges))
    {
      return TCL_ERROR;
    }

  (*skipped) += edges;

  if (to <= from)
    {
      /* empty interval */

      *length          = 0;
      *data            = 0;
      *dataIsAllocated = FALSE;

      return TCL_OK;
    }

  /*
   * Cut specified range from given string
   */

  if (to < len)
    {
      source [to] = '\0';
    }

  if (from > 0)
    {
      source += from;
    }

  len = to - from;

  /*
   * Now do the specified conversion
   */

  result = (*cvtFun) (interp, b, &len, source);
  if (result != TCL_OK)
    return TCL_ERROR;

  /*
   * The conversion was done in place, so
   * use input as output too.
   */

  *length          = len;
  *data            = source;
  *dataIsAllocated = FALSE;

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	GetFromBlob --
 *
 *	------------------------------------------------*
 *	Takes blob argument and convert to binary data
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
GetFromBlob (interp, argc, argv, b,
	     length, data, dataIsAllocated, skipped)

Tcl_Interp* interp;		/* interpreter we are working in/with */
int         argc;		/* number of arguments to process */
char**      argv;		/* reference to arguments to process */
Blob        b;		        /* handle of blob acting on generated input */
int*        length;		/* OUT: length of generated data (#bytes) */
char**      data;		/* OUT: reference to generated data */
int*        dataIsAllocated;	/* OUT: flag, indicating, wether 'data'
				 * was malloc'd on the heap or not. */
int*        skipped;            /* OUT: number of arguments processed. */
{
  /*
   * Accepted argument syntax:
   *
   * <blobname> ?from ?to??
   */

  char       *source = argv [0];
  int         from;
  int         to;
  int         len;
  Blob        srcBlob;
  int         exist;
  Tcl_CmdInfo cmdInfo;
  int         edges;

#if 0
  if (argc > 3)
    {
      Tcl_AppendResult (interp,
			"wrong # args for determination of input",
			0);
      return TCL_ERROR;
    }
#endif

  /* locate blob used as input */

  exist = Tcl_GetCommandInfo (interp, source, &cmdInfo);

  if (!exist)
    {
      Tcl_AppendResult(interp,
		       "blob input : ", source, " does not exist", 0);
      return TCL_ERROR;
    }
  else if (cmdInfo.proc != BlobObjectCmd)
    {
      Tcl_AppendResult(interp,
		       "blob input : ", source, " is not a blob", 0);
      return TCL_ERROR;
    }
  else
    {
      BlobClientData* bcd = (BlobClientData*) cmdInfo.clientData;
      srcBlob = bcd->b;
    }


  /* retrieve size of source, then determine range of input */

  if (BLOB_OK != Blob_Size (srcBlob, &len))
    {
      Tcl_AppendResult(interp, " blob input (", source,
		       "): ", Blob_LastError (srcBlob), 0);
      return TCL_ERROR;
    }

  /*
   * Skip over name of blob.
   */

  argc --;
  argv ++;
  (*skipped) ++;

  if (TCL_OK != BlobGetIntervalSpecInside (interp, argc, argv,
					   len, &from, &to, &edges))
    {
      return TCL_ERROR;
    }

  (*skipped) += edges;

  if (to <= from)
    {
      *length          = 0;
      *data            = 0;
      *dataIsAllocated = FALSE;

      return TCL_OK;
    }

  if (BLOB_OK != Blob_GetData (srcBlob, from, to, length, data))
    {
      Tcl_AppendResult(interp, "blob input (", source,
		       "): ", Blob_LastError (b), 0);
      return TCL_ERROR;
    }

  if (b == srcBlob)
    {
      /*
       * target and source are identical, so create a copy
       * of the specified range and avoid memory hazards later.
       */

      char* copy = ckalloc (*length);

      if (copy == NULL)
	{
	  char     buf [30];
	  sprintf (buf,  "%d", *length);

	  Tcl_AppendResult (interp, 
			    "could not allocate ", buf,
			    " byte(s) for copy of blob input", 0);
	  return TCL_ERROR;
	}

      memcpy ((VOID *) copy, (VOID *) *data, *length);

      *data            = copy;
      *dataIsAllocated = TRUE;
    }
  else
    {
      *dataIsAllocated = FALSE;
    }

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	GetFromFile --
 *
 *	------------------------------------------------*
 *	Takes file argument and retrieves binary data
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
GetFromFile (interp, argc, argv,
	     length, data, dataIsAllocated, skipped)
Tcl_Interp* interp;		/* interpreter we are working in/with */
int         argc;		/* number of arguments to process */
char**      argv;		/* reference to arguments to process */
int*        length;		/* OUT: length of generated data (#bytes) */
char**      data;		/* OUT: reference to generated data */
int*        dataIsAllocated;	/* OUT: flag, indicating, wether 'data'
				 * was malloc'd on the heap or not. */
int*        skipped;            /* OUT: number of arguments processed. */
{
  /*
   * Accepted argument syntax:
   *
   *  <filehandle> ?length?
   */

  FILE* input;
  int   len;
  char* store;
#if 0
  if (argc > 2)
    {
      Tcl_AppendResult (interp,
			"wrong # args for determination of input",
			0);
      return TCL_ERROR;
    }
#endif
  /* convert filehandle to FILE pointer */

  if (TCL_OK != Tcl_GetOpenFile (interp, argv [0], 0, 1, &input))
    {
      return TCL_ERROR;
    }

  (*skipped) ++;

  if (argc >= 2)
    {
      /*
       * Number of bytes to read are specified by caller.
       */

      if (TCL_OK != Tcl_GetInt (interp, argv [1], &len))
	{
	  return TCL_ERROR;
	}

      (*skipped) ++;
    }
  else
    {
      /*
       * Compute number of bytes from current
       * position to the end of file.
       */

      int where, last;

      where = ftell (input);
      fseek (input, 0, SEEK_END);

      last = ftell (input);
      fseek (input, where, SEEK_SET);

      len = last - where;
    }

  store = ckalloc (len);

  if (store == NULL)
    {
      char     buf [30];
      sprintf (buf, "%d", len);

      Tcl_AppendResult (interp, 
			"could not allocate ", buf,
			" byte(s) for file input", 0);
      return TCL_ERROR;
    }

  *length          = fread (store, 1, len, input);
  *data            = store;
  *dataIsAllocated = TRUE;

  return TCL_OK;
}
