
/*
 * bMisc.c --
 *
 *	Implementation of utility procedures.
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: bMisc.c,v 1.4 1996/02/11 00:00:51 aku Exp $
 */

#include <assert.h>
#include "blobInt.h"


/*
 * declarations of internal procedures.
 */

static int
LimitRange _ANSI_ARGS_ ((int x, int min, int max));

static int
GetValue _ANSI_ARGS_ ((Tcl_Interp* interp,
		       CONST char* arg,
		       int         limit,
		       int*        value));

/*
 *------------------------------------------------------*
 *							*
 *	BlobGetIntervalSpec --				*
 *							*
 *	------------------------------------------------*
 *	Determines an interval of bytes from the	*
 *	given arguments.  Without legal arguments the	*
 *	interval defaults to [0..limit].  Values	*
 *	outside the interval 0..limit are reset to the	*
 *	nearest edge.  The information is assumed to be	*
 *	at the end of the arg-list.			*
 *	------------------------------------------------*
 *							*
 *	Sideeffects:					*
 *		The interpreter result may change.	*
 *							*
 *	Result:						*
 *		a standard TCL error code		*
 *							*
 *------------------------------------------------------*
 */

int
BlobGetIntervalSpec (interp, argc, argv, limit, from, to)
Tcl_Interp* interp;	/* interpreter we are working in */
int         argc;	/* number of arguments to process {0, 1, 2} */
char**      argv;	/* reference to argumentsa to process */
int         limit;	/* max value allowed for from and to.
			 * necessary for "end" */
int*        from;	/* here goes the result about the interval start */
int*        to;		/* here goes the result about the interval ending */
{
  /*
   * Accepted argument syntax:
   *
   * []
   * [] <from>
   * [] <from> <to>
   *
   * with <from>, <to> integer numbers or "end"
   */

  switch (argc)
    {
    case 0:
      /* [] */
      *from = 0;
      *to   = limit;
      break;

    case 1:
      /* [] <from> */

      if (TCL_OK != GetValue (interp, argv [0], limit, from))
	{
	  return TCL_ERROR;
	}

      *from = LimitRange (*from, 0, limit);
      *to   = limit;
      break;

    case 2:
      /* [] <from> <to> */

      if (TCL_OK != GetValue (interp, argv [0], limit, from))
	{
	  return TCL_ERROR;
	}

      if (TCL_OK != GetValue (interp, argv [1], limit, to))
	{
	  return TCL_ERROR;
	}

      *from = LimitRange (*from, 0, limit);
      *to   = LimitRange (*to,   0, limit);
      break;

    default:
      /* to many (or less) arguments, error of caller */
      assert (0);
      break;
    }

  /* everything went fine */  
  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *							*
 *	BlobGetIntervalSpecInside --			*
 *							*
 *	------------------------------------------------*
 *	Determines an interval of bytes from the	*
 *	given arguments.  Without legal arguments the	*
 *	interval defaults to [0..limit].  Values	*
 *	outside the interval 0..limit are reset to the	*
 *	nearest edge.  The procedure assumes, that the	*
 *	defintion is maybe followed by additional	*
 *	arguments.					*
 *	------------------------------------------------*
 *							*
 *	Sideeffects:					*
 *		The interpreter result may change.	*
 *							*
 *	Result:						*
 *		a standard TCL error code		*
 *							*
 *------------------------------------------------------*
 */

int
BlobGetIntervalSpecInside (interp, argc, argv, limit, from, to, numEdges)
Tcl_Interp* interp;	/* interpreter we are working in */
int         argc;	/* number of arguments to process {0, 1, 2} */
char**      argv;	/* reference to argumentsa to process */
int         limit;	/* max value allowed for from and to.
			 * necessary for "end" */
int*        from;	/* here goes the result about the interval start */
int*        to;		/* here goes the result about the interval ending */
int*        numEdges;	/* here goes the number of arguments really evaluated
			 * to determine the interval. maybe 0 as not needed
			 * anytime */
{
  /*
   * Accepted argument syntax:
   *
   * []
   * []             <any-arg>...
   * [] <from>      <any-arg>...
   * [] <from> <to> <any-arg>...
   *
   * with <from>, <to> integer numbers or "end"
   * and <any-arg> anything neither integer nor "end"
   */

  int         edges;
  Tcl_DString save;

  /* Save result area first.  Maybe the caller assembles
   * its result just now and should not disturb this.
   */

  Tcl_DStringInit (&save);
  Tcl_DStringGetResult (interp, &save);


  switch (argc)
    {
    case 0:
      /* [] */
      *from = 0;
      *to   = limit;
      edges = 0;
      break;

    case 1:
      if (TCL_OK != GetValue (interp, argv [0], limit, from))
	{
	  /*
	   * NO error, just an empty interval definition,
	   * i.e. [] <any-arg>
	   * -> eliminate message possibly written by 'GetValue'
	   */

	  Tcl_ResetResult (interp);
	  *from = 0;
	  *to   = limit;
	  edges = 0;
	}
      else
	{
	  /* [] <from> */

	  *from = LimitRange (*from, 0, limit);
	  *to   = limit;
	  edges = 1;
	}
      break;

    default:
      /* argc >= 2 */

      if (TCL_OK != GetValue (interp, argv [0], limit, from))
	{
	  /*
	   * NO error, just an empty interval definition,
	   * i.e. [] <any-arg> <any-arg>...
	   * -> eliminate message possibly written by 'GetValue'
	   */

	  Tcl_ResetResult (interp);
	  *from = 0;
	  *to   = limit;
	  edges = 0;
	}
      else
	{
	  *from = LimitRange (*from, 0, limit);
	  edges = 1;

	  if (TCL_OK != GetValue (interp, argv [1], limit, to))
	    {
	      /*
	       * NO error, just an partial interval definition,
	       * i.e. [] <from> <any-arg>...
	       * -> eliminate message possibly written by 'GetValue'
	       */

	      Tcl_ResetResult (interp);
	      *to = limit;
	      /* #edges unchanged */
	    }
	  else
	    {
	      /* [] <from> <to> <any-arg>... */

	      *to = LimitRange (*to, 0, limit);
	      edges = 2;
	    }
	}
      break;
    }

  /* transfer #edges to caller, if possible */

  if (numEdges)
    {
      *numEdges = edges;
    }

  /* At last restore result area.
   */

  Tcl_DStringResult (interp, &save);
  Tcl_DStringFree (&save);

  return TCL_OK;
}


/* GetValue -- */
static int
GetValue (interp, arg, limit, value)
Tcl_Interp* interp;
CONST char* arg;
int         limit;
int*        value;
{
  if (0 == strcmp ("end", arg))
    {
      *value = limit;
      return TCL_OK;
    }
  else if (TCL_OK == Tcl_GetInt (interp, (char*) arg, value))
    {
      return TCL_OK;
    }
  else
    {
      /* neither valid integer nor "end" */
      return TCL_ERROR;
    }
}


/* LimitRange -- */
static int
LimitRange (x, min, max)
int x;
int min;
int max;
{
  if (x < min)
    return min;
  else if (x > max)
    return max;
  else
    return x;
}

