
/*
 * bPack.c --
 *
 *	Packing and Unpacking binary data.
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: bPack.c,v 1.2 1996/01/13 01:05:18 aku Exp $
 */

#include <ctype.h>
#include <netinet/in.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include "blobInt.h"


/*
 * Declarations of internal types and procedures.
 */

/*
 * Enumeration of all understood binary types
 */

typedef enum
{
  fmtIllegal   = -1, /*  ---- to be used in case of an error ----   */
  fmtAsciiNull     , /* string, padded by \0                        */
  fmtAsciiSpace    , /* string, padded by SPC                       */
  fmtSignedChar    , /* single byte, signed                         */
  fmtUnsignedChar  , /* -""-,        unsigned                       */
  fmtSignedShort   , /* 2-byte integer,   native byteorder,  signed */
  fmtUnsignedShort , /* -""-,                              unsigned */
  fmtSignedInt     , /* 2/4-byte integer, native byteorder,  signed */
  fmtUnsignedInt   , /* -""-,                              unsigned */
  fmtSignedLong    , /* 4/8-byte integer, native byteorder,  signed */
  fmtUnsignedLong  , /* -""-,                              unsigned */
  fmtNetworkShort  , /* 2-byte integer,   network byteorder, signed */
  fmtNetworkLong   , /* 4/8-byte integer, network byteorder, signed */
  fmtFloat         , /* C-float  (4 byte), native byteorder         */
  fmtDouble        , /* C-double (8 byte), native byteorder         */
  fmtNull          , /* '\0' (single byte, fixed value)             */
  fmtSkipBackward  , /* seek position backward                      */
  fmtFill            /* seek position forward (fill gaps with \0)   */
} BinaryItemType;


/*
 * Structure holding the specification of single binary item.
 *
 * Special encodings:
 * Types 'Ascii...' allow 'size' to be -1 for packing, thereby
 * accepting the complete length of the associated argument.
 */

typedef struct _BinaryItem_
{
  BinaryItemType type;    /* Type of specified item */
  int            size;    /* Size of item in bytes.
			   * Directly dependent upon item type */
  int            bytePos; /* Position of first byte of item
			   * in blob to read/create */
  /* pack ONLY:                                      */
  int            argIdx;  /* Index of argument used in this specification */
} _BinaryItem, *BinaryItem;


/*
 * Structure containing the complete internal
 * represenation of a pack/unpack format.
 */

typedef struct _Format_
{
  int         numItems;  /* Number of items in format. */
  BinaryItem* itemTable; /* Table containing the specifications
			  * of the item to pack/unpack */
  int         size;      /* size of blob to create or minimum
			  * size to be read */
  short       padFlag;   /* boolean flag indicating, wether the last
			  * specification is to be replicated to
			  * match the number of arguments or not. */
  int         numArgs;	 /* The smallest number of arguments
			  * required to satisfy the format
			  * (in case of 'pack') */
  int         bytes;     /* The number of bytes generated during 'pack'
			  * or minimum number of bytes in blob to satisfy
			  * 'unpack'-format. */
} Format;


/*
 * Internal procedures handling structures as declared above.
 */

static BinaryItem
ItemCreate _ANSI_ARGS_ ((BinaryItemType type));

static BinaryItem
ItemCopy _ANSI_ARGS_ ((BinaryItem item));

static void
ItemDelete _ANSI_ARGS_ ((BinaryItem* item));

#define ItemSize(i)       ((i)->size)
#define ItemSetSize(i,sz) ((i)->size) = (sz)

#define ItemArg(i)        ((i)->argIdx)
#define ItemSetArg(i,idx) ((i)->argIdx) = (idx)

#define ItemPos(i)        ((i)->bytePos)
#define ItemSetPos(i,pos) ((i)->bytePos) = (pos)

#define ItemType(i)       ((i)->type)

static int
ItemDefaultLength _ANSI_ARGS_ ((BinaryItemType type));

#define IsLegalType(i) ((fmtIllegal < (i)) && ((i) <= fmtFill))



static void
FormatInitialize _ANSI_ARGS_ ((Format* fmt));

static void
FormatCleanup _ANSI_ARGS_ ((Format* fmt));

static int
FormatAppend _ANSI_ARGS_ ((Format* fmt, BinaryItem bi));

#define FormatSize(fmt)       ((fmt)->size)
#define FormatSetSize(fmt,sz) ((fmt)->size) = (sz)

#define FormatPadFlag(fmt)        ((fmt)->padFlag)
#define FormatSetPadFlag(fmt,val) ((fmt)->padFlag) = (val)

#define FormatArgs(fmt)           ((fmt)->numArgs)

#define FormatNum(fmt)         ((fmt)->numItems)
#define FormatGetItem(fmt,idx) ((fmt)->itemTable [idx])

#define FormatBytes(fmt)      ((fmt)->bytes)
#define FormatSetBytes(fmt,n) ((fmt)->bytes) = (n)


static BinaryItemType
MapFormatCharacter2ItemType _ANSI_ARGS_ ((int character));


#define IMPLY(a,b) (! (a) || (b))

#define UCHAR(c) ((unsigned char) (c))

/*
 * Parsing a perl pack format.
 */

static int
ScanSingleSpec _ANSI_ARGS_ ((Tcl_Interp* interp,
			     CONST char* format,
			     int         isUnpack,
			     Format*     fmt,
			     char**      nextSpec));

static int
ScanSpecs _ANSI_ARGS_ ((Tcl_Interp* interp,
			CONST char* format,
			int         isUnpack,
			Format*     fmt));

static int
PreparePacking _ANSI_ARGS_ ((Tcl_Interp* interp,
			     int         argc,
			     char**      argv,
			     Format*     fmt));

static int
PrepareUnpack _ANSI_ARGS_ ((Tcl_Interp* interp,
			    int         blobLength,
			    Format*     fmt));

static int
DoPacking _ANSI_ARGS_ ((Tcl_Interp* interp,
			char**      argv,
			Format*     fmt,
			int*        length,
			char**      data));

static int
DoUnpack _ANSI_ARGS_ ((Tcl_Interp* interp,
		       Format*     fmt,
		       int         blobLength,
		       char*       blobData,
		       char**      result));

static int
ItemCalcBytes _ANSI_ARGS_ ((Tcl_Interp* interp,
			    int         isUnpack,
			    BinaryItem  item,
			    char**      argv,
			    int*        num,
			    int*        pos));

static int
PackSpec _ANSI_ARGS_ ((Tcl_Interp* interp,
		       BinaryItem  item,
		       char**      argv,
		       char*       d));

static int
ExtractSpec _ANSI_ARGS_ ((Tcl_Interp* interp,
			  BinaryItem  item,
			  char*       data,
			  char**      result,
			  int*        idx));

#define SHORT_SPACE (10)
#define INT_SPACE   (20)
#define LONG_SPACE  (30)


/*
 *------------------------------------------------------*
 *
 *	BlobGetFromPack --
 *
 *	------------------------------------------------*
 *	Takes a packing format and values behind, then
 *	uses the format to convert the values into binary
 *	data.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Memory is allocated
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

int
BlobGetFromPack (interp, argc, argv,
		 length, data, dataIsAllocated, skipped)
Tcl_Interp* interp;
int         argc;
char**      argv;
int*        length;
char**      data;
int*        dataIsAllocated;
int*        skipped;            /* OUT: number of arguments processed. */
{
  /*
   * Accepted syntax:
   *
   * <format> <value>...
   * for more information about <format> see external documentation.
   */

  Format fmt;
  int    res;

  if (argc < 1)
    {
      Tcl_AppendResult (interp, "wrong # args, at least 1 expected", 0);
      return TCL_ERROR;
    }

  /* parse format */
  res = ScanSpecs (interp, argv [0], FALSE, &fmt);
  if (res != TCL_OK)
    return res;

  argc --;
  argv ++;

  /* checks and normalization */
  res = PreparePacking (interp, argc, argv, &fmt);
  if (res != TCL_OK)
    return res;

  (*skipped) += 1 + FormatArgs (&fmt);

  /* generating the blob */
  res = DoPacking (interp, argv, &fmt, length, data);
  if (res != TCL_OK)
    return res;

  *dataIsAllocated = TRUE;
  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	BlobUnpack --
 *
 *	------------------------------------------------*
 *	Takes an unpack format and blob data, then uses
 *	the format to convert the data into a list of values
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Memory is allocated
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

int
BlobUnpack (interp, b, arg, data)
Tcl_Interp* interp;
Blob        b;
CONST char* arg;
char**      data;
{
  /*
   * for more information about <format> see external documentation.
   */

  Format fmt;
  int    res;
  int    blen;
  char*  bdata;

  res = Blob_Size (b, &blen);
  if (res != BLOB_OK)
    goto blob_error;

  res = Blob_GetData (b, 0, blen, &blen, &bdata);
  if (res != BLOB_OK)
    goto blob_error;

  /* parse format */
  res = ScanSpecs (interp, arg, TRUE, &fmt);
  if (res != TCL_OK)
    return res;

  /* checks and normalization */
  res = PrepareUnpack (interp, blen, &fmt);
  if (res != TCL_OK)
    return res;

  /* generating the list */
  res = DoUnpack (interp, &fmt, blen, bdata, data);
  if (res != TCL_OK)
    return res;

  return TCL_OK;

 blob_error:
  Tcl_AppendResult (interp, Blob_LastError (b), 0);
  return TCL_ERROR;
}

/*
 *------------------------------------------------------*
 *
 *	ItemCreate --
 *
 *	------------------------------------------------*
 *	Constructor of objects of type 'BinaryItem'.
 *	Type and default size are written into the
 *	object, everything else initialized with -1.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Allocates memory to hold the new object
 *
 *	Result:
 *		The newly generated object or NULL in
 *		case of an allocation failure.
 *
 *------------------------------------------------------*
 */

static BinaryItem
ItemCreate (type)
BinaryItemType type;
{
  BinaryItem item;

  assert (IsLegalType (type));

  item = (BinaryItem) ckalloc (sizeof (_BinaryItem));

  if (item)
    {
      item->type    = type;
      item->size    = ItemDefaultLength (type);
      item->bytePos = -1;
      item->argIdx  = -1;
    }

  return item;
}

/*
 *------------------------------------------------------*
 *
 *	ItemCopy --
 *
 *	------------------------------------------------*
 *	Copy constructor for objects of type BinaryItem.
 *	Generates a duplicate of the specified object.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Allocates memory to hold the clone.
 *
 *	Result:
 *		The newly generated object or NULL in
 *		case of an allocation failure.
 *
 *------------------------------------------------------*
 */

static BinaryItem
ItemCopy (item)
BinaryItem item;
{
  BinaryItem copy;

  assert (item);

  copy = (BinaryItem) ckalloc (sizeof (_BinaryItem));

  if (copy)
    {
      memcpy ((VOID*) copy, (VOID*) item, sizeof (_BinaryItem));
    }

  return copy;
}

/*
 *------------------------------------------------------*
 *
 *	ItemDelete --
 *
 *	------------------------------------------------*
 *	Destructor for objects of type 'BinaryItem'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Previously allocated memory is freed
 *		again.
 *
 *	Result:
 *		None.
 *
 *------------------------------------------------------*
 */

static void
ItemDelete (item)
BinaryItem* item;
{
  assert  (item);
  assert (*item);

  ckfree (*item);
  *item = 0;
}

/*
 *------------------------------------------------------*
 *
 *	ItemDefaultLength --
 *
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		The default size to use in a specification
 *		of a binary item.
 *
 *------------------------------------------------------*
 */

static int
ItemDefaultLength (type)
BinaryItemType type;
{
  static int length [fmtFill+1] =
    {
      1,                       /* AsciiNull     */
      1,                       /* AsciiSpace    */
      sizeof (char),           /* SignedChar    */
      sizeof (unsigned char),  /* UnsignedChar  */
      sizeof (short),          /* SignedShort   */
      sizeof (unsigned short), /* UnsignedShort */
      sizeof (int),            /* SignedInt     */
      sizeof (unsigned int),   /* UnsignedInt   */
      sizeof (long),           /* SignedLong    */
      sizeof (unsigned long),  /* UnsignedLong  */
      sizeof (short),          /* NetworkShort  */
      sizeof (long),           /* NetworkLong   */
      sizeof (float),          /* Float         */
      sizeof (double),         /* Double        */
       1,                      /* Null          */
      -1,                      /* SkipBackward  */
      -1                       /* Fill          */
    };

  assert (IsLegalType (type));

  return length [type];
}

/*
 *------------------------------------------------------*
 *
 *	FormatInitialize --
 *
 *	------------------------------------------------*
 *	Pseudo constructor of 'Format' objects
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The specified structure is initialized
 *		with values indicating an empty format.
 *
 *	Result:
 *		The changed structure.
 *
 *------------------------------------------------------*
 */

static void
FormatInitialize (fmt)
Format* fmt;
{
  assert (fmt);

  fmt->numItems  = 0;
  fmt->itemTable = 0;
  fmt->size      = 0;
  fmt->padFlag   = FALSE;
  fmt->numArgs   = 0;
  fmt->bytes     = 0;
}

/*
 *------------------------------------------------------*
 *
 *	FormatCleanup --
 *
 *	------------------------------------------------*
 *	Pseudo destructor of 'Format'-objects
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		All allocated data in the format is
 *		freed again.
 *
 *	Result:
 *		The changed structure.
 *
 *------------------------------------------------------*
 */

static void
FormatCleanup (fmt)
Format* fmt;
{
  assert (fmt);
  assert (IMPLY (fmt->numItems > 0, fmt->itemTable));
  assert (IMPLY (fmt->itemTable, fmt->numItems > 0));

  if (fmt->numItems > 0)
    {
      int i;

      for (i=0; i < fmt->numItems; i++)
	{
	  ItemDelete (& (fmt->itemTable [i]));
	}

      ckfree (fmt->itemTable);
    }
}

/*
 *------------------------------------------------------*
 *
 *	FormatAppend --
 *
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The specified 'BinaryItem'-object is
 *		appended to the 'Format' specification.
 *		Memory is allocated as needed.  At last
 *		the number of arguments required by the
 *		Format and the reference of the BinaryItem
 *		to its argument are suitably updated.
 *
 *	Result:
 *		a standard TCL error code and the
 *		changed structure (in case of TCL_OK)
 *
 *------------------------------------------------------*
 */

static int
FormatAppend (fmt, bi)
Format*    fmt;
BinaryItem bi;
{
  int last;

  assert (fmt);
  assert (bi);

  last = fmt->numItems;

  if (! fmt->itemTable)
    {
      assert (fmt->numItems == 0);

      fmt->itemTable = (BinaryItem*) ckalloc (sizeof (BinaryItem));

      if (! fmt->itemTable)
	{
	  return FALSE;
	}

      fmt->numItems  = 1;
    }
  else
    {
      BinaryItem* t;
      int         n;

      assert (fmt->numItems > 0);

      n = fmt->numItems + 1;
      t = (BinaryItem*) ckrealloc (fmt->itemTable, n * sizeof (BinaryItem));

      if (! t)
	{
	  return FALSE;
	}

      fmt->numItems  = n;
      fmt->itemTable = t;
    }

  fmt->itemTable [last] = bi;

  if ((ItemType (bi) != fmtNull) &&
      (ItemType (bi) != fmtSkipBackward) &&
      (ItemType (bi) != fmtFill))
    {
      ItemSetArg (bi, fmt->numArgs);
      fmt->numArgs ++;
    }
  else
    {
      ItemSetArg (bi, -1);
    }

  return TRUE;
}

/*
 *------------------------------------------------------*
 *
 *	MapFormatCharacter2ItemType  --
 *
 *	------------------------------------------------*
 *	A format character is mapped onto the
 *	corresponding type of a 'BinaryItem'
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		None.
 *
 *	Result:
 *		The associated type or 'fmtIllegal'.
 *
 *------------------------------------------------------*
 */

static BinaryItemType
MapFormatCharacter2ItemType (character)
int character;
{
  switch (character)
    {
    case 'a': return fmtAsciiNull;
    case 'A': return fmtAsciiSpace;
    case 'c': return fmtSignedChar;
    case 'C': return fmtUnsignedChar;
    case 's': return fmtSignedShort;
    case 'S': return fmtUnsignedShort;
    case 'i': return fmtSignedInt;
    case 'I': return fmtUnsignedInt;
    case 'l': return fmtSignedLong;
    case 'L': return fmtUnsignedLong;
    case 'n': return fmtNetworkShort;
    case 'N': return fmtNetworkLong;
    case 'f': return fmtFloat;
    case 'd': return fmtDouble;
    case 'x': return fmtNull;
    case 'X': return fmtSkipBackward;
    case '@': return fmtFill;

    default:
      return fmtIllegal;
    }
}

/*
 *------------------------------------------------------*
 *
 *	ScanSingleSpec --
 *
 *	------------------------------------------------*
 *	Parses the format specification at the beginning
 *	of the string, creates a BinaryItem to hold the
 *	extracted information and adds this to the
 *	overall 'Format'.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		
 *
 *	Result:
 *		a standard TCL error code.
 *		'nextSpec' points the first character
 *		of the next format specification.
 *
 *------------------------------------------------------*
 */

static int
ScanSingleSpec (interp, format, isUnpack, fmt, nextSpec)
Tcl_Interp* interp;
CONST char* format;
int         isUnpack;
Format*     fmt;
char**      nextSpec;
{
  /*
   * First skip other whitespace, e.g. blanks, tabs
   * and newlines.  The character afterward is the
   * format type and checked for legality.  Behind
   * the type may follow a repeat count, either a
   * positive integer or '*'.  Repeat count and
   * type are possibly separated by whitespace too.
   *
   * The procedure will gobble any whitespace behind
   * the repeat count.
   */

#define GOBBLE_WHITESPACE \
  for (; isspace(UCHAR(*p)); p++) /* Empty loop body */

  BinaryItem     bi;
  BinaryItemType type;
  CONST char*    p      = format;
  int            repeat = 1;
  char           ctype;

  GOBBLE_WHITESPACE;

  type = MapFormatCharacter2ItemType (*p);
  if (type == fmtIllegal)
    {
      char buf [2];

      buf [0] = *p;
      buf [1] = '\0';

      Tcl_AppendResult (interp, "illegal type character \"",
			buf, "\" in format", 0);
      return TCL_ERROR;
    }

  ctype = *p;

  p ++;

  bi = ItemCreate (type);
  if (! bi)
    goto memory;


  GOBBLE_WHITESPACE;

  if (*p == '*')
    {
      /*
       * Special count '*'.
       */

      if ((type == fmtAsciiNull)  ||
	  (type == fmtAsciiSpace) ||
	  (type == fmtFill))
	{
	  ItemSetSize (bi, -1);

	  if (isUnpack)
	    {
	       if (type == fmtFill)
		 {
		   Tcl_AppendResult (interp,
				     "type 'x' not allowed in unpack",
				     0);
		   goto error;
		 }
	       else
		 /* Aa: string running till end of data */
		 repeat = -1;
	    }
	}
      else if ((type == fmtNull) || (type == fmtSkipBackward))
	{
	  char buf [2];

	  buf [0] = ctype;
	  buf [1] = '\0';

	  Tcl_AppendResult (interp,
			    "type \"", buf,
			    "\" can't use * as repeat",
			    0);
	  goto error;
	}
      else
	{
	  repeat = -1;
	}

      p ++;
    }
  else if (('0' <= *p) && (*p <= '9'))
    {
      /*
       * Count given as integer.
       */

      char *end;
      int n = strtoul (p, &end, 10);

      if ((type == fmtAsciiNull)  ||
	  (type == fmtAsciiSpace) ||
	  (type == fmtFill))
	{
	  ItemSetSize (bi, n);
	}
      else
	{
	  repeat = n;
	}

      p = end;
    }
  else
    {
      /* no count specified, use default (if possible) */

      if (type == fmtFill)
	{
	  Tcl_AppendResult (interp,
			    "type \"@\" requires a count value",
			    0);
	  goto error;
	}
    }

  GOBBLE_WHITESPACE;
  *nextSpec = (char*) p;


  if (FormatPadFlag (fmt))
    {
      /*
       * Error, replication legal only at end of
       * format and we just got another spec !
       */

      Tcl_AppendResult (interp,
			"repeat '*' only usable at last specification",
			0);
      goto error;
    }


  assert ((repeat == -1) || (1 <= repeat));

  if ((repeat == 1) || (repeat == -1))
    {
      /*
       * Single repeat or replication as required.
       */

      if (! FormatAppend (fmt, bi))
	goto memory;

      if (repeat == -1)
	FormatSetPadFlag (fmt, TRUE);
    }
  else /* repeat > 1 */
    {
      /*
       * Resolve repeat by placing that
       * many specs into the format.
       */

      int i;
      for (i=0; i < repeat; i++)
	{
	  BinaryItem tmp = ItemCopy (bi);

	  if (!tmp)
	    goto memory;

	  if (! FormatAppend (fmt, tmp))
	    {
	      ItemDelete (&tmp);
	      goto memory;
	    }
	}

      ItemDelete (&bi);
    }

  return TCL_OK;

 memory:
  Tcl_AppendResult (interp, "not enough memory to parse format", 0);
  /* fall through to cleanup */

 error:
  ItemDelete (&bi);
  return TCL_ERROR;
}

/*
 *------------------------------------------------------*
 *
 *	ScanSpecs --
 *
 *	------------------------------------------------*
 *	Parses all specifications in the given string
 *	and constructs a Format object describing them.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
ScanSpecs (interp, format, isUnpack, fmt)
Tcl_Interp* interp;
CONST char* format;
int         isUnpack;
Format*     fmt;
{
  CONST char* current;
  char* next;
  int   res;

  FormatInitialize (fmt);

  current = format;

  while (*current != '\0')
    {
      res = ScanSingleSpec (interp, current, isUnpack, fmt, &next);
      if (res != TCL_OK)
	goto error;

      current = next;
    }

  return TCL_OK;

 error:
  FormatCleanup (fmt);

  return TCL_ERROR;
}

/*
 *------------------------------------------------------*
 *
 *	PreparePacking --
 *
 *	------------------------------------------------*
 *	This procedure resolves replication at the end
 *	of the format first, then computes the number of
 *	bytes to generate together with the locations of
 *	the BinaryItem's.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The format is changed.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
PreparePacking (interp, argc, argv, fmt)
Tcl_Interp* interp;
int         argc;
char**      argv;
Format*     fmt;
{
  /*
   * Preparation of pack process:
   *
   * i) check number of provided arguments and replicate
   *    last specification if requested and necessary
   *
   * ii) determine number generated bytes and the positions
   *     of the items therein.
   */

  if (argc < FormatArgs (fmt))
    {
      /*
       * The format requires values more than
       * were given, so bail out here.
       */

      char bufA [50];
      char bufB [50];

      sprintf (bufA, "%d", FormatArgs (fmt));
      sprintf (bufB, "%d", argc);

      Tcl_AppendResult (interp, "format required more values (",
			bufA, ") than given (", bufB, ")", 0);
      goto error;
    }
  else if (argc > FormatArgs (fmt))
    {
      if (FormatPadFlag (fmt))
	{
	  /*
	   * Take last item spec and replicate it as required.
	   */

	  int        i, n;
	  BinaryItem last = FormatGetItem (fmt, FormatNum (fmt)-1);

	  FormatSetPadFlag (fmt, FALSE);
	  n = argc - FormatArgs (fmt);

	  for (i=0; i <n; i++)
	    {
	      BinaryItem tmp = ItemCopy (last);

	      if (!tmp)
		goto memory;

	      if (! FormatAppend (fmt, tmp))
		goto memory;
	    }
	}
#if 0
      else
	{
	  /*
	   * More arguments given than required.
	   * This is an error too.
	   */

	  char bufA [50];
	  char bufB [50];

	  sprintf (bufA, "%d", FormatArgs (fmt));
	  sprintf (bufB, "%d", argc);

	  Tcl_AppendResult (interp, "format required less values (",
			    bufA, ") than given (", bufB, ")", 0);
	  goto error;
	}
#endif
    }
  /* (argc == FormatArgs (fmt)) */

  /*
   * The replication flag is irrelevant from now on.
   * Calc number of generated bytes + item locations.
   */

  {
    int i, num, pos, res;

    num = pos = 0;

    for (i=0; i < FormatNum (fmt); i++)
      {
	res = ItemCalcBytes (interp, FALSE, FormatGetItem (fmt, i),
			     argv, &num, &pos);
	if (res != TCL_OK)
	  goto error;
      }

    FormatSetBytes (fmt, num);
  }

  return TCL_OK;

 memory:
  Tcl_AppendResult (interp, "not enough memory to parse format", 0);
  /* fall through to cleanup */

 error:
  FormatCleanup (fmt);
  return TCL_ERROR;
}

/*
 *------------------------------------------------------*
 *
 *	PrepareUnpack --
 *
 *	------------------------------------------------*
 *	This procedure computed the number of bytes
 *	required by the format together with the locations
 *	of the BinaryItem's, then resolves replication at
 *	the end of the format.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The format is changed.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
PrepareUnpack (interp, blobLength, fmt)
Tcl_Interp* interp;
int         blobLength;
Format*     fmt;
{
  /*
   * Preparation of unpack process:
   *
   * i) determine number required bytes and the positions
   *     of the items therein.
   *
   * i) check number of provided bytes and replicate
   *    last specification if requested and necessary
   */

  {
    int i, num, pos, res;

    num = pos = 0;

    for (i=0; i < FormatNum (fmt); i++)
      {
	res = ItemCalcBytes (interp, TRUE,
			     FormatGetItem (fmt, i),
			     NULL, &num, &pos);
	if (res != TCL_OK)
	  goto error;
      }

    FormatSetBytes (fmt, num);
  }

  if (FormatPadFlag (fmt))
    {
      /*
       * Replicate last item as needed to use as much
       * of the blob data as possible.  In case of a
       * string being last, don't replicate the item
       * and set its length instead.
       */

      BinaryItem last = FormatGetItem (fmt, FormatNum (fmt)-1);

      FormatSetPadFlag (fmt, FALSE);

      if ((fmtAsciiNull  == ItemType (last)) ||
	  (fmtAsciiSpace == ItemType (last)))
	{
	  int n;

	  /* length of item should unresolved */
	  assert (ItemSize (last) < 0);

	  n = blobLength - FormatBytes (fmt);

	  if (n >= 0)
	    {
	      ItemSetSize (last, n);
	      FormatSetBytes (fmt, blobLength);
	    }
	  /* else -- blob too short for format */
	}
      else
	{
	  int n = blobLength - FormatBytes (fmt);

	  if (n > 0)
	    {
	      int sz, k, i, pos, add;

	      sz = ItemSize (last);

	      /* number of items we have place for */
	      k   = sz / n;
	      /* number of bytes required by additional items */
	      add = k * sz;
	      /* position of first new item */
	      pos = ItemPos (last) + sz;

	      for (i=0; i < k; i++, pos += sz)
		{
		  BinaryItem tmp = ItemCopy (last);
		  
		  if (!tmp)
		    goto memory;

		  if (! FormatAppend (fmt, tmp))
		    goto memory;

		  ItemSetPos (tmp, pos);
		}

	      FormatSetBytes (fmt, FormatBytes (fmt) + add);
	    }
	  /* else -- blob maybe too short for format */
	}
    }

  if (blobLength < FormatBytes (fmt))
    {
      /* -- blob too short for format */

      char bufA [50];
      char bufB [50];

      sprintf (bufA, "%d", FormatBytes (fmt));
      sprintf (bufB, "%d", blobLength);

      Tcl_AppendResult (interp, "format required more bytes (",
			bufA, ") than given (", bufB, ")", 0);
      goto error;
    }

  assert (FormatBytes (fmt) <= blobLength);

  return TCL_OK;

 memory:
  Tcl_AppendResult (interp, "not enough memory to parse format", 0);
  /* fall through to cleanup */

 error:
  FormatCleanup (fmt);
  return TCL_ERROR;
}

/*
 *------------------------------------------------------*
 *
 *	ItemCalcBytes --
 *
 *	------------------------------------------------*
 *	Calculates the location of the given item and
 *	the number of bytes to add to the total number
 *	of generated bytes.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Changes the total number of bytes and
 *		the current item-location.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
ItemCalcBytes (interp, isUnpack, item, argv, num, pos)
Tcl_Interp* interp;
int         isUnpack;
BinaryItem  item;
char**      argv;
int*        num;
int*        pos;
{
  /*
   * Determine the number of bytes required by 'item' and
   * merge it into the total number of generated bytes.
   * Compute the position of the item inside the binary
   * information too.
   */

  int n;

  assert (num);
  assert (pos);
  assert (item);

  ItemSetPos (item, *pos);

  switch (ItemType (item))
    {
    case fmtAsciiNull:
    case fmtAsciiSpace:

      if (ItemSize (item) < 0)
	{
	  if (isUnpack)
	    {
	      /* ignore item for now, replication normalization
	       * will resolve the unknown length.
	       */

	      return TCL_OK;
	    }
	  else
	    {
	      /* A*, a* -> use argument length */

	      ItemSetSize (item, strlen (argv [ItemArg (item)]));
	    }
	}
      /* FALL THROUGH into standard behaviour */


    case fmtSignedChar:
    case fmtUnsignedChar:
    case fmtSignedShort:
    case fmtUnsignedShort:
    case fmtSignedInt:
    case fmtUnsignedInt:
    case fmtSignedLong:
    case fmtUnsignedLong:
    case fmtNetworkShort:
    case fmtNetworkLong:
    case fmtFloat:
    case fmtDouble:
    case fmtNull: /* special code: move current location */

      n = ItemPos (item) + ItemSize (item);

      if (n > (*num))
	{
	  /*
	   * update top only if the end of the item is placed
	   * behind this (current) top
	   */
	  (*num) = n;
	}

      /* move current location behind actual item */
      (*pos) += ItemSize (item);
      break;


    case fmtIllegal:

      assert (0);
      break;


      /* special codes: no data, just movement of current location */

    case fmtSkipBackward:

      /*
       * this specifier is unable to extend the
       * total number of generated bytes!
       */

      (*pos) --;
      if ((*pos) < 0)
	{
	  Tcl_AppendResult (interp,
			    "malformed format: specifier 'x' ",
			    "positions to negative location", 0);
	  return TCL_ERROR;
	}
      break;


    case fmtFill:
      /* position to absolute location */

      (*pos) = ItemSize (item);

      assert ((*pos) >= 0);

      if (ItemSize (item) > (*num))
	{
	  (*num) = ItemSize (item);
	}
      break;
    }

  return TCL_OK;
}

/*
 *------------------------------------------------------*
 *
 *	DoPacking --
 *
 *	------------------------------------------------*
 *	Interprets a Format, converting the string
 *	values into binary data.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Allocates the memory to hold the
 *		generated data
 *
 *	Result:
 *		a standard TCL error code
 * 		and the generated data.
 *
 *------------------------------------------------------*
 */

static int
DoPacking (interp, argv, fmt, length, data)
Tcl_Interp* interp;
char**      argv;
Format*     fmt;
int*        length;
char**      data;
{
  int i, res;
  char* d = ckalloc (FormatBytes (fmt));

  if (! d)
    goto memory;

  *length = FormatBytes (fmt);
  *data   = d;

  /*
   * Clearing the area to operate on save us from dealing
   * with 'fmtNull', 'fmtSkipbackward' and 'fmtFill' later.
   */
  memset ((VOID*) d, '\0', FormatBytes (fmt));

  for (i=0; i < FormatNum (fmt); i++)
    {
      res = PackSpec (interp, FormatGetItem (fmt, i), argv, d);
      if (res != TCL_OK)
	goto error;
    }


  FormatCleanup (fmt);
  return TCL_OK;

 memory:
  Tcl_AppendResult (interp, "not enough memory to parse format", 0);
  /* fall through to cleanup */

 error:
  if (d)
    ckfree (d);

  FormatCleanup (fmt);
  return TCL_ERROR;
}

/*
 *------------------------------------------------------*
 *
 *	DoUnpack --
 *
 *	------------------------------------------------*
 *	Interprets a Format, converting binary data into
 *	a list of string values.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		Allocates the memory to hold the
 *		generated data
 *
 *	Result:
 *		a standard TCL error code
 * 		and the generated data.
 *
 *------------------------------------------------------*
 */

static int
DoUnpack (interp, fmt, blobLength, blobData, result)
Tcl_Interp* interp;
Format*     fmt;
int         blobLength;
char*       blobData;
char**      result;
{
  int    i, res, j;
  char** field = ckalloc (FormatArgs (fmt) * sizeof (char*));

  if (! field)
    goto memory;

  for (i=0, j=0; i < FormatNum (fmt); i++)
    {
      res = ExtractSpec (interp, FormatGetItem (fmt, i),
			 blobData, &field [j], &j);
      if (res != TCL_OK)
	goto error;
    }

  *result = Tcl_Merge (FormatArgs (fmt), field);
  if (! *result)
    goto memory;

  for (i=0; i < FormatArgs (fmt); i++)
    ckfree (field [i]);

  ckfree (field);

  FormatCleanup (fmt);
  return TCL_OK;

 memory:
  Tcl_AppendResult (interp, "not enough memory to build unpack result", 0);
  /* fall through to cleanup */

 error:
  if (field)
    {
      int i;
      for (i=0; i < FormatArgs (fmt); i++)
	if (field [i])
	  ckfree (field [i]);

      ckfree (field);
    }

  FormatCleanup (fmt);
  return TCL_ERROR;
}

/*
 *------------------------------------------------------*
 *
 *	PackSpec --
 *
 *	------------------------------------------------*
 *	Converts a single string value into binary data
 *	according to information stored in the given
 *	BinaryItem.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The binary data are is changed.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
PackSpec (interp, item, argv, d)
Tcl_Interp* interp;
BinaryItem  item;
char**      argv;
char*       d;
{
  char* source;
  char* arg = argv [ItemArg (item)];

  /*
   * Union of all possible types to convert.
   */

  typedef union
    {
      double         d;
      float          f;
      long int       li;
      unsigned long  uli;
      int            i;
      unsigned int   ui;
      short int      si;
      unsigned short usi;
      char           c;
      unsigned char  uc;
    } ConverterUnion;
  
  ConverterUnion cu;

  switch (ItemType (item))
    {
    case fmtSignedChar:
    case fmtUnsignedChar:
      if (arg [0] == '\0')
	{
	  Tcl_AppendResult (interp,
			    "empty argument for character specification",
			    0);
	  goto error;
	}
      else if (ItemType (item) == fmtSignedChar)
	{
	  cu.c   = arg [0];
	  source = &cu.c;
	}
      else
	{
	  cu.uc  = (unsigned char) arg [0];
	  source = &cu.uc;
	}
      break;


    case fmtFloat:
    case fmtDouble:
      {
	double d;
	if (TCL_OK != Tcl_GetDouble (interp, arg, &d))
	  return TCL_ERROR;

	if (ItemType (item) == fmtFloat)
	  {
	    cu.f   = d;
	    source = (char*) &cu.f;
	  }
	else
	  {
	    cu.d   = d;
	    source = (char*) &cu.d;
	  }
      }
      break;


    case fmtSignedShort:
    case fmtUnsignedShort:
    case fmtSignedInt:
    case fmtUnsignedInt:
    case fmtSignedLong:
    case fmtUnsignedLong:
    case fmtNetworkShort:
    case fmtNetworkLong:
      {
	int i;
	if (TCL_OK != Tcl_GetInt (interp, arg, &i))
	  return TCL_ERROR;

	/*
	 * Its a pity: Tcl_GetInt uses 'strtoul' (returning
	 * an 'unsigned long'), but returns itself only an int!
	 * Because of this deficiency the code here will be
	 * incorrect on DEC ALPHAs. 'cc' there says:
	 *
	 * sizeof (int)  == 4   #  normal
	 * sizeof (long) == 8   #  <-- !!
	 */

	switch (ItemType (item))
	  {
	  case fmtNetworkShort:
	  case fmtSignedShort:
	    source = (char*) &cu.si;
	    cu.si  = (short int) i;
	    break;

	  case fmtUnsignedShort:
	    source = (char*) &cu.usi;
	    cu.usi = (unsigned short) i;
	    break;

	  case fmtSignedInt:
	    source = (char*) &cu.i;
	    cu.i   = (int) i;
	    break;

	  case fmtUnsignedInt:
	    source = (char*) &cu.ui;
	    cu.ui  = (unsigned int) i;
	    break;

	  case fmtNetworkLong:
	  case fmtSignedLong:
	    source = (char*) &cu.li;
	    cu.li  = (long int) i;
	    break;

	  case fmtUnsignedLong:
	    source = (char*) &cu.uli;
	    cu.uli = (unsigned long) i;
	    break;

	  default:
	    assert (0); /* should not happen */
	  }


	if (ItemType (item) == fmtNetworkShort)
	  {
	    cu.si = htons (cu.si);
	  }
	else if (ItemType (item) == fmtNetworkLong)
	  {
	    cu.li = htonl (cu.li);
	  }
      }
      break;


    case fmtAsciiNull:
    case fmtAsciiSpace:
      {
	/*
	 * Place string into result. If value is longer than reserved,
	 * truncate the input, else pad with requested character to
	 * reach full field width.
	 */

	int len = strlen (arg);

	if (len >= ItemSize (item))
	  {
	    memcpy ((VOID*) d + ItemPos (item),
		    (VOID*) arg,
		    ItemSize (item));
	  }
	else
	  {
	    char pad    = ((ItemType(item) == fmtAsciiNull) ? '\0' : ' ');
	    int  padLen = ItemSize (item) - len;

	    memcpy ((VOID*) d + ItemPos (item),
		    (VOID*) arg,
		    len);

	    memset ((VOID*) d + (ItemPos (item) + len), pad, padLen);
	  }
	}

      /* don't fall through to standard behaviour !! */
      return TCL_OK;
      break;


    case fmtIllegal:
      assert (0);
      break;


    case fmtNull:
    case fmtSkipBackward:
    case fmtFill:
      /*
       * Ignore location movements here, as the positions
       * of the data items were computed before (-> PreparePacking).
       */

      /* don't fall through to standard behaviour !! */
      return TCL_OK;
      break;

    default:
      assert (0);
    }

  /*
   * standard behaviour:
   * copy generated binary value into defined position
   */

  memcpy ((VOID*) d + ItemPos (item),
	  (VOID*) source,
	  ItemSize (item));

  return TCL_OK;

 error:
  return TCL_ERROR;
}

/*
 *------------------------------------------------------*
 *
 *	ExtractSpec --
 *
 *	------------------------------------------------*
 *	Converts a  binary data into a string value
 *	according to information stored in the given
 *	BinaryItem.
 *	------------------------------------------------*
 *
 *	Sideeffects:
 *		The result is allocated.
 *
 *	Result:
 *		a standard TCL error code
 *
 *------------------------------------------------------*
 */

static int
ExtractSpec (interp, item, data, result, idx)
Tcl_Interp* interp;
BinaryItem  item;
char*       data;
char**      result;
int*        idx;
{
  if ((ItemType (item) == fmtAsciiNull) ||
      (ItemType (item) == fmtAsciiSpace))
    {
      *result = ckalloc (1 + ItemSize (item));
      if (! *result)
	goto memory;

      memcpy ((VOID*) *result,
	      (VOID*) data + ItemPos (item),
	      ItemSize (item));

      (*result) [ItemSize (item)] = '\0';
      (*idx) ++;
      return TCL_OK;
    }
  else
    {
      /*
       * standard behaviour:
       * copy binary value from defined position into a conversion buffer,
       * then generate string data.
       */

#define GET(buf) \
      memcpy ((VOID*) &(buf), (VOID*) data + ItemPos (item), ItemSize (item))

#define CONVERT(type, space, format, ptype) \
  {                                         \
    type x;                                 \
                                            \
    value = ckalloc (space);                \
    if (! value)                            \
      goto memory;                          \
                                            \
    GET (x);                                \
    sprintf (value, format, (ptype) x);     \
  }  

      char* value;

      switch (ItemType (item))
	{
	case fmtSignedChar:
	case fmtUnsignedChar:
	  {
	    char c;
	    GET (c);

	    /* ItemSize () == 1 (+1 '\0') */
	    value = ckalloc (2);
	    if (! value)
	      goto memory;

	    value [0] = c;
	    value [1] = '\0';
	  }
	  break;


	case fmtFloat:
	  {
	    float  f;
	    double d;

	    value = ckalloc (1 + TCL_DOUBLE_SPACE);
	    if (! value)
	      goto memory;

	    GET (f);
	    d = (double) f;
	    Tcl_PrintDouble (interp, d, value);
	  }
	  break;


	case fmtDouble:
	  {
	    double d;

	    value = ckalloc (1 + TCL_DOUBLE_SPACE);
	    if (! value)
	      goto memory;

	    GET (d);
	    Tcl_PrintDouble (interp, d, value);
	  }
	  break;

	case fmtSignedShort:
	  CONVERT (short, SHORT_SPACE, "%d", int);
	  break;

	case fmtUnsignedShort:
	  CONVERT (unsigned short, SHORT_SPACE, "%u", unsigned int);
	  break;

	case fmtSignedInt:
	  CONVERT (int, INT_SPACE, "%d", int);
	  break;

	case fmtUnsignedInt:
	  CONVERT (unsigned int, INT_SPACE, "%u", unsigned int);
	  break;

	case fmtSignedLong:
	  CONVERT (long, LONG_SPACE, "%ld", long);
	  break;

	case fmtUnsignedLong:
	  CONVERT (unsigned long, LONG_SPACE, "%lu", unsigned long);
	  break;

	case fmtNetworkShort:
	  {
	    short x;

	    value = ckalloc (SHORT_SPACE);
	    if (! value)
	      goto memory;

	    GET (x);
	    x = ntohs (x);
	    sprintf (value, "%d", (int) x);
	  }
	  break;

	case fmtNetworkLong:
	  {
	    long x;

	    value = ckalloc (LONG_SPACE);
	    if (! value)
	      goto memory;

	    GET (x);
	    x = ntohl (x);
	    sprintf (value, "%ld", (long) x);
	  }
	  break;

	case fmtAsciiNull:
	case fmtAsciiSpace:
	  /* handled before */
	  assert (0);
	  break;

	case fmtIllegal:
	  assert (0);
	  break;


	case fmtNull:
	case fmtSkipBackward:
	case fmtFill:
	  /*
	   * Ignore location movements here, as the positions
	   * of the data items were computed before (-> PreparePacking).
	   */

	  /* don't fall through to standard behaviour !! */
	  return TCL_OK;
	  break;

	default:
	  assert (0);
	}

      *result = value;
      (*idx) ++;
    }      
      
  return TCL_OK;
      
 memory:
  Tcl_AppendResult (interp, "not enough memory to build unpack result", 0);
  /* fall through to cleanup */
      
  return TCL_ERROR;
}

