#ifndef BLOB_H
#define BLOB_H

/*
 * blob.h - externally visible facilities of Tcl-BLOB
 *
 *
 * Copyright (c) 1995 Andreas Kupries (aku@kisters.de)
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL I BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL,
 * INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS
 * SOFTWARE AND ITS DOCUMENTATION, EVEN IF I HAVE BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * I SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND
 * I HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 * ENHANCEMENTS, OR MODIFICATIONS.
 *
 * CVS: $Id: blob.h,v 1.5 1996/01/21 13:52:43 aku Exp $
 */


#include <tcl.h>

/*
 * Definition of module version
 */

#define BLOB_VERSION         "1.0"
#define BLOB_MAJOR_VERSION   1
#define BLOB_MINOR_VERSION   0


/*
 * General success / failure codes.
 */

#define BLOB_OK    (0)
#define BLOB_ERROR (-1)

/*
 * Exported tcl level procedures.
 *
 * ATTENTION:
 * due to the fact that cpp - processing with gcc 2.5.8 removes any comments
 * in macro-arguments (even if called with option '-C') i have to use the
 * predefined macro __C2MAN__ to distinguish real compilation and manpage
 * generation, removing _ANSI_ARGS_ in the latter case.
 */

/* Initialize extension in standard interpreter.
 * Extends the interpreter with extension-specific
 * structures and installs the globally visible
 * command of Tcl-BLOB.
 */

#ifdef __C2MAN__
EXTERN int
Blob_Init (Tcl_Interp* interp /* interpreter to initialize */);
#else
EXTERN int
Blob_Init _ANSI_ARGS_ ((Tcl_Interp* interp));
#endif

/* Initialize extension in SAFE interpreter.
 * Same as --> Blob_Init.  The only security
 * relevant operations are reading from and
 * writing to a file.  As Tcl-Handles are
 * given to these commands it is assumed that
 * they were checked and cleared beforehand.
 */

#ifdef __C2MAN__
EXTERN int
Blob_SafeInit (Tcl_Interp* interp /* interpreter to initialize */);
#else
EXTERN int
Blob_SafeInit _ANSI_ARGS_ ((Tcl_Interp* interp));
#endif

/*
 * Exported C level facilities.
 */


/*
 * Declaration of external blob-type (as opaque handle).
 */

typedef long Blob;

/*
 * Basic manipulation, i.e. construction / destruction
 */

/*
 * Creates a new blob object.
 * Memory required by internal management is allocated
 * and initialized to contain an empty blob.
 *
 * Returns:
 * The handle of the created object or:
 *
 * BLOB_ERROR:
 * Return value in case of a failure (no memory).
 */

#ifdef __C2MAN__
EXTERN Blob
Blob_Create (void);
#else
EXTERN Blob
Blob_Create _ANSI_ARGS_ ((void));
#endif

/* Removes the specified object from memory and frees
 * all memory associated with it.
 */

#ifdef __C2MAN__
EXTERN void
Blob_Delete (Blob blob /* handle of object to destroy */);
#else
EXTERN void
Blob_Delete _ANSI_ARGS_ ((Blob blob));
#endif

/*
 * Manipulation of a blobs contents.
 */


/* Remove part or all of the data stored in the object.
 * The bytes specified through range 'from' .. 'to' are
 * removed.  The value of 'from' has to be in range
 * 0 .. (size-1), 'to' has to be in range 0 .. size.  Nothing
 * will be removed for 'to <= from'.
 *
 * Returns:
 * BLOB_OK:	Operation was successful.
 * BLOB_ERROR:	Operation failed. The error text associated
 * to the failure can be obtained via a call to --> Blob_LastError.
 */

#ifdef __C2MAN__
EXTERN int
Blob_RemoveData (Blob blob /* handle of object to operate on   */,
		 int  from /* location of first byte to be removed */,
		 int  to   /* location of byte just behind the
			      last to be removed */);
#else
EXTERN int
Blob_RemoveData _ANSI_ARGS_ ((Blob blob,
			      int  from,
			      int  to));
#endif


/* Insert additional data into the object.
 * The specified data is inserted just before the byte whose
 * position is specified via 'at'. 'at' must be in range
 * 0 ... size.  A value of 'size' is a request to append the data.
 *
 * Returns:
 * BLOB_OK:	Operation was successful.
 * BLOB_ERROR:	Operation failed. The error text associated
 * to the failure can be obtained via a call to --> Blob_LastError.
 */

#ifdef __C2MAN__
EXTERN int
Blob_InsertData (Blob  blob   /* handle of object to operate on */,
		 int   at     /* location of the first byte to be shifted */,
		 int   length /* length of data to insert    */,
		 char* data   /* reference to data to insert */);
#else
EXTERN int
Blob_InsertData _ANSI_ARGS_ ((Blob  blob,
			      int   at,
			      int   length,
			      char* data));
#endif


/* Replace part or all of the data stored in the object.
 * This is a combination of --> Blob_RemoveData
 * and --> BlobInsertData.  It is equivalent to:
 *
 * - Blob_RemoveData (blob, from, to);
 * - Blob_InsertData (blob, from, length, data)
 *
 * The constraints named there apply here too.
 *
 * Returns:
 * BLOB_OK:	Operation was successful.
 * BLOB_ERROR:	Operation failed. The error text associated
 * to the failure can be obtained via a call to --> Blob_LastError.
 */

#ifdef __C2MAN__
EXTERN int
Blob_ReplaceData (Blob  blob   /* handle of object to operate on */,
		  int   from   /* location of first byte to be replaced */,
		  int   to     /* location of byte just behind the
				  last to be removed */,
		  int   length /* length of data used as replacement */,
		  char* data   /* reference to data used as replacement */);
#else
EXTERN int
Blob_ReplaceData _ANSI_ARGS_ ((Blob blob,
			       int  from,   int   to,
			       int  length, char* data));
#endif


/* Replace all data in object.
 * This procedure is equivalent to:
 *
 * - Blob_ReplaceData (blob, 0, size, length, data);
 *
 * Returns:
 * BLOB_OK:	Operation was successful.
 * BLOB_ERROR:	Operation failed. The error text associated
 * to the failure can be obtained via a call to --> Blob_LastError.
 */

#ifdef __C2MAN__
EXTERN int
Blob_SetData (Blob  blob   /* handle of object to operate on */,
	      int   length /* length of data used as replacement */,
	      char* data   /* reference to data used as replacement */);
#else
EXTERN int
Blob_SetData _ANSI_ARGS_ ((Blob blob,
			   int  length, char* data));
#endif


/* Set the prefix character of escape sequences.
 * The value of 'escapeCharacter' will be used in conversion
 * from and to escape representation of binary data from now
 * on.  The character '\0' is not allowed.
 *
 * Returns:
 * BLOB_OK:	Operation was successful.
 * BLOB_ERROR:	Operation failed. The error text associated
 * to the failure can be obtained via a call to --> Blob_LastError.
 */

#ifdef __C2MAN__
EXTERN int
Blob_SetEscape (Blob blob            /* handle of object to operate on */,
		int  escapeCharacter /* character to use as escape prefix */);
#else
EXTERN int
Blob_SetEscape _ANSI_ARGS_ ((Blob blob,
			     int  escapeCharacter));
#endif

/*
 * Inquiry procedures.
 */

/* Retrieve part or all of data stored in object.
 * Stores the length of the retrieved range in 'length'
 * and a reference to it in 'data'.  Both arguments must
 * not be NULL.  Beware: 'data' refers to memory internal
 * to the object and is therefore to be considered
 * READ ONLY.  No allocation took place.   The value
 * of 'from' has to be in range 0 .. (size-1), 'to' has to
 * be in range 0 .. size.  Nothing will be removed for 'to <= from'.
 * 
 * Returns:
 * BLOB_OK:	Operation was successful.
 * BLOB_ERROR:	Operation failed. The error text associated
 * to the failure can be obtained via a call to --> Blob_LastError.
 */

#ifdef __C2MAN__
EXTERN int
Blob_GetData (Blob   blob   /* handle of object to query */,
	      int    from   /* location of first byte to be retrieved */,
	      int    to     /* location of byte just behind the last
			       one to be retrieved */,
	      int*   length /* length of retrieved range */,
	      char** data   /* reference to retrieved data */);
#else
EXTERN int
Blob_GetData _ANSI_ARGS_ ((Blob blob,
			   int  from,   int     to,
			   int* length, char** data));
#endif


/* Retrieve currently active escape character
 *
 * Returns:
 * char-code:	Operation was successful, the return value
 * denotes the character currently used to prefix escape
 * sequences during conversion from or to escape
 * representation of binary data.
 * BLOB_ERROR:	Operation failed. The error text associated
 * to the failure can be obtained via a call to --> Blob_LastError.
 */

#ifdef __C2MAN__
EXTERN int
Blob_GetEscape (Blob blob /* handle of object to query */);
#else
EXTERN int
Blob_GetEscape _ANSI_ARGS_ ((Blob blob));
#endif


/* Inquire current size of data stored in object.
 * Stores the length of the data currently stored inside
 * the object into 'size'.  This argument must not be NULL.
 * The unit of length is 'byte's.
 *
 * Returns:
 * BLOB_OK:	Operation was successful.
 * BLOB_ERROR:	Operation failed. The error text associated
 * to the failure can be obtained via a call to --> Blob_LastError.
 */

#ifdef __C2MAN__
EXTERN int
Blob_Size (Blob blob /* handle of object to query */,
           int* size /* variable to store the size in */);
#else
EXTERN int
Blob_Size _ANSI_ARGS_ ((Blob blob, int* size));
#endif


/* Inquire text associated to error.
 *
 * Returns:
 * A reference to the text associated with the last error
 * which occured in a call to this extension with the
 * specified handle.  A reference to the empty string
 * is returned, if no error condition is present.
 */

#ifdef __C2MAN__
EXTERN CONST char*
Blob_LastError (Blob blob /* handle of object to query */);
#else
EXTERN CONST char*
Blob_LastError _ANSI_ARGS_ ((Blob blob));
#endif


/*
 * Interface to registry of conversion procedures.
 */

/*
 * prototypes for procedures converting strings to binary data
 * and vice versa
 */

#ifdef __C2MAN__
/* Convert string into binary data.
 * This is the prototype to use for procedures converting
 * strings into binary data.  The system assumes that the
 * generated binary information is NOT longer than the
 * specified string.  Because of this the conversion can be
 * done IN PLACE.
 *
 * Returns:
 * TCL_OK:	Conversion was successful.
 * TCL_ERROR:	Conversion failed. An error text is left in the
 * interpreter result area.
 */

typedef int (*Blob_CvtString2Bin) (Tcl_Interp* interp,      /* interpreter containing the blob */
				   Blob        b,           /* blob to use to retrieve additionally
							       needed information */
				   int*        out_length   /* length of given string, length of
							       binary data afterward */,
				   char*       inout_string /* string to convert, contains binary
							       data afterward */);

/* Convert binary data into string.
 * This is the prototype to use for procedures converting
 * binary data to strings.  The generated string is
 * dynamically allocated.
 *
 * Returns:
 * TCL_OK:	Conversion was successful.
 * TCL_ERROR:	Conversion failed. An error text is left in the
 * interpreter result area.
 */

typedef int (*Blob_CvtBin2String) (Tcl_Interp* interp /* interpreter containing the blob */,
				   Blob        b      /* blob to use to retrieve additionally
							 needed information*/,
				   int         length /* length of binary data to convert */,
				   char*       input  /* binary data to convert */,
				   char**      result /* reference to conversion result*/);
#else
typedef int (*Blob_CvtString2Bin) _ANSI_ARGS_ ((Tcl_Interp* interp,
						Blob        b,
						int*        out_length,
						char*       inout_string));

typedef int (*Blob_CvtBin2String) _ANSI_ARGS_ ((Tcl_Interp* interp,
						Blob        b,
						int         length,
						char*       input,
						char**      result));
#endif

/* Add conversion method to blob extension of/in interpreter.
 * Registers a new method of converting string into binary
 * data and vice versa. The option to access this method on
 * Tcl level is the name specified here, prepended with a
 * single '-' character.
 *
 * Returns:
 * TCL_OK:	Operation was successful.
 * TCL_ERROR:	Operation failed. An error text is stored in the
 * interpreter result area.
 */

#ifdef __C2MAN__
EXTERN int
Blob_AddConversion (Tcl_Interp*       interp /* interpreter to extend */,
		    CONST char*       name   /* name of conversion method */,
		    Blob_CvtString2Bin s2bFun /* function to convert strings into binary data */,
		    Blob_CvtBin2String b2sFun /* function to convert binary data into strings */);
#else
EXTERN int
Blob_AddConversion _ANSI_ARGS_ ((Tcl_Interp*        interp,
				 CONST char*        name,
				 Blob_CvtString2Bin s2bFun,
				 Blob_CvtBin2String b2sFun));
#endif

/*
 * Utility procedures required by algorithms operating on blobs.
 */

/* Retrieve handle of blob associated to command 'name'.
 *
 * Returns:
 * TCL_OK:	Operation was successful.
 * TCL_ERROR:	Operation failed. An error text is stored in the
 * interpreter result area.
 */

#ifdef __C2MAN__
EXTERN int
Blob_GetBlobHandle (Tcl_Interp* interp, /* interpreter to use during inquiry */
		    CONST char* name    /* name of blob to retrieve */,
		    Blob*       result  /* storage to write the handle to */);
#else
EXTERN int
Blob_GetBlobHandle _ANSI_ARGS_ ((Tcl_Interp* interp,
				 CONST char* name,
				 Blob*       result));
#endif


/* Convert arguments into binary data.
 * The specified arguments are processed and then transformed
 * into binary data.  This enables people implementing commands
 * operating on blobs to use the standard syntax of specifying
 * binary information introduced by this extension.  In case of
 * 'skipped' == NULL the procedure assumes that no other arguments
 * are allowed to follow the processed ones.
 */

#ifdef __C2MAN__
EXTERN int
Blob_GetInput (Tcl_Interp* interp	   /* interpreter to work in */,
	       int         argc		   /* number of arguments to process */,
	       char**      argv		   /* arguments to process */,
	       Blob        b		   /* used as target of input, used to
					      inquire additional conversion
					      information too (if necessary) */,
	       CONST char* blobName	   /* name associated to 'b' */,
	       int*        length	   /* length of generated binary data */,
	       char**      data		   /* generated binary data */,
	       int*        dataIsAllocated /* flag indicating, wether 'data' is
					      dynamically allocated (1) or not (0) */,
	       int*        skipped	   /* number of arguments processed by
					      this procedure. Allowed to be NULL */);
#else
EXTERN int
Blob_GetInput _ANSI_ARGS_ ((Tcl_Interp* interp,
			    /* arguments to process */
			    int         argc,
			    char**      argv,
			    /* additional information */
			    Blob        b,
			    CONST char* blobName,
			    /* output */
			    int*        length,
			    char**      data,
			    int*        dataIsAllocated,
			    int*        skipped));
#endif
#endif /* BLOB_H */
