/* -*-C++-*-
 * ###################################################################
 *	EvoX - evolution in	complex	systems
 * 
 *	FILE: "tcl_args.h"
 *									  created: 16/4/96 {1:40:43	pm}	
 *								  last update: 24/7/96 {7:02:56 pm}	
 *	Author:	Vince Darley
 *	E-mail:	<vince@das.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	   www:	<http://www.fas.harvard.edu/~darley/>
 *	
 * ###################################################################
 */

/*@ ---------------------------------------------------------------------------
    ---------------------------------------------------------------------------
 */
#ifndef _EvoX_tcl_args_
#define _EvoX_tcl_args_

/*      \Section{Imports}
    Import required include files
 */
#include "vstreams.h"
#include "tcl_stream.h"
#include "CppTcl.h"

/* 
 * Note: this code works both with and without exception handling.
 * I currently avoid it, which means an	extra check	is required,
 * supplied	by the macro 'NO_EXCEPTIONS(arg,TCL_ERROR)', which should
 * follow the 'arg >> done;' statement.	 The way the code is currently
 * set up, it could	be considered to trigger exception too early. 
 * Perhaps that	should be changed so '>> done' checks if an	error
 * occurred	and	triggers an	exception at that point	only.
 */

//@ ----------------------------------------------------------------------------
/* \Section{Class Declaration}
 */

//@Section: CppTcl library
//@Man: 
/** 
 *  "class tcl_args" --
 *  
 * A C++ class designed to deal effectively with C argument lists, of
 * the form 'int argc, char ** argv', which are prevalent in C-Tcl
 * interaction.
 * 
 * Backwards compatibility is maintained with minor code modifications
 * via the functions 'orig_argc(), orig_argv(int)', although it's best
 * if you just rewrite code to use the new features.
 * 
 * More C++'y usage is of the following form:
 * 
 *       tcl_object* rrandom::make_new(tcl_args& arg){
 *           arg >> arg.tmp_chars;
 *          
 *           if(!arg) {
 *               arg >> done; // so command completion works
 *               NO_EXCEPTIONS(arg,0); // required if no exceptions
 *               return new rrandom(arg,arg.tmp_chars);
 *           } else if(arg.syntax("value")=="seed") {
 *               short seed;
 *               arg >> seed >> done;
 *               NO_EXCEPTIONS(arg,0); // required if no exceptions
 *               rrandom *r = new rrandom(arg,name);
 *               r->Srand(seed);
 *               return r;
 *           } else {
 *               return 0;
 *           }
 *       }
 *      
 *       int rrandom::parse_tcl_command(tcl_args& arg){  
 *           if (!arg) { 
 *               arg >> done; // so command completion works
 *               NO_EXCEPTIONS(arg,TCL_ERROR);
 *               tcl_ << Rand() << result;
 *               return TCL_OK;
 *           } else if(arg.syntax("max")=="int") {
 *               short n;
 *               arg >> n >> done;
 *               NO_EXCEPTIONS(arg,TCL_ERROR);
 *               tcl_ << Randint(n) << result;
 *               return TCL_OK;
 *           } else
 *               return tcl_object::parse_tcl_command(arg);
 *       }
 *  
 * These two small functions will produce error messages of the following
 * informative form, with no further effort:
 *      
 *       % load libRandom
 *       % random
 *       random: No name or other arguments supplied.
 *       % random r
 *       r
 *       % random rr seed 123456
 *       read  partial cmd 'random rr seed' but failed to convert next 
 *       argument '123456' to type 'short'; syntax should be 
 *       'random rr seed value'
 *       % random rr seed 12345
 *       rr
 *       % rr
 *       0.525446
 *       % rr
 *       0.250916
 *       % rr
 *       0.893191
 *       % rr int 2345
 *       1349
 *       % rr int
 *       wrong # args: should be 'rr int max'
 *       % rr int 3456678 757
 *       read partial cmd 'rr int' but failed to convert next argument 
 *       '3456678' to type 'short'; syntax should be 'rr int max'
 *   
 * The really cool bit is that all the comparisons and syntax messages
 * which your C++ code contains are remembered by this class, so if an
 * error occurs in a Tcl script, a useful error message is generated
 * automatically.  Therefore the Tcl interface to your C++ code is self-
 * documenting!  
 *    
 * Command completion is also supported.  To support this facility,   
 * your class must make correct use of the 'arg >> done' command.  
 * Make sure (a) you always use 'done', and (b) you never perform
 * any action at all until after the 'done' command.
 *    
 *   --Version--Author------------------Changes-------------------------------
 *     1.0     <vince@das.harvard.edu> original
 *   -------------------------------------------------------------------------
 */
class tcl_args {
  public:
	//@Man: Manipulators 
	//@{
	/** These are used to manipulate the state of the argument
	 *  parsing stream.
	 */
	/// Argument parsing complete. No more arguments should exist
	/** doc-done */
    friend tcl_args& done(tcl_args&);
	/// The next argument is optional
	/** doc-optional */
    friend tcl_args& optional(tcl_args&);
	/// Skip the next argument
	/** doc-skip */
    friend tcl_args& skip(tcl_args&);
	//@}
	
	/// This class's parse_tcl_command requires some inside information
	/// to work its magic.
	friend class cpptcl_control;
	
	/// Template friend is used for all read operations
	/** 
	 * This	would be very useful, but it doesn't work!?	However	it is in
	 * a recent	ANSI C++ draft,	so at some point I can use it and switch
	 * some	functions below	out	of this	class's	public interface.
	 */
	//template <class T> friend tcl_args& operator>>(tcl_args& arg, T& into);

  public:
	/// Main constructor.
	/** 
	 * The usual constructor --	it takes a tcl_stream for tcl-interaction,
	 * the argument	list it	obviously needs, and a tcl_object (for some
	 * other purpose).
	 */
	tcl_args(tcl_stream&, int argc=0, char* argv[]=0, const class tcl_object*o=0);
	/// Copy constructor.
	/** 
	 * Copy	constructor	-- perhaps if you're going to do something crazy with
	 * a tcl_args object which doesn't belong to you, then make	a copy and
	 * mess	with that.  This copy constructor only copies the arguments
	 * which haven't yet been used.
	 */
	tcl_args(const tcl_args&, const class tcl_object*o=0);
	virtual ~tcl_args(void);
	/// Manipulator
	//@See: done optional skip
	/** 
	 * To allow	tcl_args manipulation via 'done', 'optional' and 'skip'
	 */
	tcl_args& operator >> (tcl_args& (*f)(tcl_args&));
	/// Casting operators
	//@{
	/** 
	 * These mean you can treat	me like	a tcl_stream/Tcl_Interp if you like
	 */
	operator tcl_stream& (void) const;
	operator Tcl_Interp* (void) const;
	tcl_stream& tcl(void) const;
	Tcl_Interp* interpreter(void) const;
	//@}

	//@Man: Argument syntax checking and string comparison
	//@{
	
	//@Man: Syntax and help texts
	//@{
	/** 
	 * Use any of these four to declare the correct
	 * syntax for the following	comparison with	'=='
	 */
	/// Declare the command syntax
	tcl_args& syntax(const char* syntax);
	/// Shorthand to declare command syntax
	tcl_args& operator()(const char* syntax);
	/// Declare command syntax and help text
	tcl_args& operator()(const char* syntax, const char* help_text);
	/// Declare command help text only
	/** 
	 * Attach some help	text for a following "-help" or "-h" query
	 */
	tcl_args& help(const char* help_text);
	//@}

	
	/// Is the next command item the given string?
	bool operator == (const char* given_arg);

	/** 
	 * Typical use of the above	six	functions is as	follows:
	 *		if(arg("my syntax","my help")=="cmd"||"equivalent cmd")...
	 */
	//@}

	/// Remove the given argument.  It must match.
	/** 
	 * Remove the given	argument.  Signals a syntax	error if
	 * the next	argument doesn't match exactly.
	 */
	tcl_args& operator -= (const char*);
	/// Remove the next argument
	//@See: skip
	/** 
	 * Remove the next argument	(same as 'skip'), signalling
	 * an error	if no such argument	exists.
	 */
	tcl_args& operator --(int);
	/// How many args left to parse?
	int args_left(void) const;
	/// What's the next argument?
	const char* peek(void) const;
	/// Is the next argument a real
	bool peek_is_float(void) const;
	/// Is the next argument an integer
	bool peek_is_int(void) const;
	/// Is the next argument a string (i.e. not a number)
	bool peek_is_string(void) const;
	
	//@Man: Remaining argument list manipulation
	//@{
	/** 
	 * These deal only with unprocessed arguments; they
	 * act like	!argc_left, argv[next], argc_left, 
	 * argv+next respectively.
	 */
	/// Are there any arguments left?
	bool empty(void) const;
	/// What's the n'th argument remaining?
	char* operator[] (int n) const;
	/// How many arguments are left?
	operator int (void) const;
	/// Pointer to array of remaining arguments
	operator char** (void) const;
	//@}
	//@Man: Original argument list manipulation
	//@{
	/** 
	 * These deal with the entire argument list; they
	 * act like	!argc, argv[x], argc, argv	respectively
	 */
	/// Are there any arguments at all?
	bool operator! (void) const;
	/// Returns '#argv[item]#'
	char* orig_argv(int item) const;
	/// Returns '#argc#'
	int orig_argc(void) const;
	/// Returns '#argv#'
	char** orig_argva(void) const;
	//@}

	/// No commands remain to test.  The user gave an illegal command.
	/** 
	 * There are no	more allowed matches to	what we've seen
	 * so far.	Triggers an	exception.
	 */
	int no_match(void);

	/// Deal with an objects `name'
	//@{
	/** 
	 * Since we	deal with Tcl objects and commands,	we create
	 * things with names.  Usually the user	supplies the name
	 * as the first	argument argv[0], but sometimes	we wish
	 * to over-ride	that choice, or	generate a unique name
	 * automatically.  Calling 'name' will return 'argv[0]',
	 * unless it was over-ridden with a	prior call to 'setName'.
	 */
	void setName(const char*);
	const char* check_name(void) const;
	/** 
	 * 'name' actually removes the name	from the argument
	 * list	by calling 'skip()'.  Use 'check_name()' if	you
	 * just	wish to	examine	the	name.
	 */
	const char* name(void);
	//@}
	
	void signal_error(int err);
	/// Can be used to check if my internal stream is ok after a read
    bool arg_read_ok(void) const { return (my_args_.good() ? true: false); }
  public:
	//@Man: Items for external read functions
	/** 
	 * The following items are a part of my	interface which
	 * should only be used by 'operator	>>'	functions which
	 * extend my functionality.	 Once I	can	use	a template
	 * friend above, these will be made protected again.
	 */
	//@{
		// list of error types
		enum tcl_args_err { No_Err=0, Conversion=1, No_match=2, Syntax=3,
							Too_many_args=4, Cpp_constructor=5, Help=6,
						    Finding_completions=7 };
		// perform any post-read operations
		void read_done(void);
		// read a constant string from the argument list
		void const_string_read(const char*&);
		// inform me of the name of the thing that is trying to be read
		void set_conversion_type(const char* n) {conversion_err_type = n;}
		
		//@Man: Error streams
		//@{
		/** 
		 * These are used internally to	allow clever generation
		 * of error	messages when parsing fails.
		 */
		/// what's been parsed (used for error messages)
		ostrstream parsed_so_far;
		/// what went wrong (used for error messages)
		ostrstream failed_args;
		//@}

		//@Man: Interface for templated read operations
		//@{
		 
        /* 
         * Similarly the following two can be used to create new read
         * functions such as the following:
         *     tcl_args& operator >> (tcl_args& arg, info_observable& into) {
         *         arg.set_conversion_type("info_observable");
         *         arg.check_read() >> into;
         *         arg.check_after();
         *         arg.parsed_so_far << into << " ";
         *         arg.read_done();
         *         return arg;
         *     }
         */
		/// Read from stream, but check if there's anything to be read first
		istream& check_read(void);
		/// Check if the read went ok
		tcl_args& check_after(void);
		/// Have we been told this argument is optional?
		bool is_optional_arg(void) const;
		/// Just pretend to read (discard the argument)
		void pretend_read_from_me(void);
		//@}
	//@}
		
  protected:
	/// Storage for argument list
	strstream my_args_;
	
	/// Name of current argument type -- used for error messages
	const char* conversion_err_type;
	/// Often the first argument is an object name which may be over-ridden
    const char* myName;
	/// Skip an argument --- the friend 'skip' is for external, public use
    void skip(void);
	/// The next argument is optional
	bool optional_arg;
	/// Not sure?!
	void remove_from_buffer(void);
	
	/// We've got an error, with the given error code
	virtual void internal_signal_error(int err);
	
	//@Man: Internal methods to signal errors
	//@{
	/** 
	 * There are currently five	types of error,	signalled by the enum
	 * defined above, and each having a	different kind of error	message.
	 * They	should only	be called via their	wrapper	'signal_error'.
	 */
	/// Too many arguments were given, often called from 'done'
	void args_too_many_err(void);
	/// No command matched the given arguments
	void args_no_match_err(void);
	/// An argument failed to convert to the correct type
	void args_conversion_err(void);
	/// Too few arguments usually
	void args_syntax_err(void);
	/// ??
	void args_cpp_constructor_err(void);
	/// Not really an error. The user gave '-h' as an argument to ask for help
	/** 
	 * Finally there is a 'help-signal', if the user gives a
	 * '-h'	or '-help' argument, some help text	will be	returned if	the
	 * programmer kindly made it available.	 See 'help'	or 'operator()'
	 */
	void args_help_signal(void);
	//@}
	
	//@Man: Internal methods to deal with completions
	//@{
	///
    bool finding_completions;
	///
    enum completion_type_ { No_completion = 0, Unique_completion = 1, 
    	    Have_completion = 2, Already_complete = 3 };
    ///
    completion_type_ completion_type;
    ///
    short completion_orig_len;
    ///
    short completion_extra_len;
    ///
    ostrstream completion;
	///
	void start_completion_check(void);
	///
	void end_completion_check(void);
	/// Used internally to handle command completion
	void args_completion_signal(void);
	///
	void remember_completion(const char* s);
	//@}

	/// Add a string to a stream, removing trailing spaces or commas or eol
	static void add_without_trailing_punctuation(ostream& o, const char* ch);
	/// Empty my arguments.  So I can be reused
    void clear_args(void);
    /// Put the given ostrstream back into pristine condition
	static void reset_stream(ostrstream& o);
	
	/// My Tcl interpreter stream
	tcl_stream& tcl_;

	/// The number of arguments left to parse
	int args_left_;
	/// The command I'm currently comparing against
	const char* cur_cmd;
	/// The syntax of the current command
	const char* cmd_syntax;
	/// Help text for the current command
	const char* help_text;
	/// Already retrieved item from argument list
	bool got_cmd;

  private:
	int argc_;
	char** argv_;
	
  public:
	/// Are we doing command completion rather than real evaluation?
	bool completing(void) const { return finding_completions;}
	
	/// Has an error already been signalled?
	bool haveErr;
	/// the command we're currently checking against (with operator ==)
	const char* current_cmd(void) const { return cur_cmd; }
	
	//@Man: Temporary read variables for external use
	//@{
	/** 
	 * These are useful	to avoid incessant declarations	of local variables,
	 * so rather than '#const char* n; arg >> n >> done; ...(n)...#', you can	
	 * just	use	'#arg >> arg.tmp_chars >> done;	...(arg.tmp_chars)...#'
	 */
	/// Constant string
	const char* tmp_chars;
	///
	short tmp_short;
	///
	long tmp_long;
	///
	float tmp_float;
	///
	double tmp_double;
	///
	bool tmp_bool;
	///
	char tmp_char;
	//@}
	
};

//@Section: CppTcl library
//@Man: Reading from 'tcl_args'
//@{
/** 
 * This	is the basic 'read'	call which is used to extract arguments:
 * 
 *		template <class T> tcl_args& operator>> (tcl_args& arg, T& into);
 * 
 * I currently can't use the template _definition_ due to ambiguities.
 * However it is really	used behind	the	scenes for instantiation of these
 * functions.  The template	can	be used	to instantiate _any_ class which
 * reads correctly from an istream -- just '#\#include "tcl_args_t.cc"#'
 * when	it needs instantiating.
 */
///
tcl_args& operator >> (tcl_args& arg, char& v);
///
tcl_args& operator >> (tcl_args& arg, short& v);
///
tcl_args& operator >> (tcl_args& arg, long& v);
///
tcl_args& operator >> (tcl_args& arg, float& v);
///
tcl_args& operator >> (tcl_args& arg, double& v);

//@Man: Template specialisations for types we know more about:
//@{
///
tcl_args& operator >> (tcl_args& arg, const char*& i);
///
tcl_args& operator >> (tcl_args& arg, char*& i);
///
tcl_args& operator >> (tcl_args& arg, bool& v);
///
tcl_args& operator >> (tcl_args& arg, class tcl_object*& into);
//@}

///
/**
 * Returns the name	of a given type.  When RTTI	is working fine,
 * we can replace this with	some ANSI C++ rtti call.
 */
template <class T> const char* GETTYPE(T&);
//@}

/*
 * Include all inline functions	here.  We can do this directly in the header
 * file	because	this class doesn't interlock too heavily with others.
 */
#include "tcl_args.icc"


#endif
