/* -*-C++-*-
 * ###################################################################
 *	EvoX - evolution in	complex	systems	 
 * 
 *	FILE: "tcl_object.h"
 *								   created:		1/23/95	{11:50:03 pm}
 *								   last	update:	15/7/96 {11:02:18 pm}
 *	Author:	  Vince	Darley 
 *	 E-mail: vince@das.harvard.edu
 *	   mail:  Divison of Applied Sciences, Harvard University
 *			  Cambridge	MA 02138
 *	
 *	Description: Based on borrowed class for Tcl/C++ interfacing
 * 
 * 
 *	History
 * 
 *	modified by	 rev reason
 *	-------- --- --- -----------
 *	1/23/95	 VMD 1.0 original
 *	_/_/_
 * ###################################################################
 */

#ifndef _EvoX_tcl_object_
#define _EvoX_tcl_object_

#include <tcl.h>
#include "tcl_stream.h"
#include "CppTcl.h"

//@Section: CppTcl library
//@Man: C interface between Tcl and C++ code
/**
 * -------------------------------------------------------------------------
 *	 
 * "Tcl_ParseObjectCommand", "Tcl_DeleteObject" --
 *	
 *  These two functions are the only ones actually called by Tcl.
 *  They simply act as a go-between with the C++ class.  It would be nice
 *  if they were static member functions, but not all compilers let such
 *  functions have "C" linkage, so this is the most compatible bet.
 * -------------------------------------------------------------------------
 */
//@{
/// Tcl passes the command line of an object 'clientData' to this function.
extern "C" int Tcl_ParseObjectCommand(ClientData clientData, 
									  Tcl_Interp* interp,
                                      int argc, char* argv[]);
/// Tcl calls this when 'clientData' should be deleted.
extern "C" void Tcl_DeleteObject(ClientData clientData);
//@}

#include "tcl_args.h"
/* 
 * If there's no exception handling, you _must_	put
 * 'NO_EXCEPTIONS(arg,TCL_ERROR)' or 'NO_EXCEPTIONS(arg,0)'
 * after 'arg >> done;', as	appropriate.
 */
#ifdef NO_EXCEPTION_HANDLING
#define throw(x)
#define NO_EXCEPTIONS(a,b) if(a.haveErr) return b;
#else
#define NO_EXCEPTIONS(a,b)
#endif

//@Section: CppTcl library
/// Useful define to simplify code
#define DONE(a) a >> done; NO_EXCEPTIONS(a,TCL_ERROR);

//@Section: CppTcl library
//@Man:
/** 
 * -------------------------------------------------------------------------
 *   
 * "class tcl_object" --
 *  
 *  This file defines the interface to tcl_object.  A tcl_object is a handy 
 *  abstraction for interfacing C++ to tcl's procedural objects.  When a 
 *  tcl_object is created, it causes a tcl object to be created with 
 *  clientData pointing to the tcl_object.  When the tcl object is invoked 
 *  (from tcl), the arguments are passed to tcl_object::parse_tcl_command() 
 *  via the clientData associated with the tcl command.
 * 
 *  A tcl_object knows the name of its tcl command, and the interpreter under 
 *  which it was created.  instances of tcl_object can be created, but the
 *  basic object in Tcl would have a single method 'rename' to change its
 *  name.  Therefore you should derive from this class and add your own
 *  'parse_tcl_command' function of the following form:
 *
 *	   int my_object::parse_tcl_command(tcl_args& arg){
 *		   if (arg("syntax","help text")=="cmd") {
 *			   // extract any more arguments	
 *			   DONE(arg);
 *			   // execute 'cmd'	
 *			   return tcl_;
 *		   ...
 *		   } else if(arg(...)=="...") {
 *		   ...
 *		   } else {
 *			   return parent_object::parse_tcl_command(arg);
 *		   }
 *	   }
 *
 *  A tcl_object may be created explicitly in C++, or it may be created
 *  from Tcl by creating a companion 'tcl_class', whose sole purpose is
 *  the creation of instances of 'tcl_objects'.  In general if your class
 *  is designed as a real object of which there will be many instances,
 *  then you will want to create a tcl_class companion.  If you require
 *  just a single instance, then you should just create that in C++ and
 *  be done.  
 *   
 * --Version--Author------------------Changes-------------------------------  
 *    1.0     <vince@das.harvard.edu> original
 * -------------------------------------------------------------------------
 */
class tcl_object{
	friend int Tcl_ParseObjectCommand(ClientData clientData, 
									  Tcl_Interp* interp,
                                      int argc, char* argv[]);

  public:

	/// We need to know our interpreter and the name of our tcl command
	tcl_object(tcl_stream&, const char* name);
	/// Although these can be read from an arg list if desired.
	tcl_object(tcl_args& arg);

	/// Destroys my associated command in the Tcl interpreter
  	virtual ~tcl_object();

	/// To be supplied by derived classes
   /** 
	* Derived classes should supply	this function. It looks through 
	* the arguments	and	does whatever it wants.	 The intention is for 
	* parse_tcl_command	to decide which	member function	to call, 
	* prepare the arguments	to that	member function	(perhaps 
	* converting strings to	ints or	suchlike), then	call the member	
	* function to do the actual	work.  #parse_tcl_command# should just 
	* be a parser, as the name implies.
	*/
    virtual int parse_tcl_command(tcl_args& arg);
	/// Called after object construction to deal with command line args
	/**
	 * You may wish	to over-ride it, although if you derive	
	 * from	'cpx_with_info'	in the Cpptclextra package,	default	
	 * configuration handling is automatic
	 */
	virtual int parse_configuration_values(tcl_args& arg) {
		DONE(arg); return TCL_OK;
	}
	
    /// query our stream
    tcl_stream& get_tcl_stream(void) const  { return tcl_;}
    
	/// query the name of our tcl command
    char* get_tcl_command(void) const  { return tcl_command;}

	/// query the interpreter we were defined under
    Tcl_Interp* get_interp(void) const  { return tcl_.interpreter();}

  protected:
	/// stream to evaluate tcl commands
    tcl_stream& tcl_;

	/// the Tcl command name
    char* tcl_command;
    
    /// Copy constructor
    tcl_object(const tcl_object&);

  private:
	void change_name_to(const char* newname);
	void make_my_command(const char* name);

};


//@Section: CppTcl library
/// Find a Tcl object with a given name, or return zero.
/** 
 * -------------------------------------------------------------------------
 *	 
 * "getCpptclObjectByName" --
 *	
 *  A utility function that returns a tcl_object given the name of a tcl 
 *  command (or NULL if no such command exists).  
 * -------------------------------------------------------------------------
 */
tcl_object* getCpptclObjectByName(tcl_stream& interp, const char* name);

/* 
 * Copy	a string
 */
char* mystrdup(const char* source);

// We can include this here because the inlines don't interlock
// with other classes.
#include "tcl_object.icc"

#endif
