/* -*-C++-*-
 * ############################################################################
 *	EvoX - evolution in	complex	systems								
 *	
 *	FILE: "cpptcl_metaobject.cc"
 *										   created:	12/14/93 10:12:27 AM
 *									   last	update:	21/7/96 {11:18:45 pm}
 *	  by: Vince	Darley
 *			 E-mail: vince@das.harvard.edu
 *			   mail:   Divison of Applied Sciences,	Harvard	University
 *					Cambridge MA 02138
 * 
 *	See	header file	for	further	information
 * 
 * ############################################################################
 */

#include "cpptcl_metaobject.h"
#include "tcl_class.h"
#include "object.h"
#include "cpptcl_type.h"

#include <string.h>
#include <strstream.h>
#include <iostream.h>

int cpptcl_metaobject::new_object_type(tcl_stream& interp, const char* tcl_command, 
				    new_object_fn func, const char* type, 
				    const char* parent_name, short num_of_parents) 
{
	if(!find_type(type)) {
	    object* o = new object(tcl_command, func, type);
		return new_object_type(interp,o,parent_name, num_of_parents);
	} else {
		// we have it already
		return TCL_OK;
	}
}
    
int cpptcl_metaobject::new_object_type(tcl_stream& interp, const char* type, 
				    const char* parent_name, short num_of_parents) 
{
	if(!find_type(type)) {
	    object* o = new object(type);
		return new_object_type(interp,o,parent_name, num_of_parents);
	} else {
		// we have it already
		return TCL_OK;
	}
}
    
int cpptcl_metaobject::new_object_type(tcl_stream& interp, object* o,
									const char* parent_name, 
									short num_of_parents) 
{
    if(interp.interpreter() != get_interp()) {
		interp << tcl_command << ": Called from the wrong interpreter" << tcl_error;
		return TCL_ERROR;
    }
    
    if(!parent_name) {
		// derived from nothing, so we attach it to our base
		parent_name = _type;
    }
    if(num_of_parents ==0) {
		 return TCL_ERROR;
    }
    
    for(short i=0; i<num_of_parents;i++) {
	/* 
	 * we let ourselves	remove constness internally. We	could avoid	it with
	 * a bit of	code duplication.
	 */
		object* parent = (object*) find_object(parent_name);
		if(parent) {
			o->parent_list.append(parent);
			parent->append(o);
			parent_name += strlen(parent_name)+1;
		} else {
			cerr << "No parent '" << parent_name << "' exists "
				 << "(trying to add object type '" << o->type << "')." << ends;
		}
    }	
	
    return TCL_OK;
}


const char* cpptcl_metaobject::command_for(const char* type) const {
    const object* o = find_object(type);
    return ( o ? o->tcl_command : 0 );
}

bool cpptcl_metaobject::is_of_type(const object * o, const char* type) const {
    if(o->type == type)
		return true;
    else {
		if(o->parent_list.isNonEmpty())
			return is_of_type(o->parent_list.headConst().item(),type);
		else
			return false;
    }
}

bool cpptcl_metaobject::is_of_type(const char* t1, const char* type) const {
	// you really shouldn't call this with zero, but:
	return (t1 ? is_of_type(find_object(t1),type) : false );
}

const object* cpptcl_metaobject::find_object(const char* parent_name) const {
	// if we don't know about the base object yet, then return zero
    return (base_object ? find_object(base_object,parent_name): 0);
}

cpx_type cpptcl_metaobject::find_type(const char* type_name) const {
    // we must loop through everyone, trying 
    // to find a type which looks the same.
    const object* o = find_object(type_name);
    if(o)
		return o->type;
    else
		return 0;
    
}

const object* cpptcl_metaobject::find_object(const object* from, 
										  const char* parent_name) const {
/* 
 * Note	this first comparison could	just be	'==' not 'strcmp' except
 * we don't	need speed here, but more importantly, 'find_type' needs a 
 * strcmp, so we just use the one function for both tasks.
 */
    if(!strcmp(parent_name,from->type)) {
		return from;
	} else  {
		for(list_pos<object*> scan = from->headConst();scan;++scan) {
	    	if(const object* ch = find_object(scan.item(), parent_name))
				return ch;
		}
		return 0;
    }
}
	
object* cpptcl_metaobject::choose_object_type(void) {
    return choose_object_type(base_object);
}

object* cpptcl_metaobject::choose_object_type(object* parent) {
    if(parent->isEmpty())
		return parent;
    ;//user->new_input();
	
    ;//user->options << parent->name << ends;
    for(list_pos<object*> scan = parent->headConst();scan;++scan) {
		;//user->options << a->name;
		if(scan.item()->isNonEmpty())
			;//user->options << "...";
		;//user->options << ends;
    }
    short choice = 1;//user->which_option();
	
    if(choice==1)
		return parent;

    return choose_object_type(parent->nth_element(choice-1));
}


void cpptcl_metaobject::link_in_more_objects(void) {
/* 
 * Ask for a .cpx file	containing object information
 * Ask for a .o	file with the code to link
 * 
 * Dynamically link	in the .o
 * Loop	over information given in .cpx	declaring each
 * new object type to myself, with the pertinent pointers.
 * 
 * Finally exit
 */

}

void cpptcl_metaobject::list_descendants(const object* o) const {
	for (list_pos<object*> p = o->headConst(); p; ++p) {
		if (p.item()->length())
			tcl_ << "{" << append;
		tcl_ << p.item()->type << lappend;
		tcl_ << " " << append;
		if (p.item()->length()) {
			tcl_ << "{" << append;
			list_descendants(p.item());
			tcl_ << "}} " << append;
		}
	}
}

int cpptcl_metaobject::parse_tcl_command(tcl_args& arg){	
    if (arg("?fromType?","list of direct descendants")=="listTypes") {
		// Note this could return nothing at all
		const object* o = base_object;
		arg >> optional >> o >> done;
		NO_EXCEPTIONS(arg,TCL_ERROR);		
		for (list_pos<object*> p = o->headConst(); p; ++p)
			tcl_ <<  p.item()->type << lappend;
		return TCL_OK;
    } else if (arg("type","list all descendant types")=="listDescendants") {
		// Note this could return nothing at all
		const object* o = base_object;
		arg >> optional >> o >> done;
		NO_EXCEPTIONS(arg,TCL_ERROR);
		if(o->length()) list_descendants(o);
		return TCL_OK;
    } else if (arg("type","test if this cpx_type has further derived types")
    			=="hasDescendants") {
	  	const object* o;
	  	arg >> o >> done;
		NO_EXCEPTIONS(arg,TCL_ERROR);
		tcl_ << (o->length() != 0) << result;
		/* 
		 * if(o->length() != 0)
		 *	   tcl_	<< "1" << result;
		 * else
		 *	   tcl_	<< "0" << result;
		 */
		return TCL_OK;
    } else if (arg("type","list parent types")=="listParents") {
		// Note this could return nothing at all
	  	const object* o;
	  	arg >> o >> done;
		NO_EXCEPTIONS(arg,TCL_ERROR);
		// list parents of the object whose type we're given
		for(list_pos<object*> p(o->parent_list);p;++p){
			tcl_ <<  p.item()->type << lappend;
		}
		return TCL_OK;
    } else if (arg("type","list all ancestral types")=="listAncestry") {
		// Note this could return nothing at all
	  	const object* o;
	  	arg >> o >> done;
		NO_EXCEPTIONS(arg,TCL_ERROR);
		// list parents of the object whose type we're given
		while (o->parent_list.isNonEmpty()) {
			o = o->parent_list.headConst().item();
			tcl_ <<  o->type << lappend;
		};
		return TCL_OK;
    } else if (arg("type1 type2","tests whether one type descends from another")
    			=="isa") {
		cpx_type t1,t2;
		arg >> t1 >> t2 >> done;
		NO_EXCEPTIONS(arg,TCL_ERROR);
		tcl_ << BOOL is_of_type(t1,t2) << result;
		return TCL_OK;
    } else if (arg("type","the creation command for an object of the given type")
    			=="commandFor") {
	  	const object* o;
	  	arg >> o >> done;
		NO_EXCEPTIONS(arg,TCL_ERROR);
		if(o->tcl_command) 
			tcl_ <<  o->tcl_command << result;
    	return TCL_OK;
    } else  // if we don't recognize the command, see if cpx_with_info does
		return tcl_object::parse_tcl_command(arg);
}

const char* cpptcl_metaobject::_type = "All Objects";

cpptcl_metaobject::cpptcl_metaobject(tcl_args& arg)
	:tcl_object(arg)
{
	base_object = new object(0, (new_object_fn) 0, cpptcl_metaobject::_type);
	_global_cpptcl_metaobject = this;
}

cpptcl_metaobject::cpptcl_metaobject(tcl_args& arg, cpptcl_metaobject* from)
	:tcl_object(arg)
{
	if(_global_cpptcl_metaobject == 0) {
		base_object = new object(0, (new_object_fn) 0, cpptcl_metaobject::_type);
	} else {
		assert(from == _global_cpptcl_metaobject);
		base_object = from->base_object;
		delete from;
	}
	_global_cpptcl_metaobject = this;
}

cpptcl_metaobject::~cpptcl_metaobject(void) {
}

ostream& cpptcl_metaobject::write_to_stream(ostream& o) {
	return o;
}

istream& cpptcl_metaobject::read_from_stream(istream& i) {
	return i;
}
	
int declare_abstract_cpx_object(tcl_stream& interp, const char* type, 
								 const char* parents, short num_of_parents) {
    return _global_cpptcl_metaobject->new_object_type(interp,
    											type,
    											parents,
    											num_of_parents);
}

void setup_for_object(tcl_stream& interp, 
					  const char* type,
					  const char* &tcl_command, 
					  const char* &short_name)
{
	// We check if the user's supplied a name
	if(!tcl_command) {
		//we create one from the object's type
		
		//call the function cpxToTcl
		// it will ordinarily remove any whitespace
		ostrstream str;
		short i=0;
		while(type[i]){
			if(type[i] != ' ' && type [i] != '.')
				str << type[i];
			i++;
		}
		str << ends;
		tcl_command = str.str();
		
	}
	
	if(!short_name)  {
		// we create one of these too
		
		// call the function cpxToTclShortAndLower
		ostrstream str;
		short i=0;
		while(type[i]){
			if(type[i] != ' ' && type [i] != '.')
				str << type[i];
			i++;
		}
		str << ends;
		short_name = str.str();
		//str << "uplevel cpxToTclPickShortAndLower {" << type << "}" << ends;
		//Tcl_Eval(interp,str.str());
		
		// this returns a name, from which we must pick the shortest
		// we want. I use
		//short_name = mystrdup(interp->result);
		
	}
	
	// Now just set up a couple of tcl data structures
	// set cpxType(tcl_command)  { type }
	// set cpxName(tcl_command)  { shortname }
	/* 
	 * ostrstream str3;
	 * str3	<< "set	cpxType(" << tcl_command << ")	 {"	
	 *		<< type	<< "}" << ends;
	 * Tcl_Eval(interp.interpreter(),str3.str());
	 * Tcl_ResetResult(interp.interpreter());
	 */
	interp << "set cpxType(" << tcl_command << ")  {" 
		   << type << "}" << eval;
	interp.ResetResult();

	interp << "set cpxName(" << short_name << ")  {" 
		 << short_name << "}" << eval;
	interp.ResetResult();
	
}

int declare_cpx_object(tcl_stream& interp, const char* tcl_command, 
						new_object_fn func, const char* type, 
						const char* parents, const char* short_name, short num_of_parents) {
    // If there's no func it's an abstract class, so tcl shouldn't know
    // about it except for it's name, which it gets from other places
    if(func) {
		setup_for_object(interp,type,tcl_command,short_name);	
		new tcl_class(interp, tcl_command, func);
    }

    return _global_cpptcl_metaobject->new_object_type(interp, tcl_command, func,type,parents,num_of_parents);
}

