/* -*-C++-*-
 * ###################################################################
 *	EvoX - evolution in	complex	systems
 * 
 *	FILE: "cpx_with_info.h"
 *									  created: 21/6/95 {8:30:35	am}	
 *								  last update: 23/7/96 {10:35:41 am} 
 *	Author:	Vince Darley
 *	E-mail:	<mailto:vince@das.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	   www: <http://www.fas.harvard.edu/~darley/>
 *	
 *	Description: 
 * 
 * 
 *	History
 * 
 *	modified by	 rev reason
 *	-------- --- --- -----------
 *	21/6/95	VMD	1.0	original
 * ###################################################################
 */

#ifndef _EvoX_cpx_with_info_
#define _EvoX_cpx_with_info_

#include "cpx_base.h"

class info_observable;

//@Section: CppTclExtra library
///
/* 
 * -------------------------------------------------------------------------
 *   
 * "cpx_with_info" --
 *  
 *  An object which contains others.  In particular it may contain pieces
 *  of information (C++ data members) which are transparently accessible
 *  from Tcl.  It may also contain other complete objects with their own
 *  contents.
 * -------------------------------------------------------------------------
 */
class cpx_with_info: public cpx_base {
  friend class cpx_with_container;
  friend class info_base;

  protected:
	///
	cpx_error add_info(class info_base*);
	///
	cpx_error remove_info(class info_base*);
		
  public:
	///
	static const char* _type;
	///
	cpx_type type(void) const;
	///
	cpx_with_info(tcl_args& arg);
	///
	~cpx_with_info(void);
	///
	/** 
	 * Return a	list of	the	names of all information sources I
	 * contain,	with spaces	converted to underscores if	desired.
	 */
	void list_observables(tcl_stream& t, bool convert_spaces);

	//@Man: Adding and removing objects
	//@{
	///
	bool can_add(const char*) const { return false;}
/** 
 * The important thing is for an object	to check the type of the object
 * being added and reject it if	it is inappropriate	-- remember we could
 * be trying to	add	a simulation, any object, information, an event	or
 * a call-in.
 */
/** 
 * Actually	the	way	things are right now is	that 'can_add' must
 * be called first.	'add' and 'remove' do no checking at all.
 */
    /// 'add' can signal an error and abort the procedure
    virtual cpx_error add_object(class cpx_object*);
    /// 'remove' can only signal a warning, and must accept the removal
    virtual cpx_error remove_object(class cpx_object*);
	//@}

	///
	virtual const cpx_with_info* has_container(void) const { return 0;}
	
//@Man: Finding contained information
//@{
	///
	/** 
	 * Returns the piece of	info with this name
	 * from	my contents. Returns '0' if no such information.
	 */
    info_observable* information(const char* name) const;
	///
    info_observable* search_down_for_information(const char* name) const;
	/**
	 * These two have to compare strings to match, unlike the
	 * above which compare addresses.
	 */
	///
    info_observable* slow_information(const char* name) const;
	///
    info_observable* slow_search_down_for_information(const char* name) const;
	/** These two are used for quicker lookup */
	///
    info_observable* information(const char* name, const short &element) const;
	///
    info_observable* search_down_for_information(const char* name, const short &element) const;
//@}

	///
	int parse_tcl_command(tcl_args& arg);
	///
	int parse_configuration_values(tcl_args& arg);
	
  protected:
	///
/** 
 * This	is used	for	any	graphs or inspectors which
 * get added to	me.	It can also	be used	for	anything
 * else. Perhaps it	should be used for other stuff too.
 * Now it's also used for my information content.
 */
    list<class cpx_with_container*> contents;
	///
/** 
 * Can be used to set-up a quicker look-up method for
 * information sources.	Important in simulations which
 * have	a large	data-extraction	to computation ratio.
 *
 * The initialise function is called in any derived class
 * constructor which wishes to use the quicker methods -- they
 * require more memory which is why you won't alwyas want to use
 * them. The object passed is either 'this' or perhaps a
 * container which uniquely contains this object. E.g. if 
 * object A always contains a single B and a single C, then
 * it can be better for all data-extraction to occur through
 * class A. So both B and C will call this function with the
 * container 'A' as a parameter.
 */
    info_observable** quick_lookup;
	///
    void initialise_quick_lookup(cpx_with_info*);

	///
/** 
 * Do I	contain	anything which matches this	name?
 * Returns whatever object has this name, or zero otherwise.
 * If I contain multiple copies, then we just return the first.
 * (It would be easy to return a list of them, but I currently see no need)
 * (To get the container, which we often want to do, we just ask
 * the thing for its container.)
 */
    const cpx_base* does_contain(const char*) const;
    /// Cleverly looks for an object with this name on my list
    cpx_with_container* directly_contains(const char*) const;
    /// Compares the next argument with any possible '-option' string
	info_base* configuration_option(tcl_args& arg, bool config=true);
    /// Convert item name with '_' to one with spaces; 'delete []' the result.
    static char* uscores_to_spaces(const char* iname);
    /// Convert item name with spaces to one with '_'; 'delete []' the result.
    static char* spaces_to_uscores(const char* iname, bool with_dash=false);
    
//@Man: Scanning my contents
//@{
/** 
 * Functions to	scan through my	contents looking for objects
 * of a	certain	form. Useful mostly	for	topologies,	so that	data
 * can be extracted	from all objects matching a	certain	description.
 */
	///
	void find_matches_with(const list<const char*>& l, socket_fn f, cpx_with_info* obj) const;
	///
	bool directly_contains_a_match(const list<const char*>& l) const;
	///
	void contains_all_matches(const list<const char*>& l, socket_fn f, cpx_with_info* obj) const;
	///
	void loop_match_flags(const list<const char*>& l, unsigned short& flags) const;
//@}

  private:
	///
	short size_of_quick_lookup_array;
	///
	void quick_lookup_from(const cpx_with_info* obj, short quantity);
};

#endif
