/******************************************************************
 * list.h
 *
 * Doubly-linked list template class.
 ******************************************************************
 * Change History: 
 *	 11/4/94  JAL - Added array-related list methods.
 *	 1/31/95  VMD - Added list_pos operator methods
 ******************************************************************/


#ifndef _cpx_LIST_
#define _cpx_LIST_

#include <iostream.h>
#include <stdlib.h>

/* 
   A class of doubly-linked lists, suitable for holding pointers to things.
   We make the assumption that copying T's is cheap and sensible.
   This is a good class for Animal*'s, for example; it is
   nigh-useless for Animals.

   It is your basic classical form abstraction.  In particular, lists
   may be passed around by value freely, and assigned fairly casually.
   Be careful, though:  copying a list costs linear time (plus the time
   required to copy the elements).

   This isn't the complete abstract data type of lists.  There are any
   number of functions that should be in the abstraction -- including
   some things as simple as computing the length of the list.
   (Actually, it would be sensible and easy to maintain a count of the
   length of the list as a data, so that computing the length was
   constant-time.)  Feel free to add more list functions as you wish.

   lists are mainly manipulated via list_pos<T>'s.   These capture most
   of the uses of pointers to elements of the list; e.g., you can use them
   as iterators to loop over the list; as indices to reference, modify,
   or delete elements; and so on.  Again, the class isn't complete; it lacks
   such basic functions as operator==.  Feel free to add them.

   WARNING: list_pos's are safe as long as there are no deletions on the list.
   However, a list_pos to a deleted element is NOT told that the element
   is deleted.

*/


// Some forward references.  We're defining three
// mutually-using classes.
template<class T>
class list;
 
template<class T> 
class list_pos;


// -------------------- list_node (private class) --------------------
// list_nodes are little more than cells containing a T-value 
// and pointers to the next and previous nodes.
// They are entirely private, except that the public classes 
// can refer to them.

template<class T> struct list_node;

template<class T> ostream&  operator << (ostream&, const list<T>&);

template<class T> 
struct list_node{
private:
friend class list<T>;
friend class list_pos<T>;
friend class ostream& operator <<(ostream&, const list<T>&);
  T t;
  list_node<T> *next, *prev;
  list_node(list_node<T> *prev, const T tt,  list_node<T> *next) : t(tt)
    {this->prev = prev; this->next=next;};
  ~list_node(void);  // Just delete this node
};

// --------------------  list --------------------

template<class T> 
class list {
public:

  // Create lists with 0-4 elements....
  // These are useful in constructions like,
  // list<Sword> l = list<Sword>(excalibur, flamberge)
  // to make a two-element list.
  // It is traditional to stop at four.
  // The tradition started with Smalltalk, which
  // had a similar class and the same choice of
  // constructors.
  // Now, Smalltalk chose that number
  // because that's how many constructors they needed
  // in their kernel.
  // So ultimately, I am allowing four constructors
  // because that's how many are required by the
  // Smalltalk kernel...
  // A better approach might be to use list(...)
  // and varargs, but both are flaky and I'm
  // afraid to try them together.
  list ();
  list (T t1); 
  list (T t1, T t2); 
  list (T t1, T t2, T t3); 
  list (T t1, T t2, T t3, T t4); 
  
  // Usual Classical Form stuff...
  ~list(void);

  // Copy and assignment.  Warning; these duplicate list structure,
  // so be a bit careful about calling them; they can be expensive.
  list(const list&);
  list<T>& operator=(const list<T>& l);

  // The following constructor creates a list from a length n array of T's
  list(T *at, int  n);

// MUTATORS

  // Adding and removing things to lists
 

  void append(T t);  // Add t at the end of *this.
  void remove(T t);        //rem t from *this.
// Tests...
  bool isEmpty() const { return (this->headPtr == NULL ? true : false); }
  bool isNonEmpty() const {return (this->headPtr == NULL ? false : true); }
  // Return true iff *this contains an element == to t.
  // Requires T::operator== or the like.
  bool contains(const T& t) const;

/*
  int operator==(const list<T> l) const;
  int operator!=(const list<T> l) const{return !(*this == l);};

*/

// Returns the position of 't' in the list, or -1 if not there
// Requires T::operator== or similar (ok for pointers).
// Note: totally un-debugged because I thought I needed it and then
// came up with an easier solution to this problem
  int positionOf(const T& t) const;
  
  
// Getting list_pos's to the ends of the list.

   // Get a list_pos (a marker to an element)
   // head and tail list_pos's allow modification of the list through 
   // the list_pos.
   // headConst and tailConst can be used to look along the list,
   // but not modify it.  They can be used in const contexts, though.

  list_pos<T> head() ;
  list_pos<T> headConst() const; 
  list_pos<T> tail() ;
  list_pos<T> tailConst() const;

// OPERATIONS: 
  
  list<T> operator+(const list<T>& lt) const;
  // list concatenation.  Copies both *this and lt, so it's not 
  // cheap.
  
  void appendAndDestroy(list<T>& lt);
  // Append lt to *this.  Modifies *this (making it have lt at the end),
  // and sets lt to a null list.  This is quite cheap.
  // Hopefully, the alarming name will warn users that
  // it's got an unusual side effect.
  

  // Reverses the list so that the old head element is the new tail
  // element and vis versa
  void reverseInPlace();

  // toArray creates an array of T's from list. 
  T *toArray() const;
  // returns n'th element
  // currently performs no error checking
  T nth_element(int n) const;
  
  // The length of the list (duh).
  int length() const;

friend ostream& operator<<(ostream& o, const list<T>& l);

private: 

  // ABSTRACTION FUNCTION:
  // This implements a mutable list of T's.
  // The elements of the list are given, in order,
  // by starting with this->headPtr's T element, and following
  // next-pointers until the NULL marking the end of the list.

  // REP INVARIANT:
  // If the list is empty, both headPtr and tailPtr are null.
  // Otherwise, headPtr points to the head and tailPtr
  // to the tail.

friend class list_node<T>;
friend class list_pos<T>;

  list_node<T> *headPtr;
  // Head pointer: NULL for empty list, non-NULL for non-empty list.

  list_node<T> *tailPtr;
  // Tail pointer: NULL for empty list, non-NULL for non-empty list.



  void initToNulllist();
  // UNSAFE.
  // Turns *this into a null list, losing track of any previous
  // contents.  That is, it NULLs both pointers.
  // Afterwards, the list is a well-represented empty
  // list, and (e.g.) it is safe to iterate over and use listpos's on.

  void createFirstElement(T t);
  // UNSAFE.
  // Turns an empty list into a singleton containing t.
  // It does no checking, and will do bad things (probably
  // losing previous list contents) if the list is nonempty.

  void trashlist();
  // UNSAFE.
  // Destructor body.
  // Deletes the elements of *this,
  // leaving *this in an inconsistent state.

  void copyToNullFrom(const list<T>& l);
  // UNSAFE.
  // copy constructor and assignment component.
  // Assumes that *this is a proper null list, and copies
  // l into it.

  list_node<T> * allocNode(list_node<T> *prev, T t, list_node<T> *next);
  // Returns a new list_node<T> containing t, doing appropriate
  // memory allocation checks.  It doesn't really concern *this,
  // but we want to hide it inside the abstraction barrier, so it's
  // here.
};


// ------------------------------ list_pos ------------------------------
template<class T>
class list_pos {
public:

  // No ab nihilo constructors.
  // The ordinary way of making a list_pos on a list is,
  // list_pos<T> lp = l.head();
  // (or l.tail() if you prefer);
  // or one of the other list_pos-returning functions when and if
  // they are added.
  //
  // Basically, we consider "list_pos<T> lp(l)" to be
  // ambiguous about which end of l lp will point to.
  // However, I don't, so I've added the constructor to the head();
  
  // Assignment and copying is normal.
  list_pos(const list_pos<T>& lp);
  list_pos<T>& operator=(const list_pos<T>& l);
  list_pos(list<T>& l);
  list_pos(const list<T>& l);
  
  // Returns 1 if this list_pos is actually on the list it thinks it is.
  bool reallyOnlist() const;

  // Operations
  bool operator<(const list_pos<T>& b) const;
  bool operator==(const list_pos<T>& b) const;
  

  // Destroying the listpos doesn't do anything to the list
  ~list_pos(void);

  // get the element at this position.
  // We provide two names because (A) each made sense in
  // some phrasing, and (B) we couldn't think of a good
  // single name when we wrote that part of the code.
  // Bad style, really.
  // These functions blow up if the list_pos is offlist.
  T here() const;
  T item() const { return this->here(); };

  // Get back to the list that this list_pos points to.  
  list<T>* mylist() const {return this->tolist;};

  // The main test is, does the list_pos point somewhere
  // on the list.  The guarantee is,
  // if it is onlist, then it's here() will work right.
  // e.g., if mylist is empty, mylist.head() is offlist().
  // Note also that if lp points to an element of a list, 
  // and that element is deleted, lp will still register as
  // onlist.
  bool offlist() const {return (this->cursor ? false : true);}
  bool onlist() const {return  (this->cursor ? true : false);}
  
  // so a list_pos can be used just like a pointer in a for loop
  operator int () const  { return onlist();};
  const list_pos<T>& operator ++ (int) {goNext(); return *this;};
  list_pos<T>& operator ++ () {goNext(); return *this;};
  // return true if *this will refuse to allow modifying
  // operations on the underlying list to be used through it. 
  bool isConst() const {return this->isConstPos;};

  // Is the cursor at the head of the list (empty or not)?
  // if this->atHead() is true, this->insertPrev(t) will
  // make t the new head element of the list.
  // For example, if the list is empty, its listPos's are atHead.
  bool atHead() const;

  // Dual to atHead();
  bool atTail() const;

// MOVING THE LISTPOS AROUND
  
  // go forward or backward in the list.
  void goNext();
  void goPrev();

// MUTATING THE LIST ITSELF

  // Insert an element t after the current position.
  // Leave this list_pos pointed at the new element.
  // I'm not sure what the right behavior should be when
  // *this is off the list; so it causes an error.
  // However, when the list is empty, this adds a single element to it.
  void insertNext(T t);

  // Insert t before the current position; leave *this
  // pointing to new position.
  // Like insertNext, mutatis mutandis.
  void insertPrev(T t);


  void replaceWith(T t);
  // replace referenced list element with t.
  // Doom, if the list_pos doesn't refer to an element
  // of a non-constant list.

  void delete0();
  // Delete the node of the list *this is pointing to.
  // Leaves *this pointing to nowhere; that's what the 
  // "0" in "delete0" means.  
  
  // WARNING: Dangling list_pos's are dangerous: 
  // if you delete a list_pos to element x, you might
  // have other list_pos's to x -- and that's trouble.
  // MISSING: isReallyOnlist() -- scan list and make sure 
  // that *this is a list_pos to somewhere on the list.
  
  void deleteAndGoNext();
  // like delete0, but moves to the next list_pos.

  // Returns a list consisting of this through b.
  list<T> through(const list_pos<T> b) const;

  
private:
friend class list<T>;

  list_pos(list<T> *l, list_node<T> * n, bool consty);
  // Construct a list_pos to node n of list l, with
  // constant-ness consty.
  // Use list_pos<T> lp = l.head()   (or l.tail() if that's what you mean)
  // for internal use only.


  list_node<T> *cursor;
  list<T> *tolist;
  bool isConstPos; // True iff the list_pos should not be used for mutators.
};  


#endif

  
