/* -*-C++-*-
 * ###################################################################
 *	EvoX - evolution in	complex	systems	 
 * 
 *	FILE: "tcl_stream.h"
 *								   created:		1/19/95	{10:07:08 pm}
 *								   last	update:	12/12/96 {11:58:44 pm}
 *	Author:	  Vince	Darley 
 *	 E-mail: vince@das.harvard.edu
 *	   mail:  Divison of Applied Sciences, Harvard University
 *			  Cambridge	MA 02138
 *	
 *	Description: Implements	all	we need	to know	about
 *				   the tcl/C++ interface via a stream.
 *				   Especially useful as	everything goes	to text!
 *	History
 * 
 *	modified by	 rev reason
 *	-------- --- --- -----------
 *	1/19/95	 VMD 1.0 ammended from version from	tktree
 * ###################################################################
 */

/*@ ---------------------------------------------------------------------------
    ---------------------------------------------------------------------------
 */
#ifndef _EvoX_tcl_stream_
#define _EvoX_tcl_stream_

/* 
 * TclInterpStream.h - utility class for evaluating tcl commands
 * 
 * -----------------------------------------------------------------------------
 * Copyright 1994 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  
 * Allan Brighton make no representations about the suitability of this 
 * software for any purpose. It is provided "as is" without express or 
 * implied warranty.
 * -----------------------------------------------------------------------------
 */

/*      \Section{Imports}
    Import required include files
 */
#include <iostream.h>
#include <strstream.h>
#include "tcl.h"
 
//@ ----------------------------------------------------------------------------
/* \Section{Class Declaration}
 */

//@Section: CppTcl library
//@Man:
/**
 * -------------------------------------------------------------------------
 *   
 * "class tcl_stream" --
 *  
 *  Wraps up a Tcl interpreter inside a C++ stream.  This means we can
 *  do away with those nasty "Tcl_SetResult", "Tcl_AppendElement", ...
 *  and just use nice '<<' operators, with a number of possible actions.
 *   
 *  The available actions are:
 *     
 *      tcl_stream t(interp);
 *      
 *      t <<    "set a " << a << eval;
 *      t <<    "That action is impossible" << tcl_error;
 *      t <<    "Here's the result" << result;
 *      t <<    "add this to the current buffer" << append;
 *      t <<    "append this list element to the buffer" << lappend;
 *     
 *  The status of the stream (TCL_OK, TCL_ERROR, etc.) can be retrieved
 *  by either:
 *     
 *      return t.status();
 *      return t; //    automatically cast to TCL_OK or TCL_ERROR
 *     
 *  The status may be reset to 'TCL_OK' with: 
 *   
 *      t.reset();
 *     
 *  Finally you can gain access to the interpreter with either
 *  of the following:
 *     
 *      Tcl_Interp* i = t.interpreter();
 *      Tcl_Interp* i = t; // automatically cast
 *      
 * Side effects:
 *  Manipulates a Tcl interpreter so almost anything can happen.
 *  
 * --Version--Author------------------Changes-------------------------------  
 *    1.0     <vince@das.harvard.edu> original
 *    2.0     <vince@das.harvard.edu> future version planned to wrap up more
 *                                    interpreter functions.
 * -------------------------------------------------------------------------
 */
class tcl_stream :public ostrstream {
	//@Man: Manipulators
	//@{
	/// Evaluate the contents of the tcl_stream in its interpreter
    friend ostream& eval(ostream& outs);
	/// Signal a TCL_ERROR with message of my contents 
    friend ostream& tcl_error(ostream& outs);
	/// Set the Tcl result to my contents -- replaces any previous string
    friend ostream& result(ostream& outs);
	/// Append my contents to the Tcl result
    friend ostream& append(ostream& outs);
	/// Append my contents as a list element to the current result
    friend ostream& lappend(ostream& outs);
	/// Discard the current contents of the stream
	/// Does not affect anything already added to the interpreter.
    friend ostream& discard(ostream& outs);
	//@}
	
  private:
    // tcl interpreter 
    Tcl_Interp* interp_;

    // status of last eval
    int status_;

  public:

    /// Constructor
	tcl_stream(Tcl_Interp* interp) 
		:interp_(interp), status_(TCL_OK) {}

	~tcl_stream(void);		
	
	/// Get my interpreter
	Tcl_Interp* interpreter(void) const { return interp_;}
	/// Probably easiest just to use this implicit cast
    operator Tcl_Interp* (void) const { return interp_;}
	
	/// Returns the status of my Tcl interpreter
    int status() const {return status_;}
	/// Again we can cast if we prefer (so 'return tcl_' is ok)
	operator int () const { return status_; }

	/// Clear my status back to 'TCL_OK'
    void reset() {status_ = TCL_OK;}
	/// Returns the result string
    char* result(void) const { return interp_->result;}
    
	/// Clear the entire Tcl result.  Can be useful after an 'eval'
    tcl_stream& ResetResult(void);
	/// Very useful procedure to find you package library files
	int PackageLibraryInit(const char* libVarName, const char* envVarName, 
	                       const char* pkgName, const char* pkgInitFile, 
	                       const char* version, 
                           const char* prettyPkgName =0,
	                       const char* compiledLocation=0);
	/** 
	 * It searches in all standard places, including resource forks	and
	 * extensions folder on	MacOS, and will	load the specified file	if
	 * successful.	The	parameters are as follows:
	 * 
	 *	   libVarName -- the Tcl variable in which to store	the	library	dir.
	 *	   envVarName -- the variable env(...) in which	to look
	 *	   pkgName -- the name of the package
	 *	   pkgInitFile -- file to source; usually pkgName.tcl
	 *	   version -- string representation	of the version
	 *	   prettyPkgName --	optional to	look nice in error messages
	 *	   compiledLocation	-- string value	of a compiler flag if desired
	 * 
	 * For example,	this package uses the following	call:
	 * 
	 *	   tcl_.PackageLibraryInit("cpx::library","CPPTCL_LIBRARY","cpptcl",
	 *							   "cpptcl.tcl","1.1","CppTcl");
	 */
#ifdef ITCL_NAMESPACES
	/// Find or make the given namespace
	int MakeNamespace(const char* name);
#endif
  private:
  	
	
};

#endif
