/* -*-C++-*-
 * ###################################################################
 *	EvoX - evolution in	complex	systems
 * 
 *	FILE: "tcl_args.cc"
 *									  created: 16/4/96 {1:40:37	pm}	
 *								  last update: 10/12/96 {1:56:56 pm}	
 *	Author:	Vince Darley
 *	E-mail:	<vince@das.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	   www:	<http://www.fas.harvard.edu/~darley/>
 *	
 *	See	header file	for	further	information
 * ###################################################################
 */


#include "tcl_args.h"
#include "tcl_object.h"
#include "CppTcl.h"
#include <string.h>

// for want of a better place, these are here:
// (used by compilers which don't have ANSI bool)
#ifdef NO_ANSI_BOOL
istream& operator >> (istream& i, bool& b){
	int ii;
	i >> ii;
	b = (ii?true:false);
	return i;
}
ostream& operator << (ostream& o, bool b){
	o << BOOL b;
	return o;
}
#endif

tcl_args& skip(tcl_args& t) {
	t.skip();
	return t;
}

tcl_args& optional(tcl_args& t) {
	t.optional_arg = true;
	return t;
}

tcl_args& done(tcl_args& t) {
	// are there any more arguments left?
	if(t.finding_completions) {
		t.haveErr = false;
		t.completion_type = tcl_args::Already_complete;
		t.signal_error(tcl_args::Finding_completions);
	} else if(t.args_left() != 0) {
		t.signal_error(tcl_args::Too_many_args);
	}
	throw(TCL_ERROR);
	return t;
}
// copy constructor, copies only the un-parsed arguments
tcl_args::tcl_args(const tcl_args& a, const tcl_object *o)
:parsed_so_far(),failed_args(),
 my_args_(),
 conversion_err_type(a.conversion_err_type),
 myName(a.myName),
 finding_completions(0),
 completion_type(No_completion),
 completion_orig_len(0),completion_extra_len(0),completion(),
 tcl_(a.tcl_),cmd_syntax(a.cmd_syntax),help_text(a.help_text),
 got_cmd(false),argc_(a),
 argv_(a)
{
	// Perhaps we should let these three be inherited??
	// Can't think of a good reason one way or another right now
	// We kind of assume we copy at some default 'between argument'
	// time.
	cur_cmd = 0;
	optional_arg = false;
	haveErr = false;
	if(o) {
		parsed_so_far << o->get_tcl_command() << " ";
	}

	// store the arguments in my stream for retrieval later
	for(short i=0;i<argc_;i++) {
		my_args_ << argv_[i] << " ";
	}
	args_left_ = argc_;
}

tcl_args::tcl_args(tcl_stream& i, int argc, char* argv[], const tcl_object* o)
:parsed_so_far(),
 failed_args(),
 my_args_(),
 conversion_err_type(0), 
 myName(0),
 finding_completions(0),
 completion_type(No_completion),
 completion_orig_len(0),completion_extra_len(0),completion(),
 tcl_(i), 
 cmd_syntax(0),
 help_text(0),
 got_cmd(false),
 argc_(argc),
 argv_(argv)
{
	cur_cmd = 0;
	if(o) {
		parsed_so_far << o->get_tcl_command() << " ";
	}
	
	// store the arguments in my stream for retrieval later
	for(short i=0;i<argc_;i++) {
		my_args_ << argv_[i] << " ";
	}
	args_left_ = argc_;
	optional_arg = false;
	haveErr = false;
}

tcl_args::~tcl_args(void){
	// I think we want these
	reset_stream(failed_args);
	reset_stream(parsed_so_far);
	reset_stream(completion);
	//delete [] cur_cmd;
}

void tcl_args::setName(const char *n) {
	myName = n;
}

const char* tcl_args::check_name(void) const {
	if(myName)
		return myName;
	else
		return orig_argv(0);
}

const char* tcl_args::name(void) {
	// Only skip if we haven't been touched yet
	// We know we've been touched if our name as been set.
	if(!myName)
		skip();
	return check_name();
}

// static
void tcl_args::reset_stream(ostrstream& o) {
	// allow the ostream to continue to be reused and reset the start ptr
	#ifdef __MWERKS__
    o.rdbuf()->freeze(0); 
    o.rdbuf()->pubseekpos(0); 
	#else
    o.rdbuf()->freeze(0);
    o.seekp(0);
	#endif
}
	
void tcl_args::clear_args(void) {
	// allow the istrstream to continue to be reused and reset the start ptr
	#ifdef __MWERKS__
    my_args_.rdbuf()->freeze(0); 
    my_args_.rdbuf()->pubseekpos(0); 
	#else
    my_args_.rdbuf()->freeze(0);
    my_args_.seekp(0);
	#endif
	cmd_syntax = 0;
	reset_stream(parsed_so_far);
	reset_stream(failed_args);
}

void tcl_args::skip(void) {
	remove_from_buffer();
	args_left_--;
}

void tcl_args::remove_from_buffer(void) {
	// This is a rather poor way to remove the next item
	// And it crashes if the next item is longer than 80 chars
	char buffer[80];
	check_read() >> buffer;
}

void tcl_args::pretend_read_from_me(void) {
	optional_arg = false;
}

istream& tcl_args::check_read(void) {
	if(!args_left()) {
		if(!finding_completions) {
			signal_error(Syntax);
		}
	}
	return my_args_;
}

tcl_args& tcl_args::syntax(const char* s) {
	cmd_syntax = s;
	return *this;
}

tcl_args& tcl_args::help(const char* h) {
	help_text = h;
	return *this;
}

tcl_args& tcl_args::operator -= (const char*s) {
	if(operator==(s)) {
		// ok
	} else {
		signal_error(Syntax);
	}
	return *this;
}

bool tcl_args::_internal_match(const char* &s, const char* const arg_name) {
	if(!got_cmd) {
		// this will decrement args_left_
		if(args_left()) {
			const_string_read(cur_cmd);
			read_done();
		} else {
			cur_cmd = 0;
		}
		got_cmd = true;
	}
	if(!s) {
		// we always match, and set to the value!
		s = cur_cmd;
	}
	// We check the first character first for speed (standard practice)
	if(cur_cmd && s[0] == cur_cmd[0] && !strcmp(cur_cmd,s)) {
		// add the current command to our command-so-far
		parsed_so_far << cur_cmd << " ";
		reset_stream(failed_args);
		// we need to read another next time through.
		got_cmd = false;
		const char* hc = peek();
		if(hc && (!strcmp(hc,"-h") || !strcmp(hc,"-help")))  {
			// we were asked for help
			if(help_text) {
				parsed_so_far << ": " << help_text;
			} else {
				parsed_so_far << ": no help available.";
			}
			signal_error(Help);
		}
		cur_cmd = 0;
		return true;
	} else {
		if(!s)
			s = arg_name;
	    // If the var was optional or it doesn't match.
		if (finding_completions) {
			if(!cur_cmd ||
			   (s[0] == cur_cmd[0] && !strncmp(cur_cmd,s,strlen(cur_cmd)))) {
					remember_completion(s);
			} else if((s[0] == '\0') && !cur_cmd) {
				remember_completion(s);
			}
			
		}
		failed_args << s << ", ";
		help_text = 0;
		return false;	
	}	
}

int tcl_args::no_match(void) {
	if(finding_completions) {
		haveErr = false;
		signal_error(Finding_completions);
	} else {
		signal_error(No_match);
	}
	
	return TCL_ERROR;
}

void tcl_args::internal_signal_error(int err) {
	switch(err) {
	  case Conversion:
		args_conversion_err();
		break;
	  case No_match:
		args_no_match_err();
		break;
	  case Syntax:
		args_syntax_err();
		break;
	  case Too_many_args:
		args_too_many_err();
		break;
	  case Cpp_constructor:
		args_cpp_constructor_err();
		break;
	  case Help:
		args_help_signal();
		break;
	  case Finding_completions:
		args_completion_signal();
		break;
	  default:
		// no known error!
		break;
	}
}
	
void tcl_args::signal_error(int err){
	if(!haveErr) {
		parsed_so_far << ends;
		failed_args << ends;
		if(err) 
			internal_signal_error(err);
	}
	haveErr = true;
}

const char* tcl_args::peek(void) const {
	if (args_left() > 0)  {
		return peek_arg();
	} else {
		return 0;
	}
}

/// Is the next argument a real
bool tcl_args::peek_is_float(void) const {
	if(args_left() <=0) return false;
	double d;
	if(Tcl_GetDouble(tcl_,(char*)peek_arg(),&d) == TCL_ERROR) {
		tcl_.ResetResult();
		tcl_.reset();
		return false;
	}
	return true;
}

/// Is the next argument an integer
bool tcl_args::peek_is_int(void) const {
	if(args_left() <=0) return false;
	int l;
	if(Tcl_GetInt(tcl_,(char*)peek_arg(),&l) == TCL_ERROR) {
		tcl_.ResetResult();
		tcl_.reset();
		return false;
	}
	return true;
}

/// Is the next argument a string (i.e. not a number)
bool tcl_args::peek_is_string(void) const {
	return (peek_is_float() ? false : true);
}

/// Is the next argument the given string
bool tcl_args::peek_is(const char* const s) const {
	if(args_left() <=0) return false;
	return (!strcmp(peek_arg(),s) ? true : false);
}

tcl_args& tcl_args::check_after(void) {
	if(my_args_.bad()) {
		tcl_ << "Internal iostream error at:";
		signal_error(Syntax);
	} else if (my_args_.eof()) {
		signal_error(Syntax);
	} else if (my_args_.fail()) {
		signal_error(Conversion);
	}
	return *this;    
}

void tcl_args::read_done(void){
	optional_arg = false;
	if(args_left_>0){
		// only decrement if we actually read something
		args_left_--;
	}
}

// static
void tcl_args::add_without_trailing_punctuation(ostream& o, const char* ch) {
	if(!ch)
		return;
	short len = strlen(ch);
	while(len>0 && (ch[len-1] == ' ' || ch[len-1] == ',' 
	      || ch[len-1] == '\r' || ch[len-1] == '\n')) {
		len--;
	}
	      
	for(short i=0;i<len;i++)
		o << ch[i];
}

void tcl_args::args_too_many_err(void) {
	tcl_ << "got cmd '";
	add_without_trailing_punctuation(tcl_,parsed_so_far.str());
	tcl_ << "' with extra arguments: ";
	reset_stream(parsed_so_far);
	for(short i=argc_ - args_left();i<argc_;i++) {
		tcl_ << argv_[i] << " ";
	}
	tcl_ << tcl_error;
}

void tcl_args::args_cpp_constructor_err(void) {
	// append in case the constructor gave some info already.
	tcl_ << cmd_syntax << append;
}

void tcl_args::args_help_signal(void) {
	tcl_ << parsed_so_far.str() << tcl_error;
	reset_stream(parsed_so_far);
}

void tcl_args::start_completion_check(bool with_arguments) {
	tcl_ << "Completions:";
	finding_completions = (with_arguments ? 2 : 1);
	haveErr = true;
}

void tcl_args::end_completion_check(void) {
	finding_completions = 0;
	completion_orig_len = 0;
	completion_extra_len = 0;
	reset_stream(completion);
	reset_stream(parsed_so_far);
}

void tcl_args::args_completion_signal(void) {
	if(completion_type == No_completion) {
		tcl_ << discard << "No known completions." << result;
	} else if(completion_type == Already_complete) {
		tcl_ << discard << "Already complete." << result;
	}
	tcl_ << append;
}

void tcl_args::remember_completion(const char* s) {
	// All completions are from the point we've reached
	// so far, so we must end the string now:
 	short cur_cmd_len = (cur_cmd ? strlen(cur_cmd) : 0);
	if(!completion_orig_len) {
		parsed_so_far << ends;
	 	completion_orig_len = strlen(parsed_so_far.str()) + cur_cmd_len;
	 	completion_extra_len = strlen(s) - cur_cmd_len;
	 	completion << s;
	 	for(unsigned short i=cur_cmd_len;i<strlen(s);i++)
	 		completion << s[i];
	 	completion << ends;
	} else {
		const char* oldcomp = completion.str();
		const char* newcomp = s;
		short i;
		for(i=0;i<completion_extra_len;i++) {
			if(oldcomp[i+cur_cmd_len] != newcomp[i+cur_cmd_len])
				break;
		}
		completion_extra_len = i;
	}
	// However we don't reset it till we're totally done.
	tcl_ << endl << parsed_so_far.str() << s;
	if(finding_completions == 2) {
		if(s[0] != '\0')
			tcl_ << " ";
		tcl_ << cmd_syntax;
	}
	switch(completion_type) {
	  case No_completion:
	  	completion_type = Unique_completion;
	  	break;
	  case Have_completion:
	  	break;
	  case Unique_completion:
	  	completion_type = Have_completion;
	  	break;
	  case Already_complete:
	    assert(0);
    }
}


void tcl_args::args_conversion_err(void) {
	tcl_ << "read partial cmd '";
	add_without_trailing_punctuation(tcl_,parsed_so_far.str());
	tcl_ << "' but failed to convert next argument '"
		 << argv_[argc_ - args_left()] 
		 << "'";
	if(conversion_err_type) {
		if(conversion_err_type[0] != '\0') {
			tcl_ << " to type '" << conversion_err_type << "'";
		} else {
			// Starting with a zero means we supply the full error
			// message, and that it must be freed.
			tcl_ << conversion_err_type+1;
			delete (char*) conversion_err_type;
		}
	}
	tcl_ << "; syntax should be '";
	add_without_trailing_punctuation(tcl_,parsed_so_far.str());
	tcl_ << " " << cmd_syntax << "'" << tcl_error;
	reset_stream(parsed_so_far);
}

void tcl_args::args_no_match_err(void) {
	// Sort the failed arguments
	tcl_ << "lsort {" << failed_args.str() << "}" << eval;
	reset_stream(failed_args);
	failed_args << tcl_.result() << ends;
	tcl_.ResetResult();
	
	if(cur_cmd){
		tcl_ << "wrong option '" << cur_cmd << "'";
	} else {
		tcl_ << "No argument given";
	}
	tcl_ << " to '";
	add_without_trailing_punctuation(tcl_,parsed_so_far.str());
	tcl_ << "' should be one of: ";
	add_without_trailing_punctuation(tcl_,failed_args.str());
	tcl_ << tcl_error;
	reset_stream(parsed_so_far);
	reset_stream(failed_args);
}

void tcl_args::args_syntax_err(void) {
	if(cmd_syntax) {
		tcl_ << "wrong # args: should be '";
		add_without_trailing_punctuation(tcl_,parsed_so_far.str());
		tcl_ << " " << cmd_syntax << "'" << tcl_error;
	} else {
		tcl_ << "wrong # args: correct syntax for '";
		add_without_trailing_punctuation(tcl_,parsed_so_far.str());
		tcl_ << "' sadly unknown." << tcl_error;
	}
	reset_stream(parsed_so_far);
}

void tcl_args::const_string_read(const char*& v){
	if(!args_left()) {
		v = 0;
		if(optional_arg)
			pretend_read_from_me();
		else
			check_read();
	} else {
		check_read();
		v = (*this)[0];
		remove_from_buffer();
	}
}

