/* -*-C++-*-
 * ###################################################################
 *	EvoX - evolution in	complex	systems	 
 * 
 *	FILE: "tcl_stream.cc"
 *								   created:		1/19/95	{10:27:17 pm}
 *								   last	update:	12/12/96 {11:59:39 pm}
 *	Author:	  Vince	Darley 
 *	 E-mail: vince@das.harvard.edu
 *	   mail:  Divison of Applied Sciences, Harvard University
 *			  Cambridge	MA 02138
 *	
 *	See	header file	for	further	information
 * ###################################################################
 */

#include "tcl_stream.h"
#include "tcl.h"

/* (My code has evolved from this class)
 * TclInterpStream.C - utility class for evaluating Tcl commands from C++
 *
 * -----------------------------------------------------------------------------
 * Copyright 1994 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  
 * Allan Brighton make no representations about the suitability of this 
 * software for any purpose. It is provided "as is" without express or 
 * implied warranty.
 * -----------------------------------------------------------------------------
 *
 */

extern void cpptcl_delete_stream(tcl_stream&);

tcl_stream::~tcl_stream(void) {
	cpptcl_delete_stream(*this);
}

// allow the strstream to continue to be reused and reset the start ptr
#ifdef __MWERKS__
#define	RETURN_ \
    t.rdbuf()->freeze(0); \
    t.rdbuf()->pubseekpos(0); \
    return outs
#else
#define	RETURN_ \
    t.rdbuf()->freeze(0); \
    t.seekp(0); \
    return outs 
#endif

/* 
 * -------------------------------------------------------------------------
 *	 
 *	"eval" --
 *	
 *	   This	friend function	is used	as an iostream manipulator to send the
 *	   contents	of the stream to Tcl.  
 * -------------------------------------------------------------------------
 */
ostream&  eval(ostream& outs) {
    // outs is really a tcl_stream
    tcl_stream& t = (tcl_stream&) outs;

    // add a null char to end the string
    t << ends;

    if ((void*)t) {
		// send the string to tcl 
		t.status_ = Tcl_Eval(t.interp_, t.str());
		t.clear();		
    } else {
		t.status_ = TCL_ERROR;
		Tcl_SetResult(t.interp_, "internal iostream error", TCL_STATIC);
    }

    RETURN_;
    
}



/* 
 * -------------------------------------------------------------------------
 *	 
 *	"tcl_error"	--
 *	
 *	   This	is the same	as eval, but used for reporting	tcl	errors rather than
 *	   evaluating tcl commands.	 
 * -------------------------------------------------------------------------
 */
ostream&  tcl_error(ostream& outs) {
    // outs is really a tcl_stream
    tcl_stream& t = (tcl_stream&) outs;

    // add a null char to end the string
    t << ends;
    t.status_ = TCL_ERROR;

    if ((void*)t) {
		Tcl_SetResult(t.interp_, t.str(), TCL_VOLATILE);
    } 
    else {
		Tcl_SetResult(t.interp_, "internal iostream error", TCL_STATIC);
    }

    RETURN_;
    
}

/* 
 * -------------------------------------------------------------------------
 *	 
 *	"result" --
 *	
 *	 As	for	eval, but used to set the result string	for	the	Tcl_Interp
 * -------------------------------------------------------------------------
 */
ostream& result(ostream& outs)  {
    // outs is really a tcl_stream
    tcl_stream& t = (tcl_stream&) outs;

    // add a null char to end the string
    t << ends;

    if ((void*)t) {
		// send the string to tcl 
		t.status_ = TCL_OK;
		Tcl_SetResult(t.interp_, t.str(), TCL_VOLATILE);
    } else {
		t.status_ = TCL_ERROR;
		Tcl_SetResult(t.interp_, "internal iostream error", TCL_STATIC);
    }

    RETURN_;
}

/* 
 * -------------------------------------------------------------------------
 *	 
 *	"append" --
 *	
 *	 Adds to the result	built up so	far, in	a string like fashion.	See	
 *	 "lappend" to build	up lists.  
 * -------------------------------------------------------------------------
 */
ostream& append(ostream& outs)  {
    // outs is really a tcl_stream
    tcl_stream& t = (tcl_stream&) outs;

    // add a null char to end the string
    t << ends;

    if ((void*)t) {
		// send the string to tcl 
		t.status_ = TCL_OK;
		Tcl_AppendResult(t.interp_, t.str(), NULL);
    } else {
		t.status_ = TCL_ERROR;
		Tcl_SetResult(t.interp_, "internal iostream error", TCL_STATIC);
    }

    RETURN_;
}

/* 
 * -------------------------------------------------------------------------
 *	 
 *	"lappend" --
 *	
 *	 Function to build up a	list using a tcl_stream.
 * -------------------------------------------------------------------------
 */
ostream& lappend(ostream& outs)  {
    // outs is really a tcl_stream
    tcl_stream& t = (tcl_stream&) outs;

    // add a null char to end the string
    t << ends;

    if ((void*)t) {
		// send the string to tcl 
		t.status_ = TCL_OK;
		Tcl_AppendElement(t.interp_, t.str());
    } else {
		t.status_ = TCL_ERROR;
		Tcl_SetResult(t.interp_, "internal iostream error", TCL_STATIC);
    }

    RETURN_;
}

/* 
 * -------------------------------------------------------------------------
 *	 
 *	"discard" --
 *	
 *	 Function to build up a	list using a tcl_stream.
 * -------------------------------------------------------------------------
 */
ostream& discard(ostream& outs)  {
    // outs is really a tcl_stream
    tcl_stream& t = (tcl_stream&) outs;
    RETURN_;
}

tcl_stream& tcl_stream::ResetResult(void)  {
    Tcl_ResetResult(interp_);
    return *this;
}

int tcl_stream::PackageLibraryInit(const char* libVarName, const char* envVarName, 
                                   const char* pkgName, 
                                   const char* pkgInitFile, 
                                   const char* version, 
                                   const char* prettyPkgName,
                                   const char* compiledLocation) {
    /*
     *  Set up the library and load the init file.
     */	
	if(prettyPkgName ==0) 
		prettyPkgName = pkgName;
#ifdef ITCL_NAMESPACES
    if(compiledLocation)
    	Tcl_SetVar(interp_,(char*) libVarName, (char*)compiledLocation, ITCL_GLOBAL_VAR);
	else
    	Tcl_SetVar(interp_,(char*) libVarName, "", ITCL_GLOBAL_VAR);
#else
    if(compiledLocation)
    	Tcl_SetVar(interp_,(char*) libVarName, (char*)compiledLocation, TCL_GLOBAL_ONLY);
	else
    	Tcl_SetVar(interp_,(char*) libVarName, "", TCL_GLOBAL_ONLY);
#endif

	/* 
	 * If you are on MacOS, and get library files from the resource fork 
	 * then you will find your library variable is not set.  You must decide
	 * what to do with it.
	 */
    (*this) << "set " << pkgName << "dirs {}\n"
		    << "if [info exists env(" << envVarName << ")] {\n"
		    << "lappend " << pkgName << "dirs $env(" << envVarName << ")\n"
		    << "}\n"
		    << "if [info exists env(EXT_FOLDER)] {\n"
		    << "lappend " << pkgName << "dirs [file join $env(EXT_FOLDER) \"Tool Command Language\" lib " << pkgName << version << "]\n"
		    << "}\n"
		    << "lappend " << pkgName << "dirs ${" << libVarName << "}\n"
		    << "set " << libVarName << " {}\n"
		    << "lappend " << pkgName << "dirs [file join [file dirname [info library]] " << pkgName << version << "]\n"
		    << "set parentDir [file dirname [file dirname [info nameofexecutable]]]\n"
		    << "lappend " << pkgName << "dirs [file join $parentDir lib " << pkgName << version << "]\n"
#if 0
			<< "if {![regexp {.*[ab][12345]} $patchLevel lib]} {\n"
		    << "set lib " << pkgName << version << "\n"
		    << "}\n"
		    << "lappend " << pkgName << "dirs [file join [file dirname $parentDir] $lib library]\n"
#endif
		    << "lappend " << pkgName << "dirs [file join [file dirname $parentDir] " << pkgName << " library]\n"
		    << "lappend " << pkgName << "dirs [file join $parentDir library]\n"
		    << "if ![catch {\n"
		    << "	foreach " << pkgName << "i $" << pkgName << "dirs {\n"
		    << "		set " << libVarName << " $" << pkgName << "i\n"
		    << "		if ![catch {uplevel #0 source \\{[file join $" << pkgName << "i " << pkgInitFile << "]\\}}] {\n"
		    << "    		error {}\n"
		    << "		}\n"
		    << "	}\n"
		    << "}] {\n"
		    << "	if [catch {uplevel #0 source -rsrc " << pkgInitFile << "}] {\n"
		    << "set " << libVarName << " {}\n"
		    << "set msg \"Can't find a usable " << pkgInitFile << " in the following directories: \n\"\n"
		    << "append msg \"    $" << pkgName << "dirs\n\"\n"
		    << "append msg \"This probably means that " << prettyPkgName << " wasn't installed properly\n\"\n"
		    << "append msg \"or you need to set your " << envVarName << " environment variable.\n\"\n"
		    << "error $msg\n"
			<< "}\n"
			<< "}" << eval;
    return status_;
}

#ifdef ITCL_NAMESPACES

int tcl_stream::MakeNamespace(const char* name) {
    Itcl_Namespace ns;
    /*
     *  Find or create the namespace.
     */
    if (Itcl_FindNamesp(interp_, (char*)name, 0, &ns) != TCL_OK) {
        return TCL_ERROR;
    }
    if (ns == NULL && Itcl_CreateNamesp(interp_, (char*)name,
										(ClientData)NULL, 
										(Itcl_DeleteProc*)NULL, &ns) != TCL_OK) {
		(*this) << " (cannot initialize " << name << " namespace)" << tcl_error;
        return TCL_ERROR;
    }
    return TCL_OK;
}

#endif

