/* -*-C++-*-
 * ###################################################################
 *	EvoX - evolution in	complex	systems
 * 
 *	FILE: "cpx_with_info.cc"
 *									  created: 21/6/95 {8:30:30	am}	
 *								  last update: 12/12/96 {12:33:58 am}	
 *	Author:	Vince Darley
 *	E-mail:	<mailto:vince@das.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	www: <http://www.fas.harvard.edu/~darley/>
 *	
 *	See	header file	for	further	information
 * ###################################################################
 */

#include "cpx_with_info.h"
#include "cpx_base.icc"
#include "info_source.h"
#include "info_file.h"
#include "object_control.h"
#include <string.h>
#include "cpx_object.h"

const char * cpx_with_info::_type = "With Information";

cpx_with_info::cpx_with_info(tcl_args& arg)
:cpx_base(arg,arg.name()){
	quick_lookup = 0;
	size_of_quick_lookup_array = 0;
}

cpx_with_info::~cpx_with_info(void){
	if(quick_lookup)
		delete [] quick_lookup;
	quick_lookup = 0;
}

cpx_with_container* cpx_with_info::directly_contains(const char* item) const {
	if(!item) return (cpx_with_container*) 0;
	
    // first attempt: look through my contents
    for(list_pos<cpx_with_container*> p = contents.headConst();p;++p) {
		const cpx_base* b = p.item()->cast_to_base();
		// is it an object I contain?
		char* tc = b->get_tcl_command();
	    if (!(strcmp(tc,item)))
			return p.item();
	    // remove the root path
		for(short l = strlen(tc)-1;l>0;l--) {
			 if (tc[l] == ':') {
			 	tc += (++l);
			 	break;
			 }
		}
	    if(!(strcmp(tc,item)))
			return p.item();
	    
	    // is it some information I contain, with spaces?
		if(info_base* i = p.item()->is_info()) {
			if(!(strcmp(i->name(),item)))
				return i;
	        char * no_scores = uscores_to_spaces(i->name());
	        if (!(strcmp(no_scores,item))) {
	        	delete [] no_scores;
	        	return i;
	        }
	        delete [] no_scores;
		}
    }
    return (cpx_with_container*) 0;
}

info_base* cpx_with_info::configuration_option(tcl_args& arg, bool config) {
    for(list_pos<cpx_with_container*> p = contents.headConst();p;++p) {
		info_base* ii = p.item()->is_info();
		if(ii && (ckcast(ii,info_observable) || ckcast(ii,info_file))) {
			char* tc = ii->get_tcl_command();
		    // remove the root path
			for(short l = strlen(tc)-1;l>0;l--) {
				 if (tc[l] == ':') {
				 	tc += (++l);
				 	break;
				 }
			}
			char* opt = spaces_to_uscores(tc,true);
			if(arg == opt) {
				delete [] opt;
				if(!config) {
					// we don't try and get stuff from the next argument
					return ii;
				}
				
				// read the value into the object directly!
				if(info_observable *ib = ckcast(ii,info_observable)) {
					arg >> (*ib);
					if(!arg.haveErr) 
						archive() << ib->get_tcl_command() << " = " 
						          << arg[-1] << endl;
					return ib;
				} else if(info_file *fl = ckcast(ii,info_file)) {
					arg >> (*fl);
					if(!arg.haveErr) 
						archive() << fl->get_tcl_command() << " setFile " 
						          << arg[-1] << endl;
					return fl;
				}
			} else {
				delete [] opt;
			}
		}
		// keep looking
    }
    
    return (info_base*) 0;
}

info_observable* cpx_with_info::information(const char* iname, const short &element) const  {
	if(info_observable* i = search_down_for_information(iname, element))
		return i;
	
	// check my parents and siblings
	if(has_container()) {
		return has_container()->information(iname, element);
	}
	
	// it's nowhere
	status(0) << "failed to find '" << iname << "' quickly as desired." << ends;
    
	return information(iname);
    return (info_observable*)0;
}

info_observable* cpx_with_info::search_down_for_information(const char* iname, 
															const short &element) const {
	if(quick_lookup){
		// we're told which one it is
		info_observable* i = quick_lookup[element];
		if(i->name() == iname)
			return i;
		// check my descendants
		for(list_pos<cpx_with_container*> p(contents);p;++p)
			if(const cpx_object* obj = p.item()->is_object())
				if(info_observable* ii = obj->information(iname,element))
					return ii;
	}
	return (info_observable*)0;
}

info_observable* cpx_with_info::information(const char* iname) const {
	if(info_observable* i = search_down_for_information(iname))
		return i;
	
	// check my parents and siblings
	if(has_container()) {
		return has_container()->information(iname);
	}

    // it's nowhere
	status(0) << "failed to find '" << iname << "' at all." << ends;
    
    return (info_observable*)0;
}
		
info_observable* cpx_with_info::search_down_for_information(const char* iname) const{
	// first attempt: look through my contents
	for(list_pos<cpx_with_container*> p = contents.headConst();p;++p)
		if(info_base* ii = p.item()->is_info()) 
			if(info_observable* i = ckcast(ii,info_observable))
				if(i->name() == iname)
					return i;
	
	// try harder: look through my contents' contents, recursively
	for(list_pos<cpx_with_container*> p = contents.headConst();p;++p){
		if(const cpx_object* obj = p.item()->is_object())
			if(info_observable* ii = obj->search_down_for_information(iname))
				return ii;
	}
	// ok it's not there, so return
	return (info_observable*)0;
}

info_observable* cpx_with_info::slow_information(const char* iname) const {
	if(info_observable* i = slow_search_down_for_information(iname))
		return i;
	
	// check my parents and siblings
	if(has_container()) {
		return has_container()->slow_information(iname);
	}
	
	// perhaps it had spaces - hence underscores
	static bool underscores = false;
	if(underscores) {
		// we must have been called recursively, so we fail
		
		// it's nowhere
		status(0) << "failed to find '" << iname << "' at all." << ends;
		
		return (info_observable*)0;
	} else {
		char* noscores = uscores_to_spaces(iname);
		underscores = true;
		info_observable* inf = slow_information(noscores);
		underscores = false;
		delete [] noscores;
		return inf;
	}
		
}
	       
// static
char* cpx_with_info::uscores_to_spaces(const char* iname) {
	char* noscores = new char[strlen(iname)+1];
	for(unsigned short i=0;i<=strlen(iname);i++) {
		noscores[i] = (iname[i] == '_' ? ' ' : iname[i] );
	}
	return noscores;
}

// static
char* cpx_with_info::spaces_to_uscores(const char* iname, bool with_dash) {
	char* withscores = new char[strlen(iname)+1+(with_dash ? 1 :0)];
	if(with_dash) withscores++;
	for(unsigned short i=0;i<=strlen(iname);i++) {
		withscores[i] = (iname[i] == ' ' ? '_' : iname[i] );
	}
	if(with_dash) {
		withscores--;
		withscores[0] = '-';
	}
	return withscores;
}

info_observable* cpx_with_info::slow_search_down_for_information(const char* iname) const{
    // first attempt: look through my contents
    for(list_pos<cpx_with_container*> p = contents.headConst();p;++p)
		if(info_observable* i = ckcast(p.item(),info_observable))
		    if(!(strcmp(i->name(),iname)))
				return i;
    
    // try harder: look through my contents' contents, recursively
    for(list_pos<cpx_with_container*> p = contents.headConst();p;++p){
		if(const cpx_object* obj = p.item()->is_object())
		    if(info_observable* ii = obj->slow_search_down_for_information(iname))
				return ii;
    }
    // ok it's not there, so return
    return (info_observable*)0;
}

void cpx_with_info::initialise_quick_lookup(cpx_with_info* obj)  {
	// count how many pieces I contain
	short total_info = 0;
    for(list_pos<cpx_with_container*> p(contents);p;++p){
		if(maycast(p.item()->type(),info_observable))
			++total_info;
	}
	// tell 'obj' to put them in its quick lookup array
	//PRINT(total_info);
	//cout << type() << endl;
	obj->quick_lookup_from(this,total_info);
}

void cpx_with_info::quick_lookup_from(const cpx_with_info* obj, short quantity)  {
	info_observable** new_quick_lookup = 
		new info_observable*[quantity + size_of_quick_lookup_array];

	// copy the old ones over
	short counter = 0;
	for(;counter<size_of_quick_lookup_array;counter++)
		new_quick_lookup[counter] = quick_lookup[counter];
	
	// add the new ones
    for(list_pos<cpx_with_container*> p = obj->contents.headConst();p;++p){
		if(info_observable* i = ckcast(p.item(),info_observable)){
			new_quick_lookup[counter] = i;
			i->set_quick_index(counter);
			counter++;
			//contents.remove(i);
		}
	}
	// sort out the details
	size_of_quick_lookup_array = counter;
	if(quick_lookup)
		delete [] quick_lookup;
	quick_lookup = new_quick_lookup;
    
}


const cpx_base* cpx_with_info::does_contain(const char* thing) const  {
	for(list_pos<cpx_with_container*> o(contents);o;++o) {
		const cpx_base* res = o.item()->cast_to_base()->does_contain(thing);
		//const cpx_base* res = o.item()->does_contain(thing);
		if(res) 
			return res;
	}
	return cpx_base::does_contain(thing);	
}

cpx_error cpx_with_info::add_info(info_base* e) {
    contents.append(e);
    return cpx_error::OK;
}

cpx_error cpx_with_info::remove_info(info_base* e) {
    contents.remove(e);
    return cpx_error::OK;
}

cpx_error cpx_with_info::add_object(cpx_object* e) {
    // assumes that you can be added, and that you have already
    // set your container to point to me
    contents.append(e);
	ive_changed();
	archive() << "# (" << e->type() << ") in (" << type() << ")" << endl;
	return cpx_error::OK;
}

cpx_error cpx_with_info::remove_object(cpx_object* e) {
	// assumes that you know your container is no longer
	// going to be me
	contents.remove(e);
	ive_changed();
	return cpx_error::OK;
}

void cpx_with_info::find_matches_with(const list<const char*>& l, 
                                       socket_fn f, cpx_with_info* obj) const{
	
	// recurse through contents until we find a first match
	if(directly_contains_a_match(l)){
		contains_all_matches(l,f,obj);
	} else {
		for(list_pos<cpx_with_container*> s(contents);s;++s)
			if(cpx_with_info* obj = ckcast(s.item(),cpx_with_info))
				obj->find_matches_with(l,f,obj);
	}
}

bool cpx_with_info::directly_contains_a_match(const list<const char*>& l) const{
	for(list_pos<cpx_with_container*> s(contents);s;++s){
		if(s.item()->cast_to_base()->matches_with(l))
			//if(s.item()->matches_with(l))
			return true;
    }
    return false;
}

void cpx_with_info::contains_all_matches(const list<const char*>& l, 
                                          socket_fn f, cpx_with_info* obj) const
{
	// we have at least one match, we must check if we've got them all.
	
	// set 'flags' to be a set of 1's which get zeroed with every match
	unsigned short flags = (2 << l.length()) -1;
	loop_match_flags(l,flags);
	if(flags==0)
		(obj->*f)((cpx_with_info*)this);
    
}

void cpx_with_info::loop_match_flags(const list<const char*>& l, 
                                      unsigned short& flags) const {
	for(list_pos<cpx_with_container*> s(contents);s;++s) {
		if(short pos = s.item()->cast_to_base()->matches_with(l)) {
			//if(short pos = s.item()->matches_with(l)) {
			// note 'pos' starts at 1
			unsigned short mask = 1 << (pos-1);
			flags -= flags & mask;
			if(!flags) return;
		}
		// If it's an object, recurse down the hierarchy of containment
		if(cpx_with_info* obj = ckcast(s.item(),cpx_with_info))
			loop_match_flags(l,flags);
	}
    return;
}

void cpx_with_info::list_observables(tcl_stream& t, bool convert_spaces) {
	// return a list of all information sources
    for (list_pos<cpx_with_container*> p(contents); p; ++p)
    	if(info_base *i = p.item()->is_info())
    		if(ckcast(i,info_observable) || ckcast(i,info_file))
    			if(convert_spaces) {
    				short cc=0;
    				for(char c = i->name()[cc];c ; c = i->name()[++cc]) {
						t << (c == ' ' ? '_' : c);
					}
					t << " ";
				} else {
					t << i->name() << lappend;
				}
}

	
int cpx_with_info::parse_tcl_command(tcl_args& arg){
	if (arg("","list all things I contain")=="getContents") {
		arg >> done;
		NO_EXCEPTIONS(arg,TCL_ERROR);
		// Note this could return nothing at all
		for (list_pos<cpx_with_container*> p(contents); p; ++p)
			tcl_ << p.item()->cast_to_base()->get_tcl_command() << lappend;
		
		/* 
		 *	 Note this stuff was used when I removed quick-lookup items
		 *	 from the contents list. Currently I think that's a	bad	idea.
		 * 
		 *	 short i=0;
		 *	 while(quick_lookup[i])	{
		 *	 	tcl_ <<  quick_lookup[i]->get_tcl_command());
		 *	 	i++;
		 *	 }
		 */
		 
		return TCL_OK;
	} else if (arg("","set a configuration option")=="configure" 
				|| arg("","set a configuration option")=="config") {
		return parse_configuration_values(arg);
  	} else if (arg("-option","return value of a configuration option")=="cget") {
		info_base *inf=0;
		if((inf = configuration_option(arg, false))) {
			// found one, so continue
			DO_NOTHING;
		} else {
			// no such configuration option
			return arg.no_match();
		}
		DONE(arg);
		// return the last one as the result
		tcl_ << (*inf) << result;
  		return TCL_OK;
  	} else if (arg("item","return name of a piece of information I contain")
  	           =="getInformation") {
        const char* ev;
        arg >> ev >> done;
        NO_EXCEPTIONS(arg,TCL_ERROR);
        info_base* i = ckcast(getCpxBaseByName(tcl_, ev), info_base);
        if (!i){
            tcl_ <<  get_tcl_command() << ": No such information."<< tcl_error;
            return TCL_ERROR;
        }
        tcl_ <<  i->get_tcl_command() << result;
      	return TCL_OK;
	} else if (arg("object","Do I directly contain this object?")=="contains") {
		arg >> arg.tmp_chars >> done;
		NO_EXCEPTIONS(arg,TCL_ERROR);
		tcl_ << (directly_contains(arg.tmp_chars) != 0 ? true : false) << result;
			
		return tcl_;
	} else {
		/* We check if the current command _is_ actually the name of a contained
		   thing, and if so pass the command down in a nested way */
		const char* pk = arg.current_cmd();
		if(cpx_with_container* obj = directly_contains(pk)) {
			//tcl_args copy(arg, obj->cast_to_base());
			// to put the stream in the correct state
			arg == pk;
			int ret;
			// it must be either an object or an info
			// we can't use 'cast_to_base' because that
			// is const.
			if(info_base* i = obj->is_info()) {
				ret = i->parse_tcl_command(arg);
			} else {
				ret = obj->is_object()->parse_tcl_command(arg);
			}	 
			return ret;
		} else {
			// if we don't recognize the command, see if cpx_base does
			return cpx_base::parse_tcl_command(arg);
		}
	}
	
}

// This will give an error if there are no args at all
int cpx_with_info::parse_configuration_values(tcl_args& arg) {
	// We have this outside the loop's scope, so we can return the
	// value of the last one
	info_base *inf=0;
	do {
		if((inf = configuration_option(arg))) {
			// found one, so continue (the above line set the value too!)
			continue;
		} else {
			// no such configuration option
			return arg.no_match();
		}
	} while(arg.args_left());
	DONE(arg);
	// Return the last one as the result.  Be careful in case 'arg' was empty
	if(inf)
		tcl_ << (*inf) << result;
	return TCL_OK;
}
