# ###################################################################
#  EvoX - evolution in complex systems
# 
#  FILE: "cpxWindow.tcl"
#                                    created: 18/6/95 {12:57:47 am} 
#                                last update: 2/10/96 {8:37:38 pm} 
#  Author: Vince Darley
#  E-mail: <mailto:vince@das.harvard.edu>
#    mail: Division of Applied Sciences, Harvard University
#          Oxford Street, Cambridge MA 02138, USA
#  <http://www.fas.harvard.edu/~darley/>
#  
#  History
# 
#  modified by  rev reason
#  -------- --- --- -----------
#  18/6/95 VMD 1.0 original
#  _/_/_
# ###################################################################

##################################################################################
#																				 #
#	This file contains a definition	for	a general window class for EvoX.		 #
#	Currently window specifics necessary for system, simulation, graph,			 #
#	object,...	windows	are	just handled by	a switch statement.	Various			 #
#	methods	and	procedures are used	commonly.									 #
#																				 #
#	It might be	sensible to	use	the	same automatic subclassing used	for	the		 #
#	itcl evox object hierarchy.	 Note that the automatic subclass selection		 #
#	mechanism is already implemented in	evoxObject.tcl,	but	it currently only	 #
#	ever finds this	class (of course).											 #
#																				 #
##################################################################################

package require Tree
package require Cpptcl
package require Itcl
package require Itk

itcl::class cpx::Window {
    inherit itk::Toplevel
	
    constructor {args} {
		#eval itk::Toplevel::constructor $args
	} {
    	eval _cpx_config $args
    	import add cpx

	   	itk_component add mbar { 
    		frame $itk_interior.mbar -relief raised -bd 2
    	} {}
		pack $itk_component(mbar) -side top -fill x

		standardMenuBar

	   	itk_component add bottom { 
    		frame $itk_interior.bottom -relief groove -bd 2
    	}
		pack $itk_component(bottom) -side bottom -fill x
	}

    private method _cpx_config {option args} \
                  @itcl-builtin-configure	

    method initialise {} {
		menuBar
		createWindow
    }

	method makeMenuBar {} {
		#global ::tcl_platform
		
	}
	
	method standardMenuBar {} {
		set mb $itk_component(mbar)
		fileMenu $mb
		editMenu $mb
		helpMenu $mb

		pack $mb.file $mb.edit -side left
		pack $mb.help -side right		
	}	
	
	method postinitialise {} {
		wm title [window] [title]
		wm iconname [window] [title]
		wm minsize [window] 15 10
	}

    method window {} { return [info namespace tail $itk_component(hull)] }

	protected variable win_counter 0

    method unique_id { { prefix ".w" } }  { 
		return ${prefix}#auto
		#[unique_suffix $prefix]
	}

	method unique_suffix { { prefix ".w" } } {
		incr win_counter
		if [winfo exists ${prefix}${win_counter} ] {
			return unique_id
		}
		return ${win_counter}
	}

	method unique { prefix } {
		set a 1
		while { [::info commands "${prefix}${a}" ] != "" } {
			incr a
		}
		
		return ${prefix}${a}
	}

	method fileMenu { mb } {
		menubutton $mb.file -text File \
			-menu $mb.file.menu
		set m [menu $mb.file.menu]
		# file menu
		$m add command -label "New ..." \
			-command [code $this fileNew]
		$m add command -label "Open ..." \
			-command [code $this fileOpen]
		$m add command -label "Close window" \
			-command [code $this closeWindow]
	}

	method closeWindow {} {
		delete object $this
	}

	method editMenu { mb } {
		menubutton $mb.edit -text Edit \
			-menu $mb.edit.menu
		set m [menu $mb.edit.menu]

		# edit menu
		$m add command -label "Undo" \
			-command "editUndo"
		$m add command -label "Source ..." \
			-command "editSource"
		$m add command -label "Cut" \
			-command "editCut"
		$m add command -label "Copy" \
			-command "editCopy"
		$m add command -label "Paste" \
			-command "editPaste"
		$m add command -label "SelectAll" \
			-command "editSelectAll"
		$m add command -label "Clear" \
			-command "editClear"
	}

	method helpMenu { mb } {
		menubutton $mb.help -text Help \
			-menu $mb.help.menu
		set m [menu $mb.help.menu]
		$m add command -label "Cpptcl Class Browser" \
			-command "cpx::CpptclClassBrowser .#auto"
		$m add command -label "Itcl Class Browser" \
			-command "cpx::ItclClassBrowser .#auto"
		$m add command -label "Cpptcl HTML documentation" \
			-command "cpx::Window::gotoDocumentation"
		
	}
	
	proc gotoDocumentation {} {
		global ::tcl_platform
		switch $::tcl_platform(platform) {
		  "macintosh" {
		  }
		  "unix" {
		  	exec netscape "http://www.fas.harvard.edu/~darley/GeneratedDocs/"
		  }
		  "windows" {
		  }
		}
	}
	
	method print { args } {
		foreach v $args {
			upvar $v w
			puts "$v = $w"
		}
	}

    method title {} { pure_virtual title }
    method createWindow {} {pure_virtual createWindow }
	method menuBar {} { pure_virtual menuBar }

	method menuPretty { item } {
	    regsub -all {[A-Z0-9]} $item " &" s
		return [string toupper [string range $s 0 0]][string range $s 1 end]
	}
			
	proc createableDescendants { obj } {
		set l ""
		foreach o [cpptclControl listTypes "$obj"] {
			if { [cpptclControl commandFor "$o"] != "" } {
				lappend l $o
			}
			if [cpptclControl hasDescendants "$o"] {
				eval lappend l [createableDescendants "$o"]
			}
		}
		return $l
	}
	
	method pure_virtual { args } { error "shouldn't be in $itk_component(hull) : $args" }
	
    method zoomWindow {} {
		set w [wm screenwidth]
		set h [wm screenheight]
		wm configure $itk_component(hull) -geometry ${w}x${h}
    }

	method status { text } {
		# default is to print in the system window, can be overridden
		cpx::Status "$text"
	}
}


proc cpx::Status { text } {
    global cpx::StatusText
    set StatusText "$text"
}

