/*
 * TclDirCmd.C - tcl command to interface to directories: "Dir"
 *
 * -----------------------------------------------------------------------------
 * Copyright 1994 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  
 * Allan Brighton make no representations about the suitability of this 
 * software for any purpose. It is provided "as is" without express or 
 * implied warranty.
 * -----------------------------------------------------------------------------
 *
 */

#include "config.h"
#include <sys/types.h>
#include <pwd.h>
#include <dirent.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <string.h>
#include <iostream.h>
#include "TclDirCmd.h"

// use these standard tcl routines for cd and pwd 
extern "C" int Tcl_CdCmd(ClientData, Tcl_Interp*, int argc, char* argv[]);
extern "C" int Tcl_PwdCmd(ClientData, Tcl_Interp*, int argc, char* argv[]);

/* 
 * declare a table of tcl subcommands
 * format: name, min_args, max_args, method
 */
static class TclDirSubCmds {
public:
    char* name;      // method name
    int (TclDirCmd::*fptr)(int argc, char* argv[]); 
    int min_args;    // minimum number of args
    int max_args;    // maximum number of args
} subcmds_[] = { 
    {"ls",      &TclDirCmd::ls_,       1,  1},
    {"lsfiles", &TclDirCmd::lsfiles_,  1,  1},
    {"lsdirs",  &TclDirCmd::lsdirs_,   1,  1},
    {"cd",      &TclDirCmd::cd_,       1,  1},
    {"pwd",     &TclDirCmd::pwd_,      0,  0},
};



/*
 * Call the given method in this class with the given arguments
 */
int TclDirCmd::call(const char* name, int len, int argc, char* argv[])
{
    for(int i = 0; i < sizeof(subcmds_)/sizeof(*subcmds_); i++) {
	TclDirSubCmds* t = &subcmds_[i];
	if (strncmp(t->name, name, len) == 0) {
	    if (check_args(name, argc, t->min_args, t->max_args) != TCL_OK)
		return TCL_ERROR;
	    return (this->*t->fptr)(argc, argv);
	}
    }
    return TclCommand::call(name, len, argc, argv);
}


/*
 * A call to this function can be made from the tkAppInit file at startup
 * to install the Dir command
 */
extern "C"
int Dir_Init(Tcl_Interp* interp)  
{
    Tcl_CreateCommand(interp, "Dir", TclDirCmd::dirCmd, NULL, NULL);
    return TCL_OK;
}

/*
 * Implementation of the tcl extended command "Dir" -
 * tcl access to directories 
 * 
 * usage: see man page in dir.n in this directory
 */
int TclDirCmd::dirCmd(ClientData, Tcl_Interp* interp, int argc, char* argv[])
{
    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
			 argv[0], " name\"", NULL);
	return TCL_ERROR;
    }

    TclDirCmd* dircmd = new TclDirCmd(interp, argv[0], argv[1]);
    return dircmd->status();
}

/*
 * Constructor -
 * 
 * Create a "Dir" object in tcl for accessing the contents of directories.
 *
 * Note that the tcl command for this object is created in the
 * parent class constructor.
 */
TclDirCmd::TclDirCmd(Tcl_Interp* interp, const char* cmdname, const char* instname)
: TclCommand(interp, cmdname, instname)
{
}


/*
 * open and return a pointer to the given dir, or return null 
 * and report the error in tcl
 */
DIR* TclDirCmd::opendir_(const char* dirname)
{
    DIR* dir = opendir(dirname);
    if (dir == NULL) 
	Tcl_AppendResult(interp_, "couldn't open directory \"",
	    dirname, "\": ", Tcl_PosixError(interp_), NULL);
    return dir;
}

/*
 * Return true if dir/name is a directory
 */
int TclDirCmd::isdir_(const char* dir, const char* name)
{
    struct stat st;
    char path[1024];
    strcpy(path, dir);
    strcat(path, "/");
    strcat(path, name);
    return stat(path, &st) == 0 && S_ISDIR(st.st_mode);
}


/*
 * Return true if dir/name is a regular file
 */
int TclDirCmd::isfile_(const char* dir, const char* name)
{
    struct stat st;
    char path[1024];
    strcpy(path, dir);
    strcat(path, "/");
    strcat(path, name);
    return stat(path, &st) == 0 && S_ISREG(st.st_mode);
}


/*
 * Return a list of files and dirs in the given dir
 */
int TclDirCmd::ls_(int, char* argv[])
{
   DIR* dir = opendir_(argv[0]);
    if (dir == NULL) 
	return TCL_ERROR;

    for (struct dirent* d = readdir(dir); d; d = readdir(dir)) {
        if (strcmp(d->d_name, ".") != 0 && strcmp(d->d_name, "..") != 0)
	    Tcl_AppendElement(interp_, d->d_name);
    }
    closedir(dir);
    return TCL_OK;
}


/*
 * Return a list of files in the given dir
 */
int TclDirCmd::lsfiles_(int, char* argv[])
{
   DIR* dir = opendir_(argv[0]);
    if (dir == NULL) 
	return TCL_ERROR;

    for (struct dirent* d = readdir(dir); d; d = readdir(dir)) {
        if (strcmp(d->d_name, ".") != 0 && strcmp(d->d_name, "..") != 0 
	    && isfile_(argv[0], d->d_name))
	    Tcl_AppendElement(interp_, d->d_name);
    }
    closedir(dir);
    return TCL_OK;
}


/*
 * Return a list of directories in the given dir
 */
int TclDirCmd::lsdirs_(int, char* argv[])
{
    DIR* dir = opendir_(argv[0]);
    if (dir == NULL) 
	return TCL_ERROR;

    for (struct dirent* d = readdir(dir); d; d = readdir(dir)) {
        if (strcmp(d->d_name, ".") != 0 
	    && strcmp(d->d_name, "..") != 0 
	    && isdir_(argv[0], d->d_name))
	    Tcl_AppendElement(interp_, d->d_name);
    }
    closedir(dir);
    return TCL_OK;
}


/*
 * change dir to the given dir...
 */
int TclDirCmd::cd_(int, char* argv[])
{
    return Tcl_CdCmd(NULL, interp_, 2, --argv);
}


/*
 * return the current working directory...
 */
int TclDirCmd::pwd_(int, char* argv[])
{
    return Tcl_PwdCmd(NULL, interp_, 1, --argv);
}

