/*
 * E.S.O. - VLT project 
 * "@(#) $Id: error.C,v 1.1 1996/04/11 15:04:27 abrighto Exp $"
 *
 * error.C - error reporting routines
 * 
 * who             when      what
 * --------------  --------  ----------------------------------------
 * Allan Brighton  05/10/95  Created
 */

#include <stdlib.h>
#include <string.h>
#include <iostream.h>
#include <strstream.h>
#include <errno.h>
#include <stdio.h>
#include "error.h"

// static variable holding text of last error messages
static char errmsg_[1024];

// error code (see defines in <sys/errno.h> for values)
static int errno_ = 0;

// optional error handler, to be called with error message
static void (*errhandler_)(const char*) = NULL;


/*
 * global error reporting routine
 */
int error(const char* msg1, const char* msg2, int code)
{
    char buf[sizeof(errmsg_)];
    ostrstream os(buf, sizeof(buf));
    os << msg1 << msg2 << ends;
    
    if (errhandler_)
	(*errhandler_)(buf);

    errno_ = code;
    strcpy(errmsg_, buf);
    return ERROR;
}


/*
 * report the error, including system error code
 */
int sys_error(const char* msg1, const char* msg2)
{
    extern int sys_nerr;
    extern char *sys_errlist[];
    extern int errno;

    if (errno < 0 || errno >= sys_nerr)
	return error(msg1, msg2);

    char buf[sizeof(errmsg_)];
    ostrstream os(buf, sizeof(buf));
    os << msg1 << msg2 << ": " << sys_errlist[errno] << ends;

    if (errhandler_)
	(*errhandler_)(buf);

    errno_ = errno;
    strcpy(errmsg_, buf);
    return ERROR;
}


/* 
 * return the text of the previous error message
 */
char* last_error() 
{
    return errmsg_;
}


/* 
 * return the error code for the previous error
 */
int last_error_code() 
{
    return errno_;
}


/* 
 * reset the last_error buf to empty 
 */
void clear_error()
{
    errmsg_[0] = '\0';
}


/*
 * set a routine to be called with the text of error messages
 * when they occur. The argument is a pointer to an error 
 * handler:
 *
 *       void errhandler(const char* msg);
 *
 * The return value is a pointer to the previous error handler, or NULL,
 * if none was defined.
 */
void (*set_error_handler(void (*errhandler)(const char*)))(const char*)
{
    void (*old_handler)(const char*) = errhandler_;
    errhandler_ = errhandler;
    return old_handler;
}


/*  
 * print the given message on stderr (may be used as an error
 * handler)
 */
void print_error(const char* msg)
{
    cerr << msg << endl;
}
