# Tree.tcl - itcl widget for displaying trees
#
# This widget is based on the Tk tree widget, which is implemented 
# in C++.  It simplifies things by creating its own frame, canvas 
# and scrollbars and defines a standard layout, where each node may
# have a label, a bitmap or both.
# 
# See the man page itclTree(n) for a description.
#
# Author: Allan Brighton (abrighto@eso.org) 

itk::usual Tree {}

class util::Tree {
    inherit FrameWidget

    #  create a new Tree widget

    constructor {args} {
	FrameWidget::constructor
	
	# create Canvas object
	itk_component add canvas {
	    CanvasWidget $w_.c -relief sunken -borderwidth 3
	} {
	}
	pack $itk_component(canvas) -fill both -expand 1
	
	# get name of Tk canvas window (and the bg color)
	set canvas_ [$itk_component(canvas) component canvas]
	set canvas_bg_ [$canvas_ cget -background]

	# create the tree widget
	itk_component add tree {
	    tree $canvas_.t
	} {
	    keep -borderwidth -parentdistance -layout
	}
	set tree_ $itk_component(tree)
	set_default_bindings
	eval itk_initialize $args
    }

    
    # Set the bindings for the tree canvas
    
    method set_default_bindings {} {
	$canvas_ bind text <1> "focus %W; [code $this select_node]"
	$canvas_ bind bitmap <1> [code $this select_bitmap]
	$canvas_ bind text <Shift-1> "focus %W; [code $this select_node]"
	$canvas_ bind bitmap <Shift-1> [code $this select_bitmap]
    }
    

    # center the tree in the scrolling area
    # (Note: if the scrolling area is smaller than the canvas, how do you
    # center the tree in it ?)

    method center {} {
	if {[$tree_ cget -layout] == "horizontal"} {
	    $canvas_ yview moveto 0.5
	    $canvas_ xview moveto 0
	} else {
	    $canvas_ xview moveto 0.5
	    $canvas_ yview moveto 0
	}
    }
  
    # layout the components of the given node depending on whether
    # the tree is vertical or horizontal
    
    method layout_node {node} {
	set label $node:text
	set bitmap $node:bitmap
	
	if {[$tree_ cget -layout] != "vertical"} {
	    if {[scan [$canvas_ bbox $label] "%d %d %d %d" x1 y1 x2 y2] == 4} {
		$canvas_ itemconfig $bitmap -anchor se
		$canvas_ coords $bitmap $x1 $y2
	    }
	} else {
	    if {[scan [$canvas_ bbox $bitmap] "%d %d %d %d" x1 y1 x2 y2] == 4} {
		$canvas_ itemconfig $label -anchor n
		$canvas_ coords $label [expr "$x1+($x2-$x1)/2"] $y2
	    }
	}
    }
    
    
    # add the given node to the tree under the given parent node,
    # if not already there
    #
    # optional args: -label     - text for node label
    #                -text      - alias for -label
    #                -bitmap    - bitmap to use for nodes
    #
    
    method add_node {parent node args} {
	
	set text {}
	set label {}
	set bitmap {}
	getArgs $args

	if {"$text" != ""} {
	    set label $text
	}

	if {"$label" == "" && "$bitmap" == ""} {
	    set label $node
	}

	# don't add the node if its already there
	if [llength [$canvas_ find withtag $node]] {
	    return
	}
	
	if {"$bitmap" != ""} {
	    $canvas_ create bitmap 0 0 -bitmap $bitmap \
		    -tags [list $node bitmap $node:bitmap] \
		    -foreground $itk_option(-bitmapcolor) \
		    -background $canvas_bg_
	}
	
	if {"$label" != ""} {
	    $canvas_ create text 0 0 \
		    -text "$label" \
		    -font $itk_option(-labelfont) \
		    -tags [list $node text $node:text]
	}

	set line [$canvas_ create line 0 0 0 0 \
		-tag [list line $parent:$node:line] \
		-width 1 \
		-capstyle round \
		-fill $itk_option(-linecolor)]
	
	layout_node $node
	$tree_ addlink $parent $node $line -border 2
    }
    
    
    # select the current node's label
    
    method select_node {} {
	set tag [lindex [$canvas_ gettags current] 0]
	$canvas_ select from current 0
	$canvas_ select to current \
		[string length [$canvas_ itemcget current -text]]
	deSelect_bitmap
	set selected_node_ $tag
    }
    
    
    # de-select all node labels
    
    method deselect_node {} {
	$canvas_ select clear
	set selected_node_ {}
    }
    

    # highlight the node's bitmap
    
    method select_bitmap {} {
	set tag [lindex [$canvas_ gettags current] 0]
	deselect_node
	deSelect_bitmap
	# save bitmaps bg color
	$canvas_ itemconf current \
		-background $itk_option(-bitmapselectbackground) \
		-tags "[$canvas_ gettags current] selected" 
	set selected_node_ $tag
    }


    # stop highlighting the node's bitmap
    
    method deSelect_bitmap {} {
	$canvas_ itemconfig selected -background $canvas_bg_
	$canvas_ dtag selected
	set selected_node_ {}
    }
    

    # return the tag of the item currently selected
    
    method get_selected {} {
	return $selected_node_
    }


    # return the tag of the current item (the item under the mouse)
    
    method get_current {} {
	return [lindex [$canvas_ gettags current] 0]
    }


    # Toggle the layout of the tree between vertical and horizontal

    method toggle_layout {} {
	if {[$tree_ cget -layout] == "horizontal"} {
	    $tree_ config -layout vertical
	} else {
	    $tree_ config -layout horizontal
	}
	
	# change the layout of the nodes so that the bitmap is on top for
	# vertical trees and at left for horizontal trees
	foreach i [$canvas_ find withtag text] {
	    set node [lindex [$canvas_ gettags $i] 0]
	    layout_node $node
	    $tree_ nodeconfig $node
	}
	
	$tree_ draw
	center
    }


    # pass these methods on to the tree widget unchanged
    # (this just generates methods on the fly...)

    ::foreach i {addlink ancestors child draw isleaf isroot movelink \
	       nodeconfigure nodeconfig prune parent root rmlink \
	       sibling subnodes} {
	method $i {args} [format {return [eval "$tree_ %s $args"]} $i]
    } 


    # -- options --
    
    # default line color
    itk_option define -linecolor lineColor LineColor gray50
 
    # bitmap color
    itk_option define -bitmapselectbackground bitmapSelectBackground BitmapSelectBackground lightblue
    itk_option define -bitmapcolor bitmapColor BitmapColor gray50

    # font to use for tree labels
    itk_option define -labelfont labelFont LabelFont {-adobe-helvetica-bold-r-*-*-*-120-*-*-*-*-*-*}


    # -- protected vars --
    
    # tree widget
    protected variable tree_

    # tk canvas widget
    protected variable canvas_

    # background color of canvas
    protected variable canvas_bg_
    
    # currently selected node
    protected variable selected_node_ {}

}


