// -*-c++-*-
#ifndef _TkTree_h_
#define _TkTree_h_

/*
 * TkTree.h - class definitions for drawing trees in Tk/Tcl
 * 
 * -----------------------------------------------------------------------------
 * Copyright 1994 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  
 * Allan Brighton make no representations about the suitability of this 
 * software for any purpose. It is provided "as is" without express or 
 * implied warranty.
 * -----------------------------------------------------------------------------
 */

#include "config.h"
#include "TkWidget.h"
#include <string.h>
#include <stdlib.h>

/* 
 * tree options (used for tree configuration)
 */
class TkTreeOptions : public TkWidgetOptions {
public:
    int parentDistance;		// distance between nodes and parent
    char* layout;		// tree layout: "horizontal" or "vertical"
    int borderWidth;		// width of border around entire tree
    TkTreeOptions(int p, char* l, int b)
	:parentDistance(p), layout(strdup(l)), borderWidth(b) {}
};


// forward reference to class used to represent tree nodes
class TkTreeNode;


/*
 * Class TkTree
 * 
 * This class implements the extended Tk command "tree" for displaying
 * dynamic trees in a canvas.
 */
class TkTree: public TkWidget {
private:
    char* canvas_;			// parent of tree should be a canvas
    Tk_Window canvasWin_;		// internal Tk window of canvas
    TkTreeNode* rootNode_;		// root of the layout tree
    TkTreeOptions options_;             // holds tree config options
    int center_flag_;                   // true if tree should be centered

    // -- private member functions  --
     
    // find the named node and return a pointer to it or 0 if not found
    TkTreeNode* findNode(const char* tag);
    
    // reconfigure the node (size, options, ...)
    int setupNode(TkTreeNode*, int argc, char* argv[]);

    // copy constructor: not defined
    TkTree(const TkTree&);
    

protected:
    // redefined from parent class
    virtual int configureWidget(int argc, char* argv[], int flags);

    // redraw the widget
    virtual void redraw();

    // center the tree in the canvas
    virtual void center();


public:
    // initialize the tree with the command line args
    TkTree(Tcl_Interp*, int argc, char* argv[]);
    
    // destructor - clean up tree nodes when widget cmd is destroyed
    ~TkTree();
    
    // entry point from tcl to create a tree
    static int treeCmd(ClientData, Tcl_Interp *interp, int argc, char* argv[]);

    static void eventProc(ClientData clientData, XEvent *eventPtr);

    // call a member function by name
    virtual int call(const char* name, int len, int argc, char* argv[]);


    // -- tree subcommand methods --
    
    // add a new child node to the parent node in the tree
    int addlink(int argc, char* argv[]);
    
    // move the subtree to a new position in the tree 
    int movelink(int argc, char* argv[]);
    
    // remove a node and its subtree 
    int rmlink(int argc, char* argv[]);
    
    // recalculate the size and pos and set options
    int nodeconfigure(int argc, char* argv[]);
    
    // command to remove and delete the nodes children
    int prune(int argc, char* argv[]);
    
    // command that returns 1 if the named node is a leaf node (has no children)
    int isleaf(int argc, char* argv[]);
    
    // command that returns 1 if the named node is a root node (displays no parents)
    int isroot(int argc, char* argv[]);
    
    // command that sets a new root node
    int root(int argc, char* argv[]);
    
    // command to draw the tree on the canvas
    int draw(int argc, char* argv[]);
    
    // command to return the bounding box of the tree
    int bbox(int argc, char* argv[]);
    
    // return the name of the tree's canvas
    int getcanvas(int argc, char* argv[]);

    // command that returns the name of the child node
    int child(int argc, char* argv[]);
    
    // command that returns a list of the child nodes
    int subnodes(int argc, char* argv[]);
    
    // command that returns the name of the sibling node
    int sibling(int argc, char* argv[]);
    
    // command that returns the name of the parent node
    int parent(int argc, char* argv[]);

    // command that returns a list of ancestors of a tree node
    int ancestors(int argc, char* argv[]);

    // read-only member access
    const char* canvas() const {return canvas_;}

    int parentDistance() const { return options_.parentDistance;}
    int borderWidth() const { return options_.borderWidth;}
    
    // boolean functions to determine the layout quickly
    int horizontal() const {return (*options_.layout != 'v');}
    int vertical() const {return (*options_.layout == 'v');}

    // called with the bounding box of the tree before drawing
    void setDrawArea(int x1, int y1, int x2, int y2);
};


#endif /* _TkTree_h_ */
